package org.lsst.ccs.subsystem.focalplane;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.drivers.reb.REBException;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.imagenaming.service.ImageNameService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystem.focalplane.data.FocalPlaneDataGroup;
import org.lsst.ccs.subsystem.focalplane.states.FocalPlaneState;
import org.lsst.ccs.subsystem.rafts.GlobalProc;
import org.lsst.ccs.subsystem.rafts.ImageProc;
import org.lsst.ccs.subsystem.rafts.REBDevice;
import org.lsst.ccs.subsystem.rafts.RaftsCommands;
import org.lsst.ccs.subsystem.rafts.data.RaftException;
import org.lsst.ccs.subsystem.rafts.states.RebDeviceState;
import org.lsst.ccs.subsystem.rafts.states.RebValidationState;
import org.lsst.ccs.utilities.ccd.FocalPlane;
import org.lsst.ccs.utilities.ccd.Raft;

public class FocalPlaneSubsystem extends Subsystem implements HasLifecycle {

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private final Map<String, REBDevice> rebDevices = new HashMap();

    private Sequencers sequencers;
    private final Map<String, RaftsCommands> raftsCommands = new HashMap<>();
    private final FocalPlane focalPlaneGeometry;
    private ImageHandling imageHandling;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private GlobalProc globalProc;
        
    @LookupField(strategy = LookupField.Strategy.TREE)
    DataProviderDictionaryService dictionaryService;
    
    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private FocalPlaneConfig focalPlaneConfig;

    @LookupField(strategy = LookupField.Strategy.DESCENDANTS)
    private ImageNameService imageNameService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentStateService agentStateService;
    
    public FocalPlaneSubsystem(FocalPlane geometry) {
        super("fp", AgentInfo.AgentType.WORKER);
        this.focalPlaneGeometry = geometry;
    }

    @Override
    public void build() {
        setAgentProperty("org.lsst.ccs.use.full.paths", "true");
        setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.FOCAL_PLANE.name());
    }

    
    @Override
    public void postInit() {

        sequencers = new Sequencers(this, focalPlaneConfig);
        imageHandling = new ImageHandling(this, focalPlaneConfig);
        
        updateAgentState(FocalPlaneState.NEEDS_CLEAR);

        for (Raft raft : focalPlaneGeometry.getChildrenList()) {
            String raftName = raft.getName();
            // TODO: Shouldn't the raft know when REB devices belong to it?
            List<REBDevice> raftRebDevices = new ArrayList();
            rebDevices.forEach((rebName, device) -> {
                if (rebName.startsWith(raftName)) {
                    raftRebDevices.add(device);
                    sequencers.add(device.getSequencer());
                    final ImageProc imageProc = device.getImageProc();
                    imageProc.setPatternProperty("raftName", raftName);
                    imageHandling.addImageProc(imageProc);
                }
            });

            RaftsCommands commands = new RaftsCommands(this, raftRebDevices, globalProc, raft, null);
            raftsCommands.put(raftName, commands);

            // Add the RaftCommands to the CommandSet
            // TODO: Decide if these commands are actually useful at the focal plane level.
            addCommandsFromObject(commands, raftName);
        }

        agentStateService.addStateChangeListener((Object device, Enum newState, Enum oldState) -> {
            if (newState == RebValidationState.VALID) {
                sequencers.validate(((REBDevice) device).getSequencer());
            }
        }, RebDeviceState.class, RebValidationState.class);

        addCommandsFromObject(new LSE71Commands(this), "");
        addCommandsFromObject(new ScriptingCommands(this),"");
        
        //globalProc.initialize(rebDevices, clientFactory, sLog);
    }
    
    @Override
    public void preStart() {        
        //Manipulate the dictionary here: 
        for ( DataProviderInfo data: dictionaryService.getDataProviderDictionary().getDataProviderInfos() ) {
            FocalPlaneDataGroup dataGroup = FocalPlaneDataGroup.findFocalPlanDataGroup(data);
            if ( dataGroup != null ) {
                dataGroup.addAttributesToDataInfo(data);
            }
        }
    }
        
    

    @Override
    public void postStart() {
        try {
            sequencers.load();
        } catch (RaftException | REBException x) {
            throw new RuntimeException("Error configuring sequencers", x);
        }
    }

    public Sequencers getSequencers() {
        return sequencers;
    }

    GlobalProc getGlobalProc() {
        return globalProc;
    }

    ImageNameService getImageNameService() {
        return imageNameService;
    }

    ImageHandling getImageHandling() {
        return imageHandling;
    }
}
