package org.lsst.ccs.subsystem.focalplane.ui.view;

import java.awt.BorderLayout;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.*;
import javax.swing.Box;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTable;
import javax.swing.table.AbstractTableModel;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.base.filter.AgentChannelsFilter;
import org.lsst.ccs.gconsole.base.filter.PersistableAgentChannelsFilter;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView3;
import org.lsst.ccs.gconsole.plugins.monitor.DisplayChannel;
import org.lsst.ccs.gconsole.plugins.monitor.FormattedValue;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorCell;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorFormat;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorTableCellRenderer;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.PersistableMonitorView;
import org.lsst.ccs.gconsole.plugins.monitor.Updatable;
import org.lsst.ccs.gconsole.util.ThreadUtil;
import org.lsst.ccs.subsystem.focalplane.ui.FocalPlanePlugin;
import org.lsst.ccs.subsystem.focalplane.ui.Segment;
import static org.lsst.ccs.subsystem.focalplane.ui.Segment.*;

/**
 * {@link MonitorView} that displays focal plane schematics as a hierarchy of tables.. 
 *
 * @author onoprien
 */
public class FocalPlaneDiagramView extends AbstractMonitorView3 implements PersistableMonitorView {

// -- Fields : -----------------------------------------------------------------
    
    static private final MonitorField DEFAULT_FIELD = MonitorField.AVERAGE_VALUE;
    static private final MonitorField[] AVAILABLE_FIELDS = {MonitorField.AVERAGE_VALUE, MonitorField.MAX_VALUE, MonitorField.MIN_VALUE}; // FIXME: same array in other classes
    
    private final Descriptor descriptor = new Descriptor();
    
    private MonitorField field = MonitorField.AVERAGE_VALUE;
    private ArrayList<MonitorField> filterFields;
    private ArrayList<String> groups;
    private final EnumSet<Segment> hasSegment = EnumSet.noneOf(Segment.class);
    
    private volatile PersistableAgentChannelsFilter filter;
    
    private JScrollPane scrollPane;
    private JPanel rootPanel;
    private JButton backButton;
    private JComboBox<String> groupCombo;
    private JComboBox<MonitorField> fieldCombo;
    private Table table;
    
    private boolean armed = true;
    private Segment level = RAFT;
    
// -- Life cycle : -------------------------------------------------------------
    
//    @Create(category = FocalPlanePlugin.VIEW_CATEGORY,
//            name = "Focal Plane Diagram",
//            path = "Diagram/",
//            description = "Focal plane hierarchical diagram.")
    public FocalPlaneDiagramView() {
    }

    @Override
    public JComponent getPanel() {
        if (scrollPane == null) {
            
            rootPanel = new JPanel(new BorderLayout());
            scrollPane = new JScrollPane(rootPanel);
            
            Box top = Box.createHorizontalBox();
            rootPanel.add(top, BorderLayout.NORTH);
            top.add(Box.createRigidArea(Const.HDIM));
            backButton = new JButton("Back");
            top.add(backButton);
            backButton.addActionListener(e -> System.out.println("FIXME"));
            top.add(Box.createRigidArea(Const.HDIM));
            top.add(new JLabel("Display channel: "));
            groupCombo = new JComboBox<>();
            groupCombo.setPrototypeDisplayValue("123456789012345678901234567890");
            groupCombo.addActionListener(e -> {
                int i = groupCombo.getSelectedIndex();
                if (armed || i >= 0) {
                    descriptor.setGroup(groups.get(i));
                    if (filterFields != null) {
                        field = filterFields.get(i);
                        armed = false;
                        fieldCombo.setSelectedItem(field);
                        armed = true;
                    }
                    replaceModel();
                }
            });
            top.add(groupCombo);
            top.add(Box.createRigidArea(Const.HDIM));
            top.add(new JLabel("Display field: "));
            fieldCombo = new JComboBox<>(AVAILABLE_FIELDS);
            fieldCombo.setToolTipText("What to display for this channel");
            fieldCombo.addActionListener(e -> {
                if (armed) {
                    field = (MonitorField) fieldCombo.getSelectedItem();
                    filterFields = null;
                    descriptor.setField(field.getTitle());
                    table.getModel().clear();
                }
            });
            top.add(fieldCombo);
            top.add(Box.createRigidArea(Const.HDIM));
            top.add(Box.createHorizontalGlue());
            
            table = new Table();
            rootPanel.add(table, BorderLayout.CENTER);
            
            resetChannels();
        }
        return scrollPane;
    }

    @Override
    public PersistableAgentChannelsFilter getFilter() {
        return filter;
    }

    @Override
    public void setFilter(AgentChannelsFilter filter) {
        if (filter instanceof PersistableAgentChannelsFilter) {
            PersistableAgentChannelsFilter f = (PersistableAgentChannelsFilter) filter;
            if (FocalPlanePlugin.FILTER_CATEGORY.equals(f.getCategory())) {
                this.filter = f;
                return;
            }
        }
        throw new IllegalArgumentException("FocalPlaneDiagramView only accepts filters from FocalPlaneFilter category.");
    }

    @Override
    public void install() {
        ThreadUtil.invokeLater(this::start);
        super.install();
    }

    @Override
    public void uninstall() {
        super.uninstall();
        ThreadUtil.invokeLater(this::stop);
    }
    
    private void start() {
        if (filter == null) throw new IllegalStateException("Trying to initialize SummaryTableView without a filter.");
    }
    
    private void stop() {
        if (table != null) {
            table.getModel().destroy();
            table = null;
        }
    }
    
    
// -- Updates : ----------------------------------------------------------------

    @Override
    protected void resetChannels() {
        armed = false;
        
        groups = getGroups();
        groupCombo.setModel(new DefaultComboBoxModel<>(groups.toArray(new String[0])));
        if (groups.isEmpty()) {
            groupCombo.setEnabled(false);
        } else {
            groupCombo.setEnabled(true);
            groupCombo.setSelectedItem(descriptor.getGroup());
            if (groupCombo.getSelectedIndex() == -1) {
                groupCombo.setSelectedIndex(0);
                descriptor.setGroup(groups.get(0));
            }
        }
        
        String selectedField = descriptor.getField();
        if (selectedField == null) {
            filterFields = getGroupFields();
            if (filterFields == null || filterFields.size() != groups.size()) {
                filterFields = null;
                field = DEFAULT_FIELD;
            }
        } else {
            field = stringToField(selectedField);
        }
        fieldCombo.setSelectedItem(field);
        
        hasSegment.clear();
        for (String path : data.keySet()) {
            int[] indices = Segment.getIndices(path);
            if (indices[0] != -1) hasSegment.add(RAFT);
            if (indices[2] != -1 && indices[3] == -1) hasSegment.add(REB);
            if (indices[3] != -1) hasSegment.add(CCD);
            if (indices[4] != -1) hasSegment.add(AMP);
        }
        
        replaceModel();
        armed = true;
    }

    @Override
    protected void update() {
        if (table != null) {
            table.getModel().fireTableDataChanged();
        }
    }
    
    private void replaceModel() {
        table.getModel().destroy();
        Model model = new Model();
        table.setModel(model);
    }

    
// -- Local classes : ----------------------------------------------------------
    
    /**
     *
     */
    private final class Table extends JTable {
        
        Table() {
            super(new Model(true));
            setDefaultRenderer(FormattedValue.class, new MonitorTableCellRenderer());
            setRowSelectionAllowed(true);
            setColumnSelectionAllowed(false);
            setShowGrid(true);
            addMouseListener(new MouseAdapter() {
                @Override
                public void mouseClicked(MouseEvent e) {
                    mouseClick(e);
                }
            });
        }

        @Override
        public Model getModel() {
            return (Model) super.getModel();
        }
        
        private void mouseClick(MouseEvent e) {
            int nClick = e.getClickCount();
            Point point = e.getPoint();
            Table table = (Table) e.getSource();
            int row = table.rowAtPoint(point);
            int column = table.columnAtPoint(point);
            if (nClick == 2) {
                int[] indices = decodeSegment(descriptor.getSegment());
                switch (level) {
                    case RAFT:
                        indices[0] = row;
                        indices[1] = column;
                        break;
                    case REB:
                        indices[2] = row;
                        break;
                    case CCD:
                        indices[2] = row;
                        indices[3] = column;
                        break;
                    default:
                        return;
                }
                descriptor.setSegment(encodeSegment(indices));
                replaceModel();
            }
        }
    }
    
    private class Model extends AbstractTableModel {
        
        int nRows, nColumns; 
        Cell[][] cells;
        
        Model(boolean empty) {
            nRows = 0; nColumns = 0; cells = new Cell[0][0];
        }
        
        Model() {
            
            int[] indices = decodeSegment(descriptor.getSegment());
            if (indices[0] == -1) {
                nRows = 3;
                nColumns = 3;
                level = RAFT;
            } else if (indices[2] == -1 && hasSegment.contains(REB)) {
                nRows = 3;
                nColumns = 1;
                level = REB;
            } else  if (indices[2] == -1 || indices[3] == -1) {
                nRows = 3;
                nColumns = 3;
                level = CCD;
                indices[2] = -1;
            } else {
                nRows = 2;
                nColumns = 8;
                level = AMP;
            }
            String prefix = Segment.getPathPrefix(indices);
            cells = new Cell[nRows][nColumns];
            for (int row = 0; row < nRows; row++) {
                for (int col = 0; col < nColumns; col++) {
                    cells[row][col] = new Cell();
                }
            }
            
            for (Map.Entry<String,DisplayChannel> e : data.entrySet()) {
                String path = e.getKey();
                if (path.startsWith(prefix)) {
                    int[] ii = Segment.getIndices(path);
                    if (ii != null) {
                        try {
                            Cell cell = null;
                            switch (level) {
                                case RAFT:
                                    cell = cells[ii[0]][ii[1]]; break;
                                case REB:
                                    cell = cells[ii[2]][0]; break;
                                case CCD:
                                    cell = cells[ii[2]][ii[3]]; break;
                                case AMP:
                                    cell = cells[ii[4]][ii[5]]; break;
                            }
                            DisplayChannel dc = e.getValue();
                            cell.getChannels().add(dc);
                            dc.setTarget(cell);
                        } catch (ArrayIndexOutOfBoundsException x) {
                        }
                    }
                }
            }
            
            for (int row = 0; row < nRows; row++) {
                for (int col = 0; col < nColumns; col++) {
                    cells[row][col].getChannels().trimToSize();
                }
            }
        }

        @Override
        public int getRowCount() {
            return nRows;
        }

        @Override
        public int getColumnCount() {
            return nColumns;
        }

        @Override
        public Object getValueAt(int rowIndex, int columnIndex) {
            MonitorCell cell = cells[rowIndex][columnIndex];
            FormattedValue fv = cell.getFormattedValue();
            if (fv == null) {
                MonitorFormat.DEFAULT.format(cell);
                fv = cell.getFormattedValue();
            }
            return fv;
        }

        @Override
        public Class<?> getColumnClass(int columnIndex) {
            return FormattedValue.class;
        }
        
        void destroy() {
            for (int row = 0; row < nRows; row++) {
                for (int col = 0; col < nColumns; col++) {
                    Cell cell = cells[row][col];
                    for (DisplayChannel dc : cell.getChannels()) {
                        dc.setTarget(null);
                    }
                }
            }
        }
        
        void clear() {
            for (int row = 0; row < nRows; row++) {
                for (int col = 0; col < nColumns; col++) {
                    cells[row][col].setFormattedValue(null);
                }
            }
            fireTableDataChanged();
        }
        
    }
    
    private class Cell implements MonitorCell, Updatable {
        
        private final ArrayList<DisplayChannel> channels;
        private FormattedValue formattedValue;
        
        Cell(ArrayList<DisplayChannel> channels) {
            this.channels = channels;
        }
        
        Cell() {
            this.channels = new ArrayList<>();
        }

        @Override
        public ArrayList<DisplayChannel> getChannels() {
            return channels;
        }

        @Override
        public MonitorField getField() {
            return field;
        }

        @Override
        public FormattedValue getFormattedValue() {
            return formattedValue;
        }

        @Override
        public void setFormattedValue(FormattedValue formattedValue) {
            this.formattedValue = formattedValue;
        }

        @Override
        public void update(DisplayChannel channelHandle) {
            formattedValue = null;
        }
        
    }
    
    
// -- Saving/restoring : -------------------------------------------------------
    
    static public class Descriptor extends PersistableMonitorView.Descriptor {

        private String group;
        private String field;
        private Integer segment;

        public String getGroup() {
            return group;
        }

        public void setGroup(String group) {
            this.group = group;
        }

        public String getField() {
            return field;
        }

        public void setField(String field) {
            this.field = field;
        }

        public Integer getSegment() {
            return segment;
        }

        public void setSegment(Integer segment) {
            this.segment = segment;
        }
        
    }

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }
    
    private Integer encodeSegment(int[] indices) {
        int out = 0;
        int shift = 0;
        for (int i=0; i<4; i++) {
            int k = indices[i];
            if (k == -1) k = 9;
            out += k * 10*(shift++);
        }
        return out == 9999 ? null : out;
    }
    
    private int[] decodeSegment(Integer code) {
        int[] out = {-1, -1, -1, -1, -1, -1};
        if (code == null) return out;
        for (int i=0; i<4; i++) {
            int k = code % 10;
            if (k != 9) out[i] = k;
            code /= 10;
        }
        return out;
    }

}
