package org.lsst.ccs.subsystem.focalplane.ui.fpmap;

import org.lsst.ccs.subsystem.focalplane.ui.filter.TemplateFilter;
import java.awt.BorderLayout;
import java.awt.Color;
import java.io.Serializable;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import org.lsst.ccs.gconsole.base.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.services.persist.PersistenceService;
import org.lsst.ccs.gconsole.services.persist.Savable;
import org.lsst.ccs.subsystem.focalplane.data.FocalPlaneDataGroup;
import org.lsst.ccs.subsystem.focalplane.ui.filter.DataGroupFilter;

/**
 * Graphic component that contains {@link FocalPlaneMap} and associated controls.
 *
 * @author onoprien
 */
public final class FocalPlaneMapRegion extends JPanel implements Savable {

// -- Fields : -----------------------------------------------------------------
    
    static private final String CUSTOM = "Custom...";
    static private final String LOAD = "Load...";
    
    static private final String[] modelFactories;
    static {
        FocalPlaneDataGroup[] groups = FocalPlaneDataGroup.values();
        int n = groups.length;
        modelFactories = new String[n+3];
        modelFactories[0] = "";
        modelFactories[1] = CUSTOM;
        modelFactories[2] = LOAD;
        for (int i=0; i<n; i++) {
            modelFactories[i+3] = groups[i].getName();
        }
    }
    
    private final Descriptor descriptor = new Descriptor();
    
    private final FocalPlaneMap map;
    private final JLabel titleLabel;
    private final JComboBox<String> comboBox;
    
    private boolean restoring = false;

// -- Life cycle : -------------------------------------------------------------
    
    public FocalPlaneMapRegion(Descriptor desc) {
        super(new BorderLayout());
        
        setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createEmptyBorder(1, 1, 1, 1), 
                BorderFactory.createLineBorder(Color.BLACK, 1, true)));
        
        map = new FocalPlaneMap();
        add(map, BorderLayout.CENTER);
        
        Box header = Box.createHorizontalBox();
        add(header, BorderLayout.NORTH);
        
        comboBox = new JComboBox<>(modelFactories);
        header.add(comboBox);
        comboBox.addActionListener(e -> {
            String f = (String) comboBox.getSelectedItem();
            if (!restoring) setCurrentFactory(f);
        });
        
        header.add(Box.createRigidArea(Const.HDIM));
        titleLabel = new JLabel();
        header.add(titleLabel);
        
        header.add(Box.createHorizontalGlue());
        
        restore(desc);
    }
    
    public void opened() {
        
    }
    
    public void closed() {
        map.setModel(null);
    }
    
    
// -- Pop-up menu : ------------------------------------------------------------
        
    public JPopupMenu modifyPopupMenu(JPopupMenu menu) {
        if (descriptor.getModel() instanceof TemplateFilter.Descriptor && CUSTOM.equals(descriptor.getSelection())) {
            JMenuItem it = new JMenuItem("Save region...");
            it.addActionListener(e -> {
                try {
                    TemplateFilter.Descriptor f = (TemplateFilter.Descriptor) descriptor.getModel();
                    f.setCategory(TemplateFilter.CATEGORY);
                    PersistenceService serv = Console.getConsole().getSingleton(PersistenceService.class);
                    serv.saveAs(f, "Save focal plane map settings", this);
                } catch (Exception x) {
                }
            });
            menu.insert(it, 0);
        }
        return menu;
    }
    

// -- Local methods : ----------------------------------------------------------
    
    private void setTitle() {
        String title = map.getName();
        if (title == null || comboBox.getSelectedIndex() > 2) {
            title = "";
        }
        titleLabel.setText(title.isEmpty() ? "" : "<html><b>"+ title);
    }
    
    private void setCurrentFactory(String factory) {
        try {
            AbstractChannelsFilter filter = null;
            TemplateFilter.Descriptor d = null;
            if (factory == null || factory.isEmpty()) {
                descriptor.setSelection(null);
                descriptor.setModel(null);
                map.setModel(null);
                return;
            } else if (CUSTOM.equals(factory)) {
                d = descriptor.getModel() instanceof TemplateFilter.Descriptor ? (TemplateFilter.Descriptor) descriptor.getModel() : null;
                d = ChannelSelectorDialog.show(d, this);
                if (d != null) filter = new TemplateFilter(d);
            } else if (LOAD.equals(factory)) {
                PersistenceService serv = Console.getConsole().getSingleton(PersistenceService.class);
                d = (TemplateFilter.Descriptor) serv.load(TemplateFilter.CATEGORY, "Select saved filter", this);
                if (d != null) filter = new TemplateFilter(d);
            } else {
                FocalPlaneDataGroup g = FocalPlaneDataGroup.forName(factory);
                filter = new DataGroupFilter(descriptor.getAgent(), g);
            }
            if (filter == null) {
                throw new RuntimeException();
            } else {
                MonitorViewModel model = new MonitorViewModel();
                model.setFilter(filter);
                map.setModel(model.getFocalPlaneMapModel());
                descriptor.setSelection(factory);
                descriptor.setModel(d);
            }
        } catch (Exception x) {
            restoring = true;
            comboBox.setSelectedItem(descriptor.getSelection());
            restoring = false;
        }
        setTitle();
    }

    
// -- Saving : -----------------------------------------------------------------

    @Override
    public Descriptor save() {
        return descriptor;
    }

    @Override
    public void restore(Serializable bean) {
        if (bean == null) bean = new Descriptor();
        try {
            Descriptor desc = (Descriptor) bean;
            descriptor.setAgent(desc.getAgent());
            restoring = true;
            comboBox.setSelectedIndex(0);
            comboBox.setSelectedItem(desc.getSelection());
            AbstractChannelsFilter filter = null;
            if (comboBox.getSelectedIndex() != 0) {
                String select = (String) comboBox.getSelectedItem();
                descriptor.setSelection(select);
                if (desc.getModel() instanceof TemplateFilter.Descriptor) {
                    filter = new TemplateFilter((TemplateFilter.Descriptor) desc.getModel());
                    descriptor.setModel(filter.save());
                } else {
                    try {
                        FocalPlaneDataGroup group = FocalPlaneDataGroup.forName(select);
                        filter = new DataGroupFilter(desc.getAgent(), group);
                    } catch (IllegalArgumentException x) {
                    }
                }
            }
            if (filter == null) {
                descriptor.setSelection(null);
                descriptor.setModel(null);
                map.setModel(null);
            } else {
                MonitorViewModel model = new MonitorViewModel();
                model.setFilter(filter);
                map.setModel(model.getFocalPlaneMapModel());
            }
            setTitle();
        } catch (NullPointerException | ClassCastException x) {
        }
        restoring = false;
    }
    
    static public class Descriptor implements Serializable {

        private String selection;
        private Serializable model;
        private String agent;

        public String getSelection() {
            return selection;
        }

        public void setSelection(String selection) {
            this.selection = selection;
        }

        public Serializable getModel() {
            return model;
        }

        public void setModel(Serializable model) {
            this.model = model;
        }

        public String getAgent() {
            return agent;
        }

        public void setAgent(String agent) {
            this.agent = agent;
        }
        
    }

}
