package org.lsst.ccs.subsystem.focalplane.ui.filter;

import java.awt.Component;
import java.util.*;
import java.util.stream.Collectors;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;
import org.lsst.ccs.gconsole.annotations.services.persist.Create;
import org.lsst.ccs.gconsole.base.filter.AbstractChannelsFilter;
import org.lsst.ccs.gconsole.services.persist.Creator;
import org.lsst.ccs.gconsole.services.persist.Persistable;
import org.lsst.ccs.subsystem.focalplane.ui.FocalPlanePlugin;
import org.lsst.ccs.subsystem.focalplane.ui.Segment;

/**
 * Focal plane specific template based filter that puts accepted channels into groups, 
 * each corresponding to a single display path.
 *
 * @author onoprien
 */
public class GroupingTemplateFilter extends AbstractChannelsFilter {

// -- Fields : -----------------------------------------------------------------
    
    static public final String PATH = "Groups/Template Groups";
    
    private final Descriptor descriptor;
    private final HashMap<Template,ArrayList<String>> template2groups = new HashMap<>();

// -- Life cycle : -------------------------------------------------------------
    
    private GroupingTemplateFilter() {
        descriptor = new Descriptor();
        descriptor.setGroups(new LinkedHashMap<>(0));
    }
    
    public GroupingTemplateFilter(Descriptor desc) {
        descriptor = desc.clone();
        if (descriptor.getGroups() != null) {
            for (Map.Entry<String, ArrayList<String>> e : descriptor.getGroups().entrySet()) {
                for (String st : e.getValue()) {
                    try {
                        Template t = Template.valueOf(st);
                        ArrayList<String> groups = template2groups.get(t);
                        if (groups == null) {
                            groups = new ArrayList<>(1);
                            template2groups.put(t, groups);
                        }
                        groups.add(e.getKey());
                    } catch (IllegalArgumentException x) {
                    }
                }
            }
        }
    }
    
    @Create(category = FocalPlanePlugin.FILTER_CATEGORY,
            name = "Template Groups",
            path = PATH,
            description = "Channels filter that maps groups of focal plane channels to display paths.")
    static public GroupingTemplateFilter create() {
        GroupingTemplateFilter f = new GroupingTemplateFilter();
        Creator.Descriptor cd = new Creator.Descriptor();
        cd.setCategory(FocalPlanePlugin.FILTER_CATEGORY);
        cd.setPath(PATH);
        f.getDescriptor().setCreator(cd);
        return f.edit(null, null);
    }

// -- Filtering : --------------------------------------------------------------

    @Override
    public String getName() {
        return getDescriptor().getName();
    }

    @Override
    public List<String> getAgents() {
        boolean hasAgentless = false;
        LinkedHashSet<String> aa = new LinkedHashSet<>();
        for (ArrayList<String> tt : getDescriptor().getGroups().values()) {
            for (String t : tt) {
                String agent = Template.getAgent(t);
                if (agent.isEmpty()) {
                    hasAgentless = true;
                } else {
                    aa.add(agent);
                }
            }
        }
        if (hasAgentless) {
            String[] agents = getDescriptor().getAgents();
            if (agents == null) {
                return null;
            } else {
                List<String> out = new ArrayList<>(aa);
                out.addAll(Arrays.asList(agents));
                return out;
            }
            
        } else {
            return new ArrayList<>(aa);
        }
    }

    @Override
    public List<String> getOriginPaths(String displayPath) {
        return Collections.emptyList();
    }

    @Override
    public String getOriginPath(String displayPath) {
        return displayPath;
    }

    @Override
    public String getDisplayPath(AgentChannel channel) {
        return getDisplayPaths(channel).isEmpty() ? null : channel.getPath();
    }

    @Override
    public String getDisplayPath(String originPath) {
        return null;
    }

    @Override
    public List<String> getDisplayPaths(AgentChannel channel) {
        int[] indices = Segment.getIndices(channel);
        return getDisplayPaths(channel.getPath(), indices);
    }

    @Override
    public List<String> getDisplayPaths(String originPath) {
        int[] indices = Segment.getIndices(originPath);
        return getDisplayPaths(originPath, indices);
    }

    @Override
    public List<String> getFields(boolean compact) {
        LinkedHashMap<String, ArrayList<String>> groups = descriptor.getGroups();
        if (compact) {
            return new ArrayList<>(groups.keySet());
        } else {
            String[] ff = descriptor.getFields();
            int n = groups.size();
            if (ff == null || ff.length != n) {
                ArrayList<String> out = new ArrayList<>(n);
                String field = "AVERAGE_VALUE";
                for (int i = 0; i < n; i++) {
                    out.add(field);
                }
                return out;
            } else {
                return new ArrayList<>(Arrays.asList(ff));
            }
        }
    }

    
// -- Local methods : ----------------------------------------------------------
    
    private List<String> getDisplayPaths(String originPath, int[] indices) {
        if (indices == null) return Collections.emptyList();
        Template template = Template.valueOf(originPath, indices);
        if (template == null) return Collections.emptyList();
        ArrayList<String> groups = template2groups.get(template);
        if (groups == null) {
            String[] agents = getDescriptor().getAgents();
            if (agents == null || Arrays.asList(agents).contains(originPath.substring(0, originPath.indexOf("/")))) {
                template = template.stripAgent();
                groups = template2groups.get(template);
            }
        }
        if (groups == null) {
            return Collections.emptyList();
        } else {
            String prefix = Segment.getPathPrefix(indices);
            return groups.stream().map(s -> prefix + s).collect(Collectors.toList());
        }
    }
    
 
// -- Saving : -----------------------------------------------------------------

    @Override
    public Descriptor getDescriptor() {
        return descriptor;
    }

    @Override
    public Descriptor save() {
        return getDescriptor().clone();
    }

    @Override
    public GroupingTemplateFilter edit(String title, Component parent) {
        return GroupingTemplateFilterDialog.edit(this, parent);
    }
    
    static public class Descriptor extends Persistable.Descriptor {

        private LinkedHashMap<String, ArrayList<String>> groups;
        private String[] agents;
        private String[] fields;

        public LinkedHashMap<String, ArrayList<String>> getGroups() {
            return groups;
        }

        public void setGroups(LinkedHashMap<String, ArrayList<String>> groups) {
            this.groups = groups;
        }

        public String[] getAgents() {
            return agents;
        }

        public void setAgents(String[] agents) {
            this.agents = agents;
        }

        public String[] getFields() {
            return fields;
        }

        public void setFields(String[] fields) {
            this.fields = fields;
        }
        
        @Override
        public Descriptor clone() {
            Descriptor clone = (Descriptor) super.clone();
            if (groups == null) {
                groups = new LinkedHashMap<>(0);
            } else {
                for (Map.Entry<String, ArrayList<String>> e : groups.entrySet()) {
                    e.setValue((ArrayList<String>) e.getValue().clone());
                }
            }
            return clone;
        }
    }

}
