package org.lsst.ccs.subsystem.focalplane.ui;

import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.gconsole.services.aggregator.AgentChannel;

/**
 * Enumeration of the Focal Plane standard hardware segments.
 *
 * @author onoprien
 */
public enum Segment {
    
// -- Enum constants : ---------------------------------------------------------
    
    RAFT(3,3, DataProviderInfo.Attribute.RAFT_ID.getName(), "R") {
        @Override
        public int[] parseIndices(int[] indices, String s) {
            int n = s.length();
            if (n != (path.length()+2)) return null;
            try {
                int x = Integer.parseInt(s.substring(n-2, n-1));
                int y = Integer.parseInt(s.substring(n-1));
                if (x<0 || x>2 || y<0 || y>2) return null;
                indices[0] = x;
                indices[1] = y;
            } catch (NumberFormatException e) {
                return null;
            }
            return indices;
        }        
    },
    REB(3,1, DataProviderInfo.Attribute.REB_ID.getName(), "Reb"){
        @Override
        public int[] parseIndices(int[] indices, String s) {
            int n = s.length();
            if (n != (path.length()+1)) return null;
            try {
                int x = Integer.parseInt(s.substring(n-1));
                if (x<0 || x>2) return null;
                indices[2] = x;
            } catch (NumberFormatException e) {
                return null;
            }
            return indices;
        }        
    },
    CCD(1,3, DataProviderInfo.Attribute.SENSOR_ID.getName(), "S"){
        @Override
        public int[] parseIndices(int[] indices, String s) {
            int n = s.length();
            if (n != (path.length()+2)) return null;
            try {
                int x = Integer.parseInt(s.substring(n-2, n-1));
                int y = Integer.parseInt(s.substring(n-1));
                if (x<0 || x>2 || y<0 || y>2) return null;
                indices[2] = x;
                indices[3] = y;
            } catch (NumberFormatException e) {
                return null;
            }
            return indices;
        }        
    },
    AMP(2,8, DataProviderInfo.Attribute.AMPLIFIER_ID.getName(), "Segment"){
        @Override
        public int[] parseIndices(int[] indices, String s) {
            int n = s.length();
            if (n != (path.length()+2)) return null;
            try {
                int x = Integer.parseInt(s.substring(n-2, n-1));
                int y = Integer.parseInt(s.substring(n-1));
                if (x<0 || x>1 || y<0 || y>7) return null;
                indices[4] = x;
                indices[5] = y;
            } catch (NumberFormatException e) {
                return null;
            }
            return indices;
        }        
        
    };
    
// -- Fields : -----------------------------------------------------------------
    
    final int nX, nY;
    final String key;
    final String path;
    
// -- Life cycle : -------------------------------------------------------------
    
    private Segment(int nX, int nY, String key, String path) {
        this.nX = nX;
        this.nY = nY;
        this.key = key;
        this.path = path;
    }

// -- Getters : ----------------------------------------------------------------
    
    public int getNX() {
        return nX;
    }
    
    public int getNY() {
        return nY;
    }

    /**
     * Returns the attribute key that can be used to retrieve the corresponding segment index value from {@link DataProviderInfo}.
     * @return Attribute key.
     */
    public String getKey() {
        return key;
    }

    /**
     * Returns string representation of this segment type.
     * This representation should be used for constructing or parsing channel paths,
     * and anywhere human-readable text needs to be produced.
     * @return String representation of this segment type.
     */
    @Override
    public String toString() {
        return path;
    }
    
    public abstract int[] parseIndices(int[] indices, String s);
    
    
// -- Utility methods : --------------------------------------------------------
    
    /**
     * Computes focal plane segment indices for the given channel, based on attributes.
     * 
     * @param channel Data channel.
     * @return Array of length 6: [raftX, raftY, reb = ccdX, ccdY, ampX, ampY], or {@code null} if the channel does not belong to a valid segment.
     *         Trailing indices will be -1 if there is no corresponding segment at this level, but at least raft indices will have valid non-negative values.
     */
    static public int[] getIndices(AgentChannel channel) {
        int[] indices = {-1, -1, -1, -1, -1, -1};
        int raftX, raftY;
        String ID;
        try {
            ID = channel.get(DataProviderInfo.Attribute.RAFT_ID.getName());
            raftX = Integer.parseInt(ID.substring(1, 2));
            raftY = Integer.parseInt(ID.substring(2, 3));
            if (raftX < 0 || raftX > 2 || raftY < 0 || raftY > 2) return null;
            indices[0] = raftX;
            indices[1] = raftY;
            ID = channel.get(DataProviderInfo.Attribute.REB_ID.getName());
            int reb = -1;
            if (ID != null) {
                reb = Integer.parseInt(ID.substring(3, 4));
                if (reb < 0 || reb > 2) return null;
                indices[2] = reb;
            }
            ID = channel.get(DataProviderInfo.Attribute.SENSOR_ID.getName());
            if (ID != null) {
                int ccdX = Integer.parseInt(ID.substring(1, 2));
                int ccdY = Integer.parseInt(ID.substring(2, 3));
                if (ccdX < 0 || ccdX > 2 || ccdY < 0 || ccdY > 2 || (reb != -1 && reb != ccdX)) return null;
                indices[2] = ccdX;
                indices[3] = ccdY;
                ID = channel.get(DataProviderInfo.Attribute.AMPLIFIER_ID.getName());
                if (ID != null) {
                    int ampX = Integer.parseInt(ID.substring(0, 1));
                    int ampY = Integer.parseInt(ID.substring(1, 2));
                    if (ampX < 0 || ampX > 1 || ampY < 0 || ampY > 8) return null;
                    indices[4] = ampX;
                    indices[5] = ampY;
                }
            }
            return indices;
        } catch (IndexOutOfBoundsException | NullPointerException | NumberFormatException x) {
            return null;
        }
    }
    
    /**
     * Computes focal plane segment indices for the given path in standard format.
     * 
     * @param path Path in RXX[/RebX][/SXX[/SegmentXX]]/... format.
     * @return Array of length 6: [raftX, raftY, reb = ccdX, ccdY, ampX, ampY], or {@code null} if the channel does not belong to a valid segment.
     *         Trailing indices will be -1 if there is no corresponding segment at this level, but at least raft indices will have valid non-negative values.
     */
    static public int[] getIndices(String path) {
        int[] indices = {-1, -1, -1, -1, -1, -1};
        String[] ss = path.split("/");
        int n = ss.length - 1;
        int i = 0;
        if (i > n) return null;
        int[] out = RAFT.parseIndices(indices, ss[i++]);
        if (out == null) {
            if (i > n) return null;
            out = RAFT.parseIndices(indices, ss[i++]);
        }
        if (out == null) return null;
        if (i > n) return indices;
        out = REB.parseIndices(indices, ss[i]);
        if (out != null) {
            if (++i > n) return indices;
        }
        out = CCD.parseIndices(indices, ss[i++]);
        if (out == null || i > n) return indices;
        AMP.parseIndices(indices, ss[i]);
        return indices;
    }
    
    /**
     * Returns standardized path prefix for a channel with given segment indices.
     * 
     * @param indices Array of length 6: [raftX, raftY, reb = ccdX, ccdY, ampX, ampY].
     * @return Path prefix.
     */
    static public String getPathPrefix(int[] indices) {
        StringBuilder sb = new StringBuilder();
        if (indices[0] != -1) {
            sb.append(RAFT).append(indices[0]).append(indices[1]).append("/");
            if (indices[2] != -1) {
                if (indices[3] == -1) {
                    sb.append(REB).append(indices[2]).append("/");
                } else {
                    sb.append(CCD).append(indices[2]).append(indices[3]).append("/");
                    if (indices[4] != -1) {
                        sb.append(AMP).append(indices[4]).append(indices[5]).append("/");
                    }
                }
            }
        }
        return sb.toString();
    }
    
// -- Static constants : -------------------------------------------------------
    
    static public final int[] N = {5,5,3,3,2,8};
        
}
