package org.lsst.ccs.subsystem.focalplane.ui;

import org.lsst.ccs.subsystem.focalplane.ui.filter.DataGroupFilter;
import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.services.optpage.OptionalPage;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTableView;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusEvent;
import org.lsst.ccs.gconsole.services.aggregator.AgentStatusListener;
import org.lsst.ccs.gconsole.services.persist.DataPanelDescriptor;
import org.lsst.ccs.subsystem.focalplane.data.FocalPlaneDataGroup;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.FocalPlaneMapPage;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.FocalPlaneMapPageDialog;

/**
 * Focal plane GUI plugin.
 *
 * @author onoprien
 */
@Plugin(name="LSST CCS Focal Plane GUI",
        id="focal-plane-gui",
        description="LSST CCS Focal Plane Subsystem GUI")
public class FocalPlanePlugin extends ConsolePlugin {

// -- Fields : -----------------------------------------------------------------
    
    private final String DEFAULT_FOCAL_PLANE_MAP_PAGE_NAME = "Focal Plane";
    
    private final ArrayList<FocalPlaneMapPage> focalPlaneMapPages = new ArrayList<>();
    
    private AgentStatusListener statusListener;

// -- Life cycle : -------------------------------------------------------------
    
    public FocalPlanePlugin() {
    }
    
    @Override
    public void initialize() {
        // Pre-defined monitoring page (selectable channel group) :
        OptionalPage.getService().add(new FocalPlaneMonitoringOptionalPage());
    }

    @Override
    public void start() {
        // Initialize machinery for focal plane maps
        initFocalPlaneMaps();
    }

    @Override
    public void stop() {
        // Shut down machinery for focal plane maps
        getConsole().getStatusAggregator().removeListener(statusListener);
        statusListener = null;
    }
    
    
// -- Handling focal plane maps : ----------------------------------------------
    
    private void initFocalPlaneMaps() {
        statusListener = new AgentStatusListener() {
            @Override
            public void connect(AgentStatusEvent event) {
                AgentInfo agent = event.getSource();
                if (agent.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, "").equals(AgentCategory.FOCAL_PLANE.name())) {
                    SwingUtilities.invokeLater(() -> {
                        String agentName = agent.getName();
                        Action act = new AbstractAction("Custom...") {
                            @Override
                            public void actionPerformed(ActionEvent e) {
                                FocalPlaneMapPage.Descriptor d = FocalPlaneMapPageDialog.show(null, DEFAULT_FOCAL_PLANE_MAP_PAGE_NAME);
                                if (d != null) {
                                    d.setAgent(agentName);
                                    openFocalPlaneMap(d);
                                }
                            }
                        };
                        getServices().addMenu(act, "CCS Subsystems", agentName, "Map");
                        addFocalPlaneMapMenu(agentName, 1, 1);
                        addFocalPlaneMapMenu(agentName, 1, 2);
                        addFocalPlaneMapMenu(agentName, 2, 1);
                        addFocalPlaneMapMenu(agentName, 2, 2);
                        addFocalPlaneMapMenu(agentName, 1, 3);
                        addFocalPlaneMapMenu(agentName, 2, 3);
                        addFocalPlaneMapMenu(agentName, 3, 3);
                        addFocalPlaneMapMenu(agentName, 4, 4);
                    });
                }
            }
            @Override
            public void disconnect(AgentStatusEvent event) {
                SwingUtilities.invokeLater(() -> {
                    String agentName = event.getSource().getName();
                    getConsole().removeMenu("CCS Subsystems", agentName, "Map");
                });
            }
        };
        getConsole().getStatusAggregator().addListener(statusListener, null, Collections.singletonList(Segment.RAFT.getKey()));
    }
    
    private void openFocalPlaneMap(FocalPlaneMapPage.Descriptor descriptor) {
        FocalPlaneMapPage page = new FocalPlaneMapPage(descriptor);
        focalPlaneMapPages.add(page);
        Map<Object, Object> par = new HashMap<>();
        DataPanelDescriptor panDesc = descriptor.getPage();
        if (panDesc != null && panDesc.isOpen()) {
            Map<String, Serializable> data = panDesc.getData();
            if (data != null) {
                par.putAll(data);
            }
        }
        String title = descriptor.getName();
        if (title == null) title = DEFAULT_FOCAL_PLANE_MAP_PAGE_NAME;
        par.put(Panel.TITLE, title);
        Consumer<JComponent> onClose = c -> onClosingFocalPlaneMap((FocalPlaneMapPage)c);
        par.put(Panel.ON_CLOSE, onClose);
        Console.getConsole().getPanelManager().open(page, par);
        page.opened();
    }
    
    private void onClosingFocalPlaneMap(FocalPlaneMapPage page) {
        page.closed();
        focalPlaneMapPages.remove(page);
    }
        
    private void addFocalPlaneMapMenu(String agent, int rows, int columns) {
        Action act = new AbstractAction(columns +" x "+ rows) {
            @Override
            public void actionPerformed(ActionEvent e) {
                openFocalPlaneMap(new FocalPlaneMapPage.Descriptor(agent, rows, columns));
            }
        };
        getServices().addMenu(act, "CCS Subsystems", agent, "Map");
    }
    
    
// -- Standard monitoring page : -----------------------------------------------

    static class FocalPlaneMonitoringOptionalPage implements OptionalPage {

        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public String getPage(AgentInfo agent) {
            if (agent.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY,"").equals(AgentCategory.FOCAL_PLANE.name())) {
                return "CCS Subsystems/" + agent.getName() +"/Monitoring";
            }
            return null;
        }

        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            DataGroupMonitor dataSelection = new DataGroupMonitor(agent);
            return dataSelection;

        }
    }
    
    static private final class DataGroupMonitor extends JPanel {
        
        JComboBox dataSelector = new JComboBox(FocalPlaneDataGroup.values());
        SectionedTableView view = new SectionedTableView();
        private final AgentInfo agent;
        
        DataGroupMonitor(AgentInfo agent) {
            this.agent = agent;
            setLayout(new BorderLayout());
            add(dataSelector,BorderLayout.PAGE_START);
            makeFilteredTable();
            
            dataSelector.addActionListener(e -> {
                remove(view.getPanel());
                view = new SectionedTableView();
                makeFilteredTable();
            });
        }
        
        final void makeFilteredTable() {
            DataGroupFilter monitorFilter = new DataGroupFilter(agent.getName(),(FocalPlaneDataGroup)(dataSelector.getSelectedItem()));
            view.uninstall();
            view.setFilter(monitorFilter);
            view.install();
            add(view.getPanel(),BorderLayout.CENTER);
            revalidate();
            repaint();
        }
        
    }
    
    
// -- Saving/restoring : -------------------------------------------------------

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        
        // Close existing pages:
        
        if (!focalPlaneMapPages.isEmpty()) {
            PanelManager panMan = Console.getConsole().getPanelManager();
            ArrayList<FocalPlaneMapPage> copy = new ArrayList<>(focalPlaneMapPages);
            for (FocalPlaneMapPage page : copy) {
                panMan.close(page);
            }
        }
        
        // Open saved pages:
        
        if (storageBean instanceof Descriptor) {
            Descriptor desc = (Descriptor) storageBean;
            FocalPlaneMapPage.Descriptor[] pages = desc.getMapPages();
            if (pages != null && pages.length != 0) {
                for (FocalPlaneMapPage.Descriptor pageDescriptor : pages) {
                    openFocalPlaneMap(pageDescriptor);
                }
            }
        }
        
        return true;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        
        ArrayList<FocalPlaneMapPage.Descriptor> mapPages = new ArrayList<>(focalPlaneMapPages.size());
        for (FocalPlaneMapPage view : focalPlaneMapPages) {
            FocalPlaneMapPage.Descriptor d = view.save();
            if (d != null) {
                mapPages.add(d);
            }
        }
        if (!mapPages.isEmpty()) {
            desc.setMapPages(mapPages.toArray(new FocalPlaneMapPage.Descriptor[mapPages.size()]));
        }
        
        return desc;
    }

    static public class Descriptor extends ComponentDescriptor {

        private FocalPlaneMapPage.Descriptor[] mapPages;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public FocalPlaneMapPage.Descriptor[] getMapPages() {
            return mapPages;
        }

        public void setMapPages(FocalPlaneMapPage.Descriptor[] mapPages) {
            this.mapPages = mapPages;
        }

    }
    
}
