package org.lsst.ccs.subsystem.focalplane.ui.fpmap;

import java.io.Serializable;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.gconsole.agent.AgentChannel;

/**
 * A template that defines a class of data channels - one channel per {@link FocalPlaneMap} cell.
 *
 * @author onoprien
 */
public class Template implements Serializable, Comparable<Template> {

// -- Fields : -----------------------------------------------------------------
    
    public final String code;
    public final Segment segment;
    
    static private final Pattern fromString = Pattern.compile("(.+) \\(per (\\w+)\\)");

// -- Life cycle : -------------------------------------------------------------
    
    public Template(String code, Segment segment) {
        this.code = code;
        this.segment = segment;
    }
    
    
// -- Getters : ----------------------------------------------------------------
    
    public String getAgent() {
        return code.substring(0, code.indexOf("/"));
    }
    
    
// -- Operations : -------------------------------------------------------------
    
    /**
     * Constructs a template corresponding to the specified channel.
     * 
     * @param channel Data channel.
     * @return Template, or {@code null} if the channel does not have metadata for placement on the {@link FocalPlaneMap}.
     */
    static public Template valueOf(AgentChannel channel) {
        int[] indices = Segment.getIndices(channel);
        if (indices == null) return null;
        Segment segment = Segment.RAFT;
        String path = channel.getPath();
        String p = "/"+ Segment.RAFT.getPathPrefix() + indices[0] + indices[1];
        path = path.replaceFirst(p, "");
        if (indices[2] != -1) {
            segment = Segment.REB;
            p = "/"+ Segment.REB.getPathPrefix() + indices[2];
            path = path.replaceFirst(p, "");
            if (indices[3] != -1) {
                segment = Segment.CCD;
                p = "/"+ Segment.CCD.getPathPrefix() + indices[2] + indices[3];
                path = path.replaceFirst(p, "");
                if (indices[4] != -1) {
                    segment = Segment.AMP;
                    p = String.valueOf(indices[4]) + indices[5];
                    path = path.replaceFirst(p, "");
                }
            }
        }
        return new Template(path, segment);
    }
    
    /**
     * Constructs a template corresponding to the specified String representation.
     * 
     * @param string String representation created by {@code toString()} method.
     * @return Template.
     * @throws IllegalArgumentException if the specified String cannot be converted to a template.
     */
    static public Template valueOf(String string) {
        Matcher m = fromString.matcher(string);
        if (m.matches()) {
            String code = m.group(1);
            Segment segment = Segment.valueOf(m.group(2));
            return new Template(code, segment);
        } else {
            throw new IllegalArgumentException("Cannot convert to template: "+ string);
        }
    }
    
// -- Overriding Object : ------------------------------------------------------

    /**
     * Produces a String representation of this template.
     * The string is human-readable, and can be converted back to a {@code Template}
     * by calling {@code valueOf(String string)} method.
     * 
     * @return String representation of this template.
     */
    @Override
    public String toString() {
        return code +" (per "+ segment +")";
    }

    @Override
    public boolean equals(Object obj) {
        if (obj instanceof Template) {
            Template other = (Template) obj;
            return (code + segment).equals(other.code + other.segment);
        } else {
            return false;
        }
    }

    @Override
    public int hashCode() {
        return (code + segment).hashCode();
    }

    @Override
    public int compareTo(Template other) {
        return (code + segment).compareTo(other.code + other.segment);
    }

}
