package org.lsst.ccs.subsystem.focalplane.ui.fpmap;

import java.awt.BorderLayout;
import java.awt.Color;
import java.io.Serializable;
import java.util.Objects;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import org.lsst.ccs.gconsole.base.Const;
import org.lsst.ccs.gconsole.util.session.Savable;

/**
 * Graphic component that contains {@link FocalPlaneMap} and associated controls.
 *
 * @author onoprien
 */
public final class FocalPlaneMapRegion extends JPanel implements Savable {

// -- Fields : -----------------------------------------------------------------
    
    private Descriptor descriptor;
    private final FocalPlaneMapModelFactory[] modelFactories;
    
    private final FocalPlaneMap map;
    private final JLabel titleLabel;
    private final JComboBox<FocalPlaneMapModelFactory> comboBox;
    
    private boolean restoring = false;

// -- Life cycle : -------------------------------------------------------------
    
    public FocalPlaneMapRegion(FocalPlaneMapModelFactory[] models, Descriptor desc) {
        super(new BorderLayout());
        modelFactories = models;
        
        setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createEmptyBorder(1, 1, 1, 1), 
                BorderFactory.createLineBorder(Color.BLACK, 1, true)));
        
        map = new FocalPlaneMap();
        add(map, BorderLayout.CENTER);
        
        Box header = Box.createHorizontalBox();
        add(header, BorderLayout.NORTH);
        
        comboBox = new JComboBox<>(modelFactories);
        header.add(comboBox);
        comboBox.addActionListener(e -> {
            FocalPlaneMapModelFactory f = (FocalPlaneMapModelFactory) comboBox.getSelectedItem();
            if (!restoring) setCurrentFactory(f);
        });
        
        header.add(Box.createRigidArea(Const.HDIM));
        titleLabel = new JLabel();
        header.add(titleLabel);
        
        header.add(Box.createHorizontalGlue());
        
        restore(desc);
    }
    
    public void opened() {
        
    }
    
    public void closed() {
        map.setModel(null);
    }
    
    
// -- Setters : ----------------------------------------------------------------
    
    public void setTitle(String title) {
        if (title == null) {
            title = map.getName();
            if (title == null || ((FocalPlaneMapModelFactory) comboBox.getSelectedItem() instanceof DataGroupFactory)) {
                title = "";
            }
        }
        titleLabel.setText(title.isEmpty() ? "" : "<html><b>"+ title);
    }
    
    private void setCurrentFactory(FocalPlaneMapModelFactory factory) {
        try {
            String oldFactory = descriptor.getModelFactory();
            String newFactory = factory.toString();
            Serializable desc = Objects.equals(oldFactory, newFactory) ? descriptor.getModel() : null;
            FocalPlaneMapModel model = factory.createModel(desc);
            map.setModel(model);
            setTitle(null);
            descriptor.setModelFactory(newFactory);
            descriptor.setModel(model.save());
        } catch (NullPointerException x) {
            descriptor.setModel(null);
            descriptor.setModelFactory(null);
            setTitle("");
        }
    }
    
// -- Saving : -----------------------------------------------------------------

    @Override
    public Descriptor save() {
        descriptor.setMap(map.save());
        return descriptor;
    }

    @Override
    public void restore(Serializable desc) {
        try {
            descriptor = (Descriptor) desc;
            String currentModelName = descriptor.getModelFactory();
            FocalPlaneMapModelFactory currentFactory = null;
            for (FocalPlaneMapModelFactory fac : modelFactories) {
                if (currentModelName.equals(fac.toString())) {
                    currentFactory = fac;
                    break;
                }
            }
            restoring = true;
            comboBox.setSelectedItem(currentFactory);
            restoring = false;
            FocalPlaneMapModel model = currentFactory.getModel(descriptor.getModel());
            map.setModel(model);
            setTitle(null);
        } catch (NullPointerException | ClassCastException x) {
            descriptor = new Descriptor();
        }
    }
    
    static public class Descriptor implements Serializable {

        private FocalPlaneMap.Descriptor map;
        private String modelFactory;
        private Serializable model;

        public FocalPlaneMap.Descriptor getMap() {
            return map;
        }

        public void setMap(FocalPlaneMap.Descriptor map) {
            this.map = map;
        }

        public String getModelFactory() {
            return modelFactory;
        }

        public void setModelFactory(String modelFactory) {
            this.modelFactory = modelFactory;
        }

        public Serializable getModel() {
            return model;
        }

        public void setModel(Serializable model) {
            this.model = model;
        }
        
    }

}
