package org.lsst.ccs.subsystem.focalplane.ui.fpmap;

import java.awt.BorderLayout;
import java.awt.Component;
import java.util.*;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.DefaultListCellRenderer;
import javax.swing.DefaultListModel;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.ListSelectionModel;
import javax.swing.SwingUtilities;
import javax.swing.border.EtchedBorder;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.Const;

/**
 * Dialog that configures filters for populating {@link FocalPlaneMap}.
 *
 * @author onoprien
 */
public final class ChannelSelectorDialog extends JDialog {

// -- Fields : -----------------------------------------------------------------
    
    private String[] agents;
    private ArrayList<Template> templates;
    
    private TemplateFilter.Descriptor descriptor;
    
    private JComboBox<String> subsystemCombo;
    private JComboBox<Segment> granularityComboIn, granularityComboOut;
    private JList<Template> templatesIn, templatesOut;
    private JButton addButton, removeButton;
    private JTextField titleField;

// -- Life cycle : -------------------------------------------------------------
    
    private ChannelSelectorDialog(TemplateFilter.Descriptor desc, Component parentComponent) {
        super(parentComponent == null ? null : SwingUtilities.getWindowAncestor(parentComponent), "Select channels", ModalityType.APPLICATION_MODAL);
        descriptor = desc;
        
        List<AgentChannel> all = Console.getConsole().getStatusAggregator().getChannels(Segment.RAFT.getKey());
        TreeSet<String> aa = new TreeSet<>();
        TreeSet<Template> tt = new TreeSet<>();
        all.forEach(channel -> {
            aa.add(channel.getAgentName());
            Template t = Template.valueOf(channel);
            if (t != null)  tt.add(t);
        });
        agents = aa.toArray(new String[aa.size()]);
        templates = new ArrayList<>(tt);

        
        Box rootPanel = Box.createVerticalBox();
        add(rootPanel, BorderLayout.CENTER);
        rootPanel.setBorder(BorderFactory.createCompoundBorder(
                BorderFactory.createEtchedBorder(EtchedBorder.LOWERED), BorderFactory.createEmptyBorder(Const.VSPACE, Const.HSPACE, Const.VSPACE, Const.HSPACE)));

        Box channelsPanel = Box.createVerticalBox();
        rootPanel.add(channelsPanel);
        channelsPanel.setBorder(BorderFactory.createTitledBorder("Channel selection"));

        Box box = Box.createHorizontalBox();
        channelsPanel.add(box);
        box.add(new JLabel("Subsystem: "));
        subsystemCombo = new JComboBox<>(agents);
        box.add(subsystemCombo);
        subsystemCombo.addActionListener(e -> {
            updateTemplatesIn();
        });
        box.add(Box.createRigidArea(Const.HDIM));
        box.add(new JLabel("Granularity: "));
        granularityComboIn = new JComboBox<>(Segment.values());
        box.add(granularityComboIn);
        granularityComboIn.addActionListener(e -> {
            updateTemplatesIn();
        });
        box.add(Box.createHorizontalGlue());

        channelsPanel.add(Box.createRigidArea(Const.VDIM));

        templatesIn = new JList<>();
        channelsPanel.add(new JScrollPane(templatesIn));
        templatesIn.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        templatesIn.setCellRenderer(new DefaultListCellRenderer() {
            @Override
            public Component getListCellRendererComponent(JList<?> list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
                return super.getListCellRendererComponent(list, ((Template)value).code, index, isSelected, cellHasFocus);
            }
        });
        templatesIn.addListSelectionListener(e -> {
            addButton.setEnabled(templatesIn.getSelectedValue() != null);
        });

        box = Box.createHorizontalBox();
        rootPanel.add(box);
        box.add(Box.createHorizontalGlue());
        addButton = new JButton("  Add  ");
        box.add(addButton);
        addButton.addActionListener(e -> {
            Template template = templatesIn.getSelectedValue();
            if (template != null) {
                DefaultListModel<Template> out = (DefaultListModel) templatesOut.getModel();
                out.addElement(template);
                DefaultListModel<Template> in = (DefaultListModel) templatesIn.getModel();
                in.removeElement(template);
            }
        });
        addButton.setEnabled(false);
        box.add(Box.createRigidArea(Const.HDIM));
        removeButton = new JButton("Remove");
        box.add(removeButton);
        removeButton.addActionListener(e -> {
            Template template = templatesOut.getSelectedValue();
            if (template != null) {
                DefaultListModel<Template> out = (DefaultListModel) templatesOut.getModel();
                out.removeElement(template);
                updateTemplatesIn();
            }
        });
        removeButton.setEnabled(false);
        box.add(Box.createRigidArea(Const.HDIM));

        Box filterPanel = Box.createVerticalBox();
        rootPanel.add(filterPanel);
        filterPanel.setBorder(BorderFactory.createTitledBorder("Focal plane map"));

        box = Box.createHorizontalBox();
        filterPanel.add(box);
        box.add(new JLabel("Title: "));
        titleField = new JTextField(20);
        box.add(titleField);
        box.add(Box.createHorizontalGlue());

        filterPanel.add(Box.createRigidArea(Const.VDIM));

        templatesOut = new JList<>(new DefaultListModel<>());
        filterPanel.add(new JScrollPane(templatesOut));
        templatesOut.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
        templatesOut.addListSelectionListener(e -> {
            removeButton.setEnabled(templatesOut.getSelectedValue() != null);
        });

//        box = Box.createHorizontalBox();
//        filterPanel.add(box);
//        box.add(new JLabel("Granularity: ")); // FIXME: Disabled for now
        granularityComboOut = new JComboBox<>(Segment.values());
//        box.add(granularityComboOut);
//        box.add(Box.createHorizontalGlue());

        box = Box.createHorizontalBox();
        add(box, BorderLayout.SOUTH);
        box.add(Box.createHorizontalGlue());
        JButton cancelButton = new JButton("Cancel");
        box.add(cancelButton);
        cancelButton.addActionListener(e -> {
            descriptor = null;
            dispose();
        });
        box.add(Box.createRigidArea(Const.HDIM));
        JButton okButton = new JButton(" OK ");
        box.add(okButton);
        okButton.addActionListener(e -> {
            writeDescriptor();
            dispose();
        });

        readDescriptor();
    }


    static public TemplateFilter.Descriptor show(TemplateFilter.Descriptor descriptor, Component parentComponent) {
        if (parentComponent == null) parentComponent = Console.getConsole().getWindow();
        ChannelSelectorDialog dialog = new ChannelSelectorDialog(descriptor, parentComponent);
        dialog.pack();
        dialog.setLocationRelativeTo(parentComponent);
        dialog.setVisible(true);
        return dialog.descriptor;
    }
    
// -- Local methods : ----------------------------------------------------------
    
    private void updateTemplatesIn() {
        String agent = (String) subsystemCombo.getSelectedItem();
        Segment segment = (Segment) granularityComboIn.getSelectedItem();
        DefaultListModel<Template> inModel = new DefaultListModel<>();
        DefaultListModel outModel = (DefaultListModel) templatesOut.getModel();
        for (Template template : templates) {
            if (template.segment.equals(segment) && template.getAgent().equals(agent) && !outModel.contains(template)) {
                inModel.addElement(template);
            }
        }
        templatesIn.setModel(inModel);
    }
    
    private void readDescriptor() {
        if (descriptor == null) {
            granularityComboOut.setSelectedItem(Segment.AMP);
        } else {
            String title = descriptor.getTitle();
            if (title != null) {
                titleField.setText(title);
            }
            DefaultListModel<Template> out = (DefaultListModel) templatesOut.getModel();
            String[] tt = descriptor.getTemplates();
            if (tt == null) tt = new String[0];
            for (String s : tt) {
                try {
                    Template t = Template.valueOf(s);
                    out.addElement(t);
                } catch (IllegalArgumentException x) {
                }
            }
            String gran = descriptor.getGranularity();
            try {
                granularityComboOut.setSelectedItem(Segment.valueOf(gran));
            } catch (IllegalArgumentException | NullPointerException x) {
                granularityComboOut.setSelectedItem(Segment.AMP);
            }
        }
        updateTemplatesIn();
    }
    
    private void writeDescriptor() {
        descriptor = new TemplateFilter.Descriptor();
        String title = titleField.getText().trim();
        if (!title.isEmpty()) {
            descriptor.setTitle(title);
        }
        Segment segment = (Segment) granularityComboOut.getSelectedItem();
        if (segment != null && !Segment.AMP.equals(segment)) {
            descriptor.setGranularity(segment.name());
        }
        DefaultListModel<Template> out = (DefaultListModel) templatesOut.getModel();
        int n = out.getSize();
        if (n > 0) {
            String[] tt = new String[n];
            for (int i=0; i<n; i++) {
                tt[i] = out.get(i).toString();
            }
            descriptor.setTemplates(tt);
        }
    }
        
}
