package org.lsst.ccs.subsystem.focalplane.ui.fpmap;

import java.awt.Color;
import java.io.Serializable;
import java.util.*;
import javax.swing.JComponent;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.states.DataProviderState;
import org.lsst.ccs.gconsole.agent.AgentChannel;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;
import org.lsst.ccs.gconsole.plugins.monitor.AbstractMonitorView2;
import org.lsst.ccs.gconsole.plugins.monitor.ChannelHandle;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorCell;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorField;
import org.lsst.ccs.gconsole.plugins.monitor.MonitorFormat;
import org.lsst.ccs.gconsole.plugins.monitor.Updatable;

/**
 * {@link MonitorView} that provides {@link FocalPlaneMapModel}.
 * Can be used to create {@link FocalPlaneMapModel} based on an arbitrary {@link AgentChannelsFilter}.
 *
 * @author onoprien
 */
public class MonitorViewModel extends AbstractMonitorView2 {

// -- Fields : -----------------------------------------------------------------
    
    private final Model model = new Model();
    private Cell root;

//    static private final int[] N = {5,5,3,3,2,8};
    static private final Color COLOR_ALARM = Color.RED;
    static private final Color COLOR_WARNING = Color.ORANGE;
    static private final Color COLOR_NOMINAL = new Color(0, 200, 0);
    static private final int RGB_WARNING = COLOR_WARNING.getRGB() & 0xFFFFFF;
    static private final int RGB_NOMINAL = COLOR_NOMINAL.getRGB() & 0xFFFFFF;

// -- Life cycle : -------------------------------------------------------------
    
    public MonitorViewModel() {
    }
    
    
// -- Updates : ----------------------------------------------------------------

    @Override
    protected void disconnect(AgentInfo agent) {
        super.disconnect(agent);
        model.notifyListeners(new FocalPlaneMapModelEvent(model));
    }

    @Override
    public void statusChanged(AgentStatusEvent event) {
        super.statusChanged(event);
        SwingUtilities.invokeLater(() -> {
            model.notifyListeners(new FocalPlaneMapModelEvent(model));
        });
    }

    @Override
    protected void resetChannels() {
        
        root = new Cell();
        root.setFormattedValue(FocalPlaneMapValue.NONE);
        
        Iterator<Map.Entry<String,ChannelHandle>> it = path2data.entrySet().iterator();
        while (it.hasNext()) {
            ChannelHandle ch = it.next().getValue();
            AgentChannel channel = ch.getChannel();
            int[] indices = Segment.getIndices(channel);
            if (indices == null) {
                it.remove();
                continue;
            }
            Cell cell = addOrGetChild(indices);
            cell.add(ch);
            ch.setTarget(cell);
        }
        
        root.trim();
        model.notifyListeners(new FocalPlaneMapModelEvent(model));
    }
        
    private Cell addOrGetChild(int... indices) {
        Cell cell = root;
        for (int i = 0; i < indices.length; i++) {
            int index = indices[i];
            if (index == -1) {
                return cell;
            } else {
                if (cell.children == null) cell.children = new Cell[Segment.N[i]];
                Cell child = cell.children[index];
                if (child == null) {
                    child = new Cell();
                    child.parent = cell;
                    cell.children[index] = child;
                }
                cell = child;
            }
        }
        return cell;
    }
    
// -- Getters : ----------------------------------------------------------------

    @Override
    public JComponent getPanel() {
        throw new UnsupportedOperationException("Not supported yet.");
    }
    
    public FocalPlaneMapModel getFocalPlaneMapModel() {
        return model;
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    
    private void format(Cell cell) {
        
        if (cell.getChannels().isEmpty()) {
            cell.setFormattedValue(FocalPlaneMapValue.NONE);
            return;
        }
        
        FocalPlaneMapValue parentValue = FocalPlaneMapValue.NONE;
        Cell parent = cell.parent;
        while (parentValue == FocalPlaneMapValue.NONE && parent != null) {
            parentValue = parent.getFormattedValue();
            if (parentValue == null) {
                format(parent);
                parentValue = parent.getFormattedValue();
            }
            parent = parent.parent;
        }
        
        FocalPlaneMapValue fv = new FocalPlaneMapValue();
        fv.setSplit(cell.children != null);
        fv.bgColor = parentValue == FocalPlaneMapValue.NONE ? null : parentValue.getBgColor();
        
        if (fv.bgColor == null && cell.getChannels().size() == 1) {
            AgentChannel channel = cell.getChannels().get(0).getChannel();
            if (channel == null) {
                fv.bgColor = MonitorFormat.COLOR_OFF;
            } else {
                fv.toolTip = "<html>"+ channel.getPath() +"<br>"+ channel.get();
                fv.bgColor = computeColor(channel);
            }
        } else {
            for (ChannelHandle ch : cell.getChannels()) {
                Color c = computeColor(ch.getChannel());
                fv.bgColor = mergeColor(c, fv.bgColor);
            }
        }
        
        cell.setFormattedValue(fv);
    }
    
    private Color computeColor(AgentChannel channel) {
        DataProviderState state = channel.get(AgentChannel.Key.STATE);
        Color color = Color.LIGHT_GRAY;
        
        if (state != null) {
            switch (state) {
                case ALARM:
                    return Color.RED;
                case WARNING:
                    color = COLOR_WARNING;
                    break;
                case NOMINAL:
                    int alpha = 255;
                    try {
                        double v = getDouble(channel, AgentChannel.Key.VALUE);
                        double low, high;
                        try {
                            low = getDouble(channel, AgentChannel.Key.LOW_WARN);
                            low = getDouble(channel, AgentChannel.Key.LOW_ALARM) + low;
                        } catch (RuntimeException x) {
                            low = getDouble(channel, AgentChannel.Key.LOW_ALARM);
                        }
                        try {
                            high = getDouble(channel, AgentChannel.Key.HIGH_WARN);
                            high = getDouble(channel, AgentChannel.Key.HIGH_ALARM) - high;
                        } catch (RuntimeException x) {
                            high = getDouble(channel, AgentChannel.Key.HIGH_ALARM);
                        }
                        double d = (high-low)/4.;
                        double a = Math.min((v-low)/d, (high-v)/d);
                        alpha = (int) (a*255);
                        if (alpha >= 255) {
                            return COLOR_NOMINAL;
                        } else if (alpha < 70) {
                            alpha = 70;
                        }
                        return new Color(RGB_NOMINAL | (alpha << 24), true);
                    } catch (RuntimeException x) {
                        return COLOR_NOMINAL;
                    }
                default:
                    color = MonitorFormat.COLOR_STATE.get(state);
            }
        }
        return color;
    }
    
    private Color mergeColor(Color c1, Color c2) {
        if (c2 == null) return c1; // no default
        if (COLOR_ALARM.equals(c1) || COLOR_ALARM.equals(c2)) return COLOR_ALARM; // one is RED
        int rgb1 = c1.getRGB() & 0xffffff;
        int rgb2 = c2.getRGB() & 0xffffff;
        if (rgb1 != RGB_WARNING && rgb1 != RGB_NOMINAL) return c2; // c1 neither GREEN nor YELLOW
        if (rgb2 != RGB_WARNING && rgb2 != RGB_NOMINAL) return c1; // c2 neither GREEN nor YELLOW
        if (rgb1 == RGB_NOMINAL) {
            if (rgb2 == RGB_NOMINAL) {
                return c1.getAlpha() < c2.getAlpha() ? c1 : c2;
            } else {
                return c2;
            }
        } else {
            if (rgb2 == RGB_NOMINAL) {
                return c1;
            } else {
                return c1.getAlpha() > c2.getAlpha() ? c1 : c2;
            }
            
        }
    }
    
    private double getDouble(AgentChannel channel, String key) {
        Object o = channel.get(key);
        if (o == null) throw new RuntimeException();
        if (o instanceof ConfigurationParameterInfo) {
            o = ((ConfigurationParameterInfo)o).getCurrentValue();
        }
        double value;
        if (o instanceof Double) {
            value = (Double)o;
        } else {
            value = Double.parseDouble(o.toString());
        }
        return value;
    }
    
// -- FocalPlaneMapModel implementation : --------------------------------------
    
    private class Model extends AbstractFocalPlaneMapModel {

        @Override
        public String getTitle() {
            AgentChannelsFilter filter = MonitorViewModel.this.getFilter();
            return filter == null ? null : filter.getName();
        }

        @Override
        public void addFocalPlaneMapModelListener(FocalPlaneMapModelListener listener) {
            if (listeners.isEmpty()) {
                install();
            }
            super.addFocalPlaneMapModelListener(listener);
        }

        @Override
        public void removeFocalPlaneMapModelListener(FocalPlaneMapModelListener listener) {
            super.removeFocalPlaneMapModelListener(listener);
            if (listeners.isEmpty()) {
                uninstall();
            }
        }

        @Override
        public FocalPlaneMapValue getValue() {
            return FocalPlaneMapValue.EMPTY;
        }

        @Override
        public FocalPlaneMapValue getValue(int raftX, int raftY) {
            return get(raftX, raftY);
        }

        @Override
        public FocalPlaneMapValue getValue(int raftX, int raftY, int reb) {
           return get(raftX, raftY, reb);
        }

        @Override
        public FocalPlaneMapValue getValue(int raftX, int raftY, int reb, int ccdY) {
            return get(raftX, raftY, reb, ccdY);
        }

        @Override
        public FocalPlaneMapValue getValue(int raftX, int raftY, int reb, int ccdY, int ampX, int ampY) {
            return get(raftX, raftY, reb, ccdY, ampX, ampY);
        }
        
        private FocalPlaneMapValue get(int... indices) {
            if (root == null) return null;
            try {
                Cell cell = root.getChild(indices);
                if (cell == null) return null;
                FocalPlaneMapValue fv = cell.getFormattedValue();
                if (fv == null) format(cell);
                return cell.getFormattedValue();
            } catch (ArrayIndexOutOfBoundsException x) {
                return null;
            }
        }

        @Override
        public Serializable save() {
            try {
                TemplateFilter filter = (TemplateFilter) MonitorViewModel.this.getFilter();
                return filter == null ? null : filter.save();
            } catch (ClassCastException x) {
                return null;
            }
        }
        
    }
    
    static private class Cell extends MonitorCell implements Updatable {
                
        Cell() {
            super(new ArrayList<>(1), MonitorField.NULL);
        }
        
        Cell parent;
        Cell[] children;
        
        void add(ChannelHandle channelHandle) {
            items.add(channelHandle);
        }
        
        Cell getChild(int... indices) {
            Cell cell = this;
            for (int i = 0; i < indices.length; i++) {
                int index = indices[i];
                if (index == -1) {
                    return cell;
                } else {
                    if (cell.children == null) return null;
                    cell = cell.children[index];
                    if (cell == null) return null;
                }
            }
            return cell;
        }
        
        void trim() {
            if (items.isEmpty()) {
                items = Collections.emptyList();
            } else {
                ((ArrayList)items).trimToSize();
            }
            if (children != null) {
                for (Cell c : children) {
                    if (c != null) c.trim();
                }
            }
        }

        @Override
        public void update(ChannelHandle channelHandle, List<MonitorField> fields) {
            update(channelHandle);
        }

        @Override
        public void update(ChannelHandle channelHandle) {
            if (! items.isEmpty()) setFormattedValue(null);
            if (children != null) {
                for (Cell c : children) {
                    if (c != null) {
                        c.update(channelHandle);
                    }
                }
            }
        }

        @Override
        public FocalPlaneMapValue getFormattedValue() {
            return (FocalPlaneMapValue) super.getFormattedValue();
        }
        
    }

}
