package org.lsst.ccs.subsystem.focalplane.ui.fpmap;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.Serializable;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.python.bouncycastle.util.Arrays;

/**
 * Graphical component that displays a color-coded map of the focal plane.
 *
 * @author onoprien
 */
public final class FocalPlaneMap extends JPanel {

// -- Fields : -----------------------------------------------------------------
    
    private Descriptor descriptor;
    
    private FocalPlaneMapModel model;
    private final FocalPlaneMapModelListener modelListener = this::onModelEvent;
    
    private int sizeLimit; // shorter side of the panel
    private int size, x0, y0; // painting area size and origin
    private int ccdLineWidth, raftLineWidth; // space between RAFTs and CCDs
    private int raftSize, ccdSize, ampWidth, ampHeight; // cell sizes
    
    private final int[] currentCell = new int[6];
    private FocalPlaneMapValue currentValue;

// -- Life cycle : -------------------------------------------------------------
    
    public FocalPlaneMap() {
        
        setBackground(Color.WHITE);
        setOpaque(true);
        setToolTipText("super");
        
        Arrays.fill(currentCell, -1);
        
//        MouseHandler mh = new MouseHandler();
//        addMouseMotionListener(mh);
    }
    
    public void setModel(FocalPlaneMapModel model) {
        if (this.model != null) {
            this.model.removeFocalPlaneMapModelListener(modelListener);
        }
        this.model = model;
        if (model == null) {
            setName("");
        } else {
            model.addFocalPlaneMapModelListener(modelListener);
            setName(model.getTitle());
        }
        repaint();
    }

//    @Override
//    public void removeNotify() {
//        super.removeNotify();
////        setModel(null);
//    }
    
    
// -- Painting : ---------------------------------------------------------------

    @Override
    protected void paintComponent(Graphics g) {
        super.paintComponent(g);
        
        Dimension dim = getSize();
        int newSize = Math.min(dim.width, dim.height);
        if (newSize != sizeLimit) {
            sizeLimit = newSize;
            ccdLineWidth = Math.max(sizeLimit / 200, 1);
            raftLineWidth = ccdLineWidth * 2;
            ccdSize = (sizeLimit - raftLineWidth * 6 - ccdLineWidth * 10) / 15; // 5 rafts, 15 ccds, 6 raft lines, 10 ccd lines
            ampWidth = ccdSize/2;
            ccdSize = ampWidth*2;
            ampHeight = ccdSize/8;
            raftSize = ccdSize * 3 + ccdLineWidth * 2;
            size = raftSize * 5 + raftLineWidth * 6;
            x0 = (dim.width - size) / 2;
            y0 = (dim.height - size) / 2;
        }
        
        FocalPlaneMapValue def = model == null ? FocalPlaneMapValue.EMPTY : model.getValue();
        if (def == null) def = FocalPlaneMapValue.EMPTY;
        
        for (int raftX=0; raftX<5; raftX++) {
            for (int raftY=0; raftY<5; raftY++) { // loop over rafts
                int xRaftPos = x0 + (4 - raftX) * (raftSize + raftLineWidth) + raftLineWidth;
                int yRaftPos = y0 + (4 - raftY) * (raftSize + raftLineWidth) + raftLineWidth;
                FocalPlaneMapValue raftValue = model == null ? FocalPlaneMapValue.EMPTY : model.getValue(raftX, raftY);
                if (raftValue != null && raftValue.isSplit()) { // paint a split raft
                    if (raftValue == FocalPlaneMapValue.NONE) raftValue = def;
                    for (int reb=0; reb<3; reb++) {
                        int xRebPos = xRaftPos + (2 - reb) * (ccdSize + ccdLineWidth);
                        FocalPlaneMapValue rebValue = model.getValue(raftX, raftY, reb);
                        if (rebValue != null && rebValue.isSplit()) { // paint a split reb
                            if (rebValue == FocalPlaneMapValue.NONE) rebValue = raftValue;
                            for (int ccdY = 0; ccdY < 3; ccdY++) {
                                int yCcdPos = yRaftPos + (2 - ccdY) * (ccdSize + ccdLineWidth);
                                if (isValid(raftX, raftY, reb, ccdY)) {
                                    FocalPlaneMapValue ccdValue = model.getValue(raftX, raftY, reb, ccdY);
                                    if (ccdValue != null && ccdValue.isSplit()) { // paint a split ccd
                                        if (ccdValue == FocalPlaneMapValue.NONE) ccdValue = rebValue;
                                        for (int ampX=0; ampX<2; ampX++) {
                                            int xAmpPos = xRebPos + ampWidth*(1-ampX);
                                            for (int ampY=0; ampY<8; ampY++) {
                                                int yAmpPos = yCcdPos + ampHeight*(7-ampY);
                                                FocalPlaneMapValue ampValue = model.getValue(raftX, raftY, reb, ccdY, ampX, ampY);
                                                if (ampValue == null || ampValue == FocalPlaneMapValue.NONE) ampValue = ccdValue;
                                                g.setColor(ampValue.getColor());
                                                g.fillRect(xAmpPos, yAmpPos, ampWidth, ampHeight);
                                            }
                                        }
                                    } else { // paint a whole ccd
                                        if (ccdValue == null) ccdValue = rebValue;
                                        g.setColor(ccdValue.getColor());
                                        g.fillRect(xRebPos, yCcdPos, ccdSize, ccdSize);
                                    }
                                }
                            }
                        } else { // paint a whole reb
                            if (rebValue == null) rebValue = raftValue;
                            g.setColor(rebValue.getColor());
                            g.fillRect(xRebPos, yRaftPos, ccdSize, raftSize);
                        }
                    }
                } else { // paint a whole raft
                    if (raftValue == null) raftValue = def;
                    g.setColor(raftValue.getColor());
                    if (raftX == 4 && raftY == 0) {
                        int[] xx = {xRaftPos + ccdSize + ccdLineWidth, xRaftPos + raftSize, xRaftPos + raftSize                , xRaftPos + raftSize - ccdSize      , xRaftPos + raftSize - ccdSize, xRaftPos + ccdSize + ccdLineWidth};
                        int[] yy = {yRaftPos                         , yRaftPos           , yRaftPos + 2*ccdSize + ccdLineWidth, yRaftPos + 2*ccdSize + ccdLineWidth, yRaftPos + ccdSize, yRaftPos + ccdSize};
                        g.fillPolygon(xx, yy, 6);
                    } else if (raftX == 4 && raftY == 4) {
                        int[] xx = {xRaftPos + ccdSize + ccdLineWidth, xRaftPos + raftSize, xRaftPos + raftSize              , xRaftPos + raftSize - ccdSize    , xRaftPos + raftSize - ccdSize, xRaftPos + ccdSize + ccdLineWidth};
                        int[] yy = {yRaftPos + raftSize              , yRaftPos + raftSize, yRaftPos + ccdSize + ccdLineWidth, yRaftPos + ccdSize + ccdLineWidth, yRaftPos + raftSize - ccdSize, yRaftPos + raftSize - ccdSize};
                        g.fillPolygon(xx, yy, 6);
                    } else if (raftX == 0 && raftY == 0) {
                        int[] xx = {xRaftPos + 2*ccdSize + ccdLineWidth, xRaftPos, xRaftPos, xRaftPos + ccdSize, xRaftPos + ccdSize, xRaftPos + 2*ccdSize + ccdLineWidth};
                        int[] yy = {yRaftPos                         , yRaftPos           , yRaftPos + 2*ccdSize + ccdLineWidth, yRaftPos + 2*ccdSize + ccdLineWidth, yRaftPos + ccdSize, yRaftPos + ccdSize};
                        g.fillPolygon(xx, yy, 6);
                    } else if (raftX == 0 && raftY == 4) {
                        int[] xx = {xRaftPos + 2*ccdSize + ccdLineWidth, xRaftPos, xRaftPos, xRaftPos + ccdSize, xRaftPos + ccdSize, xRaftPos + 2*ccdSize + ccdLineWidth};
                        int[] yy = {yRaftPos + raftSize              , yRaftPos + raftSize, yRaftPos + ccdSize + ccdLineWidth, yRaftPos + ccdSize + ccdLineWidth, yRaftPos + raftSize - ccdSize, yRaftPos + raftSize - ccdSize};
                        g.fillPolygon(xx, yy, 6);
                    } else {
                        g.fillRect(xRaftPos, yRaftPos, raftSize, raftSize);
                    }
                }
            }
        }
    }
    
    private boolean isValid(int raftX, int raftY, int ccdX, int ccdY) {
        switch (raftX) {
            case 0:
                switch (raftY) {
                    case 0:
                        return ccdX + ccdY > 2;
                    case 4:
                        return ccdX > ccdY;
                    default:
                        return true;
                }
            case 4:
                switch (raftY) {
                    case 0:
                        return ccdY > ccdX;
                    case 4:
                        return ccdX + ccdY < 2;
                    default:
                        return true;
                }
            default:
                return true;
        }
    }
    
    
// -- Responding to model changes : --------------------------------------------
    
    private void onModelEvent(FocalPlaneMapModelEvent event) {
        repaint(); // FIXME: implement partial repainting
    }
    
    
// -- Responding to mouse : ----------------------------------------------------
    
    /**
     * Sets {@code currentCell} and {@code currentValue} according to the specified position.
     * @param x X
     * @param y Y
     */
    private void setCurrentCell(int x, int y) {
        clearCurrentCell();
        if (model == null || size == 0) return;
        x = (x0 + size) - (x + raftLineWidth);
        y = (y0 + size) - (y + raftLineWidth);
        if (x < 0 || y < 0) return;
        
        // Find RAFT
        
        int s = raftSize + raftLineWidth;
        int raftX = x / s;
        if (raftX > 4) return;
        int raftY = y / s;
        if (raftX > 4) return;
        x = x - s * raftX;
        y = y - s * raftY;
        if (x >= raftSize || y >= raftSize) return;
        if (raftX == 0) {
            // FIXME: handle corner rafts correctly
        } else if (raftX == 4) {
            // FIXME: handle corner rafts correctly
        }
        currentCell[0] = raftX;
        currentCell[1] = raftY;
        currentValue = model.getValue(raftX, raftY);
        if (currentValue == null) {
            currentValue = model.getValue();
            return;
        } else if (currentValue == FocalPlaneMapValue.NONE) {
            currentValue = model.getValue();
        } else {
            if (!currentValue.isSplit()) return;
        }
        
        // Find REB
        
        s = ccdSize + ccdLineWidth;
        int reb = x / s;
        x = x - reb * s;
        if (x >= ccdSize || reb < 0 || reb > 2) {
            clearCurrentCell();
            return;
        }
        currentCell[2] = reb;
        FocalPlaneMapValue v = model.getValue(raftX, raftY, reb);
        if (v == null) {
            return;
        } else if (v == FocalPlaneMapValue.NONE) {
        } else if (v.isSplit()) {
            currentValue = v;
        } else {
            currentValue = v;
            return;
        }
        
        // Find CCD
        
        int ccdY = y / s;
        y = y - ccdY * s;
        if (y >= ccdSize || ccdY < 0 || ccdY > 2) {
            clearCurrentCell();
            return;
        }
        currentCell[3] = ccdY;
        v = model.getValue(raftX, raftY, reb, ccdY);
        if (v == null) {
            return;
        } else if (v == FocalPlaneMapValue.NONE) {
        } else if (v.isSplit()) {
            currentValue = v;
        } else {
            currentValue = v;
            return;
        }
        
        // Find AMP
        
        int ampX = x / ampWidth;
        y = y - (ccdSize - ampHeight * 8);
        int ampY = y / ampHeight;
        if (ampX < 0 || ampX > 1 || ampY < 0 || ampY > 8) {
            clearCurrentCell();
            return;
        }
        currentCell[4] = ampX;
        currentCell[5] = ampY;
        v = model.getValue(raftX, raftY, reb, ccdY, ampX, ampY);
        if (v != null) {
            currentValue = v;
        }
    }
    
    private void clearCurrentCell() {
        Arrays.fill(currentCell, -1);
        currentValue = null;
    }
    
    @Override
    public String getToolTipText(MouseEvent event) {
        setCurrentCell(event.getX(), event.getY());
        if (currentCell[0] == -1) return null;
        StringBuilder sb = new StringBuilder("<html>");
        sb.append("<h4>RAFT:").append(currentCell[0]).append(currentCell[1]);
        if (currentCell[2] != -1) {
            sb.append(", ");
            if (currentCell[3] == -1) {
                sb.append("Reb:").append(currentCell[2]);
            } else {
                sb.append("CCD:").append(currentCell[2]).append(currentCell[3]);
                if (currentCell[4] != -1) {
                    sb.append(", AMP:").append(currentCell[4]).append(currentCell[5]);
                }
            }
        }
        sb.append("</h4>");
        if (currentValue != null && currentValue.toolTip != null) {
            sb.append(currentValue.toolTip);
        }
        return sb.toString();
    }
    
// -- Saving : -----------------------------------------------------------------

    public Descriptor save() {
        return descriptor;
    }
    
    static public class Descriptor implements Serializable {
    }

}
