package org.lsst.ccs.subsystem.focalplane.ui;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.Serializable;
import java.util.*;
import java.util.function.Consumer;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.JComboBox;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.gconsole.agent.AgentChannelsFilter;
import org.lsst.ccs.gconsole.agent.AgentStatusEvent;
import org.lsst.ccs.gconsole.agent.AgentStatusListener;
import org.lsst.ccs.gconsole.annotations.Plugin;
import org.lsst.ccs.gconsole.base.ComponentDescriptor;
import org.lsst.ccs.gconsole.base.Console;
import org.lsst.ccs.gconsole.base.ConsolePlugin;
import org.lsst.ccs.gconsole.base.OptionalPage;
import org.lsst.ccs.gconsole.base.panel.Panel;
import org.lsst.ccs.gconsole.base.panel.PanelManager;
import org.lsst.ccs.gconsole.plugins.monitor.SectionedTableView;
import org.lsst.ccs.subsystem.focalplane.data.FocalPlaneDataGroup;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.ChannelSelectorDialog;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.DataGroupFactory;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.FakeModel3;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.FocalPlaneMapModel;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.FocalPlaneMapModelFactory;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.FocalPlaneMapPage;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.FocalPlaneMapPageDialog;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.MonitorViewModel;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.Segment;
import org.lsst.ccs.subsystem.focalplane.ui.fpmap.TemplateFilter;

/**
 * Focal plane GUI plugin.
 *
 * @author onoprien
 */
@Plugin(name="LSST CCS Focal Plane GUI",
        id="focal-plane-gui",
        description="LSST CCS Focal Plane Subsystem GUI")
public class FocalPlanePlugin extends ConsolePlugin {

// -- Fields : -----------------------------------------------------------------
    
    private final String DEFAULT_FOCAL_PLANE_MAP_PAGE_NAME = "Focal Plane";
    private FocalPlaneMapModelFactory[] focalPlaneMapModelFactories;
    
    private String focalPlaneAgentName;
    private final ArrayList<FocalPlaneMapPage> focalPlaneMapPages = new ArrayList<>();
    
    private AgentStatusListener statusListener;

// -- Life cycle : -------------------------------------------------------------
    
    public FocalPlanePlugin() {
    }
    
    @Override
    public void initialize() {
        OptionalPage.getService().add(new FocalPlaneMonitoringOptionalPage());
    }

    @Override
    public void start() {
        statusListener = new AgentStatusListener() {
            @Override
            public void connect(AgentStatusEvent event) {
                AgentInfo agent = event.getSource();
                if (agent.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, "").equals(AgentCategory.FOCAL_PLANE.name())) {
                    SwingUtilities.invokeLater(() -> {
                        focalPlaneAgentName = agent.getName();
                        ArrayList<FocalPlaneMapModelFactory> ff = new ArrayList<>();
                        ff.add(new FocalPlaneMapModelFactory() { // no model
                            @Override
                            public String toString() {
                                return "";
                            }

                            @Override
                            public FocalPlaneMapModel getModel(Serializable descriptor) {
                                return null;
                            }
                        });
                        ff.add(new FocalPlaneMapModelFactory() { // custom model
                            @Override
                            public String toString() {
                                return "Custom...";
                            }

                            @Override
                            public FocalPlaneMapModel getModel(Serializable descriptor) {
                                if (!(descriptor instanceof TemplateFilter.Descriptor)) {
                                    return null;
                                } else {
                                    MonitorViewModel view = new MonitorViewModel();
                                    view.setFilter(new TemplateFilter((TemplateFilter.Descriptor) descriptor));
                                    return view.getFocalPlaneMapModel();
                                }
                            }

                            @Override
                            public FocalPlaneMapModel createModel(Serializable descriptor) {
                                TemplateFilter.Descriptor oldDesc = descriptor instanceof TemplateFilter.Descriptor ? (TemplateFilter.Descriptor) descriptor : null;
                                TemplateFilter.Descriptor newDesc = ChannelSelectorDialog.show(oldDesc, null);
                                return getModel(newDesc == null ? oldDesc : newDesc);
                            }
                        }); // custom...
                        for (FocalPlaneDataGroup group : FocalPlaneDataGroup.values()) {
                            ff.add(new DataGroupFactory(focalPlaneAgentName, group));
                        }
//                    ff.add(new FakeModel3("Fake 3"));
                        focalPlaneMapModelFactories = ff.toArray(new FocalPlaneMapModelFactory[ff.size()]);
                        Action act = new AbstractAction("Custom...") {
                            @Override
                            public void actionPerformed(ActionEvent e) {
                                FocalPlaneMapPage.Descriptor d = FocalPlaneMapPageDialog.show(null, DEFAULT_FOCAL_PLANE_MAP_PAGE_NAME);
                                if (d != null) {
                                    openFocalPlaneMap(d);
                                }
                            }
                        };
                        getServices().addMenu(act, "CCS Subsystems", "Focal Plane", "Map");
                        addFocalPlaneMapMenu(1, 1);
                        addFocalPlaneMapMenu(1, 2);
                        addFocalPlaneMapMenu(2, 1);
                        addFocalPlaneMapMenu(2, 2);
                        addFocalPlaneMapMenu(1, 3);
                        addFocalPlaneMapMenu(2, 3);
                        addFocalPlaneMapMenu(3, 3);
                        addFocalPlaneMapMenu(4, 4);
                    });
                }
            }
            @Override
            public void disconnect(AgentStatusEvent event) {
                SwingUtilities.invokeLater(() -> {
                    if (event.getSource().getName().equals(focalPlaneAgentName)) {
                        focalPlaneAgentName = null;
                        getConsole().removeMenu("CCS Subsystems", "Focal Plane", "Map");
                        focalPlaneMapModelFactories = null;
                    }
                });
            }
        };
        getConsole().getStatusAggregator().addListener(statusListener, null, Collections.singletonList(Segment.RAFT.getKey()));
    }

    @Override
    public void stop() {
        getConsole().getStatusAggregator().removeListener(statusListener);
    }
    
    
// -- Handling focal plane maps : ----------------------------------------------
    
    private void openFocalPlaneMap(FocalPlaneMapPage.Descriptor descriptor) {
        FocalPlaneMapPage page = new FocalPlaneMapPage(focalPlaneMapModelFactories, descriptor);
        focalPlaneMapPages.add(page);
        Map<Object, Object> par = new HashMap<>();
        String title = descriptor.getName();
        if (title == null) title = DEFAULT_FOCAL_PLANE_MAP_PAGE_NAME;
        par.put(Panel.TITLE, title);
        Consumer<JComponent> onClose = c -> onClosingFocalPlaneMap((FocalPlaneMapPage)c);
        par.put(Panel.ON_CLOSE, onClose);
        Console.getConsole().getPanelManager().open(page, par);
        page.opened();
    }
    
    private void onClosingFocalPlaneMap(FocalPlaneMapPage page) {
        page.closed();
        focalPlaneMapPages.remove(page);
    }
        
    private void addFocalPlaneMapMenu(int rows, int columns) {
        Action act = new AbstractAction(columns +" x "+ rows) {
            @Override
            public void actionPerformed(ActionEvent e) {
                openFocalPlaneMap(new FocalPlaneMapPage.Descriptor(rows, columns));
            }
        };
        getServices().addMenu(act, "CCS Subsystems", "Focal Plane", "Map");
    }
    
    
// -- Standard monitoring page : -----------------------------------------------

    static class FocalPlaneMonitoringOptionalPage implements OptionalPage {

        @Override
        public boolean isAutoOpen() {
            return false;
        }

        @Override
        public String getPage(AgentInfo agent) {
            if (agent.getAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY,"").equals(AgentCategory.FOCAL_PLANE.name())) {
                return "CCS Subsystems/Focal Plane/Monitoring/"+ agent.getName();
            }
            return null;
        }

        @Override
        public JComponent open(AgentInfo agent, JComponent existingComponent) {
            FocalPlaneDataSelection dataSelection = new FocalPlaneDataSelection(agent);
            return dataSelection;

        }
    }
    
    static private final class FocalPlaneDataSelection extends JPanel {
        
        JComboBox dataSelector = new JComboBox(FocalPlaneDataGroup.values());
        SectionedTableView view = new SectionedTableView();
        private final AgentInfo agent;
        
        FocalPlaneDataSelection(AgentInfo agent) {
            this.agent = agent;
            setLayout(new BorderLayout());
            add(dataSelector,BorderLayout.PAGE_START);
            makeFilteredTable();
            
            dataSelector.addActionListener(e -> {
                remove(view.getPanel());
                view = new SectionedTableView();
                makeFilteredTable();
            });
        }
        
        final void makeFilteredTable() {
            FocalPlaneMonitorFilter monitorFilter = new FocalPlaneMonitorFilter(agent.getName(),(FocalPlaneDataGroup)(dataSelector.getSelectedItem()));
            view.uninstall();
            view.setFilter(monitorFilter);
            view.install();
            add(view.getPanel(),BorderLayout.CENTER);            
        }
        
    }
    
    
// -- Saving/restoring : -------------------------------------------------------

    @Override
    public boolean restore(ComponentDescriptor storageBean, boolean lastRound) {
        
        // Close existing pages:
        
        if (!focalPlaneMapPages.isEmpty()) {
            PanelManager panMan = Console.getConsole().getPanelManager();
            ArrayList<FocalPlaneMapPage> copy = new ArrayList<>(focalPlaneMapPages);
            for (FocalPlaneMapPage page : copy) {
                panMan.close(page);
            }
        }
        
        // Open saved pages:
        
        if (storageBean instanceof Descriptor) {
            Descriptor desc = (Descriptor) storageBean;
            FocalPlaneMapPage.Descriptor[] pages = desc.getMapPages();
            if (pages != null && pages.length != 0) {
                for (FocalPlaneMapPage.Descriptor pageDescriptor : pages) {
                    openFocalPlaneMap(pageDescriptor);
                }
            }
        }
        
        return true;
    }

    @Override
    public Descriptor save() {
        Descriptor desc = new Descriptor(getServices().getDescriptor());
        
        ArrayList<FocalPlaneMapPage.Descriptor> mapPages = new ArrayList<>(focalPlaneMapPages.size());
        for (FocalPlaneMapPage view : focalPlaneMapPages) {
            FocalPlaneMapPage.Descriptor d = view.save();
            if (d != null) {
                mapPages.add(d);
            }
        }
        if (!mapPages.isEmpty()) {
            desc.setMapPages(mapPages.toArray(new FocalPlaneMapPage.Descriptor[mapPages.size()]));
        }
        
        return desc;
    }

    static public class Descriptor extends ComponentDescriptor {

        private FocalPlaneMapPage.Descriptor[] mapPages;
        
        public Descriptor() {
        }
        
        public Descriptor(ComponentDescriptor seed) {
            super(seed);
        }

        public FocalPlaneMapPage.Descriptor[] getMapPages() {
            return mapPages;
        }

        public void setMapPages(FocalPlaneMapPage.Descriptor[] mapPages) {
            this.mapPages = mapPages;
        }

    }
    
}
