package org.lsst.ccs.subsystem.focalplane.data;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.lsst.ccs.bus.data.DataProviderInfo;

public enum FocalPlaneDataGroup {
    ASPIC_TEMPERATURE("Aspic Temperature", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/(?<aspicId>Aspic[UL])/Temp\\d")),
    REB_TEMPERATURE("Reb Temperature", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/Temp\\d+")),
    CCD_TEMPERATURE("CCD Temperature", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/(?<sensorId>S\\d\\d)/Temp")),
    CCD_CURRENTS("CCD Currents", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/(?<sensorId>S\\d\\d)/I(?<amplifierId>\\d\\d)")),
    CCD_VOLTAGES("CCD Voltages", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/(?<sensorId>S\\d\\d)/[ORG][DG]V")),
    BIAS_VOLTAGES("Bias Voltages", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/Ref\\d+V")),
    RTD_TEMPERATURE("RTD Temperature", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/RTDTemp")),
    CLOCK_RAILS("Clock Rails", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/([PS]Clk[UL]|RG[UL]|Ref[PN]12)")),
    BOARD_POWER("Board Power", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/(Dig[VI]|Ana[VI]|Clk[HL][VI]|Power|OD[VI])")),
    HEATER("Heater", Pattern.compile("(?<raftId>R\\d\\d)/(?<rebId>Reb\\d)/Htr[V|W]"));

    private final String name;
    private final Pattern pattern;
    
    static public FocalPlaneDataGroup forName(String name) {
        for (FocalPlaneDataGroup g : values()) {
            if (g.name.equals(name)) {
                return g;
            }
        }
        throw new IllegalArgumentException("No FocalPlaneDataGroup constant with "+ name +" name.");
    }

    private FocalPlaneDataGroup(String name, Pattern pattern) {
        this.name = name;
        this.pattern = pattern;
    }

    public String getName() {
        return name;
    }

    public boolean matchesData(DataProviderInfo dataInfo) {
        return pattern.matcher(dataInfo.getPath()).matches();
    }

    public static FocalPlaneDataGroup findFocalPlanDataGroup(DataProviderInfo dataInfo) {
        for (FocalPlaneDataGroup dataGroup : FocalPlaneDataGroup.values()) {
            if (dataGroup.matchesData(dataInfo)) {
                return dataGroup;
            }
        }
        return null;
    }

    public void addAttributesToDataInfo(DataProviderInfo dataInfo) {
        Matcher m = pattern.matcher(dataInfo.getPath());
        if ( m.matches() ) {
            dataInfo.addAttribute(DataProviderInfo.Attribute.DATA_GROUP, getName());
            try {
                String raftId = m.group("raftId");
                if (raftId != null) {
                    dataInfo.addAttribute(DataProviderInfo.Attribute.RAFT_ID, raftId);
                }
            } catch (Exception e) {
            }
            try {
                String rebId = m.group("rebId");
                if (rebId != null) {
                    dataInfo.addAttribute(DataProviderInfo.Attribute.REB_ID, rebId);
                }
            } catch (Exception e) {
            }
            try {
                String aspicId = m.group("aspicId");
                if (aspicId != null) {
                    dataInfo.addAttribute(DataProviderInfo.Attribute.ASPIC_ID, aspicId);
                }
            } catch (Exception e) {
            }
            try {
                String sensorId = m.group("sensorId");
                if (sensorId != null) {
                    dataInfo.addAttribute(DataProviderInfo.Attribute.SENSOR_ID, sensorId);
                }
            } catch (Exception e) {
            }
            try {
                String sensorId = m.group("amplifierId");
                if (sensorId != null) {
                    dataInfo.addAttribute(DataProviderInfo.Attribute.AMPLIFIER_ID, sensorId);
                }
            } catch (Exception e) {
            }
        }
        
    }
    
    public String getRaftId(DataProviderInfo dataInfo) {
        try {
            Matcher m = pattern.matcher(dataInfo.getPath());
            return m.matches() ? m.group("raftId") : null;
        } catch (Exception e) {
            return null;
        }
    }

    public String getRebId(DataProviderInfo dataInfo) {
        try {
            Matcher m = pattern.matcher(dataInfo.getPath());
            return m.matches() ? m.group("rebId") : null;
        } catch (Exception e) {
            return null;
        }
    }

    public String getAspicId(DataProviderInfo dataInfo) {
        try {
            Matcher m = pattern.matcher(dataInfo.getPath());
            return m.matches() ? m.group("aspicId") : null;
        } catch (Exception e) {
            return null;
        }
    }

    public String getSensorId(DataProviderInfo dataInfo) {
        try {
            Matcher m = pattern.matcher(dataInfo.getPath());
            return m.matches() ? m.group("sensorId") : null;
        } catch (Exception e) {
            return null;
        }
    }

}
