package org.lsst.ccs.subsystem.focalplane.data;

import java.io.Serializable;
import java.util.Arrays;
import org.lsst.ccs.bus.annotations.DataAttributes;
import org.lsst.ccs.bus.annotations.DoNotTrend;
import org.lsst.ccs.bus.annotations.SkipEncoding;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.location.Location;
import org.lsst.ccs.utilities.location.LocationSet;

/**
 * An event sent by focal-plane shortly after readout starts, containing
 * meta-data to be forwarded to SAL.
 *
 * @author tonyj
 */
@SkipEncoding
//To be uncommented with toolkit 5.3.15
//@DoNotTrend
@DataAttributes(units = "unitless")
public class ImageReadoutParametersEvent implements Serializable {
    
    private static final long serialVersionUID = -68951332963342L;

    public static String EVENT_KEY = "ImageReadoutParameters";

    //Which fields should be registered?
    private final ImageName imageName;
    private final String[] ccdType;
    private final int[] overRows;
    private final int[] overCols;
    private final int[] readRows;
    private final int[] readCols;
    private final int[] readCols2;
    private final int[] preCols;
    private final int[] preRows;
    private final int[] postCols;
    private final int[] postRows;
    private final int[] underCols;
    private final String annotation;
    private final LocationSet locations;
    private final int currentOpcode;
    private final String daqFolder;

    public ImageReadoutParametersEvent(ImageName imageName, String annotation, LocationSet locations, int currentOpcode, String daqFolder) {
        this.imageName = imageName;
        this.annotation = annotation;
        this.locations = new LocationSet(locations);
        this.currentOpcode = currentOpcode;
        this.daqFolder = daqFolder;
        int nLocations = locations.size();
        this.ccdType = new String[nLocations];
        this.overRows = new int[nLocations];
        this.overCols = new int[nLocations];
        this.readRows = new int[nLocations];
        this.readCols = new int[nLocations];
        this.readCols2 = new int[nLocations];
        this.preCols = new int[nLocations];
        this.preRows = new int[nLocations];
        this.postCols = new int[nLocations];
        this.postRows = new int[nLocations];
        this.underCols = new int[nLocations];
    }

    public void setParameters(Location location, CCDType ccdType, int preCols, int underCols, int readCols, int postCols, int readCols2, int overCols, int preRows, int readRows, int postRows, int overRows) {
        int currentLocation = indexOf(location, locations);
        this.ccdType[currentLocation] = ccdType.getManufacturer();
        this.overRows[currentLocation] = overRows;
        this.overCols[currentLocation] = overCols;
        this.underCols[currentLocation] = underCols;
        this.readRows[currentLocation] = readRows;
        this.readCols[currentLocation] = readCols;
        this.readCols2[currentLocation] = readCols2;
        this.preCols[currentLocation] = preCols;
        this.preRows[currentLocation] = preRows;
        this.postCols[currentLocation] = postCols;
        this.postRows[currentLocation] = postRows;
        this.underCols[currentLocation] = underCols;
    }

    private int indexOf(Location location, LocationSet locations) {
        int i=0;
        for (Location l : locations) {
            if (l.equals(location)) { 
                return i;
            }
            i++;
        }
        return -1;
    }

    public ImageName getImageName() {
        return imageName;
    }

    public String[] getCcdType() {
        return ccdType;
    }

    public int[] getOverRows() {
        return overRows;
    }

    public int[] getOverCols() {
        return overCols;
    }

    public int[] getReadRows() {
        return readRows;
    }

    public int[] getReadCols() {
        return readCols;
    }

    public int[] getReadCols2() {
        return readCols2;
    }

    public int[] getPreCols() {
        return preCols;
    }

    public int[] getPreRows() {
        return preRows;
    }

    public int[] getPostCols() {
        return postCols;
    }

    public int[] getPostRows() {
        return postRows;
    }

    public int[] getUnderCols() {
        return underCols;
    }

    public String getAnnotation() {
        return annotation;
    }

    public LocationSet getLocations() {
        return locations;
    }

    public int getCurrentOpcode() {
        return currentOpcode;
    }

    public String getDaqFolder() {
        return daqFolder;
    }

    @Override
    public String toString() {
        return "ImageReadoutParametersEvent{" + "imageName=" + imageName + ", ccdType=" + Arrays.toString(ccdType) + ", overRows=" + Arrays.toString(overRows) + ", overCols=" + Arrays.toString(overCols) + ", readRows=" + Arrays.toString(readRows) + ", readCols=" + Arrays.toString(readCols) + ", readCols2=" + Arrays.toString(readCols2) + ", preCols=" + Arrays.toString(preCols) + ", preRows=" + Arrays.toString(preRows) + ", postCols=" + Arrays.toString(postCols) + ", postRows=" + Arrays.toString(postRows) + ", underCols=" + Arrays.toString(underCols) + ", annotation=" + annotation + ", locations=" + locations + ", currentOpcode=" + currentOpcode + ", daqFolder=" + daqFolder + '}';
    }
    
}
