
package org.lsst.ccs.subsystems.fcs.testbench;

import org.lsst.ccs.subsystems.fcs.drivers.CanOpenNode;
import org.lsst.ccs.subsystems.fcs.drivers.ApcTcpProxy;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystem.proxies.HardwareControlNotReady;
import org.lsst.ccs.subsystems.fcs.errors.BootingHardwareError;
import org.lsst.ccs.subsystems.fcs.errors.ReadingADCError;

/**
 * This is the main module for the control software of the APC test bench.
 * 
 * When the TestBenchMainModule starts, it starts a a TCP-IP server.
 * the can open wrapper software will connect. 
 * Then the TestBenchProxy and the Wrapper will communicate over this TCP connection 
 * with a protocol to be define.
 * In a first version :
 * 1- The TCP server is started and waits for a client connection,
 * 2- When the client connects, it sends first a key word "APCTestBench".
 * 3- When the server receives the key word "APCTestBench", it's ready to boot the hardware 
 * and receive commands.
 * @author virieux
 */
public class TestBenchMainModule extends Module  {
    

    private ApcTcpProxy tcpProxy;
    /*This is the key word that has to be sent by the client when it connects to the tcp proxy. */
    private String myClientName = "APCTestBench";
    /*number of can open nodes expected on the device */
    private int nodeNB = 2;
    /*number of booted nodes*/
    private int bootednodeNB = 0;
    /*number of received command with "info"*/
    private int infoNB = 0;
    /*An array of booted hardware*/
    private CanOpenNode[] nodes = new CanOpenNode[nodeNB];
    private boolean hardwareBooted = false;
    private boolean adcInitialized = false;
    /**
     * One bit read on the input of the ADC represents 0.3125 mV
     */
    private static double adcVoltsPerBit = 0.3125;
    private static double waterFreezeTemp = 273.15;
    
    /**
     * @return the myTcpProxy
     */
    public ApcTcpProxy getTcpProxy() {
        return tcpProxy;
    }

    /**
     * @param myTcpProxy the myTcpProxy to set
     */
    public void setTcpProxy(ApcTcpProxy myTcpProxy) {
        this.tcpProxy = myTcpProxy;
    }

    /**
     * @return the myClientName
     */
    public String getMyClientName() {
        return myClientName;
    }

    /**
     * @param myClientName the myClientName to set
     */
    public void setMyClientName(String myClientName) {
        this.myClientName = myClientName;
    }
    
    @Override
    public void initModule() {
       nodes = new CanOpenNode[nodeNB];
    }
    
    @Override
    public void tick() {
        //INITIALISATION 
        //first we want to check the hardware is booted.
        if (!hardwareBooted && tcpProxy.isReady(myClientName)) {
            try {
                bootHardware();
            } catch (BootingHardwareError ex) {
                Logger.getLogger(TestBenchMainModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        
        //we want to confifure the ADC so it sends its data every SYNC.
        if (hardwareBooted && !adcInitialized) {
            try {
                initADC(1);
                adcInitialized = true;
                log.debug("ADC is initialized");
            } catch (ReadingADCError ex) {
                Logger.getLogger(TestBenchMainModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        
    }
    
    /**
     * Returns the value read on the socket for the input of the ADC given as an argument.
     * @param responseLine A string read on the socket.
     * @param inputNB  The numero if the input on the ADC. (1<=inputNB<=8)
     * @return the value (int) given 
     * @throws ReadingADCError 
     */
    public int getReturnValue(String responseLine, int inputNB) throws ReadingADCError {
            if (inputNB <1 || inputNB >8) throw new IllegalArgumentException();
            String[] words = ((String) responseLine).split(",");
            String command = words[0];
            log.debug("Value in HEXA:" + words[inputNB]);
            int value = Integer.parseInt(words[inputNB],16);
            log.debug("Value in decimal:" + words[inputNB]);
            return value;
    }
    
    /**
     * Read the temperature : in the test bench, a thermometer is plugged on the input number 1
     * of the ADC.
     * @return 
     * @throws ReadingADCError 
     */
    public double readTemperature() throws ReadingADCError {
        int value = getReturnValue(readADC(),1);
        log.debug("Input Value read on the ADC:");
        double voltage = value * TestBenchMainModule.adcVoltsPerBit;
        log.debug("Voltage = " + voltage + " volts");
        double temperature = (voltage / 10) - TestBenchMainModule.waterFreezeTemp;
        return temperature;
    }
    
    /**
     * Read the values of the 8 channels of the ADC.
     * It just send the command "adc" to the Can Open Wrapper and the Can OPen Wrapper sends 
     * a SYNC to the ADC node.
     * @return
     * @throws ReadingADCError 
     */
    public String readADC() throws ReadingADCError {
        Object response = null;
        log.debug("Nom du client= " + myClientName);
        if (!tcpProxy.isReady(myClientName)) throw new HardwareControlNotReady(myClientName); 
        response = getTcpProxy().call(myClientName, "adc,");
        if (response instanceof String) {           
                log.debug("Reponse= " + response); 
                return (String) response;
        } else throw new ReadingADCError(response);
    }
    
    /*just for debug purpose*/
    public String readFromSocket(String clientName) {
        String response = null;
        response = tcpProxy.readFromSocket(clientName);
        log.debug("READ Reponse= " + response);
        return response;
    }
    
    
    public void turnOnLED() {
        if (!tcpProxy.isReady(myClientName)) throw new HardwareControlNotReady(getMyClientName());
        getTcpProxy().call(getMyClientName(), "dac,3000,0000,0000,0000");
    }
    
    public void turnOffLED() {
        if (!tcpProxy.isReady(myClientName)) throw new HardwareControlNotReady(getMyClientName());
        getTcpProxy().call(getMyClientName(), "dac,0,0,0,0");
    }
    
    
    
    public void stopAutomaticMode() {
        throw new UnsupportedOperationException("Not supported yet.");
    }

    public void stop() {
        this.getTcpProxy().stop();
    }

    /**
     * This method listens to the Socket and waits for the reading of 
     * a "boot" line and a "info" line for each can open nodes.
     * Then it creates an array (nodes) with all the booted nodes.
     */
    private void bootHardware() throws BootingHardwareError {
       
       //TODO there must be a timeout.
       while (this.bootednodeNB < this.nodeNB || this.infoNB < this.nodeNB) {
                String bootLine = tcpProxy.readFromSocket(myClientName);
                String[] words = bootLine.split(",");
                String command = words[0];
                int nodeID = Integer.parseInt(words[1]);
                log.debug("Received on socket command = " + command + "," + nodeID);
                if (command.equals("boot")) {                      
                    nodes[bootednodeNB] = new CanOpenNode(nodeID); 
                    bootednodeNB++;
                    log.debug("Node " + nodeID + " added");
                    log.debug("Hardware booted nb=" + bootednodeNB);
                } else if (command.equals("info")) {
                    int type = Integer.parseInt(words[2]);
                    int vendor = Integer.parseInt(words[3]);
                    int productCode = Integer.parseInt(words[4]);
                    int revision = Integer.parseInt(words[5]);
                    addNodeInfo(nodeID,type,vendor,productCode,revision);
                    this.infoNB++;
                    log.debug("Node Info" + nodeID + " added");
                } //TODO what if we receive a command which is no "boot" neither "info" before the end
                  //of the booting process ?                            
            }
       if (bootednodeNB == nodeNB && infoNB == nodeNB) {
           hardwareBooted = true;
       } else throw new BootingHardwareError(myClientName);
      
    }


    private void addNodeInfo(int nodeID, int type, int vendor, int productCode, int revision) throws BootingHardwareError  {
        
        boolean nodeFound = false;
        for (int i=0; i < nodes.length; i++ ) {
            if (!(nodes[i] == null) &&  (nodes[i].getNodeID() == nodeID)) {
                nodes[i].setNodeInfo(type, vendor, productCode, revision);
                nodeFound = true;
            }
        }
        if (!nodeFound) throw new BootingHardwareError(myClientName, nodeID);       
    }

    /**
     * List the can open nodes which are booted.
     * It means that we have received  a "boot,nodeID" and a "info,nodeID" messages
     * for these nodes.
     * @return the list of can open nodes and the information stored in this.nodes.
     */
    public String listNodes() {
        
        StringBuilder sb = new StringBuilder("Nodes LIST = ");
        for (int i=0; i < nodes.length; i++) {
            sb.append("\n");
            sb.append(nodes[i].toString()) ;
        }
        log.debug(sb.toString());
        return sb.toString();
    }

    /**
     * Initialize the ADC and configure the way it will send its data : 
     * we want it to send data each time it receives syncNB SYNC.
     * if syncNB = 1, it will send data every SYNC,
     * if syncNB = 2, it will send data every 2 SYNC, etc...
     * @param syncNB 
     */
    private void initADC(int syncNB) throws ReadingADCError {
        //if (!hardwareBooted) throw new HardwareNotBootedError();
        //for the first 4 INPUT channels from 1 to 4
        Object ack = getTcpProxy().call(getMyClientName(), "wsdo,1,1801,2,1,1");
        if (ack instanceof String) {           
                log.debug("initADC-step1 = " + ack); 
                Object ack2 = getTcpProxy().call(getMyClientName(), "wsdo,1,1802,2,1,1");
                log.debug("initADC-step2 = " + ack2); 
        } else throw new ReadingADCError(ack);
        
    }
    
    /**
     * For engineering mode, this method can be used to send Can Open commands
     * to the Wrapper. 
     * Not very useful as we can send the same command from the console with:
     * invoke apctestbench/tcpProxy call APCTestBench command
     * @param command A Can Open command that the Wrapper should understand.
     * @return the response from the Wrapper
     */
    public Object sendCanOpen(String command) {
        return getTcpProxy().call(getMyClientName(), command);
    }


    
}
