/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.subsystems.fcs.drivers;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.Actuator;

/**
 * This is a model for the clamp actuator in the Single Filter Test.
 * The actuator is used to unlock or release a clamp.
 * To unlock a clamp, the actuator must be ON and to release the clamp
 * it has to be OFF.
 * There is 2 clamps actuators on the carousel : 
 * one for side X-, one for side X+
 * For the Single Filter Test the actuator will be Maxon Motor EPOS2 24/5
 * There is many operation modes for this actuators. 
 * For the Single Filter Test, we will use VELOCITY mode or CURRENT mode.
 * 
 * @author virieux
 */
public abstract class ClampActuatorModule extends Module implements Actuator {
    
    /*limit values for parameters in VELOCITY mode*/
    static int MAXI_VELOCITY = 0;
    static int MAXI_ACCELERATION = 0;
    
    /*limit values for parameters in CURRENT mode*/
    static int MAXI_CURRENTLIMIT = 0;
    static int MAXI_SPEED = 0;
    static int MAXI_CONSTANTWINDING = 0;
    
    CanOpenProxy carouselProxy;

    /*
     * Node ID : the Can Open node ID of the DAC.
     */
    int nodeID;
    
    /*
     * The identifier number of the input on the DAC.
     */
    int inputID;
    
    /* Mode of Operation */
    ActuatorMode mode;
    
    /*parameters for Velocity mode*/
    int maxProfileVelocity;
    int maxAcceleration;
    
    /*parameters for Current mode*/
    int continuousCurrentLimit;
    int maxSpeedInCurrentMode;
    int thermalTimeConstantWinding;
    
    
    private int currentValue; 
    private int velocityValue;
    
    private boolean isOn;
    
    public ClampActuatorModule(int nodeID, int inputID) {
        this.carouselProxy = (CanOpenProxy) this.getModule("carouselProxy");
        this.nodeID = nodeID;
        this.inputID = inputID;
        this.mode = ActuatorMode.CURRENT;
        this.maxProfileVelocity = MAXI_VELOCITY;
        this.maxAcceleration = MAXI_ACCELERATION;
        this.continuousCurrentLimit = MAXI_CURRENTLIMIT;
        this.maxSpeedInCurrentMode = MAXI_SPEED;
        this.thermalTimeConstantWinding = MAXI_CONSTANTWINDING; 
        this.currentValue = 0;
        this.velocityValue = 0;
        this.isOn = false;
    }

    /**
     * @return the currentValue
     */
    public int getCurrentValue() {
        return currentValue;
    }

    /**
     * @param currentValue the currentValue to set
     */
    public void setCurrentValue(int currentValue) {
        this.currentValue = currentValue;
    }

    /**
     * @return the velocityValue
     */
    public int getVelocityValue() {
        return velocityValue;
    }

    /**
     * @param velocityValue the velocityValue to set
     */
    public void setVelocityValue(int velocityValue) {
        this.velocityValue = velocityValue;
    }
    
    public enum ActuatorMode {
	VELOCITY, CURRENT, 
    }
    
    public void initModule() {
        carouselProxy = (CanOpenProxy) this.getModule("carouselProxy");
    }
  
    public void enable () {
        /*shutdown*/
        writeDAC(6040,00,0X0006); 
        /*Switch-On*/
        writeDAC(6040,00,0X000F);
    }
    
    /*Methods available in VELOCITY mode*/ 
    public void setMaxiVelocity(int velocity) {
        if (velocity > MAXI_VELOCITY) 
            throw new IllegalArgumentException(getName() + ": can't set parameters greater than maxi value");
        this.maxProfileVelocity = velocity;
        writeDAC(0x607F,00,velocity);
    }
    
    public void setMaxiAcceleration(int acceleration) {
        if (acceleration > MAXI_ACCELERATION ) 
            throw new IllegalArgumentException(getName() + ": can't set parameters greater than maxi value");
        this.maxAcceleration = acceleration;
        writeDAC(0x60C5,00,acceleration);
    }
    
    public void setVelocity() throws BadCommandException {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            writeDAC(0x206B,00,this.velocityValue);
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");

    }
    
    public void stopVelocity() throws BadCommandException {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            writeDAC(0x206B,00,0);
        } else throw new BadCommandException(getName() + "is not in VELOCITY mode");
        
    }
    
    /*Methods available in CURRENT mode*/ 
    public void setCurrent() throws BadCommandException {
        if (mode.equals(ActuatorMode.CURRENT)) {
            writeDAC(0x2030,00,this.currentValue);
        } else throw new BadCommandException(getName() + "is not in CURRENT mode");
    }
    
    public void stopCurrent() throws BadCommandException {
        if (mode.equals(ActuatorMode.CURRENT)) {
            writeDAC(0x2030,00,0);
        } else throw new BadCommandException(getName() + "is not in CURRENT mode");
        
    }
    
    
    public void quickStop() {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            writeDAC(0x6040,00,0x000B);
        } else if (mode.equals(ActuatorMode.CURRENT)) {
            writeDAC(0x6040,00,0x0002);
        } // else throw an exception
    }
    

    public String on() {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            try {
                setVelocity();
            } catch (BadCommandException ex) {
                Logger.getLogger(ClampActuatorModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        } else if (mode.equals(ActuatorMode.CURRENT)) {
            try {
                setCurrent();
            } catch (BadCommandException ex) {
                Logger.getLogger(ClampActuatorModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        this.isOn = true;
        return getName() + "Actuator ON";
    }

    public String off() {
        if (mode.equals(ActuatorMode.VELOCITY)) {
            try {
                stopVelocity();
            } catch (BadCommandException ex) {
                Logger.getLogger(ClampActuatorModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        } else if (mode.equals(ActuatorMode.CURRENT)) {
            try {
                stopCurrent();
            } catch (BadCommandException ex) {
                Logger.getLogger(ClampActuatorModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        this.isOn = false;
        return getName() + "Actuator OFF";
    }

    public boolean isOn() {
        return isOn;
    }

    public boolean isOff() {
        return !isOn;
    }
    
    public int writeDAC(int index, int subindex,int newValue) {
        return carouselProxy.writeDAC(nodeID,inputID,index,subindex,newValue);
    }
    
}
