package org.lsst.ccs.subsystems.fcs;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.common.FilterClamp;
import org.lsst.ccs.subsystems.fcs.common.FilterClampState;
import org.lsst.ccs.subsystems.fcs.common.FilterPresenceStateOnClamp;
import org.lsst.ccs.subsystems.fcs.common.Sensor12bits;
import org.lsst.ccs.subsystems.fcs.errors.SensorValueOutOfRangeException;

/**
 * An abstract class which extends Module to model a clamp that holds a filter on the carousel.
 * It implements pattern state.
 * See <a href="CarouselClampState.png"> the state machine diagram for a carousel clamp.</a>
 * Each clamp on the carousel is coupled with a sensor which detects the presence of a filter.
 * This abstract class provides a method to lock, unlock or release the clamps, and to compute the state 
 * of the clamp. 
 * The state of the clamp is computed, updated and published on the status bus each tick of the timer.
 * 
 * 
 * @author virieux
 */
public abstract class FilterClampModule extends Module implements FilterClamp {

    protected FilterClampState state;
    protected FilterPresenceStateOnClamp filterPresence;
    
    /**
     * The filterPresenceSensor detects where is the filter within the clamp.
     * @See enum PresenceFilterStateOnClamp
     */
    private Sensor12bits filterPresenceSensor;
    
    /**
     * if the filter presence sensor returns a value between 0 and valueA : 
     * the sensor is in error.
     */
    private int valueA;
    
    /**
     * if filter presence sensor returns a value between valueA and valueB : 
     * the filter is engaged and lockable.
     */
    private int valueB;
    
       /**
     * if filter presence sensor returns a value between valueB and valueC : 
     * we don't know. And if the value is greater than valuC, there's no filter.
     */   
    private int valueC;
    
    
    public String publishedByClampOutputName = "publishedByClamp";
    
    /**
     * what the clamp publishes on the status bus.
     */
    public StatusDataPublishedByClamp publishedByClamp;
    
     /* This is used when we update the clamp state with the values returned 
     *  by the sensors.
     */
    private volatile boolean updateCompleted = true;

    /**
     * @param valueA the valueA to set
     */
    public void setValueA(int valueA) {
        this.valueA = valueA;
    }

    /**
     * @param valueB the valueB to set
     */
    public void setValueB(int valueB) {
        this.valueB = valueB;
    }

    /**
     * @param valueC the valueC to set
     */
    public void setValueC(int valueC) {
        this.valueC = valueC;
    }

    @Override
    public FilterClampState getState() {
        return state;
    }

    /**
     * @param state the state to set
     */
    @Override
    public void setState(FilterClampState state) {
        this.state = state;
    }

    /**
     * @return the updateCompleted
     */
    protected boolean isUpdateCompleted() {
        return updateCompleted;
    }

    /**
     * @param updateCompleted the updateCompleted to set
     */
    protected void setUpdateCompleted(boolean updateCompleted) {
        this.updateCompleted = updateCompleted;
    }
    


    /**
     * @return the valueA
     */
    public int getValueA() {
        return valueA;
    }

    /**
     * @return the valueB
     */
    public int getValueB() {
        return valueB;
    }

    /**
     * @return the valueC
     */
    public int getValueC() {
        return valueC;
    }
    
     /**
     * For encapsulation : we want to access to filterPresenceSensor in sub-packages.
     * @return the filterPresenceSensor
     */
    public Sensor12bits getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    /**
     * for Spring
     * @param filterPresenceSensor the filterPresenceSensor to set
     */
    public void setFilterPresenceSensor(Sensor12bits filterPresenceSensor) {
        this.filterPresenceSensor = filterPresenceSensor;
    }
    
    public synchronized boolean isFilterEngaged() {
        return (this.filterPresence.equals(FilterPresenceStateOnClamp.ENGAGED));
    }

    /**
     * Method to be overrided in the concrete classes :
     * It depends on the type of the clamp.
     * @return 
     */
    @Override
    public abstract boolean isLocked();

    /**
     * This method updates the clamp state regarding the value returned by the 
     * filter presence sensor and the value returned by the method isLocked().
     */
    
    @Override
    public synchronized void updateStateWithSensors() {
        
        setUpdateCompleted(false);
        
        this.filterPresenceSensor.updateValue();
        

        if (this.filterPresenceSensor.getValue() >= this.getValueC()) {
            this.filterPresence = FilterPresenceStateOnClamp.NOFILTER;
        } else if (this.filterPresenceSensor.getValue() >= this.getValueB()) {
            this.filterPresence = FilterPresenceStateOnClamp.UNKNOWN;          
        } else if (this.filterPresenceSensor.getValue() >= this.getValueA()) {
            this.filterPresence = FilterPresenceStateOnClamp.ENGAGED;
        } else {
            try {
                throw new SensorValueOutOfRangeException(this.filterPresenceSensor,this.filterPresenceSensor.getValue());
            } catch (SensorValueOutOfRangeException ex) {
                Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, "should not append", ex);
            }
        }
       
        
        // a filter is in the socket
        if (this.filterPresence.equals(FilterPresenceStateOnClamp.ENGAGED)) {
            if (isLocked()) {
                state = (FilterClampState) FilterClampState.CLAMPEDONFILTER;
            } else {
                state = (FilterClampState) FilterClampState.UNCLAMPEDONFILTER;
            }
        // no filter
        } else {
            if (isLocked()) {
                state = (FilterClampState) FilterClampState.READYTOCLAMP;
            } else {
                state = (FilterClampState) FilterClampState.UNCLAMPEDEMPTY;
            }

        }
        
        setUpdateCompleted(true);

    }




    @Override
    public synchronized void tick() {
       //TODO when the carousel is not at standby position, it's not necessary to read the sensors.
       updateStateWithSensors();
       log.debug(getName() + " : filterPresence =" + filterPresence.toString());
       this.publishedByClamp.update(this);
       setChanged();
       notifyObservers(new ValueUpdate(publishedByClampOutputName,this.publishedByClamp));
    }

    @Override
    public void initModule() {
        initPublishedData();
        this.filterPresence = FilterPresenceStateOnClamp.UNKNOWN;
    }

    public void initPublishedData() {
	this.publishedByClamp = new StatusDataPublishedByClamp();
    }


    @Override
    public StatusDataPublishedByClamp getStatusData() {
         this.publishedByClamp.update(this);
         return this.publishedByClamp;
    }
    
//This could be useful only in Engineering mode.
//    public String lock(Actuator actuator) throws BadCommandException, ErrorInCommandExecutionException {
//        log.info(getName() + ": is going to be locked.");
//        actuator.off();
//        return state.lock(this);
//    }

    /**
     * The clamps on the carousel are locked automaticaly when the filter comes
     * at the standby position. To be able to lock automaticaly again, it has to be
     * released after each time it has been unlocked.
     *
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     */
    @Override
    public String release(Actuator actuator) throws BadCommandException, ErrorInCommandExecutionException {

        updateStateWithSensors();

        if (!state.equals(FilterClampState.UNCLAMPEDEMPTY)) 
            throw new BadCommandException("Can't release a clamp if isn't unclamped and empty.");
        
        actuator.off();
        while (state.equals(FilterClampState.UNCLAMPEDEMPTY)) {
            try {
                Thread.sleep(this.tickMillis);
                updateStateWithSensors();
            } catch (InterruptedException ex) {
                Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        }  
        
        if (state.equals(FilterClampState.READYTOCLAMP)) {
            return getName() + " is released";
        } else {
            throw new ErrorInCommandExecutionException(getName() + ": Could not release clamp.");
        }
     
        
    }

    /**
     * Unlock the clamp when a filter is locked by the clamp.
     * @return
     * @throws BadCommandException
     * @throws ErrorInCommandExecutionException
     */
    @Override
    public String unlock(Actuator actuator) 
            throws BadCommandException, ErrorInCommandExecutionException {
        
        log.debug(getName() + "State1 = " + state.toString());
        updateStateWithSensors();
        
        if (!state.equals(FilterClampState.CLAMPEDONFILTER)) 
            throw new BadCommandException("Can't unlock a clamp if isn't clamped on filter.");
        
        actuator.on();
               
        while (state.equals(FilterClampState.CLAMPEDONFILTER)) {
            try {
                Thread.sleep(this.tickMillis);
                updateStateWithSensors();
            } catch (InterruptedException ex) {
                Logger.getLogger(FilterClampModule.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        
        log.debug(getName() + "State2 = " + state.toString());
        
        if (state.equals(FilterClampState.UNCLAMPEDONFILTER)) {
            return getName() + " is unlocked";
        } else {
            throw new ErrorInCommandExecutionException(getName() + ": Could not unlock clamp.");
        }
        
    }






    
	
}
