
package org.lsst.ccs.subsystems.fcs;

import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.subsystems.fcs.errors.HardwareErrorDetectedException;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.subsystems.fcs.common.FilterClamp;
import org.lsst.ccs.subsystems.fcs.common.FilterClampState;
import org.lsst.ccs.subsystems.fcs.common.FilterLocation;


/**
 * This is a socket on the carousel : there is 5 sockets on a carousel.
 * When a filter is on the carousel, it is attached at a socket.
 * 
 * @author virieux
 *
 */
public class CarouselSocket {

    /**
    * The clamps to hold the filter in the socket.
    * clampX+ and clampX- are the same except for the method isLocked()
    * See FilterClampXminusModule and FilterClampXplusModule
    */

    private FilterClamp clampXminus;

    private FilterClamp clampXplus;

    /**
     * socket position (angle) in degrees on the carousel
     * 
     */
    private double position;

    /**
     * carousel position (angle) when this socket is at standby position
     * should be 360 - position, but could be different
     */
    private double standbyPosition;

    private Filter filter;
    
    /*
     * This is used by the method updateClampsStateWithSensors.
     * This is the minimum number of milliseconds the socket waits when it's 
     * updating the clamps state.
     */
    private int tickMillis = 500;


    public CarouselSocket() {}

    /**
     * This constructor is for tests.
     * @param position
     */
    public CarouselSocket(double position) {
            this.position = position;
            if (position == 0) {
                    standbyPosition = 0;
                    } else {
                            standbyPosition = 360 - position;
                    }
    }
    
    
    /** 
    * @return the clampXminus
    */
    public FilterClamp getClampXminus() {
        return clampXminus;
    }

    /**
     * //FOR SPRING
     * @param clampXminus the clampXminus to set
     */
    public void setClampXminus(FilterClamp clampXminus) {
        this.clampXminus = clampXminus;
    }

    /**
     * @return the clampXplus
     */
    public FilterClamp getClampXplus() {
        return clampXplus;
    }

    /**
     * //FOR SPRING
     * @param clampXplus the clampXplus to set
     */
    public synchronized void setClampXplus(FilterClamp clampXplus) {
        this.clampXplus = clampXplus;
    }


    public double getPosition() {
            return position;
    }
    
    ////FOR SPRING
    public void setPosition(double position) {
            this.position = position;
    }

    public Filter getFilter() {
            return filter;
    }

    protected void setFilter(Filter filter) {
            this.filter = filter;
    }

    public double getStandbyPosition() {
            return standbyPosition;
    }

    //FOR SPRING has to be public but it's stupid
    public void setStandbyPosition(double standbyPosition) {
            this.standbyPosition = standbyPosition;
    }

    /*
     * The carousel socket is empty if there is no filter in the socket.
     * That appends when clamps of both side of the socket are empty.
     * A clamp is empty when no filter is engaged in the clamp.
     * @throws HardwareErrorDetectedException
     * If the 2 clamps are not in the same state this method throws
     * a HardwareErrorDetectedException : this should not append except if 
     * the sensors are broken.
     */
    public synchronized boolean isEmpty() throws HardwareErrorDetectedException {

 
            if (clampXminus.isFilterEngaged() &&
                    (clampXplus.isFilterEngaged() )) return false;
            if (!clampXminus.isFilterEngaged()  &&
                    !clampXplus.isFilterEngaged() ) return true;
            throw new HardwareErrorDetectedException("Error in filter presence detection at standby position : "
                    + "the clamps don't agree.");
    }
   

    public synchronized boolean isClampedOnFilter() {
        return (clampXminus.getState().equals(FilterClampState.CLAMPEDONFILTER)
                && clampXplus.getState().equals(FilterClampState.CLAMPEDONFILTER));
    }
    
    boolean isUnclampedOnFilter() {
        return (clampXminus.getState().equals(FilterClampState.UNCLAMPEDONFILTER)
                && clampXplus.getState().equals(FilterClampState.UNCLAMPEDONFILTER));
    }

    public synchronized boolean isUnclampedEmpty() {
        return (clampXminus.getState().equals(FilterClampState.UNCLAMPEDEMPTY)
                && clampXplus.getState().equals(FilterClampState.UNCLAMPEDEMPTY));
    }
    
    public synchronized boolean isReadyToClamp() {
        return (clampXminus.getState().equals(FilterClampState.READYTOCLAMP)
                && clampXplus.getState().equals(FilterClampState.READYTOCLAMP));
    }
    
    /**
     * This is used to synchronize the update of the clamps state with the values
     * returned by the sensors.
     * The update of the socket is completed if both clamps have completed their update.
     */
    public synchronized boolean isUpdatedCompleted() {
        return ((FilterClampModule) getClampXminus()).isUpdateCompleted() &&
                ((FilterClampModule) getClampXplus()).isUpdateCompleted();
    }
    
     /**
     * This methods updates the state of the 2 clamps at standby position. 
     * This is done bychecking the values returned by the sensors.
     * It waits until the 2 clamps have completed the update.
     */
    public synchronized void updateClampsStateWithSensors()  {
        //TODO timeout
        ((FilterClampModule) getClampXminus()).setUpdateCompleted(false);
        ((FilterClampModule) getClampXplus()).setUpdateCompleted(false);
        
        getClampXminus().updateStateWithSensors();
        getClampXplus().updateStateWithSensors();
        
        //wait until the 2 clamps are updated
        while (!isUpdatedCompleted()) {
            try {
                wait(this.tickMillis);
            } catch (InterruptedException ex) {
                Logger.getLogger(CarouselSocket.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        
 
        
    }
    
    
    public void removeFilter() throws BadCommandException {
        if (this.filter == null) {
                throw new BadCommandException("Carousel socket : there is no filter to remove at standby position");
        }
        this.filter = null;
    }

    public synchronized void putFilterOnSocket(Filter filter) {
        setFilter(filter);
        filter.setFilterLocation(FilterLocation.ONCAROUSEL);
    }

    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder("Filter in socket : ");
        if (getFilter() == null) {
            sb.append(" NO FILTER").append("\n");
        } else {
            sb.append(getFilter().getName()).append("\n");
        }
        sb.append(this.clampXminus.getStatusData().toString()).append("\n");
        sb.append(this.clampXplus.getStatusData().toString()).append("\n");
        return sb.toString();
    }


}
