
/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.AutoChanger;
import org.lsst.ccs.subsystems.fcs.common.FilterLocation;
import org.lsst.ccs.subsystems.fcs.common.Latch;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;

/**
 * An abstract changer module without a fliprail, without online clamp, 
 * and without a motor.
 * This Module is in charge of the 2 latches which hold a filter in the autochanger.
 * This basic autochanger is used in single filter test.
 * @author virieux
 */
public abstract class BasicAutoChangerModule extends Module implements AutoChanger {
    private boolean isEmpty;
    Filter filterOnTrucks;

        /**
         * A latch which hold a filter on the trucks.
         */
    private Latch latchXminus;

        /**
         * A latch which hold a filter on the trucks.
         */
    private Latch latchXplus;

    public static String publishedByAutoChangerOutputName = "publishedByAutoChanger";
    private StatusDataPublishedByBasicAutoChanger publishedByBasicAutoChanger;
    protected volatile ModuleState state = ModuleState.HALTED;

    public BasicAutoChangerModule() {
    }

    @Override
    public Filter getFilterOnTrucks() {
        return filterOnTrucks;
    }

    public String getFilterOnTrucksName() {
        if (this.filterOnTrucks == null) {
            return "none";
        } else {
            return this.filterOnTrucks.getName();
        }
    }



    public StatusDataPublishedByBasicAutoChanger getStatusData() {
        this.publishedByBasicAutoChanger.update(this);
        return this.publishedByBasicAutoChanger;
    }

    /**
     * @return the state
     */
    public ModuleState getState() {
        return state;
    }

    /**
     * @param state the state to set
     */
    public void setState(ModuleState state) {
        this.state = state;
    }

    public abstract String goToPosition(double requiredPosition) throws IllegalArgumentException, BadCommandException;
    
    public abstract String goToStandby();
    
    public String goToStandback() {
        return null;
    }


    @Override
    public String grabbeFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException {
        if (!(this.filterOnTrucks == null)) {
            throw new BadCommandException(getName() + ": can't grabbe a filter " + "when a filter is already loaded in trucks ");
            //TODO some refactoring for this
            //		if (!(filter == this.getFilterAtStandbyOnCarousel())) {
            //				throw new BadCommandException("autochanger can't grabbe a filter that " +
            //					"is not at standby position " +
            //                                        filter.getName() + "/ " + this.getFilterAtStandbyOnCarousel().getName());
            //		}
        }
        log.info(getName() + ": grabbing " + filter.getName() + " at standby position.");
        unlockLatchesAtStandby();
        //this.goToPosition(this.getTrucksPositionAtStandby());
        goToStandby();
        lockLatchesAtStandby(filter);      
        String ack = getName() + ": " + filter.getName() + " is grabbed on autochanger";
        log.info(ack);
        return ack;
    }

    @Override
    public void initModule() {
        log.info("[AutoChangerModule] Initializing the Auto Changer module ");
        
        this.initPublishedData();
//        this.motor.setMinimalPosition(getTrucksPositionAtStandby());
//        this.motor.setMaximalPosition(getTrucksPositionOnline());
    }

    @Override
    public void initPublishedData() {
        this.publishedByBasicAutoChanger = new StatusDataPublishedByBasicAutoChanger();
    }

    public boolean isEmpty() {
        return isEmpty;
    }

    public abstract boolean isMoving();

    /**
     * This methods locks the trucks latches at standby position and updates the 
     * field filterOnTrucks, the field filterLocation of the object Filter and updates
     * the data that we publish on the status bus.
     * It sends the command "lock" to the hardware of the latches and if this command is 
     * executed with success it goes on with the updates otherwise it returns an exception.
     * @return
     * @throws ErrorInCommandExecutionException 
     */
    @Override
    public String lockLatchesAtStandby(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException {
        try {
            //TODO: have to test if autochanger truck is HALTED, otherwise throw an exception
            //TODO refactorise this method : is it Thread safe ?
            latchXminus.lock();
            latchXplus.lock();
            //TODO: implement a timeout 
            while (!(latchXminus.isLocked() && latchXplus.isLocked())) {
                try {
                    log.info("Waiting for the AutoChanger latches to be locked on the filter.");
                    Thread.sleep(tickMillis);
                } catch (InterruptedException ex) {
                    Logger.getLogger(BasicAutoChangerModule.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
            
            if (latchXminus.isLocked() && latchXplus.isLocked()) {
            //Filter is going now on autochanger.
                log.info("Filter is now locked on auto-changer trucks.");
                synchronized (this) {
                    this.filterOnTrucks = aFilter;
                    setEmpty(false);
                    aFilter.setFilterLocation(FilterLocation.ONAUTOCHANGER);
                    updatePublishedDataAndNotifyObservers();
                }
            } else {
                throw new ErrorInCommandExecutionException(getName() + ": standby latches couldn't be locked.");
            }
            String ack = getName() + ": standby latches are locked";
            log.info(ack);
            return ack;
        } catch (BadCommandException ex) {
            Logger.getLogger(BasicAutoChangerModule.class.getName()).log(Level.SEVERE, null, ex);
        }
        return null;
    }

        public abstract String moveFilterToOnline(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException;

        public abstract String moveFilterToStandby(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException;
        
        public abstract String moveFilterToStandback(Filter aFilter) throws BadCommandException, ErrorInCommandExecutionException;

    /*
     * what to do when the Modules we observe change their values.
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        this.updatePublishedDataAndNotifyObservers();
    }

    public void setEmpty(boolean isEmpty) {
        this.isEmpty = isEmpty;
    }

    public void setFilterOnTrucks(Filter filterOnTrucks) {
        this.filterOnTrucks = filterOnTrucks;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n");
        sb.append("Trucks position = ");
        sb.append("Filter on trucks:");
        if (this.filterOnTrucks == null) {
            sb.append(" NONE").append("\n");
        } else {
            sb.append(this.filterOnTrucks.getName()).append("\n");
        }
        return sb.toString();
    }

    @Override
    public String unGrabbeFilterAtStandby() throws ErrorInCommandExecutionException, BadCommandException {
        //TODO to check : carousel clamp must be locked
        //
        log.info(getName() + ": ungrabbing filter at standby position.");
        unlockLatchesAtStandby();
        //eject filter from autochanger
        synchronized (this) {
            this.filterOnTrucks = null;
            setEmpty(true);
            this.updatePublishedDataAndNotifyObservers();
        }
        //when a filter is at standby position the trucks goes empty to SWAPOUT position
        log.info(getName() + " trucks going empty to SWAPOUT position");
        //goToPosition(this.getTrucksPositionSwapout());
        goToStandback();
        
        String ack = getName() + ": trucks are empty at SWAPOUT position ";
        return ack;
    }

    @Override
    public String unlockLatchesAtStandby() throws BadCommandException, ErrorInCommandExecutionException {
        try {
            //TODO : harmoniser cette methode avec lockLatchesAtStandby.
            latchXminus.unlock();
            latchXplus.unlock();
            if (latchXminus.isLocked() || latchXplus.isLocked()) {
                throw new ErrorInCommandExecutionException(getName() + ": standby latches could not be unlocked.");
            }
            String ack = getName() + ": standby latches are unlocked";
            log.info(ack);
            return ack;
        } catch (BadCommandException ex) {
            Logger.getLogger(BasicAutoChangerModule.class.getName()).log(Level.SEVERE, null, ex);
        }
        return null;
    }




    public void updatePublishedDataAndNotifyObservers() {
            this.publishedByBasicAutoChanger.update(this);
            setChanged();
            notifyObservers(new ValueUpdate(publishedByAutoChangerOutputName, this.publishedByBasicAutoChanger));
    }

    
}
