package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.subsystems.fcs.common.FilterClamp;
import org.lsst.ccs.subsystems.fcs.common.FilterClampState;
import org.lsst.ccs.subsystems.fcs.common.FilterLocation;
import org.lsst.ccs.subsystems.fcs.simulation.SimuFilterClampModule;


/**
 * This is a socket on the carousel : there is 5 sockets on a carousel.
 * When a filter is on the carousel, it is attached at a socket.
 * 
 * @author virieux
 *
 */
public abstract class CarouselSocket {

        /**
        * The clamps to hold the filter in the socket.
        * clampX+ and clampX- are the same except for the method isLocked()
        * See FilterClampXminusModule and FilterClampXplusModule
        */

        private FilterClamp clampXminus;

        private FilterClamp clampXplus;
	
	/**
	 * socket position (angle) in degrees on the carousel
	 * 
	 */
	private double position;
	
	/**
	 * carousel position (angle) when this socket is at standby position
	 * should be 360 - position, but could be different
	 */
	private double standbyPosition;
	
	private Filter filter;
	//private boolean isEmpty;
	
	public CarouselSocket() {}
	
	/**
	 * This constructor is for tests.
	 * @param position
	 */
	public CarouselSocket(double position) {
		this.position = position;
		if (position == 0) {
			standbyPosition = 0;
			} else {
				standbyPosition = 360 - position;
			}
	}




	public double getPosition() {
		return position;
	}

	public void setPosition(double position) {
		this.position = position;
	}

	public Filter getFilter() {
		return filter;
	}

	protected void setFilter(Filter filter) {
		this.filter = filter;
	}

	public double getStandbyPosition() {
		return standbyPosition;
	}

	public void setStandbyPosition(double standbyPosition) {
		this.standbyPosition = standbyPosition;
	}

        /*
         * The carousel socket is empty if there is no physical filter in the socket :
         * the 2 sensors for presence filter must both return the value 0.
         * If both the 2 filter presence sensors return the value 1 : there is a filter
         * in the socket, so the socket is not empty.
         * If the 2 sensors disagree we send and error message to the operator.
         */
	public boolean isEmpty() throws HardwareErrorDetectedException {
                if ((clampXminus.getFilterPresenceSensor().getValue() == 1 ) &&
                        (clampXplus.getFilterPresenceSensor().getValue() == 1 )) return false;
                if ((clampXminus.getFilterPresenceSensor().getValue() == 0 ) &&
                        (clampXplus.getFilterPresenceSensor().getValue() == 0 )) return true;
                throw new HardwareErrorDetectedException("Error in filter presence detection at standby position : "
                        + "the clamps don't agree.");
	}

            /**
     * @return the clampXminus
     */
    public FilterClamp getClampXminus() {
        return clampXminus;
    }

    /**
     * @param clampXminus the clampXminus to set
     */
    public void setClampXminus(FilterClamp clampXminus) {
        this.clampXminus = clampXminus;
    }

    /**
     * @return the clampXplus
     */
    public FilterClamp getClampXplus() {
        return clampXplus;
    }

    /**
     * @param clampXplus the clampXplus to set
     */
    public void setClampXplus(FilterClamp clampXplus) {
        this.clampXplus = clampXplus;
    }

    public String unlockClamps() throws BadCommandException, ErrorInCommandExecutionException {
        clampXminus.unlock();
        clampXplus.unlock();
        return "Clamps are unlocked at standby position";
    }

    public String releaseClamps() throws BadCommandException, ErrorInCommandExecutionException  {
            clampXminus.release();
            clampXplus.release();
            return "Simulated Clamps are released at standby position";

    }

    public boolean isClampedOnFilter() {
        return (clampXminus.getState().equals(FilterClampState.CLAMPEDONFILTER)
                && clampXplus.getState().equals(FilterClampState.CLAMPEDONFILTER));
    }

    public boolean isUnclampedAndEmpty() {
        return (clampXminus.getState().equals(FilterClampState.UNCLAMPEDEMPTY)
                && clampXplus.getState().equals(FilterClampState.UNCLAMPEDEMPTY));
    }

    public synchronized void removeFilter() {
        setFilter(null);
    }

    public synchronized void putFilterOnSocket(Filter filter) {
        setFilter(filter);
        filter.setFilterLocation(FilterLocation.ONCAROUSEL);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("position on the carousel:");
        sb.append("/ filter on socket : ");
        if (getFilter() == null) {
            sb.append(" NONE").append("\n");
        } else {
            sb.append(getFilter().getName()).append("\n");
        }
        sb.append(this.clampXminus.getStatusData().toString()).append("\n");
        sb.append(this.clampXplus.getStatusData().toString()).append("\n");
        return sb.toString();
    }


      public abstract void releaseClampsContact();

      public abstract void engageClampsContact();



}
