/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.lsst.ccs.subsystems.fcs;

import java.util.Observable;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.CommandReply;
import org.lsst.ccs.bus.CommandReply.CommandStatus;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.AutoChanger;
import org.lsst.ccs.subsystems.fcs.common.EngineState;
import org.lsst.ccs.subsystems.fcs.common.FilterLocation;
import org.lsst.ccs.subsystems.fcs.common.Latch;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.common.Motor;

/**
 * An basic autochanger module without a fliprail, without online clamp.
 * This basic autochanger is used in single filter test.
 * @author virieux
 */
public abstract class BasicAutoChangerModule extends Module implements AutoChanger {

    private double trucksPositionOnline;
    private double trucksPositionAtStandby;
    private double trucksPositionSwapout;
    private boolean isEmpty;
    Filter filterOnTrucks;
    Motor motor;

        /**
         * To be replaced by latchX- and latchX+
         */
    private Latch standbyLatch;

        /**
         * A latch which hold a filter on the trucks.
         */
    private FilterClampModule latchXminus;

        /**
         * A latch which hold a filter on the trucks.
         */
    private FilterClampModule latchXplus;

    public static String publishedByAutoChangerOutputName = "publishedByAutoChanger";
    private StatusDataPublishedByBasicAutoChanger publishedByBasicAutoChanger;

    private volatile ModuleState state = ModuleState.HALTED;

    public BasicAutoChangerModule() {
    }

    @Override
    public Filter getFilterOnTrucks() {
        return filterOnTrucks;
    }

    public String getFilterOnTrucksName() {
        if (this.filterOnTrucks == null) {
            return "none";
        } else {
            return this.filterOnTrucks.getName();
        }
    }

    @Override
    public Filter getFilterOnline() {
        if ((getTrucksPosition() == this.getTrucksPositionOnline()) && (!this.isMoving())) {
            return filterOnTrucks;
        } else {
            return null;
        }
    }

    public String getFilterOnlineName() {
        if (getFilterOnline() == null) {
            return "none";
        } else {
            return getFilterOnline().getName();
        }
    }

    public Motor getMotor() {
        return motor;
    }

    public Latch getStandbyLatch() {
        return standbyLatch;
    }

    public StatusDataPublishedByBasicAutoChanger getStatusData() {
        this.publishedByBasicAutoChanger.update(this);
        return this.publishedByBasicAutoChanger;
    }

    @Override
    public double getTrucksPosition() {
        return this.motor.getPosition();
    } 

    public double getTrucksPositionAtStandby() {
        return trucksPositionAtStandby;
    }

    public double getTrucksPositionOnline() {
        return trucksPositionOnline;
    }

    public double getTrucksPositionSwapout() {
        return trucksPositionSwapout;
    }

    private boolean isAbleToMove() {
	return true;
    }

    @Override
    public String goToPosition(double requiredPosition) throws IllegalArgumentException, BadCommandException {
        String message = null;
        //availables values for a position : [Standby,Online]
        if (requiredPosition > this.getTrucksPositionOnline() || requiredPosition < this.getTrucksPositionAtStandby()) {
            throw new IllegalArgumentException("Please enter a position between " + this.getTrucksPositionAtStandby() + " and " + this.getTrucksPositionOnline());
        }
        if (isMoving()) {
            // Thread-safe because of encapsulation for carousel.state
            //if (getState() == ModuleState.RUNNING) {
            message = "Auto-changer is moving, stop it before sending a new motion command.";
            log.error(message);
            throw new BadCommandException(message);
        }
        if (!this.isAbleToMove()) {
            message = "Auto-changer is unable to move because " + "fliprail is open or online clamp is locked.";
            log.error(message);
            throw new BadCommandException(message);
        }
        setState(ModuleState.RUNNING);
        this.motor.setRequiredPosition(requiredPosition);
        log.info(getName() + " is moving to position: " + requiredPosition);
        this.sendToReply(new CommandReply("Please wait during auto-changer motion...", CommandStatus.OK));
        message = "Moving to required position: " + requiredPosition;
        double displacement = requiredPosition - getTrucksPosition();
        motor.move(displacement);
        log.info("AutoChanger moving to required position: " + requiredPosition + ", please wait...");
        while (isMoving()) {
            try {
                log.info("...auto-changer moving, please wait.....");
                log.info(getName() + " WAITING TO BE AT REQUIRED POSITION=" + requiredPosition);
                log.info(state.toString());
                Thread.sleep(tickMillis);
            } catch (InterruptedException e) {
                return "Sleeping interrupted";
            }
        }
        //auto changer is not moving any more.
        if (!isMoving()) {
            log.info("THREAD=" + Thread.currentThread().getName());
            setState(ModuleState.HALTED);
            //we have to let know the observers that autochanger has changed : state and position.
            //and to update the new value of the data we publish.
            //                        this.publishedByAutoChanger.update(this);
            //                        setChanged();
            //			notifyObservers(new ValueUpdate(FcsMainModule.publishedByAutoChangerOutputName, this.publishedByAutoChanger));
            log.info("auto-changer is now halted.");
            log.info("filter on truck: " + this.getFilterOnTrucksName());
            if (getTrucksPosition() == requiredPosition) {
                message = "goToPosition command completed";
                log.info(message);
            } else {
                message = "goToPosition command non completed: auto-changer is not stopped at the required position";
                log.info(message);
            }
        }
        log.error(message);
        return message;
    }

    @Override
    public String grabbeFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException {
        if (!(this.filterOnTrucks == null)) {
            throw new BadCommandException(getName() + ": can't grabbe a filter " + "when a filter is already loaded in trucks ");
            //TODO some refactoring for this
            //		if (!(filter == this.getFilterAtStandbyOnCarousel())) {
            //				throw new BadCommandException("autochanger can't grabbe a filter that " +
            //					"is not at standby position " +
            //                                        filter.getName() + "/ " + this.getFilterAtStandbyOnCarousel().getName());
            //		}
        }
        log.info(getName() + ": grabbing " + filter.getName() + " at standby position.");
        closeFliprail();
        unlockLatchStandby();
        this.goToPosition(this.getTrucksPositionAtStandby());
        lockLatchStandby();
        //TODO in real life : test if the standby latches are really locked
        //otherwise throw an exception.
        //Filter is going now on autochanger.
        synchronized (this) {
            this.filterOnTrucks = filter;
            setEmpty(false);
            filter.setFilterLocation(FilterLocation.ONAUTOCHANGER);
            updatePublishedDataAndNotifyObservers();
        }
        this.sendToReply(new CommandReply(getName() + ": " + filter.getName() + " is grabbed on autochanger", CommandStatus.OK));
        String ack = getName() + ": " + filter.getName() + " is grabbed on autochanger";
        log.info(ack);
        return ack;
    }

    @Override
    public void initModule() {
        log.info("[AutoChangerModule] Initializing the Auto Changer module ");
        setState(ModuleState.HALTED);
        this.initPublishedData();
        this.motor.setMinimalPosition(getTrucksPositionAtStandby());
        this.motor.setMaximalPosition(getTrucksPositionOnline());
    }

    @Override
    public void initPublishedData() {
        this.publishedByBasicAutoChanger = new StatusDataPublishedByBasicAutoChanger();
    }

    public boolean isEmpty() {
        return isEmpty;
    }

    public boolean isMoving() {
        return this.motor.getEngineState().equals(EngineState.RUNNING);
    }

    /**
     * This method has to be executed at the FCS startup.
     * It checks the hardware trucks and set the trucks position.
     */
    @Override
    public synchronized void locateTrucks() {
        //TODO IN REAL LIFE : check if the trucks are empty or not
        this.filterOnTrucks = null;
        setEmpty(true);
    }

    @Override
    public String lockLatchStandby() throws ErrorInCommandExecutionException {
        standbyLatch.lock();
        if (!standbyLatch.isLocked()) {
            throw new ErrorInCommandExecutionException(getName() + ": standby latch couldn't be locked.");
        }
        String ack = getName() + ": standby latch is locked";
        log.info(ack);
        return ack;
    }

    @Override
    public String moveFilterToOnline() throws BadCommandException, ErrorInCommandExecutionException {
        if (isEmpty()) {
            throw new BadCommandException("Autochanger is empty");
        }
        if (getTrucksPosition() == this.getTrucksPositionOnline()) {
            throw new BadCommandException("Filter " + this.filterOnTrucks.getName() + " is already at online position.");
        }
        closeFliprail();
        goToPosition(this.getTrucksPositionOnline());
        if (getTrucksPosition() == this.getTrucksPositionOnline()) {
            log.info(getName() + ": " + this.filterOnTrucks.getName() + " is at online position");
            closeOnlineClamp();
            return openFliprail();
        }
        log.error("");
        return null;
    }

    @Override
    public String moveFilterToStandby() throws BadCommandException, ErrorInCommandExecutionException {
        if (isEmpty()) {
            throw new BadCommandException("Autochanger is empty");
        }
        if (getTrucksPosition() == this.getTrucksPositionAtStandby()) {
            throw new BadCommandException("Filter " + this.filterOnTrucks.getName() + " is already at standby position.");
        }
        if (!(getTrucksPosition() == this.trucksPositionOnline)) {
            throw new BadCommandException("Wrong position for autochanger trucks. Change to engineering mode.");
        }
        closeFliprail();
        openOnlineClamp();
        goToPosition(this.getTrucksPositionAtStandby());
        if (getTrucksPosition() == this.getTrucksPositionAtStandby()) {
            log.info(getName() + ": " + this.filterOnTrucks.getName() + " is at standby position.");
            return getName() + ": " + this.filterOnTrucks.getName() + " is at standby position.";
        } else {
            String message = "the autochanger truck is not at the required position.";
            log.error(message);
            throw new ErrorInCommandExecutionException(message);
        }
    }

    /*
     * what to do when the Modules we observe change their values.
     */
    @Override
    public void processUpdate(Observable source, ValueUpdate v) {
        this.updatePublishedDataAndNotifyObservers();
    }

    public void setEmpty(boolean isEmpty) {
        this.isEmpty = isEmpty;
    }

    public void setFilterOnTrucks(Filter filterOnTrucks) {
        this.filterOnTrucks = filterOnTrucks;
    }

    public void setMotor(Motor motor) {
        this.motor = motor;
    }

    public void setStandbyLatch(Latch standbyLatch) {
        this.standbyLatch = standbyLatch;
    }

    private void setState(ModuleState s) {
	this.state = s;

    }

    public void setTrucksPositionAtStandby(double trucksPositionAtStandby) {
        this.trucksPositionAtStandby = trucksPositionAtStandby;
    }

    public void setTrucksPositionOnline(double trucksPositionOnline) {
        this.trucksPositionOnline = trucksPositionOnline;
    }

    public void setTrucksPositionSwapout(double trucksPositionSwapout) {
        this.trucksPositionSwapout = trucksPositionSwapout;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(getName());
        sb.append("\n");
        sb.append("Trucks position = ");
        sb.append(this.getTrucksPosition()).append(" / ");
        sb.append("trucks position at standby = ");
        sb.append(this.getTrucksPositionAtStandby()).append(" / ");
        sb.append("trucks position online = ");
        sb.append(this.getTrucksPositionOnline()).append(" / ");
        sb.append("trucks position swapout = ");
        sb.append(this.getTrucksPositionSwapout()).append("\n");
        sb.append("Filter on trucks:");
        if (this.filterOnTrucks == null) {
            sb.append(" NONE").append("\n");
        } else {
            sb.append(this.filterOnTrucks.getName()).append("\n");
        }
        return sb.toString();
    }

    @Override
    public String stop() {
        String message;
        if (isMoving()) {
            message = "Stopping auto-changer";
            motor.stop();
            setState(ModuleState.HALTED);
        } else {
            message = "Auto-changer is already stopped";
        }
        return message;
    }

    @Override
    public String unGrabbeFilterAtStandby() throws ErrorInCommandExecutionException, BadCommandException {
        //TODO to check : carousel clamp must be locked
        //
        log.info(getName() + ": ungrabbing filter at standby position.");
        unlockLatchStandby();
        //eject filter from autochanger
        synchronized (this) {
            this.filterOnTrucks = null;
            setEmpty(true);
            this.updatePublishedDataAndNotifyObservers();
        }
        //when a filter is at standby position the trucks goes empty to SWAPOUT position
        log.info(getName() + " trucks going empty to SWAPOUT position");
        goToPosition(this.getTrucksPositionSwapout());
        openFliprail();
        String ack = getName() + ": trucks are empty at SWAPOUT position ";
        return ack;
    }

    @Override
    public String unlockLatchStandby() throws ErrorInCommandExecutionException {
        standbyLatch.unlock();
        if (standbyLatch.isLocked()) {
            throw new ErrorInCommandExecutionException(getName() + ": standby latch could not be unlocked.");
        }
        String ack = getName() + ": standby latch is unlocked";
        log.info(ack);
        return ack;
    }


    public abstract String openFliprail() throws ErrorInCommandExecutionException, BadCommandException;
 
    public abstract String closeFliprail() throws ErrorInCommandExecutionException;

    public abstract String openOnlineClamp() throws ErrorInCommandExecutionException;

    public abstract String closeOnlineClamp() throws ErrorInCommandExecutionException;

    public void updatePublishedDataAndNotifyObservers() {
            this.publishedByBasicAutoChanger.update(this);
            setChanged();
            notifyObservers(new ValueUpdate(publishedByAutoChangerOutputName, this.publishedByBasicAutoChanger));
    }

}
