package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.subsystems.fcs.common.NumericSensor;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.Actuator;
import org.lsst.ccs.subsystems.fcs.common.FilterClamp;
import org.lsst.ccs.subsystems.fcs.common.FilterClampState;

/**
 * An abstract class which extends Module to model a clamp that holds a filter on the carousel.
 * It implements pattern state.
 * See <a href="CarouselClampState.png"> the state machine diagram for a carousel clamp.</a>
 * 
 * @author virieux
 */
public abstract class FilterClampModule extends Module implements FilterClamp {

    protected FilterClampState state;
    
    /**
     * The filterPresenceSensor detects if the filter is in the clamp or not. 
     */
    protected NumericSensor filterPresenceSensor;
    public String publishedByClampOutputName = "publishedByClamp";
    public StatusDataPublishedByClamp publishedByClamp;

    @Override
    public FilterClampState getState() {
        return state;
    }

    /**
     * @param state the state to set
     */
    @Override
    public void setState(FilterClampState state) {
        this.state = state;
    }

    @Override
    public String release() throws BadCommandException, ErrorInCommandExecutionException  {
        log.info(getName() + ": is going to be released.");
        return state.release(this);
    }

    @Override
    public String unlock() throws BadCommandException, ErrorInCommandExecutionException {
        log.info(getName() + ": is going to be unlocked.");
        log.info(getName() + ": is in state " + state.toString());
        return state.unlock(this);
    }

    @Override
    public String lock() throws BadCommandException, ErrorInCommandExecutionException {
        log.info(getName() + ": is going to be locked.");
        return state.lock(this);
    }

    @Override
    public abstract boolean isLocked();

    @Override
    public void updateStateWithSensors() {
        // a filter is in the socket
        if (this.filterPresenceSensor.getValue() == 1) {
            if (isLocked()) {
                state = (FilterClampState) FilterClampState.CLAMPEDONFILTER;
            } else {
                state = (FilterClampState) FilterClampState.UNCLAMPEDONFILTER;
            }
        // no filter
        } else {
            if (isLocked()) {
                state = (FilterClampState) FilterClampState.READYTOCLAMP;
            } else {
                state = (FilterClampState) FilterClampState.UNCLAMPEDEMPTY;
            }

        }

    }


    /**
     * @return the filterPresence
     */
    @Override
    public NumericSensor getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    /**
     * @param filterPresence the filterPresence to set
     */
    @Override
    public void setFilterPresenceSensor (NumericSensor filterPresence) {
        this.filterPresenceSensor = filterPresence;
    }

    @Override
    public synchronized void tick() {
//       log.info(getName() + ": is in state " + state.toString());
//       log.info(getName() + ": actuator " + actuator.toString());
//       log.info(getName() + ": presence sensor= " + this.filterPresenceSensor.toString());
       updateStateWithSensors();
       this.publishedByClamp.update(this);
       setChanged();
       notifyObservers(new ValueUpdate(publishedByClampOutputName,this.publishedByClamp));
    }

    @Override
    public void initModule() {
        initPublishedData();
    }

    public void initPublishedData() {
	this.publishedByClamp = new StatusDataPublishedByClamp();
    }


    @Override
    public StatusDataPublishedByClamp getStatusData() {
         this.publishedByClamp.update(this);
         return this.publishedByClamp;
    }


    
	
}
