package org.lsst.ccs.subsystems.fcs;


import java.util.Observable;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.bus.CommandReply;
import org.lsst.ccs.bus.CommandReply.CommandStatus;
import org.lsst.ccs.subsystems.fcs.common.Carousel;
import org.lsst.ccs.subsystems.fcs.common.FlipRailState;
import org.lsst.ccs.subsystems.fcs.common.Motor;
import org.lsst.ccs.subsystems.fcs.common.EngineState;
import org.lsst.ccs.subsystems.fcs.common.Latch;
import org.lsst.ccs.subsystems.fcs.common.ModuleState;
import org.lsst.ccs.subsystems.fcs.common.RunningWay;
import org.lsst.ccs.bus.BadCommandException;
import org.lsst.ccs.bus.ErrorInCommandExecutionException;
import org.lsst.ccs.framework.Module;
import org.lsst.ccs.subsystems.fcs.common.Actuator;

/**
* This is a representation of the hardware of the carousel.
* It receives commands from the FCSMAinModule and send back an anknowledge.
* It publishes data on the status bus.
* In engineering mode it can receive commands from the engineering console.
* It has a latch, an engine which are connected to real or simulated hardware. 
* @author virieux
*
*/
public class CarouselModule extends Module implements Carousel {
	
	/**
	 * 
	 */
	private static final long serialVersionUID = -2376279469784152348L;

	private Motor carouselMotor;

        private Actuator clampsActuator;

        protected int nbSockets;
	
	private CarouselSocket[] sockets = new CarouselSocket[nbSockets];
	
	/**
	 * This latch is to hold the carousel in position.
	 * It prevents it from rotating. It should be a brake.
	 */
	private Latch lock;

        public static String publishedByCarouselOutputName = "publishedByCarousel";
	private StatusDataPublishedByCarousel publishedByCarousel;
	
	private volatile ModuleState state = ModuleState.HALTED;
		
	private boolean stopped = false;
		
	
	public CarouselModule() {}
	

	@Override
	public void initModule()  {
		log.info("[CarouselModule] Initializing the carousel module ");
		initPublishedData();
		setState(ModuleState.HALTED);


		try {
			// checks carousel position, engine state, carousel latch, clamps on filters must be locked
			// when every hardware pieces are OK, CarouselModule  is READY

		} catch (final RuntimeException e) {
			log.fatal("[CarouselModule] ERROR: " + this.name +  " Initialization failed for Carousel module : " + e.getMessage());
			throw e;
			
		}

	}

	
	public void initPublishedData() {
		this.publishedByCarousel = new StatusDataPublishedByCarousel(nbSockets);
	}


        public StatusDataPublishedByCarousel getStatusData() {
            this.publishedByCarousel.update(this);
            return this.publishedByCarousel;
        }


        public String toString() {
            StringBuilder sb = new StringBuilder(getName());
            sb.append("\n Numbers of sockets: ");
            sb.append(nbSockets);
            for (int i=0; i< nbSockets; i++ ) {
                sb.append("\n Socket number ");
                sb.append(i);
                sb.append(" / ");
                sb.append(sockets[i].toString());
            }
            sb.append("\nClamps actuator  ").append(clampsActuator.toString()).append("\n");
            return sb.toString();
        }

    /**
    * What to do when the Modules we observe send there new values : we notify our own observers
    */
    @Override
	public void processUpdate(Observable source, ValueUpdate v) {

            this.publishedByCarousel.update(this);
            setChanged();
            notifyObservers(new ValueUpdate(publishedByCarouselOutputName, this.publishedByCarousel));

	}

    @Override
	public double getPosition() {
		return carouselMotor.getPosition();
	}
	



    @Override
        public synchronized CarouselSocket getSocketAtStandby() {
            if (this.isRotating()) {
			return null;
		} else {
			for (CarouselSocket socket: sockets) {
				if (this.getPosition() == socket.getStandbyPosition()) {
					return socket;
				}
			}
		}
		return null;
        }

        
	
	public synchronized Filter getFilterAtStandby() {
		if (getSocketAtStandby() == null) {
			return null;
		} else {
			
			return getSocketAtStandby().getFilter();

		}
	}


        public synchronized String getFilterInStandbyName() {
		if (getFilterAtStandby() == null) {
			return "none";
		} else {
			return getFilterAtStandby().getName();
		}

	}


        /**
         * This method let us know if the carousel is ready to receive a filter at
         * standby position : 
         *  - the carousel must not rotate
         *  - an empty socket is at standby position.
         * @return true if the position of the carousel matches
         *              the position when one of its sockets is at standby position
         *              and this socket is empty.
         *         false
         */
        public boolean isReadyToGrabbeAFilterAtStandby() throws HardwareErrorDetectedException {
            if (this.isRotating()) return false;
            if (getSocketAtStandby() == null) return false;
            if (getSocketAtStandby().isEmpty()) return true;
            return false;
        }
	
	
	public boolean isRotating () {
		return carouselMotor.getEngineState().equals(EngineState.RUNNING);
	}
	

        @Override
	public boolean isAbleToMove() {
		FlipRailModule fliprail = (FlipRailModule) this.getModule("fliprail");	
		return !lock.isLocked() && (fliprail.state.equals(FlipRailState.OPEN));
		//return !latch.isLocked();
	}

        public boolean isLocked() {
            return lock.isLocked();
        }

    @Override
	public String stop() {
		String message;
		if (isRotating()) {
			stopped = true;		
			message = "Stopping carousel";
			carouselMotor.stop();
			setState(ModuleState.HALTED);
		} else {
			message = "Carousel is already stopped";
		}
				
		return message;
	}
	
    @Override
	public String lock() {
		return lock.lock();
	}

    @Override
	public String unlock() {		
		CommandReply myReply = new CommandReply("unlocking carousel", CommandStatus.OK);
		this.sendToReply(myReply);
                return lock.unlock();
	}
	
	/**
	 * Rotates carousel for an angle given as a parameter.
	 * This method can be launched by the user from the TestConsole.
	 * It checks the value of the rotation angle entered by the user,
	 * converts it in a double,
	 * and send the result of the method rotate(double angle).
	 * @param angleEntered : rotation angle
	 * 			The rotation angle can be positive or negative.
	 * 			It must be a double and have a value between -360.0 and 360.0
	 * @return the result of the command rotate(double angle)
	 * @throws IllegalArgumentException, BadCommandException 
	 */			
	public String rotate(double angle) throws IllegalArgumentException, BadCommandException {
		
		
		//availables values for an angle : [-360,+360]
		if (angle < -360 || angle > 360) {
			throw new IllegalArgumentException( "Please enter an angle between -360 and +360");
		}
		
		RunningWay way = null;
                double absAngle;
		if (angle >= 0) {
                        absAngle = angle;
			way = RunningWay.POSITIVE;				
		} else {
                        absAngle = -angle;
			way = RunningWay.NEGATIVE;
		}
		double requiredPosition = addAngle(getPosition(),angle);
		return rotate(absAngle,way,requiredPosition);
	}
	

	
	/**
	 * Rotate the carousel to a final position.
	 * 
	 * @param angle of the rotation : 0 <= angle <= 360
         * @param the final position has to be : 0 <= finalPosition <= 360
	 * @param runningWay rotation way (enum)
	 * @param finalPosition is the position the carousel must have
	 * 			when the command is completed
	 * The method rotate is delegated to the engine, which is an interface.
	 * An engine can be a simulated engine or a real hardware driver.
	 * @throws BadCommandException, IllegalArgumentException
	 *
	 */
	
	private String rotate(double angle, RunningWay runningWay, double finalPosition) 
                throws BadCommandException, IllegalArgumentException {
		//assertion 0 <= angle <= 360
		//assertion 0 <= finalPosition <= 360
                if (angle < 0 || angle > 360) throw new IllegalArgumentException(getName()
                            + " command rotate accepts angle only between 0 and 360)");
                if (finalPosition < 0 || finalPosition > 360) throw new IllegalArgumentException(getName()
                            + " command rotate accepts finalPosition only between 0 and 360)");
                
		String message = null;
		
		if 	(isRotating()) {
		// Thread-safe because of encapsulation for carousel.state		
		//if (getState() == ModuleState.RUNNING) {		
			message = "Carousel is rotating, stop it before sending a new rotate command.";
			log.error(message);
			throw new BadCommandException(message);
		} 
		
		if (!this.isAbleToMove()) {
			message = "Carousel is unable to move. Check lock or flip rail.";
			log.error(message);
			throw new BadCommandException(message);
		}


		stopped = false;	
		setState(ModuleState.RUNNING);

		getCarouselMotor().setRequiredPosition(finalPosition);
		this.sendToReply(new CommandReply("Please wait during carousel rotation...", CommandStatus.OK));
		carouselMotor.move(angle,runningWay);	
		message = "Rotating to required position: " + finalPosition;
		log.info(message);
		
		while (isRotating()) {
			try {
				log.info("...carousel rotating, please wait.....");
				log.info(getName() + " WAITING TO BE AT REQUIRED POSITION="  +  finalPosition);
				log.info(state.toString());
				Thread.sleep(tickMillis);
			} catch (InterruptedException e) {
				return "Sleeping interrupted";
			}
		}
		if (!isRotating()) {		
			log.info("THREAD=" + Thread.currentThread().getName());
			setState(ModuleState.HALTED);
			log.info("carousel is now stopped.");
			log.info("filter in standby position: " + getFilterInStandbyName());
			if ( getPosition() == finalPosition) {
				message = "Command completed";
				log.info(message);
			} else {
				message = "Command non completed: carousel is not stopped at the required position";
				log.info(message);
			}
		}			
		
		if (stopped) {
				message = "Command rotate carousel interrupted by command stop";
				log.info(message);
		}
		
		if (!stopped && (!( getPosition() == finalPosition))) {
			message = "Command non completed: carousel is not stopped at the required position";
			log.error(message);
			//TODO
			//throw new ErrorCommandException(message);
		}
		
		return message;	
	
	}


	/**
	 * What has to be done for each tick of the timer.
	 * We have to notify our observers we have changed 
	 * and give them the new values of the update data. 
	 */
	@Override
	public void tick() {
                log.info(getName() + ": socket at standby state : " + getSocketAtStandby().toString());
		try {
			
		} catch (Exception e) {
			log.error(e);
		}

	}
	
	/**
         * This method has no action on hardware.
	 * This method updates the socket at standby position when a filter is put on the carousel
	 * at standby position.
	 * A filter must be detected in the socket at standby position (socket not empty).
         * 
	 * @param filter
	 * @throws ErrorInCommandExecutionException 
	 * @throws BadCommandException 
	 */
	private void putFilterOnCarousel(Filter filter) throws ErrorInCommandExecutionException, BadCommandException, HardwareErrorDetectedException {
	
                if (getSocketAtStandby() == null) 
                        throw new BadCommandException(getName() + ": there is no socket at standby position to put a filter in.");

                //
		if ((getSocketAtStandby().isEmpty()))
			throw new BadCommandException(getName() + ": no filter is detected in socket at standby position so the operation putFilterOnCarousel is not available.");
		
		getSocketAtStandby().putFilterOnSocket(filter);
                this.publishedByCarousel.update(this);
                setChanged();
                notifyObservers(new ValueUpdate(publishedByCarouselOutputName, this.publishedByCarousel));
                
	}
	
	
	/*
         * Removes a filter from carousel. This method has no action on hardware.
         * It updates the objects Socket and Filter.
         * A filter can be removed from Carousel only when the socket 
         * is at standby position.
         */
    public void removeFilterFromCarousel(Filter filter) throws BadCommandException, HardwareErrorDetectedException, ErrorInCommandExecutionException {
//                if (!filter.isOnCarousel())
//                    throw new BadCommandException(getName() + " : can't remove a filter which is not on carousel");
        if (!getSocketAtStandby().getFilter().equals(filter))
                    throw new BadCommandException(getName() + " : can't remove a filter which is not at standby position");

        getSocketAtStandby().removeFilter();
        this.publishedByCarousel.update(this);
        setChanged();
        notifyObservers(new ValueUpdate(publishedByCarouselOutputName, this.publishedByCarousel));
               
    }
	
	//TODO exception if filter is not on Carousel ?
	public double getFilterPosition(Filter filter) {
		double filterPosition = 0;
		for (int ix = 0; ix < sockets.length; ix++) {
				if (sockets[ix].getFilter() == filter) {
					filterPosition = sockets[ix].getPosition();					
				}
		}
		return filterPosition;
	}



	public double getStandbyPositionForFilter(Filter filter) {
		double position = 0;
		for (CarouselSocket socket : sockets) {
			if (socket.getFilter() == filter) {
				position = socket.getStandbyPosition();
			}
			
		}
		return position;
	}
	
	//TODO exception if filter is not on Carousel ?
        /**
         *
         * @param filter
         * @return the position of the filter in the sockets array
         * I'm not sure this method is useful actually.
         * Only used is FcsMainModule in method printFilters.
         */
	public int getSocketNumber(Filter filter) {
		int socketNumber = 0;
		for (int ix = 0; ix < sockets.length; ix++) {
				if (sockets[ix].getFilter() == filter) {
					socketNumber = ix;					
				}
		}
		return socketNumber;
	}

	
	
	/**
	 * This method rotates the carousel to put the given filter in front of the autochanger
	 * (standby position).
	 * This method can't be launched by the user from the console.
	 * It computes the shortest way for the carousel. So the running way can be positive or
	 * negative.
	 * This could be more complicated if the carousel was not able to rotate
	 * more than TBD turns in one way.
	 * @params filter : the filter can not be null and must be on the carousel
	 * @throws BadCommandException
	 */
	protected String moveFilterToStandby(Filter filter) throws InterruptedException, BadCommandException {
		
		//assert filter is on carousel
		if (!filter.isOnCarousel()) {
			throw new IllegalArgumentException("filter: " + filter.getName() + 
					" is not on carousel.");
		}
		String message = " [CarouselModule]";
		double requiredPosition = getStandbyPositionForFilter(filter);
					
		
		if (state == ModuleState.RUNNING) {
			throw new BadCommandException("Carousel is running, stop it before sending a new command.");
		}
		
		if ( getPosition() == requiredPosition) {
			log.info(filter.getName() + " is at STANDBY position on carousel.");
			return(filter.getName() + " is at STANDBY position on carousel.");
		} 

		//if carousel is locked, we have to unlock it before running.		
		if ( getLatch().isLocked()) {
				log.info("Carousel is locked. Unlocking carousel");
				unlock(); 					
		} 
			
		double angle = addAngle(getPosition(),getFilterPosition(filter));
		
		//
		RunningWay runningWay;
		if (angle < 180 ) {
				runningWay = RunningWay.NEGATIVE;                       
		} else {
				angle = 360 - angle;
				runningWay = RunningWay.POSITIVE;
		}
                //Release The clamps contact before the rotation
                releaseClampsContact();
                //rotate the carousel
                rotate(angle, runningWay, requiredPosition);
                //engage the clamps contact when it's stopped at standby position
                engageClampsContact();
                //lock the carousel to hold it in position
		return message += lock();
	}
	

	

	/**
	 * Print for every socket on carousel, the name of the filter it contains.
	 */
	public void printSockets() {
		
		for (int ix = 0; ix < sockets.length; ix++) {
			log.info("Socket number: " + ix + " socket position: "+ sockets[ix].getPosition()
                   + " contains filter: " + sockets[ix].getFilter().getName());

		}
	}
	
	public String grabbeFilterAtStandby(Object filterName) throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException {
		
		FcsMainModule fcsMainModule =  (FcsMainModule) this.getModule("main");
		fcsMainModule.controlFilterName(filterName);
		
		// the entered filter name is correct
		log.info("Filter to move : " + filterName);
		log.info("Filter location : " + fcsMainModule.getFilterByName((String) filterName).getFilterLocation());				
		return getName() + grabbeFilterAtStandby(fcsMainModule.getFilterByName((String) filterName));
	}
        
	
	public String grabbeFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException {
		if (this.isRotating()) {
			throw new BadCommandException(getName() + ": can't grabbe a filter while rotating");
		}
		log.info(getName() + ": grabbing " + filter.getName() + " at standby position.");
		
		putFilterOnCarousel(filter);
                
		this.sendToReply(new CommandReply(getName() + ": " + filter.getName() + " is grabbed on carousel.", CommandStatus.OK));
		String ack = getName() + ": " + filter.getName() + " is grabbed at standby position";
		return ack;
		
	}


    public void  ungrabbeFilterAtStandby(Filter filter) throws BadCommandException, ErrorInCommandExecutionException, HardwareErrorDetectedException {
            
//            if (!autochangerLatchesAreLocked())
//                throw new BadCommandException("Carousel can't ungrabbe filter if it's not held by autochanger");
        if (!isOnStandby(filter))
			throw new BadCommandException(getName() + ": filter has to be at standby position to be ejected");
        if (!isClampedOnFilterAtStandby())
                throw new BadCommandException(getName() + ": can't ungrabbe filter which is already unclamped.");

        log.info(getName() + ": ungrabbing " + filter.getName() + " at standby position.");

        // The clamps can be opened.
        unlockClamps();
        log.info(this.toString());
        log.info(getName() + ": socket at standby state : " + getSocketAtStandby().toString());
        log.info(this.toString());
        removeFilterFromCarousel(filter);
        log.info(getName() + ": filter is removed from carousel at standby / " + getSocketAtStandby().toString());
        this.sendToReply(new CommandReply(getName() + ": " + filter.getName() + " is ungrabbed from carousel.", CommandStatus.OK));
    }


        @Override
        public String  releaseClamps() throws BadCommandException, ErrorInCommandExecutionException {
            log.info(getName() + ": releasing clamps on socket at standby position.");
            this.getClampsActuator().off();
            return getSocketAtStandby().releaseClamps();
        }

        @Override
        public String unlockClamps() throws BadCommandException, ErrorInCommandExecutionException {
            log.info(getName() + ": unlocking clamps on socket at standby position.");
            this.getClampsActuator().on();
            return getSocketAtStandby().unlockClamps();
        }


        private boolean isClampedOnFilterAtStandby() {
            return getSocketAtStandby().isClampedOnFilter();
        }


	public synchronized boolean isOnStandby(Filter filter) {
		return this.getPosition() == this.getStandbyPositionForFilter(filter);		
	}
	
	/**
	 * Add 2 angles : an angle is a double which value is between -360 and +360
	 * @param angle1
	 * @param angle2
	 * @return result of the angles addition. Always > 0
	 */
	public static double addAngle(double angle1, double angle2) {
		//if ( abs(angle1) > 360 || abs(angle2) > 360) throw new AssertionError("Adding ");

		double angle = angle1 + angle2;
		if (angle >= 360) {angle -= 360; 	}
		if (angle < 0) {angle += 360;     }	
		return angle;
	}
	
	//SETTERS and GETTERS

	public CarouselSocket[] getSockets() {
		return sockets;
	}

	public void setSockets(CarouselSocket[] sockets) {
		this.sockets = sockets;
	}
	


	public Motor getCarouselMotor() {
		return carouselMotor;
	}

	public void setCarouselMotor(Motor engine) {
		this.carouselMotor = engine;
	}


	public Latch getLatch() {
		return lock;
		
	}

	public void setLatch(Latch latch) {
		this.lock = latch;
	}


	public synchronized ModuleState getState() {
		return state;
	}


	public synchronized void setState(ModuleState state) {
		this.state = state;
	}

    /**
     * @return the nbSockets
     */
    public int getNbSockets() {
        return nbSockets;
    }

    /**
     * @param nbSockets the nbSockets to set
     */
    public void setNbSockets(int nbSockets) {
        this.nbSockets = nbSockets;
    }

    /*
     * Before rotating the carousel we have to release the power contact between the
     * moving part and the fixed part.
     */
    private void releaseClampsContact() {
        this.getSocketAtStandby().releaseClampsContact();
    }

    /*
     * After a rotation, when the carousel is stopped the power contact can be engaged
     * again.
     */
    private void engageClampsContact() {
        this.getSocketAtStandby().engageClampsContact();
    }

    /**
     * @return the clampsActuator
     */
    public Actuator getClampsActuator() {
        return clampsActuator;
    }

    /**
     * @param clampsActuator the clampsActuator to set
     */
    public void setClampsActuator(Actuator clampsActuator) {
        this.clampsActuator = clampsActuator;
    }



}
