/*
 * Decompiled with CFR 0.152.
 */
package org.lsst.ccs.subsystems.fcs;

import java.io.Serializable;
import java.util.EnumMap;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.Autochanger;
import org.lsst.ccs.subsystems.fcs.AutochangerTruck;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.FcsActions;
import org.lsst.ccs.subsystems.fcs.FcsAlerts;
import org.lsst.ccs.subsystems.fcs.MainModule;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByAutochangerTwoTrucks;
import org.lsst.ccs.subsystems.fcs.StatusDataPublishedByOnlineProximityProbe;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.states.FcsState;
import org.lsst.ccs.subsystems.fcs.utils.ActionGuard;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

public class AutochangerTwoTrucks
extends MobileItem {
    private static final Logger FCSLOG = Logger.getLogger(AutochangerTwoTrucks.class.getName());
    @LookupField(strategy=LookupField.Strategy.ANCESTORS)
    private Subsystem subs;
    private final AutochangerTruck truckXminus;
    private final AutochangerTruck truckXplus;
    @ConfigurationParameter(description="Which of the two trucks is master. Should be 'Xminus' if acTruckXminus is master, or 'Xplus' if acTruckXplus is master. Used in the build method", units="unitless", category="autochanger")
    private volatile String driverSide = "Xminus";
    @ConfigurationParameter(range="0..1000", description="Minimal position for autochanger trucks", units="micron", category="autochanger")
    private volatile int minActualPositionValue = 0;
    @ConfigurationParameter(range="0..1200000", description="Maximal position for autochanger trucks", units="micron", category="autochanger")
    private volatile int maxActualPositionValue = 1007000;
    @ConfigurationParameter(range="900000..1010000", description="STANDBY position to be used to store a filter on carousel", units="micron", category="autochanger")
    private volatile int standbyPosition = 997500;
    @LookupField(strategy=LookupField.Strategy.TREE)
    private AlertService alertService;
    @ConfigurationParameter(range="900000..1010000", description="STANDBY position to be used when grabing a filter on Carousel", units="micron", category="autochanger")
    private volatile int standbyPositionEmpty = 997500;
    @ConfigurationParameter(range="60000..80000", description="HANDOFF position", units="micron", category="autochanger")
    private volatile int handoffPosition = 70000;
    @ConfigurationParameter(range="0..5000", description="ONLINE position", units="micron", category="autochanger")
    private volatile int onlinePosition = 2600;
    @ConfigurationParameter(range="0..100000", description="Time for trucks motion from ONLINE to STANDBY", units="millisecond", category="autochanger")
    private volatile long timeoutForTrucksMotion = 20000L;
    @ConfigurationParameter(range="0..100000", description="Time to align follower on driver controller", units="millisecond", category="autochanger")
    private volatile long timeoutForAlignFollower = 5000L;
    @ConfigurationParameter(range="0..5000", description="Preliminary ONLINE position, used by process whichmoves a filter at ONLINE position.", units="micron", category="autochanger")
    private volatile int prelimaryTargetPosition = 3367;
    @ConfigurationParameter(range="0..5000", description="Natural position at ONLINE position used by process whichmoves a filter at ONLINE position.", units="micron", category="autochanger")
    private volatile int naturalPosition = 3215;
    @ConfigurationParameter(range="0..3000", description="Lowest position to be used by command moveAndClampFilterOnline.", units="micron", category="autochanger")
    private volatile int minTargetPosition = 1800;
    @ConfigurationParameter(range="0..6", description="Minimum value of voltage read on proximitySensor.", units="V", category="autochanger")
    private volatile double umin = 2.0;
    @ConfigurationParameter(range="0..8", description="Maximum value of voltage read on proximitySensor.", units="V", category="autochanger")
    private volatile double umax = 4.6;
    @ConfigurationParameter(range="0..8", description="Voltage target on proximitySensor.", units="V", category="autochanger")
    private volatile double utarget = 2.5;
    @ConfigurationParameter(range="0..2000", description="Adjustment factor.", units="micron/V", category="autochanger")
    private volatile double adjustmentFactor = 1200.0;
    @ConfigurationParameter(range="0..150", description="Low speed to approach STANDBY position", units="unitless", category="autochanger")
    private volatile int lowSpeed = 50;
    @ConfigurationParameter(range="150..500", description="High speed to approach STANDBY position", units="unitless", category="autochanger")
    private volatile int highSpeed = 260;
    @ConfigurationParameter(range="0..150", description="Low acceleration to approach STANDBY position", units="unitless", category="autochanger")
    private volatile int lowAcceleration = 50;
    @ConfigurationParameter(range="50..500", description="High acceleration to approach STANDBY position", units="unitless", category="autochanger")
    private volatile int highAcceleration = 310;
    @ConfigurationParameter(range="0..150", description="Low deceleration to approach STANDBY position", units="unitless", category="autochanger")
    private volatile int lowDeceleration = 50;
    @ConfigurationParameter(range="50..500", description="High deceleration to approach STANDBY position", units="unitless", category="autochanger")
    private volatile int highDeceleration = 310;
    @ConfigurationParameter(range="900000..1000000", description="Approach STANDBY position", units="micron", category="autochanger")
    private volatile int approachStandbyPosition = 950000;
    @ConfigurationParameter(range="5000..15000", description="Approach ONLINE position", units="micron", category="autochanger")
    private volatile int approachOnlinePosition = 9000;
    @ConfigurationParameter(range="10000..300000", description="Slow motion point until online", units="micron", category="autochanger")
    private volatile int slowMotionPosition = 200000;
    private int driverPosition;
    private int followerPosition;
    private int deltaPosition;
    private int positionRange = 100;
    @ConfigurationParameter(range="0..300", description="Position is reached at STANDBY if |targetPosition - driverPosition| < positionRangeAtStandby", units="micron", category="autochanger")
    private volatile int positionRangeAtStandby = 100;
    @ConfigurationParameter(range="0..100", description="Position is reached at ONLINE if |targetPosition - driverPosition| < positionRangeAtOnline", units="micron", category="autochanger")
    private volatile int positionRangeAtOnline = 50;
    @ConfigurationParameter(range="0..300", description="Position is reached at HANDOFF if |targetPosition - driverPosition| < positionRangeAtOnHandoff", units="micron", category="autochanger")
    private volatile int positionRangeAtHandoff = 100;
    private int positionRangeAtHandoffForLoader = 50;
    private int positionRangeForRelativeMotion = 50;
    @ConfigurationParameter(range="0..300", description="Position is reached in TRAVEL if |targetPosition - driverPosition| < positionRangeAtOnline", units="micron", category="autochanger")
    private volatile int positionRangeInTravel = 150;
    @ConfigurationParameter(range="0..300", description="Value in alignFollower when we want a strict alignment : alignment of follower is done when |deltaPosition| > deltaPositionForAlignStrict.", units="micron", category="autochanger")
    private volatile int deltaPositionForAlignStrict = 200;
    @ConfigurationParameter(range="0..500", description="Value in alignFollower when we want a lenient alignment : alignment of follower is done when |deltaPosition| > deltaPositionForAlignLenient.", units="micron", category="autochanger")
    private volatile int deltaPositionForAlignLenient = 400;
    @ConfigurationParameter(range="0..100", description="Value in alignFollower when we want a strict alignment for interactions with Loader : alignment of follower is done when |deltaPosition| > deltaPositionForAlignStrictForLoader.", units="micron", category="autochanger")
    private volatile int deltaPositionForAlignStrictLoader = 50;
    @ConfigurationParameter(range="0..400", description="Proximity to preliminaryTargetPosition to allow to find the true ONLINE position", units="micron", category="autochanger")
    private volatile int deltaPreliminaryTargetPosition = 200;
    private int absoluteTargetPosition = 0;
    private int followerTargetPosition = 0;
    private final EPOSControllerForLinearRail acTruckXminusController;
    private final EPOSControllerForLinearRail acTruckXplusController;
    private EPOSControllerForLinearRail linearRailDriverController;
    private EPOSControllerForLinearRail linearRailFollowerController;
    @LookupField(strategy=LookupField.Strategy.TREE)
    private MainModule main;
    @LookupField(strategy=LookupField.Strategy.TREE)
    private Autochanger autochanger;
    @LookupField(strategy=LookupField.Strategy.TREE, pathFilter=".*\\/proximitySensorsDevice")
    private ADCInterface proximitySensorsDevice;
    @ConfigurationParameter(description="Input number on the ADC where proximity sensor is plugged", range="1..4", units="unitless", category="autochanger")
    private volatile int proximitySensorInput = 1;
    private double proximityVoltage = 0.0;
    private boolean atHandoff = false;
    private boolean atOnline = false;
    private boolean atStandby = false;

    public AutochangerTwoTrucks(AutochangerTruck truckXminus, AutochangerTruck truckXplus, EPOSControllerForLinearRail acTruckXminusController, EPOSControllerForLinearRail acTruckXplusController) {
        this.truckXminus = truckXminus;
        this.truckXplus = truckXplus;
        this.acTruckXminusController = acTruckXminusController;
        this.acTruckXplusController = acTruckXplusController;
    }

    public void build() {
        this.dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerTwoTrucks.class, this.path);
        this.dataProviderDictionaryService.registerClass(StatusDataPublishedByOnlineProximityProbe.class, this.path);
        this.movementCounter = new EnumMap(FcsActions.MobileItemAction.class);
        for (FcsActions.MobileItemAction action : new FcsActions.MobileItemAction[]{FcsActions.MobileItemAction.ALIGN_FOLLOWER, FcsActions.MobileItemAction.MOVE_FOLLOWER_TRUCK_ALONE, FcsActions.MobileItemAction.MOVE_DRIVER_TRUCK_ALONE, FcsActions.MobileItemAction.MOVE_TO_ABSOLUTE_POSITION}) {
            action.registerDurationPerElement(this.dataProviderDictionaryService, this.path);
            this.movementCounter.put(action, PersistentCounter.newCounter(action.getCounterPath(this.path), this.subs, action.name()));
        }
        if ("Xminus".equals(this.driverSide)) {
            this.linearRailDriverController = this.acTruckXminusController;
            this.linearRailFollowerController = this.acTruckXplusController;
        } else {
            this.linearRailFollowerController = this.acTruckXminusController;
            this.linearRailDriverController = this.acTruckXplusController;
        }
    }

    public void init() {
        super.init();
        ClearAlertHandler alwaysClear = new ClearAlertHandler(){

            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        this.alertService.registerAlert(FcsAlerts.HARDWARE_ERROR.getAlert(this.name), alwaysClear);
        this.alertService.registerAlert(FcsAlerts.SDO_ERROR.getAlert(this.name), alwaysClear);
        this.alertService.registerAlert(FcsAlerts.AC_TRUCKS_ERROR.getAlert(), alwaysClear);
    }

    @ConfigurationParameterChanger
    public void setNaturalPosition(int pos) {
        int minLimit = this.onlinePosition - 1500;
        int maxLimit = this.onlinePosition + 1500;
        if (pos < minLimit || pos > maxLimit) {
            throw new IllegalArgumentException(pos + " bad value for natural position must be > " + minLimit + " and < " + maxLimit);
        }
        this.naturalPosition = pos;
    }

    public String getDriverSide() {
        return this.driverSide;
    }

    @Command(type=Command.CommandType.QUERY, level=1, description="Return actual trucks position (master position). Doesn't read again the position on the controller.")
    public int getPosition() {
        return this.driverPosition;
    }

    @Command(type=Command.CommandType.QUERY, level=1, description="Return HANDOFF position. Doesn't read again the position on the controller.")
    public int getHandoffPosition() {
        return this.handoffPosition;
    }

    @Command(type=Command.CommandType.QUERY, level=1, description="Return STANDBY position. Doesn't read again the position on the controller.")
    public int getStandbyPosition() {
        return this.standbyPosition;
    }

    @Command(type=Command.CommandType.QUERY, level=1, description="Return ONLINE position. Doesn't read again the position on the controller.")
    public int getOnlinePosition() {
        return this.onlinePosition;
    }

    public int getApproachStandbyPosition() {
        return this.approachStandbyPosition;
    }

    public int getApproachOnlinePosition() {
        return this.approachOnlinePosition;
    }

    public int getOnlineProximityDistance() {
        return (int)(this.adjustmentFactor * (this.utarget - this.proximityVoltage));
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the carrier is at handoff position. This command doesn't read again the sensors.")
    public boolean isAtHandoff() {
        return this.atHandoff;
    }

    public boolean isAtHandoffForLoader() {
        return Math.abs(this.driverPosition - this.handoffPosition) < this.positionRangeAtHandoffForLoader;
    }

    public boolean isBetweenSlowmoAndOnline() {
        return this.driverPosition < this.slowMotionPosition && this.driverPosition > this.onlinePosition;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the carrier is at ONLINE position. This command doesn't read again the sensors.")
    public boolean isAtOnline() {
        return this.atOnline;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return true if the carrier is at STANDBY position. This command doesn't read again the sensors.")
    public boolean isAtStandby() {
        return this.atStandby;
    }

    @Command(type=Command.CommandType.QUERY, level=1, description="Return true if the homing of the trucks has been done.")
    public boolean isHomingDone() {
        return this.linearRailFollowerController.isHomingDone() && this.acTruckXminusController.isHomingDone();
    }

    @Command(type=Command.CommandType.QUERY, level=1, description="Return true if TruckXminus position sensors or TruckXminus position sensors are in error.")
    public boolean isPositionSensorsInError() {
        return this.truckXminus.isPositionSensorsInError() || this.truckXplus.isPositionSensorsInError() || this.positionSensorsNonConsistant();
    }

    private boolean positionSensorsNonConsistant() {
        return this.isAtStandby() && this.isAtHandoff() || this.isAtStandby() && this.isAtOnline() || this.isAtHandoff() && this.isAtOnline();
    }

    public boolean isPositionSensorErrorsTransient() {
        int delta = 5000;
        boolean errorHandoff = this.truckXminus.isHandoffSensorsInError() || this.truckXplus.isHandoffSensorsInError();
        boolean errorOnline = this.truckXminus.isOnlineSensorsInError() || this.truckXplus.isOnlineSensorsInError();
        boolean errorStandby = this.truckXminus.isStandbySensorsInError() || this.truckXplus.isStandbySensorsInError();
        return errorHandoff && Math.abs(this.driverPosition - this.handoffPosition) < delta || errorOnline && Math.abs(this.driverPosition - this.onlinePosition) < delta || errorStandby && Math.abs(this.driverPosition - this.standbyPosition) < delta;
    }

    public void postStart() {
        FCSLOG.fine(() -> this.name + " BEGIN postStart.");
        if (this.acTruckXplusController.isBooted() && this.acTruckXminusController.isBooted()) {
            this.initializeControllers();
            this.acTruckXplusController.updateEposState();
            this.acTruckXminusController.updateEposState();
            this.acTruckXminusController.setBrakeActivatedPub(this.acTruckXminusController.isBrakeActivated());
            this.acTruckXplusController.setBrakeActivatedPub(this.acTruckXplusController.isBrakeActivated());
            this.acTruckXminusController.readProfileVelocity();
            this.acTruckXminusController.readProfileAcceleration();
            this.acTruckXminusController.readProfileDeceleration();
            this.acTruckXminusController.publishData();
            this.acTruckXplusController.publishData();
        }
        if (this.proximitySensorsDevice.isBooted()) {
            this.proximitySensorsDevice.initializeAndCheckHardware();
            this.proximitySensorsDevice.publishData();
        } else {
            this.raiseAlarm(FcsAlerts.HARDWARE_ERROR, " proximitySensorsDevice not booted: it will be impossible to clamp at ONLINE.", this.name);
        }
        FCSLOG.fine(() -> this.name + " END postStart.");
    }

    public void initializeControllers() {
        try {
            this.acTruckXplusController.initializeAndCheckHardware();
            this.acTruckXminusController.initializeAndCheckHardware();
            this.acTruckXplusController.checkHomingNeeded();
            this.acTruckXminusController.checkHomingNeeded();
        }
        catch (FailedCommandException | FcsHardwareException ex) {
            this.raiseAlarm(FcsAlerts.HARDWARE_ERROR, "Couldn't initialize controllers", this.name, (Exception)ex);
        }
    }

    @Command(type=Command.CommandType.ACTION, level=3, description="Configure Autochanger trucks driver and follower controllers.")
    public void configureControllers() {
        FCSLOG.info(this.name + " Begin configuration of the controllers");
        this.acTruckXminusController.configureDigitalInputOfLinearRails();
        this.acTruckXplusController.configureDigitalInputOfLinearRails();
        this.acTruckXminusController.configureDigitalOutputOfLinearRails();
        this.acTruckXplusController.configureDigitalOutputOfLinearRails();
        FCSLOG.info(this.name + " End configuration of the controllers");
    }

    @Command(type=Command.CommandType.QUERY, level=1, description="Return true if both controllers are initialized and homing of trucks are done.")
    public boolean isInitialized() {
        return this.acTruckXminusController.isInitialized() && this.acTruckXplusController.isInitialized() && this.isHomingDone();
    }

    @Command(type=Command.CommandType.QUERY, level=1, description="Check if the motion of trucks are allowed.")
    public void checkConditionsForTrucksMotion() {
        this.autochanger.checkFilterSafetyBeforeMotion();
        this.checkPrerequisitesForTrucksMotion();
    }

    private void checkPrerequisitesForTrucksMotion() {
        if (!this.acTruckXplusController.isInitialized() || !this.acTruckXminusController.isInitialized()) {
            String msg = this.name + " can't move autochanger trucks because controllers are not both initialized:\n";
            FCSLOG.severe(msg);
            throw new RejectedCommandException(msg);
        }
        this.alignFollowerLenient();
    }

    public void goToHandOff() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("goToHandOff");){
            this.moveToAbsoluteTargetPosition(this.handoffPosition);
        }
    }

    public void goToHandOffForLoader() {
        if (this.isAtHandoffForLoader()) {
            FCSLOG.info(this.name + " goToHandOffForLoader has nothing to do because position of trucks are already at a good position for loader. positionRangeAtHandoffForLoader = " + this.positionRangeAtHandoffForLoader);
            return;
        }
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("goToHandOffForLoader");){
            this.positionRange = this.positionRangeAtHandoffForLoader;
            this.absoluteTargetPosition = this.handoffPosition;
            FCSLOG.info(this.name + " accuracy of motion = " + this.positionRange + "microns");
            try (ReleaseAutochangerBrakes g = new ReleaseAutochangerBrakes();){
                this.checkConditionsForTrucksMotion();
                this.checkEposMode();
                this.checkReadyForAction();
                this.goToPosition(this.handoffPosition);
                this.checkPositionSensors();
                this.main.updateAgentState(FcsState.TRUCKS_STOPPED);
                this.publishData();
            }
        }
        this.alignFollowerForLoader();
    }

    public void goToOnline() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("goToOnline");){
            this.moveToAbsoluteTargetPosition(this.onlinePosition);
        }
    }

    public void goToStandby() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("goToOnline");){
            this.moveToAbsoluteTargetPosition(this.standbyPosition);
        }
    }

    @Command(type=Command.CommandType.ACTION, level=2, description="Move Autochanger trucks to the absolute position given as argument. At the end of motion : doesn't align follower then activate brakes and disable controllers.", alias="moveABSPos", timeout=20000)
    public void moveToAbsoluteTargetPosition(int targetPosition) {
        block22: {
            try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToAbsoluteTargetPosition");){
                if (targetPosition < this.minActualPositionValue || targetPosition > this.maxActualPositionValue) {
                    throw new IllegalArgumentException(this.name + "argument has to be between " + this.minActualPositionValue + " and " + this.maxActualPositionValue);
                }
                this.updatePositionAndPublish();
                if (this.driverPosition == targetPosition) {
                    FCSLOG.info(this.name + " is already at target position=" + targetPosition);
                    break block22;
                }
                try (ReleaseAutochangerBrakes g = new ReleaseAutochangerBrakes();){
                    this.checkConditionsForTrucksMotion();
                    this.checkEposMode();
                    this.checkReadyForAction();
                    this.absoluteTargetPosition = targetPosition;
                    FCSLOG.info(this.name + " going to absolute position: " + this.absoluteTargetPosition);
                    if (this.absoluteTargetPosition == this.handoffPosition) {
                        this.main.updateAgentState(FcsState.MOVING_TRUCKS_TO_HANDOFF);
                        this.positionRange = this.positionRangeAtHandoff;
                    } else if (this.absoluteTargetPosition == this.standbyPosition) {
                        this.main.updateAgentState(FcsState.MOVING_TRUCKS_TO_STANDBY);
                        this.positionRange = this.positionRangeAtStandby;
                    } else if (this.absoluteTargetPosition == this.onlinePosition) {
                        this.main.updateAgentState(FcsState.MOVING_TRUCKS_TO_ONLINE);
                        this.positionRange = this.positionRangeAtOnline;
                    } else if (this.absoluteTargetPosition == this.prelimaryTargetPosition) {
                        this.main.updateAgentState(FcsState.MOVING_TRUCKS_TO_ONLINE);
                        this.positionRange = this.deltaPreliminaryTargetPosition;
                    } else {
                        if (this.driverPosition > this.absoluteTargetPosition) {
                            this.main.updateAgentState(FcsState.MOVING_TRUCKS_TO_ONLINE);
                        } else {
                            this.main.updateAgentState(FcsState.MOVING_TRUCKS_TO_STANDBY);
                        }
                        this.positionRange = this.positionRangeInTravel;
                    }
                    FCSLOG.info(this.name + " accuracy of motion = " + this.positionRange + "microns");
                    this.goToPosition(this.absoluteTargetPosition);
                    this.checkPositionSensors();
                    this.main.updateAgentState(FcsState.TRUCKS_STOPPED);
                    this.publishData();
                }
            }
        }
    }

    private void moveFilterOnlineGeneric(boolean withClamping) {
        if (!this.isAtOnline()) {
            try (ReleaseAutochangerBrakes g = new ReleaseAutochangerBrakes();){
                this.moveToSlowMotionPositionWithHighVelocity();
                this.moveToApproachOnlinePositionWithLowVelocity();
                this.autochanger.waitForProtectionSystemUpdate();
                this.alignFollowerStrict();
                this.moveToTargetOnlinePosition();
            }
        }
        if (withClamping) {
            this.autochanger.lockFilterAtOnline();
            this.releaseTruckTensionOnClamps();
        }
        this.autochanger.updateFCSStateToReady();
        this.publishData();
    }

    public void moveAndClampFilterOnline() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveAndClampFilterOnline");){
            this.moveFilterOnlineGeneric(true);
        }
    }

    public void moveFilterOnlineNoClamping() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveFilterOnlineNoClamping");){
            this.moveFilterOnlineGeneric(false);
        }
    }

    public void unclampAndMoveFilterToHandoff() {
        if (!this.autochanger.isAtOnline()) {
            throw new RejectedCommandException(this.name + " autochanger has to be at ONLINE for this command.");
        }
        this.autochanger.getOnlineClamps().unlockAndOpen();
        if (this.autochanger.getOnlineClamps().isOpened()) {
            try (ReleaseAutochangerBrakes g = new ReleaseAutochangerBrakes();){
                this.moveToApproachOnlinePositionWithLowVelocity();
                this.slowProfile();
                this.goToHandOffForLoader();
            }
        }
    }

    @Command(type=Command.CommandType.QUERY, level=2, description="Compute the actual online target position")
    public int findTargetOnlinePosition() {
        this.updatePositionAndPublish();
        if (Math.abs(this.driverPosition - this.prelimaryTargetPosition) > this.deltaPreliminaryTargetPosition) {
            throw new RejectedCommandException(this.name + "Invalid position for this command. Should be within a distance " + this.deltaPreliminaryTargetPosition + " of preliminaryTargetPosition = " + this.prelimaryTargetPosition + " to be executed.");
        }
        double umeasured = this.readProximitySensorDevice();
        if (umeasured > this.umax) {
            throw new FcsHardwareException(this.name + " voltage returned by proximity sensor is too high.");
        }
        if (umeasured < this.umin) {
            throw new FcsHardwareException(this.name + " voltage returned by proximity sensor is too low.");
        }
        int newTargetPosition = (int)((double)this.prelimaryTargetPosition + this.adjustmentFactor * (this.utarget - umeasured));
        FCSLOG.info(this.name + " computed new target online position=" + newTargetPosition);
        newTargetPosition = Math.max(this.minTargetPosition, newTargetPosition);
        FCSLOG.info(this.name + " final new target online position=" + newTargetPosition);
        return newTargetPosition;
    }

    @Command(type=Command.CommandType.ACTION, level=2, description="Move a filter slowly to preliminaryTargetPosition, then computes the new target position and finally reaches it", timeout=5000)
    public void moveToTargetOnlinePosition() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToTargetOnlinePosition");
             ReleaseAutochangerBrakes g = new ReleaseAutochangerBrakes();){
            this.slowProfile();
            this.moveToAbsoluteTargetPosition(this.prelimaryTargetPosition);
            int newTargetPosition = this.findTargetOnlinePosition();
            this.moveToAbsoluteTargetPosition(newTargetPosition);
        }
    }

    @Command(type=Command.CommandType.ACTION, level=2, description="Release the tension on the clamps after they are locked by moving the trucks to a move natural position than the computed Online position.", timeout=2000)
    public void releaseTruckTensionOnClamps() {
        double umeasured = this.readProximitySensorDevice();
        if (umeasured > this.umax) {
            this.raiseAlarm(FcsAlerts.AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too high.");
        } else if (umeasured < this.umin) {
            this.raiseAlarm(FcsAlerts.AC_TRUCKS_ERROR, " voltage returned by proximity sensor is too low.");
        }
        this.goToPosition(this.naturalPosition);
        FCSLOG.info(this.name + " END process docking ONLINE.");
    }

    public void moveFilterToStandby() {
        this.moveFilterToStandbyPlusDelta(0);
    }

    public void moveFilterToStandbyPlusDelta(int deltaStandbyPosition) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveFilterToStandbyPlusDelta");){
            this.updatePositionAndPublish();
            this.autochanger.updateStateWithSensors();
            if (!this.atHandoff && !this.atOnline) {
                throw new RejectedCommandException(this.name + "autochanger should be at Online or Handoff position");
            }
            this.main.updateAgentState(FcsState.MOVING_TRUCKS_TO_STANDBY);
            if (!this.atOnline) {
                this.alignFollowerLenient();
            }
            try (ReleaseAutochangerBrakes g = new ReleaseAutochangerBrakes();){
                if (this.atOnline) {
                    this.moveToApproachOnlinePositionWithLowVelocity();
                    this.alignFollowerLenient();
                }
                if (this.isBetweenSlowmoAndOnline()) {
                    this.moveToSlowMotionPositionWithLowVelocity();
                }
                this.moveToApproachStandbyPositionWithHighVelocity();
                this.alignFollowerStrict();
                this.moveToStandbyPlusDeltaWithLowVelocity(deltaStandbyPosition);
                FCSLOG.info("filter ID on autochanger after moveToStandbyWithLowVelocity = " + this.autochanger.getFilterID());
                this.publishData();
            }
        }
    }

    public void moveToApproachStandbyPositionWithLowVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachStandbyPositionWithLowVelocity");){
            this.slowProfile();
            this.moveToAbsoluteTargetPosition(this.approachStandbyPosition);
        }
    }

    public void moveToApproachOnlinePositionWithLowVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachOnlinePositionWithLowVelocity");){
            this.slowProfile();
            this.moveToAbsoluteTargetPosition(this.approachOnlinePosition);
        }
    }

    public void moveToSlowMotionPositionWithLowVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachOnlinePositionWithLowVelocity");){
            this.slowProfile();
            this.moveToAbsoluteTargetPosition(this.slowMotionPosition);
        }
    }

    public void alignFollowerAndMoveEmptyFromApproachToHandoff() {
        this.alignFollowerAndMoveEmptyFromApproachToHandoff(false);
    }

    public void alignFollowerAndMoveEmptyFromApproachToHandoff(boolean withPrecision) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("alignFollowerAndMoveEmptyFromApproachToHandoff");){
            this.alignFollowerStrict();
            this.fastProfile();
            if (withPrecision) {
                this.goToHandOffForLoader();
            } else {
                this.goToHandOff();
            }
        }
    }

    public void moveToApproachStandbyPositionWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachStandbyPositionWithHighVelocity");){
            this.fastProfile();
            this.moveToAbsoluteTargetPosition(this.approachStandbyPosition);
        }
    }

    public void moveToApproachOnlinePositionWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachOnlinePositionWithHighVelocity");){
            this.fastProfile();
            this.moveToAbsoluteTargetPosition(this.approachOnlinePosition);
        }
    }

    public void moveToSlowMotionPositionWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachOnlinePositionWithHighVelocity");){
            this.fastProfile();
            this.moveToAbsoluteTargetPosition(this.slowMotionPosition);
        }
    }

    public void moveToStandbyPlusDeltaWithLowVelocity(int deltaStandbyPosition) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToStandbyWithLowVelocity");){
            FCSLOG.info(this.name + " === moveToStandbyWithLowVelocity ===");
            this.autochanger.checkCarouselDeltaPosition();
            this.slowProfile();
            if (this.autochanger.isEmpty()) {
                this.moveToAbsoluteTargetPosition(this.standbyPositionEmpty);
            } else {
                this.moveToAbsoluteTargetPosition(this.standbyPosition + deltaStandbyPosition);
            }
        }
    }

    public void moveToStandbyWithLowVelocity() {
        this.moveToStandbyPlusDeltaWithLowVelocity(0);
    }

    public void moveEmptyToStandbyWithLowVelocity() {
        FCSLOG.info(this.name + " === moveToStandbyEmptyWithLowVelocity ===");
        this.autochanger.checkCarouselDeltaPosition();
        this.slowProfile();
        this.moveToAbsoluteTargetPosition(this.standbyPositionEmpty);
    }

    public void moveToHandoffWithHighVelocity() {
        this.fastProfile();
        this.moveToAbsoluteTargetPosition(this.handoffPosition);
    }

    public void moveToHandoffWithLowVelocity() {
        this.slowProfile();
        this.moveToAbsoluteTargetPosition(this.handoffPosition);
    }

    private void goToPosition(int targetPosition) {
        try (ReleaseAutochangerBrakes g = new ReleaseAutochangerBrakes();){
            FCSLOG.info(this.name + " moving to position " + targetPosition);
            if (!this.isHomingDone()) {
                this.homing();
            }
            this.absoluteTargetPosition = targetPosition;
            this.executeAction(FcsActions.MobileItemAction.MOVE_TO_ABSOLUTE_POSITION, this.timeoutForTrucksMotion);
            this.main.updateAgentState(FcsState.TRUCKS_STOPPED);
            this.main.updateStateWithSensors();
        }
    }

    @Command(type=Command.CommandType.ACTION, level=3, description="Move to a position, bypassing all safety tests. Use this command only if you know what you are doing.")
    public void moveToAbsoluteTargetPositionDangerous(int absolutePosition) {
        this.goToPosition(absolutePosition);
    }

    @Command(type=Command.CommandType.ACTION, level=2, description="Activate brakes and disable the 2 controllers.")
    public void activateBrakesAndDisable() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("2Trucks-activateBrakesAndDisable");){
            FcsUtils.parallelRun(() -> this.linearRailDriverController.activateBrake(), () -> this.linearRailFollowerController.activateBrake());
            FcsUtils.sleep(this.autochanger.getWaitTimeForBrakeLR(), this.name);
            FcsUtils.parallelRun(() -> this.linearRailDriverController.goToSwitchOnDisabled(), () -> this.linearRailFollowerController.goToSwitchOnDisabled());
        }
    }

    @Override
    public void startAction(FcsActions.MobileItemAction action) {
        this.autochanger.checkLinearRailMotionAllowed();
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION: {
                this.checkEposMode();
                this.linearRailDriverController.writeTargetPosition(this.absoluteTargetPosition);
                this.linearRailDriverController.writeControlWord(EPOSEnumerations.ControlWord.ABSOLUTE_POSITION_AND_MOVE);
                break;
            }
            case ALIGN_FOLLOWER: {
                this.linearRailFollowerController.enableAndReleaseBrake();
                this.linearRailFollowerController.writeTargetPosition(this.followerTargetPosition);
                this.linearRailFollowerController.writeControlWord(EPOSEnumerations.ControlWord.ABSOLUTE_POSITION_AND_MOVE);
                break;
            }
            case MOVE_DRIVER_TRUCK_ALONE: {
                int relativeTargetPosition = this.absoluteTargetPosition - this.linearRailDriverController.readPosition();
                this.linearRailFollowerController.changeMode(EPOSEnumerations.EposMode.PROFILE_POSITION);
                this.linearRailDriverController.enableAndReleaseBrake();
                this.linearRailDriverController.writeTargetPosition(relativeTargetPosition);
                this.linearRailDriverController.writeControlWord(EPOSEnumerations.ControlWord.RELATIVE_POSITION_AND_MOVE);
                break;
            }
            case MOVE_FOLLOWER_TRUCK_ALONE: {
                int relativeTargetPosition = this.absoluteTargetPosition - this.linearRailFollowerController.readPosition();
                this.linearRailFollowerController.changeMode(EPOSEnumerations.EposMode.PROFILE_POSITION);
                this.linearRailFollowerController.enableAndReleaseBrake();
                this.linearRailFollowerController.writeTargetPosition(relativeTargetPosition);
                this.linearRailFollowerController.writeControlWord(EPOSEnumerations.ControlWord.RELATIVE_POSITION_AND_MOVE);
                break;
            }
            default: {
                throw new IllegalArgumentException("Action " + action.toString() + " undefined for autochanger trucks");
            }
        }
    }

    @Override
    public boolean isActionCompleted(FcsActions.MobileItemAction action) {
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION: {
                FCSLOG.info(this.name + " /driverPosition=" + this.driverPosition + " /position to reach = " + this.absoluteTargetPosition + " /positionRange = " + this.positionRange);
                return Math.abs(this.driverPosition - this.absoluteTargetPosition) < this.positionRange;
            }
            case ALIGN_FOLLOWER: {
                FCSLOG.info(this.name + " linearRailFollowerController.isTargetReached()=" + this.linearRailFollowerController.isTargetReached() + " /driverPosition=" + this.driverPosition);
                return this.linearRailFollowerController.isTargetReached();
            }
            case MOVE_DRIVER_TRUCK_ALONE: {
                boolean targetReached = this.linearRailDriverController.isTargetReached();
                boolean positionAttained = Math.abs(this.driverPosition - this.absoluteTargetPosition) < this.positionRangeForRelativeMotion;
                FCSLOG.info(String.format("%s: targetPosition=%d, driverPosition=%d, isTargetReached=%s", this.name, this.absoluteTargetPosition, this.driverPosition, targetReached));
                return targetReached && positionAttained;
            }
            case MOVE_FOLLOWER_TRUCK_ALONE: {
                boolean targetReached = this.linearRailFollowerController.isTargetReached();
                boolean positionAttained = Math.abs(this.followerPosition - this.absoluteTargetPosition) < this.positionRangeForRelativeMotion;
                FCSLOG.info(String.format("%s: targetPosition=%d, followerPosition=%d, isTargetReached=%s", this.name, this.absoluteTargetPosition, this.followerPosition, targetReached));
                return targetReached && positionAttained;
            }
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            this.acTruckXminusController.checkFault(false);
            this.acTruckXplusController.checkFault(false);
            this.updatePositionAndPublish();
            this.autochanger.carouselUpdateStandbyState();
        }
        catch (SDORequestException ex) {
            this.raiseWarning(FcsAlerts.SDO_ERROR, "=> ERROR IN READING CONTROLLERS:", this.name, (Exception)((Object)ex));
        }
    }

    @Override
    public void endAction(FcsActions.MobileItemAction action) {
        FCSLOG.info(this.name + " ENDING action: " + action.toString());
        this.autochanger.updateStateWithSensors();
        this.autochanger.carouselUpdateStandbyState();
        switch (action) {
            case MOVE_TO_ABSOLUTE_POSITION: {
                break;
            }
            case ALIGN_FOLLOWER: {
                this.stopAlignFollower();
                break;
            }
            case MOVE_DRIVER_TRUCK_ALONE: {
                this.linearRailDriverController.activateBrakeAndDisable();
                this.linearRailFollowerController.changeMode(EPOSEnumerations.EposMode.MASTER_ENCODER);
                break;
            }
            case MOVE_FOLLOWER_TRUCK_ALONE: {
                this.linearRailFollowerController.activateBrakeAndDisable();
                this.linearRailFollowerController.changeMode(EPOSEnumerations.EposMode.MASTER_ENCODER);
                break;
            }
        }
        this.linearRailDriverController.checkFault();
        this.linearRailFollowerController.checkFault();
    }

    @Override
    public void abortAction(FcsActions.MobileItemAction action, long delay) {
        FCSLOG.info(this.name + " ABORTING action: " + action.toString() + " within delay " + delay);
        switch (action) {
            case ALIGN_FOLLOWER: {
                this.stopAlignFollower();
                break;
            }
            default: {
                this.quickStopAction(action, 0L);
            }
        }
    }

    @Override
    public void quickStopAction(FcsActions.MobileItemAction action, long delay) {
        this.linearRailDriverController.quickStop();
        this.linearRailFollowerController.quickStop();
        switch (action) {
            case MOVE_DRIVER_TRUCK_ALONE: 
            case MOVE_FOLLOWER_TRUCK_ALONE: {
                this.linearRailFollowerController.changeMode(EPOSEnumerations.EposMode.MASTER_ENCODER);
                break;
            }
        }
    }

    @Command(type=Command.CommandType.ACTION, level=2, description="Enable and release brake for the 2 controllers.")
    public void enableControllersAndReleaseBrake() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("2Trucks-enableControllersAndReleaseBrake");){
            FcsUtils.parallelRun(() -> {
                this.linearRailDriverController.goToOperationEnable();
                this.linearRailDriverController.doReleaseBrake();
            }, () -> {
                this.linearRailFollowerController.goToOperationEnable();
                this.linearRailFollowerController.doReleaseBrake();
            });
        }
    }

    private void checkEposMode() {
        if (!this.linearRailDriverController.isInMode(EPOSEnumerations.EposMode.PROFILE_POSITION)) {
            throw new FcsHardwareException(this.name + " : linearRailDriverController is not in PROFILE_POSITION mode, can't move.");
        }
        if (!this.linearRailFollowerController.isInMode(EPOSEnumerations.EposMode.MASTER_ENCODER)) {
            throw new FcsHardwareException(this.name + " : linearRailFollowerController is not in MASTER_ENCODER mode, can't move. Follower controller mode =" + this.linearRailFollowerController.getMode());
        }
    }

    public void alignFollower(int deltaMax) {
        block14: {
            try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("alignFollower");){
                this.updatePositionAndPublish();
                if (!this.isHomingDone()) {
                    this.homing();
                }
                int startDeltaPosition = this.deltaPosition;
                FCSLOG.info("alignFollower delta " + this.deltaPosition + " (max " + deltaMax + ") " + (Math.abs(this.deltaPosition) > deltaMax ? "ALIGN" : "no align"));
                if (Math.abs(this.deltaPosition) <= deltaMax) break block14;
                try (ActivateAutochangerBrakes g = new ActivateAutochangerBrakes();){
                    if (this.linearRailDriverController.isEnabled()) {
                        FCSLOG.info("!!! driver controller was enabled");
                        this.activateBrakesAndDisable();
                    }
                    if (this.linearRailDriverController.isEnabled()) {
                        FCSLOG.warning("!!! driver controller was STILL enabled");
                        throw new FcsHardwareException(this.name + " driver controller must be disabled.");
                    }
                    this.doAlignFollower(this.driverPosition);
                    if (Math.abs(this.deltaPosition - startDeltaPosition) < 5) {
                        throw new FcsHardwareException(this.name + " alignFollower did nothing : is dirty power on ?");
                    }
                }
            }
        }
    }

    public void alignFollowerLenient() {
        this.alignFollower(this.deltaPositionForAlignLenient);
    }

    public void alignFollowerStrict() {
        this.alignFollower(this.deltaPositionForAlignStrict);
    }

    public void alignFollowerForLoader() {
        this.alignFollower(this.deltaPositionForAlignStrictLoader);
    }

    private void stopAlignFollower() {
        FCSLOG.info(this.name + " STOP ALIGN FOLLOWER");
        this.linearRailFollowerController.activateBrake();
        FcsUtils.sleep(this.autochanger.getWaitTimeForBrakeLR(), this.name);
        this.linearRailFollowerController.changeMode(EPOSEnumerations.EposMode.MASTER_ENCODER);
        this.linearRailFollowerController.goToSwitchOnDisabled();
        FCSLOG.info(this.name + " END STOP ALIGN FOLLOWER");
    }

    private void doAlignFollower(int pos) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("doAlignFollower");){
            this.linearRailFollowerController.changeMode(EPOSEnumerations.EposMode.PROFILE_POSITION);
            FCSLOG.info(this.name + " follower's alignment to be done. deltaPosition= " + this.deltaPosition);
            if (!this.linearRailFollowerController.isInMode(EPOSEnumerations.EposMode.PROFILE_POSITION)) {
                throw new FcsHardwareException(this.name + " can't align follower truck because couldn't change its mode to PROFILE_POSITION.");
            }
            this.followerTargetPosition = pos;
            this.executeAction(FcsActions.MobileItemAction.ALIGN_FOLLOWER, this.timeoutForAlignFollower);
        }
    }

    @Command(type=Command.CommandType.ACTION, level=2, timeout=3000, description="Move the truck attached to the follower controller, separately from the one attached to the driving controller. Motion is limited to a relative distance within 400 microns of the follower truck.")
    public void moveFollowerTruckAlone(int absolutePosition) {
        if (Math.abs(absolutePosition - this.linearRailDriverController.readPosition()) >= 400) {
            throw new RejectedCommandException("The requested position for the follower truck would send it more that 400 microns away from the driver truck. Aborting.");
        }
        FCSLOG.info(this.name + ": moving follower truck alone to absolute position " + absolutePosition);
        this.activateBrakesAndDisable();
        this.absoluteTargetPosition = absolutePosition;
        this.executeAction(FcsActions.MobileItemAction.MOVE_FOLLOWER_TRUCK_ALONE, 3000L);
        FCSLOG.info(this.name + ": follower truck position at the end of the move " + this.followerPosition);
    }

    @Command(type=Command.CommandType.ACTION, level=2, timeout=3000, description="Move the truck attached to the driver controller, separately from the one attached to the follower controller. Motion is limited to a relative distance within 400 microns of the follower truck.")
    public void moveDriverTruckAlone(int absolutePosition) {
        if (Math.abs(absolutePosition - this.linearRailFollowerController.readPosition()) >= 400) {
            throw new RejectedCommandException("The requested position for the driver truck would send it more that 400 microns away from the follower truck. Aborting.");
        }
        FCSLOG.info(this.name + ": moving driver truck alone to absolute position " + absolutePosition);
        this.activateBrakesAndDisable();
        this.absoluteTargetPosition = absolutePosition;
        this.executeAction(FcsActions.MobileItemAction.MOVE_DRIVER_TRUCK_ALONE, 3000L);
        FCSLOG.info(this.name + ": driver truck position at the end of the move " + this.driverPosition);
    }

    public void checkPositionSensors() {
        this.autochanger.updateStateWithSensors();
        if (this.absoluteTargetPosition == this.standbyPosition && !this.atStandby) {
            throw new FailedCommandException(this.name + ": check with sensors: STANDBY sensors don't confirm trucks position.");
        }
        if (this.absoluteTargetPosition == this.handoffPosition && !this.atHandoff) {
            FCSLOG.info("this.absoluteTargetPosition=" + this.absoluteTargetPosition);
            FCSLOG.info("this.handoffPosition=" + this.handoffPosition);
            FCSLOG.info("this.atHandoff=" + this.atHandoff);
            throw new FailedCommandException(this.name + ": check with sensors: HANDOFF sensors don't confirm trucks position.");
        }
        if (this.absoluteTargetPosition == this.onlinePosition && !this.atOnline) {
            throw new FailedCommandException(this.name + ": check with sensors: ONLINE sensors don't confirm trucks position.");
        }
    }

    @Command(type=Command.CommandType.ACTION, level=2, description="Do homing for both controllers.")
    public void homing() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("AC2Trucks-homing");){
            this.linearRailDriverController.homing();
            this.linearRailFollowerController.homing();
            this.updatePositionAndPublish();
        }
    }

    @Command(type=Command.CommandType.QUERY, level=1, description="Update trucks position in reading controller.")
    public void updatePositionAndPublish() {
        this.autochanger.getTcpProxy().updatePDOData();
        this.driverPosition = this.linearRailDriverController.getPosition();
        this.followerPosition = this.linearRailFollowerController.getPosition();
        this.deltaPosition = this.driverPosition - this.followerPosition;
        this.publishData();
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Update trucks position in reading controller and return position.")
    public int readPosition() {
        this.updatePositionAndPublish();
        return this.driverPosition;
    }

    @Command(type=Command.CommandType.QUERY, level=0, description="Return a voltage in mV which represents a distance between between filterand filter beam.")
    public double readProximitySensorDevice() {
        this.safeReadProximitySensorDevice();
        return this.proximityVoltage;
    }

    private void readProximityVoltage() {
        this.proximityVoltage = this.proximitySensorsDevice.readVoltage(this.proximitySensorInput);
    }

    private void safeReadProximitySensorDevice() {
        this.readProximityVoltage();
        FcsUtils.waitCondition(() -> this.proximityVoltage > 0.0, () -> this.readProximityVoltage(), "safeReadProximityVoltage", 200L);
    }

    @Override
    public boolean myDevicesReady() {
        return this.isInitialized();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-ac2Trucks");){
            AutochangerTwoTrucks autochangerTwoTrucks = this;
            synchronized (autochangerTwoTrucks) {
                this.atHandoff = this.truckXminus.isAtHandoff() && this.truckXplus.isAtHandoff();
                this.atOnline = this.truckXminus.isAtOnline() && this.truckXplus.isAtOnline();
                this.atStandby = this.truckXminus.isAtStandby() && this.truckXplus.isAtStandby();
            }
            this.updatePositionAndPublish();
        }
        this.truckXminus.publishData();
        this.truckXplus.publishData();
    }

    public void slowProfile() {
        this.slowProfileVelocity();
        this.slowProfileAcceleration();
        this.slowProfileDeceleration();
        this.checkProfileVelocity();
        this.checkProfileAcceleration();
        this.checkProfileDeceleration();
        this.linearRailDriverController.publishData();
        this.linearRailFollowerController.publishData();
    }

    public void fastProfile() {
        this.raiseProfileVelocity();
        this.raiseProfileAcceleration();
        this.raiseProfileDeceleration();
        this.checkProfileVelocity();
        this.checkProfileAcceleration();
        this.checkProfileDeceleration();
        this.linearRailDriverController.publishData();
        this.linearRailFollowerController.publishData();
    }

    public void slowProfileVelocity() {
        this.linearRailDriverController.changeProfileVelocity(this.lowSpeed);
        this.linearRailFollowerController.changeProfileVelocity(this.lowSpeed);
        FCSLOG.info(this.name + " new ProfileVelocity=" + this.lowSpeed);
    }

    public void raiseProfileVelocity() {
        this.linearRailDriverController.changeProfileVelocity(this.highSpeed);
        this.linearRailFollowerController.changeProfileVelocity(this.highSpeed);
        FCSLOG.info(this.name + " new ProfileVelocity=" + this.highSpeed);
    }

    private void checkProfileVelocity() {
        int driverV;
        int followerV = (int)this.linearRailFollowerController.readParameter(EPOSEnumerations.EposParameter.ProfileVelocity);
        if (followerV != (driverV = (int)this.linearRailDriverController.readParameter(EPOSEnumerations.EposParameter.ProfileVelocity))) {
            throw new FcsHardwareException(this.name + " driver and follower ProfileVelocity are not the same. DRIVER velocity =" + driverV + " and FOLLOWER velocity =" + followerV);
        }
    }

    private void checkProfileAcceleration() {
        long driverAcc;
        long followerAcc = this.linearRailFollowerController.readParameter(EPOSEnumerations.EposParameter.ProfileAcceleration);
        if (followerAcc != (driverAcc = this.linearRailDriverController.readParameter(EPOSEnumerations.EposParameter.ProfileAcceleration))) {
            throw new FcsHardwareException(this.name + " driver and follower ProfileAcceleration are not the same. DRIVER acceleration =" + driverAcc + " and FOLLOWER acceleration =" + followerAcc);
        }
    }

    private void checkProfileDeceleration() {
        long driverDec;
        long followerDec = this.linearRailFollowerController.readParameter(EPOSEnumerations.EposParameter.ProfileDeceleration);
        if (followerDec != (driverDec = this.linearRailDriverController.readParameter(EPOSEnumerations.EposParameter.ProfileDeceleration))) {
            throw new FcsHardwareException(this.name + " driver and follower ProfileDeceleration are not the same. DRIVER deceleration =" + driverDec + " and FOLLOWER deceleration =" + followerDec);
        }
    }

    public void raiseProfileAcceleration() {
        this.linearRailDriverController.changeProfileAcceleration(this.highAcceleration);
        this.linearRailFollowerController.changeProfileAcceleration(this.highAcceleration);
        FCSLOG.info(this.name + " new ProfileAcceleration=" + this.highAcceleration);
    }

    public void slowProfileAcceleration() {
        this.linearRailDriverController.changeProfileAcceleration(this.lowAcceleration);
        this.linearRailFollowerController.changeProfileAcceleration(this.lowAcceleration);
        FCSLOG.info(this.name + " new ProfileAcceleration=" + this.lowAcceleration);
    }

    public void raiseProfileDeceleration() {
        this.linearRailDriverController.changeProfileDeceleration(this.highDeceleration);
        this.linearRailFollowerController.changeProfileDeceleration(this.highDeceleration);
        FCSLOG.info(this.name + " new ProfileDeceleration=" + this.highDeceleration);
    }

    public void slowProfileDeceleration() {
        this.linearRailDriverController.changeProfileDeceleration(this.lowDeceleration);
        this.linearRailFollowerController.changeProfileDeceleration(this.lowDeceleration);
        FCSLOG.info(this.name + " new ProfileDeceleration=" + this.lowDeceleration);
    }

    public StatusDataPublishedByAutochangerTwoTrucks createStatusDataPublishedByAutochangerTwoTrucks() {
        StatusDataPublishedByAutochangerTwoTrucks s = new StatusDataPublishedByAutochangerTwoTrucks();
        s.setDriverPosition(this.driverPosition);
        s.setFollowerPosition(this.followerPosition);
        s.setDeltaDriverFollowerPosition();
        s.setHomingDone(this.isHomingDone());
        s.setAtHandoff(this.atHandoff);
        s.setAtOnline(this.atOnline);
        s.setAtStandby(this.atStandby);
        s.setInError(this.isPositionSensorsInError());
        return s;
    }

    public StatusDataPublishedByOnlineProximityProbe createStatusDataPublishedByOnlineProximityProbe() {
        StatusDataPublishedByOnlineProximityProbe s = new StatusDataPublishedByOnlineProximityProbe();
        s.setProximityVoltage(this.readProximitySensorDevice());
        s.setProximityDistance(this.getOnlineProximityDistance());
        return s;
    }

    @Override
    public void publishData() {
        this.subs.publishSubsystemDataOnStatusBus(new KeyValueData(this.path, (Serializable)this.createStatusDataPublishedByAutochangerTwoTrucks()));
        this.subs.publishSubsystemDataOnStatusBus(new KeyValueData(this.path, (Serializable)this.createStatusDataPublishedByOnlineProximityProbe()));
    }

    @Override
    public void shutdown() {
        super.shutdown();
        if (this.linearRailDriverController.isBooted() && this.linearRailFollowerController.isBooted()) {
            this.activateBrakesAndDisable();
        }
        FCSLOG.info(this.name + " is shutdown.");
    }

    public class ReleaseAutochangerBrakes
    extends ActionGuard {
        @Override
        public Runnable getPreAction() {
            return () -> AutochangerTwoTrucks.this.enableControllersAndReleaseBrake();
        }

        @Override
        public Runnable getPostAction() {
            return () -> AutochangerTwoTrucks.this.activateBrakesAndDisable();
        }
    }

    public class ActivateAutochangerBrakes
    extends ActionGuard {
        @Override
        public Runnable getPreAction() {
            return () -> AutochangerTwoTrucks.this.activateBrakesAndDisable();
        }

        @Override
        public Runnable getPostAction() {
            return () -> AutochangerTwoTrucks.this.enableControllersAndReleaseBrake();
        }

        @Override
        public Class<? extends ActionGuard> requiredEnclosure() {
            return ReleaseAutochangerBrakes.class;
        }
    }
}

