
package org.lsst.ccs.subsystems.fcs.common;

import java.time.Instant;
import java.util.logging.Logger;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.services.alert.AlertService.RaiseAlertStrategy;
import org.lsst.ccs.subsystems.fcs.FcsAlerts;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertHistory;
import org.lsst.ccs.bus.states.AlertState;

/**
 * This interface is implemented by each class that can raises an Alert.
 * @author virieux
 */
public interface AlertRaiser {
    Logger FCSLOG = Logger.getLogger(AlertRaiser.class.getName());
    /**
     * to separate Alert Name or ID from device name.
     */
    char alertSeparator = '/';

    Subsystem getSubsystem();

    /**
     * return alertService
     * @return
     */
    AlertService getAlertService();

    default char getAlertSeparator() {
        return alertSeparator;
    }


    /**
     * Raises an ALARM with a FcsAlert and a cause.
     * @param fcsAlert
     * @param cause
     */
    default void raiseAlarm(FcsAlerts fcsAlert, String cause) {
        getAlertService().raiseAlert(fcsAlert.getAlert(), AlertState.ALARM, cause);
    }

    /**
     * Raises an ALARM with a FcsAlert, a cause and an Exception.
     * @param fcsAlert
     * @param cause
     * @param ex
     */
    default void raiseAlarm(FcsAlerts fcsAlert, String cause, Exception ex) {
        FCSLOG.severe(ex.toString());
        raiseAlarm(fcsAlert, cause + " due to " + ex);
    }

    /**
     * Raises an ALARM with a FcsAlert, a cause and a deviceName.
     * @param fcsAlert
     * @param cause
     * @param deviceName
     */
    default void raiseAlarm(FcsAlerts fcsAlert, String cause, String deviceName) {
        getAlertService().raiseAlert(fcsAlert.getAlert(deviceName), AlertState.ALARM, cause);
    }

    /**
     * Raises an ALARM with a FcsAlert, a cause, a deviceName and an Exception
     * @param fcsAlert
     * @param cause
     * @param deviceName
     * @param ex
     */
    default void raiseAlarm(FcsAlerts fcsAlert, String cause, String deviceName, Exception ex) {
        raiseAlarm(fcsAlert, cause + " due to " + ex, deviceName);
    }

    /**
     * Clear Alarm
     * @param fcsAlert
     * @param deviceName
     */
    default void clearAlarm(FcsAlerts fcsAlert, String deviceName) {
        getAlertService().clearAlerts(fcsAlert.getAlert(deviceName).getAlertId());
    }

    /**
     * Raises an WARNING with a FcsAlert and a cause.
     * @param fcsAlert
     * @param cause
     */
    default void raiseWarning(FcsAlerts fcsAlert, String cause) {
        getAlertService().raiseAlert(fcsAlert.getAlert(), AlertState.WARNING, cause);
    }

    /**
     * Raises a WARNING with a FcsAlert, a cause and an Exception.
     * @param fcsAlert
     * @param cause
     * @param ex
     */
    default void raiseWarning(FcsAlerts fcsAlert, String cause, Exception ex) {
        FCSLOG.severe(ex.toString());
        raiseWarning(fcsAlert, cause + " due to " + ex);
    }

    /**
     * Raises an WARNING with a FcsAlert, a cause and a deviceName.
     * @param fcsAlert
     * @param cause
     * @param deviceName
     */
    default void raiseWarning(FcsAlerts fcsAlert, String cause, String deviceName) {
        getAlertService().raiseAlert(fcsAlert.getAlert(deviceName), AlertState.WARNING, cause);
    }

    /**
     * Raises an WARNING with a FcsAlert, a cause, a deviceName and an exception.
     * @param fcsAlert
     * @param cause
     * @param deviceName
     * @param ex
     */
    default void raiseWarning(FcsAlerts fcsAlert, String cause, String deviceName, Exception ex) {
        FCSLOG.warning(ex.toString());
        raiseWarning(fcsAlert, cause + " due to " + ex, deviceName);
    }

    /**
     * raise an alert only if it's a new alert. this method has to be used in
     * periodic tasks to avoid having a too big volume of raised alerts.
     *
     * @param fcsAlert
     * @param cause
     * @param deviceName
     * @param state
     */
    default void raiseAlertOnlyIfNew(FcsAlerts fcsAlert, String cause, String deviceName, AlertState state) {
        Alert alert = fcsAlert.getAlert(deviceName);
        getAlertService().raiseAlert(alert, state, cause, RaiseAlertStrategy.ON_SEVERITY_CHANGE);
    }

    default void raiseAlertOnlyEveryTenMinutes(FcsAlerts fcsAlert, String cause, String deviceName, AlertState state) {
        Alert alert = fcsAlert.getAlert(deviceName);
        String alertId = alert.getAlertId();
        boolean alertToBeRaised = true;
        RaisedAlertHistory history = getAlertService().getRaisedAlertSummary().getRaisedAlert(alertId);
        if (history != null) {
            if (history.getLatestAlertState() == state) {
                Instant lastAlertInstant = history.getLatestAlertCCSTimeStamp().getUTCInstant();
                alertToBeRaised = Instant.now().minusMillis(600000).isAfter(lastAlertInstant);
            }
        }
        if (alertToBeRaised) {
            getAlertService().raiseAlert(alert, state, cause);
        }
    }

    default void raiseAlarmOnlyIfNew(FcsAlerts fcsAlert, String cause, String deviceName) {
        raiseAlertOnlyIfNew(fcsAlert, cause, deviceName, AlertState.ALARM);
        FCSLOG.severe(() -> deviceName + ": " + cause);
    }

    default void raiseWarningOnlyIfNew(FcsAlerts fcsAlert, String cause, String deviceName) {
        raiseAlertOnlyIfNew(fcsAlert, cause, deviceName, AlertState.WARNING);
        FCSLOG.warning(() -> deviceName + ": " + fcsAlert + ": " + cause);
    }

    default void raiseWarningOnlyEveryTenMinutes(FcsAlerts fcsAlert, String cause, String deviceName) {
        raiseAlertOnlyEveryTenMinutes(fcsAlert, cause, deviceName, AlertState.WARNING);
        FCSLOG.warning(() -> deviceName + ": " + fcsAlert + ": " + cause);
    }

    default void raiseAlertOnlyEveryTenMinutes(FcsAlerts fcsAlert, String cause, String deviceName, Exception ex) {
        raiseAlertOnlyEveryTenMinutes(fcsAlert, cause + " due to " + ex, deviceName, AlertState.WARNING);
    }

}
