
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnDevice;

import java.util.logging.Logger;

/**
 * This is a model for the Inclinometer used in CPPM test bench and on whole
 * size prototype.
 *
 * Because for mechanical raisons, the filter changer system X axis has to be
 * horizontal in order to be able to rotate carousel or move autochanger trucks
 * along its rails. So we have 2 inclinometers which give redondant information.
 * We must read permanently the 2 inclinometers to check that inclinaison is
 * correct. We read the inclinaison on the plutoGateway.
 *
 * @author virieux
 */
public class Inclinometer implements SensorPluggedOnDevice, AlertRaiser, HasLifecycle {
    private static final Logger FCSLOG = Logger.getLogger(Inclinometer.class.getName());

    @LookupField(strategy = LookupField.Strategy.TOP)
    private Subsystem subs;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupName
    private String name = "unset";

    @ConfigurationParameter(range = "0..16", description = "Inclinometer byte number", units = "unitless", category = "sensor")
    private volatile int byteNumero;

    @ConfigurationParameter(description = "Inclinometer device name", units = "unitless", category = "sensor")
    private volatile String deviceName;

    private double voltage;

    private double inclinaison = 0;

    @Override
    public void build() {
    }

    @Override
    public void init() {
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(FcsAlerts.HARDWARE_ERROR.getAlert(getName()), alwaysClear);
    }

    /**
     *
     * methods overridden from interface SensorPLuggedOnDevice
     * @return
     */
    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDeviceName() {
        return deviceName;
    }

    @Override
    public int getByteNumero() {
        return byteNumero;
    }

    @Override
    public void updateValue(int newValue) {
        FCSLOG.fine(() -> name + " updating inclinaison with newValue read on gateway (unit=0,1Volt): " + newValue);
        voltage = newValue;
        inclinaison = convertVoltage2Degree(voltage);
    }

    /**
     *
     * end of methods overridden from interface SensorPLuggedOnDevice
     */

    /**
     * Returns inclinaison value in degrees. Doesn't read again plutoGateway.
     *
     * @return inclinaison
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns inclinaison value in degrees. Doesn't read again CANopen device.")
    public double getInclinaison() {
        return inclinaison;
    }

    /**
     *
     * @return voltage read on autochanger PlutoGateway
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Raw voltage read on autochanger Pluto gateway in units of 0.1 Volts")
    public double getVoltage() {
        return voltage;
    }

    /**
     * Convert a voltage unit = 0.1volt to degrees.
     *
     * @param voltage
     * @return angle in degrees or NaN if voltage value is not correct.
     */
    private double convertVoltage2Degree(double voltage) {
        // voltage unit= 0.1Volt
        double voltageInVolt = voltage / 10;
        double sin_angle = (voltageInVolt - 2.5) / 11.52;
        // if |sin_angle| > 1 this returns NaN.
        return Math.toDegrees(Math.asin(sin_angle));
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

}
