package org.lsst.ccs.subsystems.fcs;

import java.time.Duration;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystem.ocsbridge.states.CameraMotionState;
import org.lsst.ccs.subsystems.fcs.FcsActions.GeneralAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.CarouselPowerMode;
import org.lsst.ccs.subsystems.fcs.common.BridgeToLoader;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.states.AutochangerInclinationState;
import org.lsst.ccs.subsystems.fcs.states.CarouselPowerState;
import org.lsst.ccs.subsystems.fcs.states.FcsState;
import org.lsst.ccs.subsystems.fcs.states.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.states.ObservatoryFilterState;
import org.lsst.ccs.subsystems.fcs.states.TmaRotatorState;
import org.lsst.ccs.subsystems.fcs.utils.ActionGuard;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;

/**
 * The main module in FCS. Commands can be sent to the FcsMain from the CCS
 * Master Control Module (MCM) in normal operation mode or from a Console.
 *
 * The main goals of the FcsMain is :
 * - receive commands from MCM or ccs-console for the whole subsystem
 * - execute commands received from MCM or ccs-console
 * - handle the logic of the whole subsystem
 * - publish on the status bus the status of the whole subsystem
 *
 * @author FCS team
 *
 */
public class FcsMain extends MainModule implements HasDataProviderInfos {

    @SuppressWarnings("unused")
    private static final long serialVersionUID = 7669526660659959402L;
    private static final Logger FCSLOG = Logger.getLogger(FcsMain.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private BridgeToLoader bridgeToLoader;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private Carousel carousel;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private Autochanger autochanger;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private Loader loader;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private FilterManager filterManager;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    // Initialize the state of the software lock between TMA / rotator and FCS
    private volatile TmaRotatorState tmaAndRotatorMotionStatus = TmaRotatorState.UNKNOWN;

    // Predicate used to check the presence of the OCS-Bridge on the buses
    private final Predicate<AgentInfo> ocsBridgePredicate = (ai) ->  {
            return ai.getAgentProperty("agentType").equals(AgentType.OCS_BRIDGE.name());
    };

    /**
     * Average duration of commands issued by the observatory
     * These are not supposed to change much but have been set as configuration parameters for clarity
     * and ease of long term maintenance
     */
    @ConfigurationParameter(range = "0..200000", description = "Expected max duration for a filter exchange at zenith", units = "millisecond")
    public volatile int setFilterFastDuration = 95000;
    @ConfigurationParameter(range = "0..200000", description = "Expected max duration for a filter exchange at horizon", units = "millisecond")
    public volatile int setFilterSlowDuration = 120000;
    @ConfigurationParameter(range = "0..500000", description = "Expected max duration of a filter swap with loader (load or unload)", units = "millisecond")
    public volatile int loadUnloadFilterMaxDuration = 480000;

    /* previous socket ID of the filter which is on the autochanger. */
    @Persist
    private volatile int previousSocketID = -1;

    private boolean firstMcmPublication = true;

    private Map<GeneralAction, PersistentCounter> movementCounter;

    /* Store last time a function was executed to check for a minimal interval between execution */
    private AtomicLong lastUpdateStateWithSensors = new AtomicLong(0);

    @Override
    public void build() {
        if (bridgeToLoader == null) {
            throw new RuntimeException("The FcsMain expected a bridge to the Loader, but none was found.");
        }
        if (bridgeToLoader.equals(bridge)) {
            throw new RuntimeException(
                    "In FcsMain both bridges are the same instance. We expected two different bridges.");
        }
        super.build();

        //Registering the KeyValueData published below
        KeyValueData data = getDataForMcm();
        dataProviderDictionaryService.registerData(data);
        String mainPath = data.getKey();
        KeyValueDataList dataList = (KeyValueDataList) data.getValue();
        for ( KeyValueData d : dataList ) {
            String path = mainPath + "/" + d.getKey();
            DataProviderInfo info = dataProviderDictionaryService.getDataProviderDictionary().getDataProviderInfoForPath(path);
            String units = "unitless";
            String description = "";
            switch (d.getKey()) {
                case "filter_on_autochanger":
                    description = "OCS name of the filter currently on the autochanger";
                    break;
                case "filter_previous_socketID":
                    description = "ID of the socket on which the current filter used to sit";
                    break;
                case "autochanger_trucks_position":
                    description = "Absolute position of the autochanger trucks";
                    units = "micron";
                    break;
                case "autochanger_trucks_state":
                    description = "Known autochanger position (STANDBY, HANDOFF, ONLINE) or IN_TRAVEL";
                    break;
                case "proximity":
                    description = "Value of the autochanger proximity sensor";
                    break;
                default:
                    units = "unitless";
                    description = "";
            }
            info.addAttribute(DataProviderInfo.Attribute.UNITS, units);
            info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, description);
        }
        agentStateService.updateAgentState(ObservatoryFilterState.UNKNOWN);
    }

    @Override
    public void init() {
        super.init();
        /* define a role for my subsystem in order to make FcsGUI listen to my subsystem */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.wholefcs", "fcs");

        // register counters and durations for high level actions
        movementCounter = new EnumMap<>(GeneralAction.class);
        for (GeneralAction action : new GeneralAction[]{
            GeneralAction.SET_FILTER,
            GeneralAction.SET_FILTER_AT_HANDOFF_FOR_LOADER,
            GeneralAction.STORE_FILTER_ON_CAROUSEL,
            GeneralAction.SET_NO_FILTER,
            GeneralAction.DISENGAGE_FILTER_FROM_CAROUSEL,
            GeneralAction.LOAD_FILTER,
            GeneralAction.UNLOAD_FILTER,
            GeneralAction.CONNECT_LOADER,
            GeneralAction.DISCONNECT_LOADER}) {
            movementCounter.put(action, PersistentCounter.newCounter(action.getCounterPath(), subs, action.name()));
            action.registerDurationTopLevel(dataProviderDictionaryService);
        }

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(FcsAlerts.HARDWARE_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlerts.CA_LOCKING_ISSUE.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlerts.HARDWARE_ERROR.getAlert("carousel"), alwaysClear);
    }

    private void updateTmaAndRotatorStatus(TmaRotatorState tmaRotatorState) {
        TmaRotatorState currentMotionStatus = TmaRotatorState.valueOf(tmaAndRotatorMotionStatus.name());
        tmaAndRotatorMotionStatus = tmaRotatorState;
        if ( !currentMotionStatus.equals(tmaAndRotatorMotionStatus) ) {
            FCSLOG.info(name
                    + String.format(
                            " Registering a Camera Motion State (software lock) change from %s to %s",
                            currentMotionStatus.name(),
                            tmaAndRotatorMotionStatus.name()
                    )
            );
        }
    }
    private final StatusMessageListener cameraMotionStatusMessageListener = new StatusMessageListener() {

        private final AtomicBoolean isFirstStatusStateChangeNotification = new AtomicBoolean(true);

        @SuppressWarnings("rawtypes")
        @Override
        public void onStatusMessage(StatusMessage msg) {
            StateBundle stateToProcess = null;
            if (isFirstStatusStateChangeNotification.getAndSet(false)) {
                stateToProcess = msg.getState();
            } else if (msg instanceof StatusStateChangeNotification) {
                StatusStateChangeNotification stateChange = (StatusStateChangeNotification) msg;
                stateToProcess = stateChange.getNewState().diffState(stateChange.getOldState());
            }
            if (stateToProcess != null) {
                CameraMotionState cameraMotionState = stateToProcess.getState(CameraMotionState.class);
                if (cameraMotionState != null) {
                    FCSLOG.finer("Camera Motion State received from OCS-Bridge = " + cameraMotionState.name());
                    switch (cameraMotionState) {
                        case LOCKED:
                            updateTmaAndRotatorStatus(TmaRotatorState.LOCKED);
                            break;
                        case UNLOCKED:
                            updateTmaAndRotatorStatus(TmaRotatorState.UNLOCKED);
                            break;
                        default:
                            FCSLOG.warning(name + "Unknown Camera Motion (software lock) state: " + cameraMotionState.name());
                            updateTmaAndRotatorStatus(TmaRotatorState.UNKNOWN);
                    }
                }
            }
        }
    };

    @Override
    public void postInit() {
        // In case postInit is defined in MainModule, make sure to execute it here
        super.postInit();

        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(new AgentPresenceListener() {
            @Override
            public void connecting(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    if (ocsBridgePredicate.test(agent)) {
                        getMessagingAccess().addStatusMessageListener(
                                cameraMotionStatusMessageListener,
                                (m) -> {
                                    return ocsBridgePredicate.test(m.getOriginAgentInfo());
                                }
                        );
                        break;
                    }
                }
            }

            @Override
            public void disconnecting(AgentInfo agent) {
                if ( ocsBridgePredicate.test(agent) ) {
                    getMessagingAccess().removeStatusMessageListener(cameraMotionStatusMessageListener);
                    updateTmaAndRotatorStatus(TmaRotatorState.UNKNOWN);
                }
            }
        });
    }

    @Override
    public void postStart() {
        // In case postStart is defined in MainModule, make sure to execute it here
        super.postStart();

        // Request the OCS-Bridge to publish its states when FCS joins the buses
        for (AgentInfo agent : getMessagingAccess().getAgentPresenceManager().listConnectedAgents()) {
            if (ocsBridgePredicate.test(agent)) {
                ConcurrentMessagingUtils messagingUtils = new ConcurrentMessagingUtils(getMessagingAccess());
                CommandRequest cmd = new CommandRequest(agent.getName(), "publishState");
                FCSLOG.info("Requesting the OCS-Bridge to publish its states");
                messagingUtils.sendAsynchronousCommand(cmd);
            }
            break;
        }

        if ( FcsUtils.isProto() ) {
            // If we are running on prototype (in Paris) then bypass the software lock on telescope
            updateTmaAndRotatorStatus(TmaRotatorState.PROTOTYPE);
        }
    }

    @Override
    public void shutdown() {
        /* Force Carousel to wakeup before the FCS is shutdown to avoid reading bad values on the controller after restart */
        if (isPowerSaveActivated()) {
            carousel.powerOn();
        }
        // Stop listening to the buses
        getMessagingAccess().removeStatusMessageListener(cameraMotionStatusMessageListener);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Print the current software lock status on the telescope and rotator motion")
    public String getTelescopeSoftwareLockStatus() {
        return tmaAndRotatorMotionStatus.name();
    }

    @Override
    public void updateFCSStateToReady() {
        if (carousel.isInitialized() && autochanger.isInitialized() && loader.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            updateAgentState(FcsState.READY);
            updateAgentState(FilterReadinessState.READY);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Sets the filters state to READY even if some sensors are still missing in autochanger.")
    public void forceFilterReadinessStateToReady() {
        // because of some sensors still missing, autochanger is never initialized
        // but we need the Filter to be ready for filter exchanges
        // TODO GLOBAL REVIEW should be removed  - was still used in Nov 2023 !!
        updateAgentState(FcsState.READY);
        updateAgentState(FilterReadinessState.READY);
    }

    /**
     * Return true if the changer is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the changer is connected.")
    public boolean isChangerConnected() {
        return this.bridge.isReady();
    }

    /**
     * Return true if the loader is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected.")
    public boolean isLoaderConnected() {
        return autochanger.isLoaderConnected();
    }

    /**
     * Return true if the hardware of the changer is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the changer is ready.")
    public boolean isChangerReady() {
        return this.bridge.allDevicesBooted();
    }

    /**
     * Return true if the hardware of the loader is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the loader is ready.")
    public boolean isLoaderReady() {
        return this.bridgeToLoader.allDevicesBooted();
    }

    /**
     * Disconnect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Disconnect the loader hardware.")
    public void disconnectLoaderCANbus() {
        movementCounter.get(GeneralAction.DISCONNECT_LOADER).increment();
        long beginTime = System.currentTimeMillis();
        loader.disconnectLoaderCANbus();
        GeneralAction.DISCONNECT_LOADER.publishDurationTopLevel(subs, System.currentTimeMillis() - beginTime);
        /* after loader disconnection update state for GUI*/
        updateStateWithSensors();
    }

    /**
     * Connect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Connect the loader hardware.")
    public void connectLoaderCANbus() {
        movementCounter.get(GeneralAction.CONNECT_LOADER).increment();
        long beginTime = System.currentTimeMillis();
        loader.connectLoaderCANbus();
        GeneralAction.CONNECT_LOADER.publishDurationTopLevel(subs, System.currentTimeMillis() - beginTime);
        /* after loader connection update state for GUI*/
        updateStateWithSensors();
    }

    /**
     * For Whole FCS GUI. Has to be overridden in FcsMain.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the list of LOADER CANopen hardware that this subsystem manages.")
    @Override
    public List<String> listLoaderHardwareNames() {
        return this.bridgeToLoader.listHardwareNames();
    }

    /**
     * For whole FCS GUI.
     * Override listLoSensorsNames of MainModule
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the list of names of sensors plugged on Loader pluto gateway.")
    @Override
    public List<String> listLoSensorsNames() {
        return bridgeToLoader.listLoSensorsNames();
    }

    public boolean isFilterInCamera(int filterID) {
        return carousel.isFilterOnCarousel(filterID) || autochanger.isFilterOnTrucks(filterID);
    }

    public boolean isFilterAvailable(int filterID) {
        return getAvailableFilterMap().entrySet().stream().anyMatch(e -> e.getKey() == filterID);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Store the filter on a given socket of the carousel and move the empty autochanger to HANDOFF position.", autoAck = false, timeout = 60000)
    public void storeFilterOnSocket(String socketName) {
        storeFilterOnSocket(socketName, true);
    }

    private void storeFilterOnSocket(String socketName, boolean withPrecision) {
        try (AutoTimed at = new AutoTimed("storeFilterOnSocket")) {
            updateStateWithSensors();
            FCSLOG.info(name + " === About to store filter on Carousel and go empty to HANDOFF " + (withPrecision?"for loader ":"") + "===");

            String debugMsg = "\n" + name + " This command is used to store on Carousel the filter currently on the Autochanger.\n";

            // increment the action in the database
            movementCounter.get(GeneralAction.STORE_FILTER_ON_CAROUSEL).increment();
            carousel.getSocketAtStandby().storeFilterOnCarouselCounter.increment();
            final long beginTime = System.currentTimeMillis();

            // Check all conditions to store filter on the given socket
            if (!FilterReadinessState.READY.equals(this.getFilterReadinessState())) {
                throw new RejectedCommandException(debugMsg + "FilterReadinessState must be READY");
            }
            if (!autochanger.isAvailable() ) {
                throw new RejectedCommandException(debugMsg + "Autochanger is currently unavailable");
            }
            if (!autochanger.isHoldingFilter()) {
                throw new RejectedCommandException(debugMsg + "Autochanger is not currently holding a filter");
            }

            /* If the autochanger is at STANDBY and its filter is already clamped on Carousel, we can't be sure what the user want:
             *  + could be putting the filter in the current socket at STANDBY and move autochanger to HANDOFF
             *  + OR could be to put the filter into another socket.
             * Thus we prefer to let the user go to a non-ambiguous situation.
             */
            if (autochanger.isAtStandby() && autochanger.isCarouselHoldingFilterAtStandby()) {
                throw new RejectedCommandException(debugMsg + "Autochanger is at STANDBY and its filter is already clamped on the Carousel. Please do things manually.");
            }

            CarouselSocket desiredSocket = carousel.getSocketByName(socketName);

            if (!desiredSocket.isAvailable()) {
                throw new RejectedCommandException(debugMsg + socketName + " is unavailable");
            }
            if (!desiredSocket.isEmpty()) {
                throw new RejectedCommandException(debugMsg + socketName + " should be empty");
            }
            if (!desiredSocket.isAtStandby() && !carousel.isAvailable() ) {
                throw new RejectedCommandException(debugMsg + "The desired socket is not at STANDBY and the Carousel rotation is currently unavailable");
            }

            // Do the movements
            FCSLOG.info(name + " === All preconditions passed, starting the movements ===");

            agentStateService.updateAgentState(ObservatoryFilterState.UNLOADING);

            // Move autochanger to HANDOFF (no precision) if autochanger not already at HANDOFF or ONLINE
            if ( !(autochanger.isAtHandoff() || autochanger.isAtOnline())) {
                FCSLOG.info(name + " === Moving Autochanger to HANDOFF to enable further movements ===");
                autochanger.goToHandOff();
            }

            // Rotate to desired socket (do nothing if already at STANDBY && carousel correctly aligned)
            // We rotate before opening ONLINE clamps to keep the filter in safe position for telescope as long as possible
            carousel.rotateSocketToStandby(socketName);
            carousel.checkDeltaPosition();
            updateStateWithSensors();

            FCSLOG.info(name + " === Carousel is ready to receive a filter at STANDBY ===");

            // Make sure the autochanger is free to move
            if (autochanger.isAtOnline()) {
                AutochangerThreeOnlineClamps onlineClamps = autochanger.getOnlineClamps();
                onlineClamps.unlockAndOpen();
                updateStateWithSensors();
            }

            /* Move the Autochanger into the Carousel
             * we MUST wait for autochanger PLC update otherwise trucks motions would not be allowed
             */
            autochanger.waitForProtectionSystemUpdate();
            autochanger.moveFilterToStandbyPlusDelta(carousel.getSocketAtStandby().getDeltaAutochangerStandbyPosition());
            // because both subsystems sensors have changed we have to update again
            updateStateWithSensors();

            // We check filter is indeed locked in the socket
            if (!carousel.isHoldingFilterAtStandby()) {
                recoveryLockingProcess();
                if (!carousel.isHoldingFilterAtStandby()) {
                    String msg = name + ": Carousel should be LOCKED_ON_FILTER when Autochanger is at STANDBY with a filter; recovery process didn't work";
                    this.raiseAlarm(FcsAlerts.CA_LOCKING_ISSUE, msg, name);
                    throw new FcsHardwareException(msg);
                }
            }
            carousel.getSocketAtStandby().updateFilterID();
            FCSLOG.info("filter " + filterManager.getObservatoryNameByID(carousel.getSocketAtStandby().getFilterID()) + " is now locked in the carousel");

            /* Unlock latches and move Autochanger to HANDOFF
             * we MUST wait for autochanger PLC update otherwise trucks motions would not be allowed
             */
            FCSLOG.info(name + ": is going to moveEmptyFromStandbyToHandoff");
            autochanger.waitForProtectionSystemUpdate();
            autochanger.moveEmptyFromStandbyToHandoff(withPrecision);

            agentStateService.updateAgentState(ObservatoryFilterState.UNLOADED);

            previousSocketID = -1;
            publishDataForMcm();

            // for duration
            GeneralAction.STORE_FILTER_ON_CAROUSEL.publishDurationTopLevel(subs, System.currentTimeMillis() - beginTime);
            GeneralAction.STORE_FILTER_ON_CAROUSEL.publishDurationPerElement(carousel.getSocketAtStandby().getSubsystem(), System.currentTimeMillis() - beginTime, carousel.getSocketAtStandby().path);
        }
    }

    private void storeFilterOnCarouselRelaxed() {
        storeFilterOnCarousel(false);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Store the filter in the carousel and move the empty autochanger to HANDOFF position for loader."
            + " Initial State for autochanger: a filter at HANDOFF or ONLINE. Final State for autochanger : empty at HANDOFF for loader.", autoAck = false, timeout = 60000)
    public void storeFilterOnCarousel() {
        storeFilterOnCarousel(true);
    }

    private void storeFilterOnCarousel(boolean withPrecision) {
        try (AutoTimed at = new AutoTimed("storeFilterOnCarousel")) {
            CarouselSocket desiredSocket = carousel.getSocketAtStandby();
            if (desiredSocket == null) {
                //TODO ALTERNATIVE: Should we create an alert for that ?
                String msg = name + ": there is no socket at STANDBY. Please rotate Carousel to the desired socket or use 'storeFilterOnSocket()'";
                throw new FcsHardwareException(msg);
            }
            storeFilterOnSocket(desiredSocket.getName(), withPrecision);
        }
    }

    private void recoveryLockingProcess() {
        try (AutoTimed at = new AutoTimed("recoveryLockingProcess")) {
            if (carousel.getClampXplus().isLocked() && !carousel.getClampXminus().isLocked()) {
                FcsUtils.sleep(200, name);
                carousel.recoveryLockingXminus();
            }
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, autoAck = false,
             description = "Unclamp filter from carousel, move autochanger to handoff position and release the carousel clamps",
             timeout = 15000)
    public void disengageFilterFromCarousel() {
        disengageFilterFromCarousel(false, false);
    }

    private void disengageFilterFromCarousel(boolean toOnline, boolean doClampOnline) {
        try (AutoTimed at = new AutoTimed("disengageFilterFromCarousel")) {
            final long beginTime = System.currentTimeMillis();
            movementCounter.get(GeneralAction.DISENGAGE_FILTER_FROM_CAROUSEL).increment();
            updateStateWithSensors();
            if (!FilterReadinessState.READY.equals(getFilterReadinessState())) {
                throw new RejectedCommandException(name + " FilterReadinessState must be READY to start movement, currently = " + getFilterReadinessState());
            }
            if (!carousel.isAtStandby()) {
                throw new RejectedCommandException(name + " Carousel should have a socket at STANDBY before issuing this command");
            }
            if (!(autochanger.isAtStandby())) {
                throw new RejectedCommandException(name + " Autochanger should be at STANDBY position before issuing this command");
            }
            if (!autochanger.isAvailable()) {
                throw new RejectedCommandException(name + " Autochanger is not available, canceling this command");
            }
            // Unclamp the filter from the carousel
            carousel.unlockClamps();
            carousel.updateStateWithSensors();
            carousel.waitForProtectionSystemUpdate();

            if (!carousel.isUnclampedOnFilterAtStandby()) {
                String msg = String.format(
                    "%s carousel clamps still locked. Aborting autochanger movement because carousel is still holding the filter.", name);
                String debug = String.format(
                    "\nsocketAtStandby should be UNLOCKED_ON_FILTER but current state is %s", carousel.getClampsStateAtStandby());
                raiseAlarm(FcsAlerts.HARDWARE_ERROR, msg + debug, name);
                throw new FcsHardwareException(msg + debug);
            }
            autochanger.waitForProtectionSystemUpdate();

            /* Published for the MCM */
            previousSocketID = carousel.getSocketAtStandbyID();

            try (ActionGuard g = autochanger.getAutochangerTrucks().new ReleaseAutochangerBrakes()) {
                // autochanger begins to leave standby position : Move autochanger to approach standby position
                autochanger.getAutochangerTrucks().moveToApproachStandbyPositionWithLowVelocity();
                publishDataForMcm();

                updateStateWithSensors();
                if (!carousel.isEmptyAtStandby()) {
                    String msg = String.format(
                        "Something went wrong when disengaging filter %s from carousel socket%d, aborting autochanger movement.",
                        filterManager.getObservatoryNameByID(autochanger.getFilterID()), previousSocketID
                    );
                    String help = "\nThe autochanger was sent to approachStandby position after unlocking the carousel clamps ";
                    help += "but the carousel clamps are still sensing the presence of the filter.";
                    help += "\nReach out immediately to a FES expert to assess the current status of the Filter Exchange System.";
                    raiseAlarm(FcsAlerts.HARDWARE_ERROR, msg + help, "carousel");
                    throw new FcsHardwareException(msg);
                }

                /**
                 * Release the clamps asynchronously, to prevent halting the movement of the trucks
                 * This way the autochanger can
                 * 1. go to approachStandby for safety
                 * 2. verify the clamp status and if ok start releasing async
                 * 3. continue movement without stopping (brakes are released) to its destination
                 */
                FcsUtils.asyncRun(() -> {
                    try {
                        carousel.releaseClamps();
                    } catch (Exception ex) {
                        String msg = "A problem has appeared during carousel clamp release : " + ex.getMessage();
                        String debug = String.format(
                            "\nCurrent clamp status is %s, X- is %s, X+ is %s",
                            carousel.getSocketAtStandby().getClampsState(),
                            carousel.getClampXminus().getClampState(),
                            carousel.getClampXplus().getClampState()
                        );
                        raiseWarning(FcsAlerts.HARDWARE_ERROR, msg + debug, "carousel");
                    }
                });

                if (toOnline) {
                    /* Move the filter to online position
                     * The methods below are in charge of setting the state to LOADED */
                    if (doClampOnline) {
                        autochanger.moveAndClampFilterOnline();
                    } else {
                        autochanger.goToOnline();
                    }
                } else {
                    // Move the filter to handoff position
                    autochanger.moveFilterToHandoff();
                }
            }
            updateStateWithSensors();
            GeneralAction.DISENGAGE_FILTER_FROM_CAROUSEL.publishDurationTopLevel(subs , System.currentTimeMillis() - beginTime);
        }
    }

    // SET FILTER PRECONDITIONS
    public boolean autochangerNotInTravel() {
        return autochanger.isAtHandoff() || autochanger.isAtOnline() || autochanger.isAtStandby();
    }

    public boolean latchesOpenOrClosed() {
        return autochanger.getLatches().isClosed() || autochanger.getLatches().isOpened();
    }

    public boolean onlineClampsOpenOrLocked() {
        return autochanger.getOnlineClamps().isLocked() || autochanger.getOnlineClamps().isOpened();
    }

    public boolean filterAtOnlineMustBeLocked() {
        return !autochanger.isAtOnline() || autochanger.isEmpty() || autochanger.getOnlineClamps().isLocked();
    }

    public boolean filterAtStandbyMustBeHeld() {
        return !autochanger.isAtStandby() || autochanger.isEmpty() || autochanger.isHoldingFilter()
                || carousel.isHoldingFilterAtStandby();
    }

    public boolean carouselHoldingFilterOrReadyToGrab() {
        return carousel.isHoldingFilterAtStandby() || carousel.isReadyToGrabAFilterAtStandby();
    }

    public boolean carouselReadyToClampAtStandby() {
        return autochanger.isAtStandby() || autochanger.isEmpty() || carousel.isReadyToGrabAFilterAtStandby();
    }

    // Are we in a situation for which a filter exchange is safe from the telescope point of view
    public boolean areTmaAndRotatorLocked() {
        return tmaAndRotatorMotionStatus.equals(TmaRotatorState.LOCKED) ||
                tmaAndRotatorMotionStatus.equals(TmaRotatorState.PROTOTYPE) ||
                isLoaderConnected() ||
                FcsUtils.isSimu();
    }

    // Are the trucks in a position that allows for the moves ?
    public boolean trucksMustBeStraight() {
        return autochanger.getAutochangerInclinationState().equals(AutochangerInclinationState.STRAIGHT);
    }

    /**
     * Main method to change to the desired filter and place it online to be ready for image acquisition.
     *
     * This command is both valid for glass filters or for no filter (ID = 0)
     * It does all the required checks to ensure the presence, availability of the filters and hardware,
     * and safety through the software interlock with the TMA and rotator.
     *
     * @param filterID ID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Move filter to ONLINE position.", timeout = 180000, autoAck = false)
    public void setFilter(int filterID) {
        /* At LPNHE, Paris, on the prototype, the autochanger clamps work differently. Hence for now we should not clamp */
        boolean doClampOnline = "autochanger-PROTO".equals(autochanger.getIdentifier()) ? false : true;

        /* Fail early if input filter ID is incorrect (avoid waking up the system) */
        if ( !filterManager.containsFilterID(filterID) ) {
            throw new RejectedCommandException("Unknown filter ID: " + filterID);
        }

        if ( filterID == 0 ) {
            setNoFilterAtHandoffOrOnline(true);
        } else {
            setFilterAtHandoffOrOnline(filterID, true, doClampOnline);
        }
    }

    /**
     * Main method to move the desired filter to the HANDOFF position used for filter swap using the loader.
     *
     * This command is both valid for glass filters or for no filter (ID = 0)
     * It does all the required checks to ensure the presence, availability of the filters and hardware,
     * and safety through the software interlock with the TMA and rotator.
     *
     * It also makes sure the precise alignment of the autochanger trucks at HANDOFF is done to prepare
     * for the exchange with the loader during filter swap.
     *
     * @param filterID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Select filter and move to HANDOFF position without going to ONLINE.", timeout = 180000, autoAck = false)
    public void setFilterAtHandoff(int filterID) {
        /* Fail early if input filter ID is incorrect (avoid waking up the system) */
        if ( !filterManager.containsFilterID(filterID) ) {
            throw new RejectedCommandException("Unknown filter ID: " + filterID);
        }

        if ( filterID == 0 ) {
            setNoFilterAtHandoffOrOnline(false);
        } else {
            setFilterAtHandoffOrOnline(filterID, false);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Store all filters on carousel and move autochanger trucks to ONLINE.", timeout = 180000, autoAck = false)
    public void setNoFilter() {
        setFilter(0);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Store all filters on carousel and move autochanger trucks to HANDOFF with loader precision.", timeout = 180000, autoAck = false)
    public void setNoFilterAtHandoff() {
        setFilterAtHandoff(0);
    }

    /**
     * Test whether the filter change is necessary or not given the current state of the system
     *
     * This test is used to exit early and silently the setFilter and setFilterAtHandoff commands
     * if the requested filter is already at the desired position
     *
     * @param filterID ID of the requested filter
     * @param toOnline whether we expected the requested filter ONLINE or at HANDOFF
     * @return true if we are already in position and can dismiss further movement
     */
    private boolean isRequestedFilterAlreadyInPosition(int filterID, boolean toOnline) {
        // First scenario: setNoFilter
        if (filterID == 0 ) {
            if ( toOnline ) {
                return autochanger.isEmptyOnline();
            } else {
                return autochanger.isEmpty() && autochanger.isAtHandoffForLoader();
            }
        // Second scenario: setFilter
        } else {
            if ( toOnline ) {
                return autochanger.isFilterClampedOnline(filterID);
            } else {
                return autochanger.isFilterOnTrucks(filterID) && autochanger.isAtHandoffForLoader();
            }
        }
    }

    /**
     * Set the desired filter at HANDOFF or ONLINE
     *
     * @param filterID ID of the selected filter
     * @param toOnline if true, move to ONLINE, else stop at HANDOFF with precision
     * @param doClampOnline if true, clamp filter ONLINE, else don't use the clamps (PROTO)
     */
    @SuppressWarnings("unchecked")
    private void setFilterAtHandoffOrOnline(int filterID, boolean toOnline, boolean doClampOnline) {
        updateStateWithSensors();

        if ( filterID == 0 ) {
            throw new RejectedCommandException(name + " This method is invalid for the No Filter case. Should use setNoFilterAtHandoffOrOnline instead");
        }

        String filterObsName = filterManager.getObservatoryNameByID(filterID);

        /**
         * Exit the method as early as possible if requested filter is already in position,
         * without waking up the carousel
         */
        if ( isRequestedFilterAlreadyInPosition(filterID, toOnline) ) {
            FCSLOG.info(name + " requested filter " + filterObsName + " already in position");
            return;
        }

        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setFilter")) {
            if (isPowerSaveActivated()) {
                wakeFilterChanger(CarouselPowerMode.WAKE_UP);
                autochanger.waitForProtectionSystemUpdate();
            }

            /* MCM information */
            publishDataForMcm();

            /* if a controller is in fault, FCS goes in ALARM state */
            checkControllers();


            FCSLOG.info(name + " filter requested online: " + filterObsName);

            subs.helper()
                    .precondition(!agentStateService.isInState(AlertState.ALARM), "Can't execute commands in ALARM state.")
                    .precondition(areTmaAndRotatorLocked(),
                            "The telescope motion and rotator need to be locked before performing a filter exchange. Current status of the lock is %s",
                            getTelescopeSoftwareLockStatus())
                    .precondition(trucksMustBeStraight(),
                            "The FCS does not allow a filter exchange to be executed because the camera is currently rotated. Ensure the rotator is set back to 0 before trying again.")
                    .precondition(filterManager.containsFilterID(filterID), "Unknown filter id : %s", filterID)
                    .precondition(isFilterInCamera(filterID), "The filter %s is currently out of the camera", filterObsName)
                    .precondition(isFilterAvailable(filterID), "The filter %s is currently not available", filterObsName)
                    .precondition(carousel.isAtStandby(), "The carousel has no socket currently at STANDBY position")
                    .precondition(autochangerNotInTravel(), "The autochanger trucks are not at STANDBY, HANDOFF or ONLINE position")
                    .precondition(autochanger.isAvailable(), "The autochanger is not available")
                    .precondition(latchesOpenOrClosed(),
                            "The autochanger latches should either be OPENED or CLOSED to proceed. Current state: %s",
                            autochanger.getLatches().getLockStatus())
                    .precondition(onlineClampsOpenOrLocked(),
                            "The autochanger ONLINE clamps should either be OPENED or LOCKED during normal operations. Current state: %s",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtStandbyMustBeHeld(),
                            "When a filter is at STANDBY it must be held by the autochanger or by the carousel.")
                    .precondition(autochanger.isNotHoldingFilter() || carouselHoldingFilterOrReadyToGrab(),
                            "The carousel socket at STANDBY should be holding a filter or ready to receive a filter. Current state: %s",
                            carousel.getClampsStateAtStandby())
                    .precondition(autochanger.isNotHoldingFilter() || carouselReadyToClampAtStandby(),
                            "The carousel state for the socket at STANDBY should be READYTOCLAMP " +
                            "when a filter is on the autochanger at HANDOFF or ONLINE. Current state: %s",
                            carousel.getClampsStateAtStandby())
                    .precondition(FilterReadinessState.READY)
                    .duration(Duration.ofMillis(setFilterSlowDuration))
                    .enterFaultOnException(true)
                    .action(() -> {
                        GeneralAction action = toOnline ? GeneralAction.SET_FILTER : GeneralAction.SET_FILTER_AT_HANDOFF_FOR_LOADER;
                        movementCounter.get(action).increment();

                        // For duration
                        final long beginTime = System.currentTimeMillis();

                        if (autochanger.isHoldingFilter()) {
                            FCSLOG.info(name + ": autochanger is currently holding filter " + autochanger.getFilterOnTrucksObservatoryName());
                        } else {
                            FCSLOG.info(name
                                + ": autochanger is empty, filter " + filterManager.getObservatoryNameByID(filterID)
                                + " is on carousel socket " + carousel.getFilterSocket(filterID).getId()
                            );
                        }
                        /**
                         * If the filter we want to move ONLINE is NOT on autochanger :
                         * - if autochanger is at STANDBY it has to be moved empty to HANDOFF
                         * - if autochanger is at HANDOFF or ONLINE, and another filter is on autochanger,
                         * this filter has to be stored on carousel, then autochanger can be moved empty at HANDOFF.
                         */
                        carousel.setControllerPositionSensorTypeEncoderSSI();
                        if (!autochanger.isFilterOnTrucks(filterID)) {
                            if (autochanger.isAtStandby()) {
                                FCSLOG.info(name + " autochanger is at STANDBY, it has to be moved empty to HANDOFF");
                                autochanger.moveEmptyFromStandbyToHandoff();
                            } else {
                                /* autochanger can be at HANDOFF or ONLINE with a filter or empty */
                                /* if a filter is on autochanger */
                                if (autochanger.isHoldingFilter()) {
                                    FCSLOG.info(name + ": is going to store filter " + autochanger.getFilterOnTrucksObservatoryName() + " on carousel");
                                    // No need here to use precision at handoff since the autochanger
                                    // is going back to the carousel next
                                    storeFilterOnCarouselRelaxed();
                                }
                            }

                            /* MCM information */
                            publishDataForMcm();

                            /* Now autochanger is empty at Handoff or at ONLINE*/
                            if (!(autochanger.isEmpty() && (autochanger.isAtHandoff() || autochanger.isAtOnline()))) {
                                throw new FcsHardwareException(name + ": autochanger should be empty at HANDOFF or ONLINE");
                            }
                            // Rotate desired filter to standby
                            carousel.rotateSocketToStandby(carousel.getFilterSocket(filterID).getName());

                            /* MCM information */
                            publishDataForMcm();

                            FcsUtils.sleep(100, name);
                            updateStateWithSensors();
                            if ( !carousel.isAtStandby() ) {
                                agentStateService.updateAgentState(ObservatoryFilterState.UNLOADED);
                                throw new FcsHardwareException(name + ": carousel should be at STANDBY after the rotateSocketToStandby command.");
                            }

                            /* At this point filterID should be on carousel at STANDBY */
                            if (carousel.getFilterAtStandbyID() == filterID) {
                                // Carousel could have continued rotation due to unbalanced state
                                carousel.checkDeltaPosition();
                                // Go and grab filter on autochanger
                                // wait for autochanger PLC to update after carousel rotation
                                // otherwise trucks controller is in UNDER_VOLTAGE fault.
                                autochanger.waitForProtectionSystemUpdate();

                                /* Go get the filter at STANDBY */
                                if ( toOnline ) {
                                    agentStateService.updateAgentState(ObservatoryFilterState.LOADING);
                                }
                                autochanger.grabFilterAtStandby();
                                updateStateWithSensors();

                                /* MCM information */
                                previousSocketID = carousel.getSocketAtStandbyID();
                                publishDataForMcm();

                            } else {
                                throw new FcsHardwareException(name
                                + " filter in carousel at STANDBY is " + filterManager.getObservatoryNameByID(carousel.getFilterAtStandbyID())
                                + " but it should be " + filterObsName);
                            }

                            updateStateWithSensors();
                            if (!autochanger.isFilterOnTrucks(filterID)) {
                                throw new FcsHardwareException(name + " filter " + filterObsName + " should now be on the autochanger");
                            }
                        }
                        /* now filterID is on autochanger. autochanger is at STANDBY or at HANDOFF or ONLINE. */
                        if (autochanger.isAtStandby()) {
                            // Unclamp filter from carousel and move to approach position
                            disengageFilterFromCarousel(toOnline, doClampOnline);
                        } else {
                            if (toOnline) {
                                if (doClampOnline) {
                                    if (autochanger.isAtHandoff()) {
                                        autochanger.moveAndClampFilterOnline();

                                    } else if (autochanger.isAtOnline()) {
                                        autochanger.lockFilterAtOnline();
                                    }
                                } else {
                                    autochanger.goToOnline();
                                }
                                agentStateService.updateAgentState(ObservatoryFilterState.LOADED);
                            } else {
                                autochanger.moveFilterToHandoff();
                                agentStateService.updateAgentState(ObservatoryFilterState.UNLOADED);
                            }
                        }
                        FCSLOG.info(name + " the filter " + filterObsName + " is now at " + (toOnline ? "ONLINE" : "HANDOFF"));
                        updateAgentState(FcsState.valueOf("ONLINE_" + filterManager.getFilterNameByID(filterID).toUpperCase()));

                        this.publishData();

                        /* go back to power save after a normal set filter if allowed */
                        if ( toOnline && isPowerSaveWanted() ) {
                            wakeFilterChanger(CarouselPowerMode.GO_TO_SLEEP);
                        }

                        action.publishDurationTopLevel(subs, System.currentTimeMillis() - beginTime);
                        this.publishData();

                        publishDataForMcm();

                    });
        }
    }

    /**
     * Default method should clamp when going ONLINE
     *
     * @param filterID
     * @param toOnline
     */
    private void setFilterAtHandoffOrOnline(int filterID, boolean toOnline) {
        setFilterAtHandoffOrOnline(filterID, toOnline, true);
    }

    /**
     * Set the autochanger empty at ONLINE for observations or HANDOFF for exchange with loader
     *
     * If a filter is currently on the autochanger, the command will first make sure to put it back
     * safely in the carousel before going empty to its destination.
     *
     * @param toOnline if true goes to ONLINE, else goes to HANDOFF
     */
    @SuppressWarnings("unchecked")
    private void setNoFilterAtHandoffOrOnline(boolean toOnline) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setNoFilter")) {
            updateStateWithSensors();
            if (isPowerSaveActivated()) {
                wakeFilterChanger(CarouselPowerMode.WAKE_UP);
            }
            publishDataForMcm();
            /* if a controller is in fault, FCS goes in ALARM state */
            checkControllers();
            subs.helper()
                    .precondition(!agentStateService.isInState(AlertState.ALARM),
                            "can't execute commands in ALARM state.")
                    .precondition(areTmaAndRotatorLocked(),
                            "The telescope motion and rotator need to be locked before performing a filter exchange. Current status of the lock is %s", getTelescopeSoftwareLockStatus())
                    .precondition(trucksMustBeStraight(),
                            "The FCS does not allow a filter exchange to be executed because the autochanger/camera is currently tilted. Ensure the rotator is set back to 0 before trying again.")
                    .precondition(carousel.isAtStandby(), "carousel not stopped at STANDBY position")
                    .precondition(autochangerNotInTravel(), "autochanger trucks are not at a HANDOFF or ONLINE or STANDBY")
                    .precondition(autochanger.isAvailable(), "autochanger is not available")
                    .precondition(latchesOpenOrClosed(),
                            "%s: bad state for autochanger latches - have to be OPENED or CLOSED",
                            autochanger.getLatches().getLockStatus())
                    .precondition(onlineClampsOpenOrLocked(),
                            "%s: bad state for autochanger ONLINE clamps - have to be OPENED or LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtOnlineMustBeLocked(),
                            "%s: bad state for autochanger ONLINE clamps - at ONLINE with a filter should be LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtStandbyMustBeHeld(),
                            "When a filter is at STANDBY it must be held by autochanger or by carousel.")
                    .precondition(autochanger.isNotHoldingFilter() || carouselHoldingFilterOrReadyToGrab(),
                            "%s: bad state for carousel : should be holding a filter or ready to receive a filter",
                            carousel.getClampsStateAtStandby())
                    .precondition(autochanger.isNotHoldingFilter() || carouselReadyToClampAtStandby(),
                            "%s: bad state for carousel when a filter is on autochanger at HANDOFF or ONLINE: should be READYTOCLAMP",
                            carousel.getClampsStateAtStandby())
                    .precondition(FilterReadinessState.READY)
                    .duration(Duration.ofMillis(setFilterSlowDuration))
                    .enterFaultOnException(true)
                    .action(() -> {
                        movementCounter.get(GeneralAction.SET_NO_FILTER).increment();
                        final long beginTime = System.currentTimeMillis();
                        if (autochanger.isHoldingFilter()) {
                    FCSLOG.info(
                            name + ": autochanger is holding filter: " + autochanger.getFilterOnTrucksObservatoryName());
                    FCSLOG.info(name + ": is going to store a filter on carousel");
                    storeFilterOnCarousel();
                } else {
                    FCSLOG.info(name + " no filter to store on carousel");
                }
                /* Now autochanger is empty at HANDOFF or Online */
                if (!(autochanger.isEmpty() && (autochanger.isAtHandoff() || autochanger.isAtOnline()))) {
                    throw new FcsHardwareException(name + ": autochanger is not empty at handoff");
                }
                if (toOnline) {
                    agentStateService.updateAgentState(ObservatoryFilterState.LOADING);
                    autochanger.goToOnline();
                }
                updateAgentState(FcsState.ONLINE_NONE);
                agentStateService.updateAgentState(ObservatoryFilterState.LOADED);

                if (isPowerSaveWanted()) {
                    wakeFilterChanger(CarouselPowerMode.GO_TO_SLEEP);
                }
                publishDataForMcm();
                this.publishData();
                GeneralAction.SET_NO_FILTER.publishDurationTopLevel(subs, System.currentTimeMillis() - beginTime);
                    });
        }
    }

    /**
     * just for mcm testing
     *
     * @param state
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "change subsystem state.")
    public void changeState(String state) {
        updateAgentState(FcsState.valueOf(state));
        FCSLOG.fine(() -> "SUBSYSTEM STATE=" + isInState(FcsState.valueOf(state)));
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "List the possible filters names as defined in the filter manager.")
    public List<String> listAllFilterNames() {
        return filterManager.getFilterNames();
    }

    /**
     * Produce a synthetic map of the filters in the FES organised by position (either autochanger or a socket)
     *
     * @return Map of FES position and associated filter
     */
    public Map<String, Filter> getFcsFilterMap() {
        Map<String, Filter> lf = new TreeMap<String, Filter>();
        if (autochanger.getFilterID() != 0) {
            lf.put("autochanger", filterManager.getFilterByID(autochanger.getFilterID()));
        }
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty() && socket.getFilterID() != 0) {
                lf.put(socket.getName(), filterManager.getFilterByID(socket.getFilterID()));
            }
        });
        return lf;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Print information on the filters physically in the LSSTCam.")
    public String printFilterMap() {
        StringBuilder sb = new StringBuilder("Filters on FES: \n");
        getFcsFilterMap().entrySet().stream().forEach(e -> {
            String socketName = e.getKey();
            if ( socketName != "autochanger" && !carousel.getSocketByName(socketName).isAvailable() ) {
                sb.append("* ");
            }
            sb.append(socketName);
            sb.append("\n\t");
            sb.append(e.getValue().toString());
            sb.append("\n");
        });
        sb.append("\n'*' marks the currently unavailable sockets with filters.\n");
        return sb.toString();
    }

    /**
     * This methods prints information on a given filter
     *
     * @param filterName
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Print general information on a filter.")
    public String printFilterInfo(String filterName) {
        return filterManager.getFilterByName(filterName).toString();
    }

    /**
     * This method lists the available filters installed in the FES
     * It starts with the list of filters installed in the camera and can be affected by the availability
     * of several elements of the Filter Exchange System such as
     * - the carousel unable to rotate
     * - the autochanger trucks unable to move
     * - one or several carousel sockets malfunctioning
     *
     * Such elements will be marked as unavailable during daytime or at night if an issue is encountered and the
     * system will thus switch to a degraded mode where the availability of filters is reduced.
     *
     * @return the list of available filters in a HashMap { filterID : filtername }
     *
     */
    private Map<Integer, String> getAvailableFilterMap() {
        Map<Integer, String> availableFilterMap = new TreeMap<Integer, String>();

        // First scenario, the autochanger is flagged as unavailable
        if (!autochanger.isAvailable()) {
            if (autochanger.isAtOnline()) {
                // either a filter is at ONLINE or it's empty and the filter will be NONE
                availableFilterMap.put(autochanger.getFilterID(), autochanger.getFilterOnTrucksName());
            }
            return availableFilterMap;
        }

        // Second scenario, the carousel rotation is not available but the autochanger is working fine
        //   - 1) a filter is on the current socket at STANDBY and the socket is available it can be used
        //   - 2) a filter is on the autochanger so it can be used
        //   - in both cases, if the socket is available, setNoFilter can also be used
        if (!carousel.isAvailable()) {
            if ( !carousel.socketAtStandby.isEmpty() ) {
                if ( carousel.socketAtStandby.isAvailable() ) {
                    availableFilterMap.put(carousel.socketAtStandby.getFilterID(), carousel.socketAtStandby.getFilterName());
                }
                availableFilterMap.put(0, "NONE");
            } else if ( !autochanger.isEmpty() ) {
                availableFilterMap.put(autochanger.getFilterID(), autochanger.getFilterOnTrucksName());
                if ( carousel.socketAtStandby.isAvailable() ) {
                   availableFilterMap.put(0, "NONE");
                }
            }
            return availableFilterMap;
        }

        // Third scenario, the carousel rotation is available and the autochanger is working fine, but some socket might be unavailable.
        // - start with a filter in autochanger
        if (!autochanger.isEmpty() && !autochanger.isAtStandby()) {
            availableFilterMap.put(autochanger.getFilterID(), autochanger.getFilterOnTrucksName());
            // If there is no free and available socket on the carousel to drop the filter in, we are stuck with the filter on autochanger
            if (!carousel.socketsMap.values().stream().anyMatch((socket) -> (socket.isEmpty() && socket.isAvailable()))) {
                return availableFilterMap;
            }
        };
        // - continue with available filters on the carousel
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty() && socket.getFilterID() != 0 && socket.isAvailable()) {
                Filter f = filterManager.getFilterByID(socket.getFilterID());
                availableFilterMap.put(f.getFilterID(), f.getName());
            }
        });
        // - end with the "NONE" filter
        availableFilterMap.put(0, "NONE");

        return availableFilterMap;
    }

    /**
     * @return List of filter IDs
     */
    private List<Integer> getAvailableFiltersID() {
        return getAvailableFilterMap().keySet().stream()
            .filter(filterID -> filterID != 0)
            .toList();
    }


    /***************************************************************
     *  __  __  ____ __  __      __                                *
     * |  \/  |/ ___|  \/  |    / /                                *
     * | |\/| | |   | |\/| |   / /                                 *
     * | |  | | |___| |  | |  / /                                  *
     * |_|  |_|\____|_|  |_| /_/                                   *
     *                                                             *
     *   ___   ____ ____        ____  ____  ___ ____   ____ _____  *
     *  / _ \ / ___/ ___|      | __ )|  _ \|_ _|  _ \ / ___| ____| *
     * | | | | |   \___ \ _____|  _ \| |_) || || | | | |  _|  _|   *
     * | |_| | |___ ___) |_____| |_) |  _ < | || |_| | |_| | |___  *
     *  \___/ \____|____/      |____/|_| \_\___|____/ \____|_____| *
     *                                                             *
     ***************************************************************/

    /**
     * The main command used by the Observatory to issue a filter change.
     *
     * It uses as input the filter name as defined for the Observatory, e.g. "g_6"
     * The method tests that the name is listed among the known filters and that the FES
     * is READY to move, and then delegates to logic to setFilter.
     *
     * @param filterName the name of the filter, which can either be in the form <name> or <name>_<id>
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Move filter to ONLINE position.", timeout = 180000, autoAck = false)
    public void setFilterByName(String filterName) {
        subs.helper()
                .precondition(filterManager.containsFilterName(filterName), "%s: Unknown filter name : %s", name, filterName)
                .precondition(isInState(FilterReadinessState.READY))
                .action(() -> {
                    setFilter(filterManager.getFilterID(filterName));
                });
    }

    /**
     * Command used by scripts to get the mapping of available sockets and associated filters
     *
     * @return dictionary of mapping between carousel sockets and filters
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide a mapping carousel sockets to available filters ID in the Filter Exchange System")
    public Map<String, Integer> getSocketsToAvailableFiltersID() {
        List<Integer> availableFiltersID = getAvailableFiltersID();
        Map<String, Integer> fm = new TreeMap<String, Integer>();
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if ( socket.isEmpty() ) {
                if ( socket.isAtStandby() && availableFiltersID.contains(autochanger.getFilterID()) ) {
                    fm.put(socket.getName(), autochanger.getFilterID());
                }
            } else if ( availableFiltersID.contains(socket.getFilterID()) ) {
                fm.put(socket.getName(), socket.getFilterID());
            }
        });
        return fm;
    }

    /**
     * This command is used on a daily basis by the OCS-bridge to request the list of available filters
     * in order to produce an observing schedule for the night.
     *
     * @return List of filter names
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the list of available filters in the Filter Exchange System formatted as wanted by the OCS Bridge")
    public List<String> getAvailableFilters() {
        return getAvailableFilterMap().keySet().stream()
            .map(filterID -> filterManager.getObservatoryNameByID(filterID))
            .collect(Collectors.toList());
    }

    /**
     * This command is called by the OCS-bridge to know the installed filters in the camera
     *
     * @return List of filter names
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the list of installed filters in the Filter Exchange System formatted as wanted by the OCS Bridge.")
    public List<String> getInstalledFilters() {
        return getFcsFilterMap().entrySet().stream()
            .map(entry -> entry.getValue().getObservatoryName())
            .collect(Collectors.toList());
    }

    /**
     * This command is called by the OCS-bridge to know the name of the filter in operation
     *
     * @return The filter name
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the name of the filter currently online, formatted as wanted by the OCS Bridge")
    public String getOnlineFilterName() {
        int filterId = 0;  // No filter
        if ( autochanger.isAtOnline() ) {
            filterId = autochanger.getFilterID();
        }

        return filterManager.getObservatoryNameByID(filterId);
    }

    /**
     * This command is called by the OCS-bridge to know the maximum angle (degrees) from vertical at which a filter exchange can be executed
     *
     * @return angle in degrees
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the maximum angle from vertical at which a filter exchange can be executed as wanted by the OCS Bridge.")
    public double getMaxAngleForFilterChange() {
        // TODO Alex: currently a placeholder, update this value
        return 90;
    }

    /**
     * This command is called by the OCS-bridge to know the maximum angle (degrees) from vertical at which a filter exchange can be executed in normal speed
     *
     * @return duration
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the maximum angle from vertical at which a filter exchange can be executed in normal speed (fast) as wanted by the OCS Bridge.")
    public double getMaxAngleForFastFilterChange() {
        /* The carousel is set to use the slow velocity for an airmass > 3
        zenith = arccos(1/airmass) => elevation ~ 70.5 degrees */
        int airmass = 3;
        return Math.acos(1. / airmass) * 180 / Math.PI;
    }

    /**
     * This command is called by the OCS-bridge to know the maximum time of a slow filter exchange during operations
     *
     * @return duration
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the maximum duration of a filter exchange in slow-mode conditions as wanted by the OCS Bridge.")
    public Duration getDurationForSlowFilterChange(String filterName) {
        return Duration.ofMillis(setFilterSlowDuration);
    }

    /**
     * This command is called by the OCS-bridge to know the maximum time of a normal filter exchange during operations
     *
     * @return duration
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the maximum duration of a filter exchange in normal conditions as wanted by the OCS Bridge.")
    public Duration getDurationForFastFilterChange(String filterName) {
        return Duration.ofMillis(setFilterFastDuration);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = " Is the power save (sleep) mode authorized in the FES configuration")
    public boolean isPowerSaveEnabled() {
        return carousel.isPowerSaveEnabled();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Does the carousel automatically goes to sleep after a filter change")
    public boolean isPowerSaveWanted() {
        return carousel.isPowerSaveWanted();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Is the carousel in power save (sleep) mode")
    public boolean isPowerSaveActivated() {
        return carousel.isPowerSaveActivated();
    }

    /**
     * This command is called by the OCS-bridge to control the power state of the carousel and get the system ready for a filter change in advance.
     *
     * @param mode is the wake up mode decided by the observatory 0: go to sleep, 1: wake up and go back to sleep after setFilter, 2: wake up and stay up
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Control the power state of the filter changer carousel. "
        + "Options are 0: go to sleep, 1: wake up and go back to sleep after setFilter, 2: wake up and stay up")
    public void wakeFilterChanger(int mode) {
        try {
            CarouselPowerMode wakeUpMode = CarouselPowerMode.values()[mode];
            FCSLOG.info("wakeFilterChanger was issued with mode : " + wakeUpMode);
            wakeFilterChanger(wakeUpMode);
        } catch (ArrayIndexOutOfBoundsException ex) {
            throw new RejectedCommandException("The only available modes for wakeFilterChanger are 0: go to sleep, 1: wake up, 2: wake up and stay up");
        }
    }

    private void wakeFilterChanger(CarouselPowerMode mode) {
        switch (mode) {
            case GO_TO_SLEEP -> {
                // First make sure going to power save is authorized from the configuration
                if ( !isPowerSaveEnabled() ) {
                    FCSLOG.info(name + " power save was requested but the current configuration does not allow it");
                    return;
                }
                carousel.setPowerSaveWanted(true);
                if ( agentStateService.isInState(CarouselPowerState.REGULAR) ) {
                    carousel.powerSave();
                } else {
                    FCSLOG.info("Filter changer already in power save");
                }
            }
            case WAKE_UP -> {
                carousel.setPowerSaveWanted(true);
                if ( agentStateService.isInState(CarouselPowerState.LOW_POWER) ) {
                    carousel.powerOn();
                } else {
                    FCSLOG.info("Filter changer already waken up");
                }
            }
            case STAY_UP -> {
                carousel.setPowerSaveWanted(false);
                if ( agentStateService.isInState(CarouselPowerState.LOW_POWER) ) {
                    carousel.powerOn();
                } else {
                    FCSLOG.info("Filter changer already waken up");
                }
            }
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Approximate duration of the wake up operation for the filter exchange system")
    public Duration getDurationForWakeUp(int mode) {
        return Duration.ofMillis(carousel.getWakeUpTimeout());
    }

    // END OF MCM / OCS-BRIDGE COMMANDS

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Put the filter changer in sleep mode. It will naturally wake up before a filter change.")
    public void sleep() {
        wakeFilterChanger(CarouselPowerMode.GO_TO_SLEEP);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Wake up the filter changer. It will naturally go back to sleep mode after a filter change.")
    public void wakeUp() {
        wakeFilterChanger(CarouselPowerMode.WAKE_UP);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Wake up the filter changer and make sure it DOES NOT go back to sleep mode after a filter change.")
    public void wakeUpAndStayUp() {
        wakeFilterChanger(CarouselPowerMode.STAY_UP);
    }

    /**
     * Load a filter from the loader to the camera. The loader must hold a filter at
     * STORAGE position. The autochanger must be empty at HANDOFF, latches open. At
     * the end of this command the filter is held by autochanger at Handoff
     * position, and the loader carrier is empty at STORAGE.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @SuppressWarnings("unchecked")
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Load a filter from the loader to the camera. "
            + "The loader must hold a filter at STORAGE position."
            + "The autochanger must be empty at HANDOFF, latches open. At the end of this command the"
            + "filter is held by autochanger at Handoff position, and the loader carrier is empty at STORAGE.",
            timeout = 480000, autoAck = false)
    public void loadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("loadFilter")) {
        updateStateWithSensors();
        subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM),
                        "can't execute commands in ALARM state.")
                // autochanger trucks must be empty at "Hand-Off" position
                .precondition(autochanger.isAtHandoff() && autochanger.isEmpty(),
                        " autochanger is not empty at HANDOFF position; can't load a filter.")
                // autochanger latches must be open.
                .precondition(autochanger.getLatches().isOpened(),
                        "Autochanger latches must be open before loadFilter command.")
                .duration(Duration.ofMillis(loadUnloadFilterMaxDuration))
                .enterFaultOnException(true)
                .action(() -> {
                    movementCounter.get(GeneralAction.LOAD_FILTER).increment();
                    final long beginTime = System.currentTimeMillis();

                    if (!autochanger.isAtHandoffForLoader()) {
                        FCSLOG.info(name + " autochanger not in correct position for exchange with loader, trying to align the trucks");
                        autochanger.goToHandOffForLoader();
                    }

                    /* When the loader is moved around between the clean room and the telescope, the vibrations may
                     * affect the force with which the hooks are clamped on the filter. The following ensures that the
                     * the loader hooks gets properly clamped again to secure the filter and continue the loading operation */

                    if ( loader.getCarrier().isLooslyBetweenStorageAndEngaged() && !loader.isClampedOnFilter() ) {
                        FCSLOG.info(name + " loader between STORAGE and ENGAGED with hooks not CLAMPED. "
                            + "Starting recovery to get back to a CLAMPED state to continue the load operation. ");
                        loader.getClamp().recoveryUnclamp();
                        FcsUtils.sleep(50, name);
                        updateStateWithSensors();
                        loader.getClamp().recoveryClamp();
                        FcsUtils.sleep(50, name);
                        updateStateWithSensors();
                    }

                    /* The following method is able to start from any position of the carrier
                     * between STORAGE and HANDOFF. It will bring the filter down to ENGAGED
                     * and unclamp the hooks before moving slowly to HANDOFF */

                    loader.moveFilterToHandoff();
                    FcsUtils.sleep(50, name);
                    updateStateWithSensors();

                    autochanger.closeLatches();
                    FcsUtils.sleep(50, name);
                    updateStateWithSensors();

                    loader.openClampAndMoveEmptyToStorage();
                    FcsUtils.sleep(50, name);
                    updateStateWithSensors();
                    GeneralAction.LOAD_FILTER.publishDurationTopLevel(subs, System.currentTimeMillis() - beginTime);
                });
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Load a filter from the loader to the camera"
            + "and store it on socket which name is given as argument. The loader must hold a filter at STORAGE position."
            + "The autochanger must be empty at HANDOFF, latches open. At the end of this command the"
            + "filter is store on carousel socket which name is given as argument, autochanger is empty at Handoff position, "
            + "and the loader carrier is empty at STORAGE.")
    public void loadFilterOnSocket(String socketName) {
        // note: this is a composite action LOAD_FILTER --> ROTATE_SOCKET_TO_STANDBY --> STORE_FILTER_ON_CAROUSEL
        // with 3 counters/durations
        updateStateWithSensors();
        FcsUtils.checkSocketName(socketName);
        CarouselSocket socket = carousel.getSocketsMap().get(socketName);
        if (!socket.isEmpty()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not empty. "
                    + "Select an empty socket.", socketName));
        }
        if (!socket.isAvailable()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not available. "
                    + "Select an available socket.", socketName));
        }
        loadFilter();
        carousel.rotateSocketToStandby(socketName);
        storeFilterOnCarousel();
    }

    /**
     * Unload a filter from the camera to the loader. The camera has to be at
     * horizontal position. The filter has to be on Autochanger at HANDOFF. Loader
     * carrier must be empty at STORAGE. At the end of this command, autochanger is
     * empty at HANDOFF, latches are open, and loader carrier is holding the filter
     * at STORAGE (hooks are clamped).
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @SuppressWarnings("unchecked")
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Unload a filter from the camera to the loader. The camera has to be at horizontal position. "
            + "The filter has to be on Autochanger at HANDOFF. Loader carrier must be empty at STORAGE. "
            + "\nAt the end of this command, autochanger is empty at HANDOFF, latches are open, "
            + "and loader carrier is holding the filter at STORAGE (hooks are clamped)",
            timeout = 480000, autoAck = false)
    public void unloadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("unloadFilter")) {
        updateStateWithSensors();
        subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM),
                        "can't execute commands in ALARM state.")
                // the auto changer trucks must hold a filter at "Hand-Off" position
                .precondition(autochanger.isAtHandoff() && autochanger.isHoldingFilter(),
                        " autochanger is not holding a filter at HANDOFF position; can't unload a filter.")
                .precondition(autochanger.isAvailable(), "autochanger is not available")
                .duration(Duration.ofMillis(loadUnloadFilterMaxDuration))
                .enterFaultOnException(true)
                .action(() -> {
                    movementCounter.get(GeneralAction.UNLOAD_FILTER).increment();
                    final long beginTime = System.currentTimeMillis();

            if (!autochanger.isAtHandoffForLoader()) {
                if (autochanger.isLinearRailMotionAllowed()) {
                    autochanger.goToHandOffForLoader();
                } else {
                    throw new RejectedCommandException("Autochanger trucks position is not correct for exchange with loader "
                            + " but command goToHandOffForLoader can't be executed because PLC doesn't allow to move trucks.");
                }
            }

            loader.moveEmptyToHandoffAndClose();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            autochanger.openLatches();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            loader.moveFilterToStorage();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            GeneralAction.UNLOAD_FILTER.publishDurationTopLevel(subs, System.currentTimeMillis() - beginTime);
                    }); // END of action
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Recover from a timeout or an exception raised during the carousel clamping process. "
                + "This command moves the filter back to handoff position and releases both carousel clamps.")
    public void recoveryCarouselClamping() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("recoveryCarouselClamping")) {
            carousel.unlockClamps();
            autochanger.moveToApproachStandbyPositionWithLowVelocity();
            carousel.releaseClamps();
            autochanger.moveToHandoffWithHighVelocity();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "check if controllers are in fault")
    @Override
    public void checkControllers() {
        this.bridge.checkControllers();
    }

    /**
     * Update state in reading sensors for all components of Filter Changer :
     * carousel, autochanger and loader. Read also autochanger trucks and ONLINE
     * clamps position.
     *
     * @throws FcsHardwareException
     */
    @Override
    public synchronized void updateStateWithSensors() {

        long beginTime = System.currentTimeMillis();

        // Check if the system is in simulation mode
        if (!FcsUtils.isSimu()) {
            // Check the time elapsed since the last sync was sent on the bus
            // If this amounts to less that 100ms, use the current values and do not send a sync
            // The 100 ms rate was decided so that it would be useful for the trending and still
            // be higher than the rate at which the system is able to receive consecutive syncs
            // which is ~ 50 ms according to Guillaume Daubard
            if (beginTime - lastUpdateStateWithSensors.get() < 100) {
                FCSLOG.info(name + " do not repeat updateStateWithSensors; last one was executed less than 100ms ago.");
                return;
            }
        }
        lastUpdateStateWithSensors.set(beginTime);

        // synchronized : parallel operations should do it just once and n-1 will just
        // wait.

        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-fcs")) {
            // Carousel
            carousel.updateStateWithSensors();
            if (carousel.isAtStandby()) {
                carousel.getSocketAtStandby().updateFilterID();
            }
            // Autochanger
            autochanger.updateStateWithSensors();
            if (autochanger.getAutochangerTrucks().isAtOnline() && autochanger.isHoldingFilter()
                    && autochanger.getOnlineClamps().isLocked()) {
                String filterName = filterManager.getFilterNameByID(autochanger.getFilterID());
                updateAgentState(FcsState.valueOf("ONLINE_" + filterName.toUpperCase()));
            } else {
                updateAgentState(FcsState.valueOf("ONLINE_NONE"));
            }
            // update loader sensors is loader is connected on camera
            if (loader.isCANbusConnected()) {
                loader.updateStateWithSensors();
            }
        }
        /* publish for MCM for the first time */
        if (firstMcmPublication) {
            this.publishDataForMcm();
            firstMcmPublication = false;
        }
    }

    @Override
    public void initializeHardware() {
        carousel.initializeHardware();
        autochanger.initializeHardware();
        loader.initializeHardware();
        postStart();
        FCSLOG.info("initializeHardware done");
    }

    @Override
    public void publishData() {
        super.publishData();
        bridgeToLoader.publishData();
    }

    public void publishDataForMcm() {
        // The following is used at startup to make sure the system is awake
        if (autochanger.getAutochangerTrucks().getPosition() == 0) {
            FcsUtils.sleep(100, "main");
            autochanger.updateStateWithSensors();
        }
        FCSLOG.info(name + " publishing data for MCM..");
        updateAgentState(autochanger.getAutochangerInclinationState());
        subs.publishSubsystemDataOnStatusBus(getDataForMcm());
    }

    /**
     * publishes data needed by MCM when MCM connects.
     * @param agents
     */
    @Override
    public void publishDataProviderCurrentData(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (AgentInfo.AgentType.MCM.equals(agent.getType())) {
                FCSLOG.info(name + " published for MCM that just joined the buses.");
                publishDataForMcm();
                break;
            }
        }
    }

    private KeyValueData getDataForMcm() {
        KeyValueDataList kvdl = new KeyValueDataList();
        kvdl.addData("filter_on_autochanger", filterManager.getObservatoryNameByID(autochanger.getFilterID()));
        kvdl.addData("filter_previous_socketID", previousSocketID);
        kvdl.addData("autochanger_trucks_position", autochanger.getAutochangerTrucks().getPosition());
        kvdl.addData("autochanger_trucks_state", autochanger.getAutochangerTrucksState());
        kvdl.addData("proximity", autochanger.getOnlineProximityDistance());
        return new KeyValueData(FCSCst.PUBLICATION_KEY_FOR_MCM, kvdl);
    }

    @Override
    public void postShutdown() {
        FCSLOG.info(name + " is shutting down.");
        bridge.doShutdown();
        if (loader.isCANbusConnected()) {
            bridgeToLoader.doShutdown();
        }
        FCSLOG.info(name + " is shutdown.");
    }

    @Override
    public String vetoTransitionToNormalMode() {
        // Ensure the filter state is correctly defined if FCS is starting up, assuming nothing is in motion
        if ( autochanger.isEmptyOnline() || autochanger.isFilterClampedOnline() ) {
            updateAgentState(ObservatoryFilterState.LOADED);
        } else {
            updateAgentState(ObservatoryFilterState.UNLOADED);
        }

        if ( !isChangerReady() ) {
            String msg = "The filter exchange system has devices not fully booted, cannot switch to Normal mode";
            FCSLOG.info(name + " vetoTransitionToNormalMode failed:\n" + msg);
            return msg;
        } else if ( !agentStateService.isInState(ObservatoryFilterState.LOADED) ) {
            String msg = String.format(
                "The filter exchange system is currently in filter state = %s.\n" +
                "Switching to Normal mode requires it to be in a LOADED state.\n" +
                "Use the autochanger level 1 command moveOnlineForScience, with or without filter, to prepare the system.",
                agentStateService.getState(ObservatoryFilterState.class)
            );
            FCSLOG.info(name + " vetoTransitionToNormalMode failed:\n" + msg);
            return msg;
        } else {
            FCSLOG.info(name + " vetoTransitionToNormalMode succeeded: current filter ONLINE = " + getOnlineFilterName());
            /* Publish information on the filter currently installed at ONLINE to the observatory */
            publishDataForMcm();
            /* Put the system to sleep to avoid forgetting it - It will also set powerSaveWanted = true which is what we want */
            sleep();
            /* Finally switch to Normal mode */
            return super.vetoTransitionToNormalMode();
        }
    }

}
