package org.lsst.ccs.subsystems.fcs;

import java.util.EnumMap;
import java.util.logging.Logger;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.FcsActions.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;


/**
 * This class is a model for the latch which lock a filter on the autochanger
 * trucks. The ACTION methods : open and close of this class is used in
 * ENGINEERING mode only when an operator may need to open or close only one
 * Latch. In NORMAL mode, when we have to open or close both Latches in the same
 * time, the class AutochangerTwoLatches is used. On the autochanger there are 2
 * latches : one on each side of the filter : at X+ and X-.
 *
 * @author virieux
 */
public class AutochangerLatch extends MobileItem implements MovedByEPOSController, ControlledBySensors {
    private static final Logger FCSLOG = Logger.getLogger(AutochangerLatch.class.getName());

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Autochanger autochanger;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    private final EPOSController latchController;
    private final ComplementarySensors filterEngagedSensors;
    private final ComplementarySensors closeSensors;
    private final ComplementarySensors openSensors;

    @ConfigurationParameter(range = "-1000..1000", description = "Current to be sent to the controller to open the latch.", units = "mA", category = "autochanger")
    private volatile int currentToOpen = -300;

    @ConfigurationParameter(range = "0..10000", description = "Timeout for opening or closing latch.", units = "millisecond", category = "autochanger")
    private volatile long timeoutForLatchMotion = 3000;

    private LockStatus lockStatus = LockStatus.UNKNOWN;

    /**
     * Builds a AutochangerLatchModule with a controller and 3 sensors.
     *
     * @param latchController
     * @param closeSensors
     * @param openSensors
     * @param filterEngagedSensors
     */
    public AutochangerLatch(EPOSController latchController, ComplementarySensors closeSensors,
            ComplementarySensors openSensors, ComplementarySensors filterEngagedSensors) {
        this.latchController = latchController;
        this.closeSensors = closeSensors;
        this.openSensors = openSensors;
        this.filterEngagedSensors = filterEngagedSensors;
    }
    /**
     * Initialize field autochanger and latchController
     */
    @Override
    public void init() {
        if (latchController == null) {
            FCSLOG.severe(() -> name + "==>>> latchController == null - Please fix groovy description file.");
            throw new IllegalArgumentException(name + "==>>> null latchController - fix groovy description file.");
        }

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(FcsAlerts.HARDWARE_ERROR.getAlert(latchController.getName()), alwaysClear);
        alertService.registerAlert(FcsAlerts.SDO_ERROR.getAlert(latchController.getName()), alwaysClear);
        alertService.registerAlert(FcsAlerts.AC_SENSOR_ERROR.getAlert(), alwaysClear);
    }

    @Override
    public void build() {
        // duration data
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerLatch.class, path);
        MobileItemAction.OPEN.registerDurationPerElement(dataProviderDictionaryService, path);
        MobileItemAction.CLOSE.registerDurationPerElement(dataProviderDictionaryService, path);
        // counter data
        movementCounter = new EnumMap<>(MobileItemAction.class);
        movementCounter.put(MobileItemAction.OPEN, PersistentCounter.newCounter(
            MobileItemAction.OPEN.getCounterPath(path), subs, MobileItemAction.OPEN.name()));
        movementCounter.put(MobileItemAction.CLOSE, PersistentCounter.newCounter(
            MobileItemAction.CLOSE.getCounterPath(path), subs, MobileItemAction.CLOSE.name()));
    }

    /**
     * Executed during INITIALIZATION phase. Initialize and check latchController.
     */
    @Override
    public void postStart() {
        FCSLOG.finer(() -> name + " BEGIN postStart.");
        if (latchController.isBooted()) {
            initializeController();
            latchController.updateEposState();
            latchController.publishData();
        }
        FCSLOG.finer(() -> name + " END postStart.");
    }

    private void initializeController() {
        try {
            latchController.initializeAndCheckHardware();

        } catch (FcsHardwareException ex) {
            this.raiseAlarm(FcsAlerts.HARDWARE_ERROR, name + " could not initialize controller", latchController.getName(), ex);
        }
        try {
            latchController.changeMode(EposMode.CURRENT);
        } catch (FcsHardwareException ex) {
            this.raiseWarning(FcsAlerts.HARDWARE_ERROR, name + " could not change mode of controller", latchController.getName(), ex);
        }
    }

    /**
     * Returns true if autochanger CANopen hardware is connected and ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if autochanger CANopen hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return latchController.isInitialized();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     *   ____      _   _                 __  ____       _   _              *
     *  / ___| ___| |_| |_ ___ _ __     / / / ___|  ___| |_| |_ ___ _ __   *
     * | |  _ / _ \ __| __/ _ \ '__|   / /  \___ \ / _ \ __| __/ _ \ '__|  *
     * | |_| |  __/ |_| ||  __/ |     / /    ___) |  __/ |_| ||  __/ |     *
     *  \____|\___|\__|\__\___|_|    /_/    |____/ \___|\__|\__\___|_|     *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Return latchController
     *
     * @return
     */
    @Override
    public EPOSController getController() {
        return latchController;
    }


    /**
     * For simulation, return currentToClose
     *
     * @return
     */
    public int getCurrentToClose() {
        return -currentToOpen;
    }

    /**
     * For simulation, return currentToClose
     *
     * @return
     */
    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * Return lockStatus. Used for simulation and tests.
     *
     * @return lockStatus
     */
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     *   _   _               _                          _            _       *
     * | | | | __ _ _ __ __| |_      ____ _ _ __ ___  | |_ ___  ___| |_ ___  *
     * | |_| |/ _` | '__/ _` \ \ /\ / / _` | '__/ _ \ | __/ _ \/ __| __/ __| *
     * |  _  | (_| | | | (_| |\ V  V / (_| | | |  __/ | ||  __/\__ \ |_\__ \ *
     * |_| |_|\__,_|_|  \__,_| \_/\_/ \__,_|_|  \___|  \__\___||___/\__|___/ *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Returns true if LockStatus=LOCKED Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if LOCKED. Doesn't read again sensors.", alias = "isClosed")
    public boolean isClosed() {
        return this.closeSensors.isOn();
    }

    /**
     * Returns true if LockStatus=UNLOCKED Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if UNLOCKED. Doesn't read again sensors.", alias = "isOpen")
    public boolean isOpened() {
        return this.openSensors.isOn();
    }

    /**
     * Returns true if latch is initialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if latch is initialized.")
    public boolean isInitialized() {
        return latchController.isInitialized();
    }

    /**
     * Returns true if LockStatus=ERROR, this means that openSensor and closeSensor
     * return non consistant values. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if in ERROR, this means that "
            + "openSensor and closeSensor return non consistant values. Doesn't read again sensors.")
    @Override
    public boolean isInError() {
        return this.lockStatus == LockStatus.ERROR;
    }

    /**
     * Returns true if autochanger is empty.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if autochanger is empty. Doesn't read again sensors.")
    public boolean isEmpty() {
        return !this.filterEngagedSensors.isOn();
    }

    public boolean isFilterEngagedInError() {
        return filterEngagedSensors.isInError();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     *   ____                                          _       *
     *  / ___|___  _ __ ___  _ __ ___   __ _ _ __   __| |___   *
     * | |   / _ \| '_ ` _ \| '_ ` _ \ / _` | '_ \ / _` / __|  *
     * | |__| (_) | | | | | | | | | | | (_| | | | | (_| \__ \  *
     *  \____\___/|_| |_| |_|_| |_| |_|\__,_|_| |_|\__,_|___/  *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Reads sensors, updates state and checks if sensors are in error. If sensors
     * are in error, raises an ALERT.
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(FcsAlerts.AC_SENSOR_ERROR, name);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     */
    protected void updateState() {
        boolean closed = this.closeSensors.isOn();
        boolean opened = this.openSensors.isOn();

        boolean someSensorsInError = this.filterEngagedSensors.isInError() || this.closeSensors.isInError()
                || this.openSensors.isInError();

        boolean inError = (closed && opened) || someSensorsInError;

        if (inError) {
            lockStatus = LockStatus.ERROR;
        } else if (closed) {
            lockStatus = LockStatus.CLOSED;
        } else if (opened) {
            lockStatus = LockStatus.OPENED;
        } else {
            lockStatus = LockStatus.INTRAVEL;
        }

        this.publishData();
    }

    /**
     * Updates the field readCurrent of the latch in reading the CPU of the
     * controller. This method is used in the simulator mainly.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update latch current in reading controller.")
    public void updateCurrent() {
        try {
            latchController.readCurrent();
            this.publishData();

            /*
             * we don't want to have an ALARM when monitor-current could not read current.
             */
        } catch (Exception ex) {
            raiseWarning(FcsAlerts.SDO_ERROR, " could not updateCurrent", latchController.getName(), ex);
        }
    }

    /**
     * Close this latch.
     *
     * @throws FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Close latch.")
    public void close() {
        try (AutoTimed at = new AutoTimed("ACLatch-close")) {
            updateStateAndCheckSensors();
            if (isClosed()) {
                FCSLOG.info(() -> name + " is already CLOSED. No action.");
                return;
            }
            autochanger.checkConditionsForClosingLatches();
            this.executeAction(MobileItemAction.CLOSE, timeoutForLatchMotion);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Close latch even if latch sensors are in error.")
    public void closeDangerous() {
        try (AutoTimed at = new AutoTimed("ACLatch-recoveryClose")) {
            updateStateAndCheckSensors();
            if (isClosed()) {
                FCSLOG.info(() -> name + " is already CLOSED. No action.");
                return;
            }
            autochanger.checkConditionsForClosingLatches(true);
            this.executeAction(MobileItemAction.CLOSE, timeoutForLatchMotion);
        }
    }

    /**
     * Open this latch.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Open latch.")
    public void open() {
        try (AutoTimed at = new AutoTimed("ACLatch-open")) {
            updateStateAndCheckSensors();
            if (isOpened()) {
                FCSLOG.info(() -> name + " is already OPENED. No action.");
                return;
            }
            autochanger.checkConditionsForOpeningLatches();
            this.executeAction(MobileItemAction.OPEN, timeoutForLatchMotion);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Open latch even if latch sensors are in error.")
    public void openDangerous() {
        try (AutoTimed at = new AutoTimed("ACLatch-recoveryOpen")) {
            updateStateAndCheckSensors();
            if (isOpened()) {
                FCSLOG.info(() -> name + " is already OPENED. No action.");
                return;
            }
            autochanger.checkConditionsForOpeningLatches(true);
            this.executeAction(MobileItemAction.OPEN, timeoutForLatchMotion);
        }
    }


    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     *  __  __       _     _ _         _        _   _              *
     * |  \/  | ___ | |__ (_) | ___   / \   ___| |_(_) ___  _ __   *
     * | |\/| |/ _ \| '_ \| | |/ _ \ / _ \ / __| __| |/ _ \| '_ \  *
     * | |  | | (_) | |_) | | |  __// ___ \ (__| |_| | (_) | | | | *
     * |_|  |_|\___/|_.__/|_|_|\___/_/   \_\___|\__|_|\___/|_| |_| *
     *                                                             *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Start Action OPEN or CLOSE
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void startAction(MobileItemAction action) {
        switch (action) {
            case OPEN -> {
                latchController.enableAndWriteCurrent((short) getCurrentToOpen());
            }
            case CLOSE -> {
                latchController.enableAndWriteCurrent((short) getCurrentToClose());
            }
            default -> {
                throw new IllegalArgumentException("Action on latch must be OPEN or CLOSE");
            }
        }
    }

    /**
     * Return true if action is completed with success.
     *
     * @param action
     * @return
     */
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        return switch (action) {
            case OPEN -> lockStatus == LockStatus.OPENED;
            case CLOSE -> lockStatus == LockStatus.CLOSED;
            default -> false;
        };
    }

    /**
     * Read sensors and update State. Used during an action.
     */
    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        autochanger.updateStateWithSensors();
        latchController.updateDataDuringMotionFromSDO();
        latchController.publishDataDuringMotion();
    }

    /**
     * Ending action OPEN or CLOSE
     *
     * @param action
     * @throws FcsHardwareException
     */
    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.finer(() -> name + " ENDING action " + action.toString());
        this.latchController.stopAction();
    }

    /**
     * Stop action OPEN or CLOSE
     *
     * @param action
     * @param delay
     * @throws FcsHardwareException
     */
    @Override
    public void abortAction(MobileItemAction action, long delay) {
        FCSLOG.finer(() -> name + " STOPPING action " + action.toString() + " within delay " + delay);
        this.latchController.stopAction();
    }

    /**
     * @throws FcsHardwareException
     * @param action
     * @param delay
     */
    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        abortAction(action, delay);
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * ____        _                      _     _ _           _   _                *
     * |  _ \  __ _| |_ __ _   _ __  _   _| |__ | (_) ___ __ _| |_(_) ___  _ __    *
     * | | | |/ _` | __/ _` | | '_ \| | | | '_ \| | |/ __/ _` | __| |/ _ \| '_ \   *
     * | |_| | (_| | || (_| | | |_) | |_| | |_) | | | (_| (_| | |_| | (_) | | | |  *
     * |____/ \__,_|\__\__,_| | .__/ \__,_|_.__/|_|_|\___\__,_|\__|_|\___/|_| |_|  *
     *                        |_|                                                  *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Creates an return an object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerLatch createStatusDataPublishedByLatch() {
        StatusDataPublishedByAutochangerLatch status = new StatusDataPublishedByAutochangerLatch();
        status.setLockSensorOn(closeSensors.isOn());
        status.setUnlockSensorOn(openSensors.isOn());
        status.setFilterPresenceSensorOn(filterEngagedSensors.isOn());
        status.setLockStatus(lockStatus);
        status.setLockSensorsInError(closeSensors.isInError());
        status.setUnlockSensorsInError(openSensors.isInError());
        status.setFilterEngagedSensorsInError(filterEngagedSensors.isInError());
        status.setInError(lockStatus == LockStatus.ERROR);
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Publish Data on the Status Bus.")
    @Override
    public void publishData() {
        StatusDataPublishedByAutochangerLatch status = createStatusDataPublishedByLatch();
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, status));
    }

}
