package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnDevice;

/**
 * This class represents a sensor which sends 0 or 1. This is a model for the
 * digital sensors we have in single-filter-test, the loader, to monitor the
 * latches in the autochanger, the autochanger trucks position in
 * single-filter-test and the loader hooks. Value sent by this type of sensor is
 * 0 (false) or 1 (true).
 *
 *
 * This type of sensors are used in the autochanger for the rail filter presence
 * sensor : the sensors which detects if a filter is there or not on each side
 * of the rail and at standby and standback position. This is used too to
 * monitor the autochanger latches : to know if a latch is locked or unlocked.
 *
 *
 * This is also used in the loader test bench where the sensors are connected to
 * a Pluto PLC for the camera protection system. We read the sensors through a
 * Gateway (CanOpenPlutoGateway) byte by byte. So we need to know on which byte
 * this sensor is coded.
 *
 * @author virieux
 */
public class DigitalSensor implements BinarySensor, SensorPluggedOnDevice, HasLifecycle {

    @LookupName
    private String name = "unset";

    @LookupField(strategy = LookupField.Strategy.TOP)
    protected Subsystem subs;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;


    /**
     * This is the name of the CANOpen DIO (digital input output) device in the
     * groovy file. Used in the toString method, just for tracing purpose when the
     * subsystem starts from groovy configuration.
     *
     */
    @ConfigurationParameter(isFinal = true, description = "The name of the Digital Input Output device "
            + "where this sensor is plugged on.", units = "unitless", category = "sensor")
    private volatile String dioName;

    /**
     * Input channel number on the device where this sensor is plugged.
     */
    @ConfigurationParameter(range = "0..7", description = "The input number on DIO device "
            + "where this sensor is plugged on.", units = "unitless", category = "sensor")
    private volatile int inputNumero;

    /**
     * Gateway byte number where the sensor is coded. value = 1 or 2 in january 2014
     * value = 0,1,2,3,4,5 in december 2014
     */
    @ConfigurationParameter(range = "0..15", description = "The byte number"
            + "where the value of this sensor is stored in.", units = "unitless", category = "sensor")
    protected volatile int byteNumero;

    /**
     * this represents the value sent by the sensor.
     * If sensor sent value 1, on = true.
     * If sensor sends value 0, on = false;
     */
    protected boolean on = false;

    public DigitalSensor() {

    }

    @Override
    public void build() {
    }
    /**
     * Build a new DigitalSensor which is plugged on a plutoGateway for example.
     *
     * @param dioName
     * @param inputNumero
     * @param byteNumero
     */
    public DigitalSensor(String dioName, int inputNumero, int byteNumero) {
        this.dioName = dioName;
        this.inputNumero = inputNumero;
        this.byteNumero = byteNumero;
    }

    /**
     * The name of this component.
     *
     * @return the name of this component
     */
    @Override
    public String getName() {
        return name;
    }

    @Override
    public int getByteNumero() {
        return byteNumero;
    }

    public int getInputNumero() {
        return inputNumero;
    }

    @Override
    public String getDeviceName() {
        return dioName;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE,
             description="Return True if the sensor is ON")
    @Override
    public boolean isOn() {
        return this.on;
    }

    /**
     * This methods updates the digital value for this sensor. It takes as an
     * argument the value sent by the adc device. This value represents the 8 values
     * for the 8 channels. example if newValue = 80, and inputNumero=7, digital
     * value=1 otherwise digital value=0.
     *
     * @param newValue
     */
    @Override
    public synchronized void updateValue(int newValue) {
        /**
         * to retrieve the digital value for this sensor
         * (the digits are given from left to right
         * 1010000 => DIO[7]=1,DIO[6]=0,DIO[5]=1,DIO[4]=0,DIO[3]=0,DIO[2]=0,DIO[1]=0,DIO[0]=0
         */
        if (newValue > 256)
            throw new IllegalArgumentException("hex cannot be superior to 256");
        on = ((newValue >> inputNumero) & 1) == 1;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE,
            description = "Returns a String representation of DigitalSensor", alias = "printValue")
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(name);
        sb.append("/DIO module=");
        sb.append(this.dioName);
        sb.append("/Input numero=");
        sb.append(String.valueOf(this.inputNumero));
        sb.append("/byteNumero=");
        sb.append(String.valueOf(this.byteNumero));
        sb.append("/value=");
        sb.append(String.valueOf(this.on));
        return sb.toString();
    }


}
