package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class represents a model for the 2 Latches of the autochanger. The
 * Latches hold a filter on the trucks. There is one latch on each side of the
 * filter. Because the ACTION open (or close) for 2 latches is different from
 * opening (or closing) one Latch, then the other one, it's more convenient to
 * have a class to represent 2 Latches at a time.
 *
 * @author virieux
 */
public class AutochangerTwoLatches implements AlertRaiser, SignalHandler, HasLifecycle {

    @LookupName
    protected String name;

    @LookupPath
    protected String path;

    @LookupField(strategy = LookupField.Strategy.ANCESTORS)
    private Subsystem subs;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Autochanger autochanger;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = LookupField.Strategy.TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    private final AutochangerLatch latchXminus;
    private final AutochangerLatch latchXplus;

    private volatile LockStatus lockStatus = LockStatus.UNKNOWN;
    private volatile FilterPresenceStatus filterPresenceStatus = FilterPresenceStatus.UNKNOWN;

    /**
     * Build an Object AutochangerTwoLatches from 2 AutochangerLatch.
     *
     * @param latchXminus
     * @param latchXplus
     */
    public AutochangerTwoLatches(AutochangerLatch latchXminus, AutochangerLatch latchXplus) {
        this.latchXminus = latchXminus;
        this.latchXplus = latchXplus;
    }

    @Override
    public void init() {
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(FcsAlerts.AC_SENSOR_ERROR.getAlert(), alwaysClear);
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerTwoLatches.class, path);
    }

    /**
     * initialize autochanger latches hardware after initialization. to be executed if during
     * boot process some hardware is missing.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ADVANCED, description = "Initialize autochanger latches hardware after initialization. To be executed if during boot process some hardware is missing.")
    public void initializeHardware() {
        latchXminus.postStart();
        latchXplus.postStart();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     *   ____      _   _                 __  ____       _   _              *
     *  / ___| ___| |_| |_ ___ _ __     / / / ___|  ___| |_| |_ ___ _ __   *
     * | |  _ / _ \ __| __/ _ \ '__|   / /  \___ \ / _ \ __| __/ _ \ '__|  *
     * | |_| |  __/ |_| ||  __/ |     / /    ___) |  __/ |_| ||  __/ |     *
     *  \____|\___|\__|\__\___|_|    /_/    |____/ \___|\__|\__\___|_|     *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    /**
     * return filterPresenceStatus
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "return filterPresenceStatus")
    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     *   _   _               _                          _            _       *
     * | | | | __ _ _ __ __| |_      ____ _ _ __ ___  | |_ ___  ___| |_ ___  *
     * | |_| |/ _` | '__/ _` \ \ /\ / / _` | '__/ _ \ | __/ _ \/ __| __/ __| *
     * |  _  | (_| | | | (_| |\ V  V / (_| | | |  __/ | ||  __/\__ \ |_\__ \ *
     * |_| |_|\__,_|_|  \__,_| \_/\_/ \__,_|_|  \___|  \__\___||___/\__|___/ *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Returns true if the 2 latches controllers are initialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if the 2 latches are initialized.")
    public boolean isInitialized() {
        return latchXminus.isInitialized() && latchXplus.isInitialized();
    }

    /**
     * Return if both latches are CLOSED.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if both latches are CLOSED.")
    public boolean isClosed() {
        return this.latchXminus.getLockStatus() == LockStatus.CLOSED && this.latchXplus.getLockStatus() == LockStatus.CLOSED;
    }

    /**
     * Return if both latches are OPENED.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if both latches are OPENED.")
    public boolean isOpened() {
        return this.latchXminus.getLockStatus() == LockStatus.OPENED && this.latchXplus.getLockStatus() == LockStatus.OPENED;
    }

    /**
     * Returns true if one of the latches is in ERROR state. Doesn't read again
     * sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if one of the latches is in ERROR state.")
    public boolean isInError() {
        return this.latchXminus.getLockStatus() == LockStatus.ERROR || this.latchXplus.getLockStatus() == LockStatus.ERROR;
    }

    /**
     *
     * @return true if both latches are in Travel
     */
    private boolean isInTravel() {
        return this.latchXminus.getLockStatus() == LockStatus.INTRAVEL && this.latchXplus.getLockStatus() == LockStatus.INTRAVEL;
    }

    /**
     * Returns true if there is no filter is latches. Doesn't read again sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if there is no filter is latches.")
    public boolean isEmpty() {
        return this.latchXminus.isEmpty() && this.latchXplus.isEmpty();
    }

    public boolean isFilterEngagedInError() {
        return latchXminus.isFilterEngagedInError() || latchXplus.isFilterEngagedInError();
    }

    /**
     * Return true if autochanger is holding a filter. A filter is ENGAGED and
     * latches are CLOSED. Does not read again sensors values.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if autochanger is holding a filter.")
    public boolean isHoldingFilter() {
        return isFilterEngaged() && this.isClosed();
    }

    /**
     *
     * @return true if both latches detect filter presence.
     */
    private boolean isFilterEngaged() {
        return !this.latchXminus.isEmpty() && !this.latchXplus.isEmpty();
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     *   ____                                          _       *
     *  / ___|___  _ __ ___  _ __ ___   __ _ _ __   __| |___   *
     * | |   / _ \| '_ ` _ \| '_ ` _ \ / _` | '_ \ / _` / __|  *
     * | |__| (_) | | | | | | | | | | | (_| | | | | (_| \__ \  *
     *  \____\___/|_| |_| |_|_| |_| |_|\__,_|_| |_|\__,_|___/  *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Checks sensors.
     *
     * @throws FcsHardwareException if some latch sensors are in error or if latches
     *                              state is not equal on both side of the filter.
     */
    public void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        latchXminus.checkSensors(FcsAlerts.AC_SENSOR_ERROR, "latchXminus");
        latchXplus.checkSensors(FcsAlerts.AC_SENSOR_ERROR, "latchXplus");
        if (latchXminus.getLockStatus() != latchXplus.getLockStatus()) {
            String msg = name + " sensors are in ERROR: latchXminus state=" + latchXminus.getLockStatus()
                    + " latchXplus state=" + latchXplus.getLockStatus();
            this.raiseAlarm(FcsAlerts.AC_SENSOR_ERROR, msg);
            throw new FcsHardwareException(
                    name + " LockStatus is different on both side. Can't close neither open latches.");
        }
        if (latchXminus.isEmpty() != latchXplus.isEmpty()) {
            String msg = name + " Error in filterPresenceSensors of autochanger : the 2 sensors send different values."
                    + " latchXminus.isEmpty=" + this.latchXminus.isEmpty() + " latchXplus.isEmpty="
                    + this.latchXplus.isEmpty();
            this.raiseAlarm(FcsAlerts.AC_SENSOR_ERROR, msg);
            throw new FcsHardwareException(msg + "Can't close neither open latches.");
        }
    }

    /**
     * Update lockStatus from latchXminus state and latchXplus state.
     *
     */
    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-latches")) {
            synchronized (this) {
                this.latchXminus.updateState();
                this.latchXplus.updateState();
                computeLockStatus();
                computeFilterPresenceStatus();
            }
            this.publishData();
        }
    }

    /**
     * Compute a global lock status for the 2 latches.
     */
    private void computeLockStatus() {

        if (isInError()) {
            this.lockStatus = LockStatus.ERROR;

        } else if (isClosed()) {
            this.lockStatus = LockStatus.CLOSED;

        } else if (isOpened()) {
            this.lockStatus = LockStatus.OPENED;

        } else if (isInTravel()) {
            this.lockStatus = LockStatus.INTRAVEL;

        } else {
            /* if lockStatus is not the same for the 2 latches */
            /*
             * can happen when closing (or opening) when a latch is already closed and not
             * the other
             */
            this.lockStatus = LockStatus.UNKNOWN;
        }
    }

    private void computeFilterPresenceStatus() {
        /* compute filter presence */
        if (this.latchXminus.isEmpty() != this.latchXplus.isEmpty()) {
            filterPresenceStatus = FilterPresenceStatus.ERROR;

        } else if (this.isEmpty()) {
            filterPresenceStatus = FilterPresenceStatus.NOFILTER;

        } else if (isFilterEngaged()) {
            filterPresenceStatus = FilterPresenceStatus.ENGAGED;

        } else {
            filterPresenceStatus = FilterPresenceStatus.UNKNOWN;
        }
    }

    /**
     * Close latches
     *
     * @throws FcsHardwareException
     */
    public void close() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("closeLatches")) {
            FcsUtils.parallelRun(() -> latchXminus.close(), () -> latchXplus.close());
        }
    }

    /**
     * Open the 2 latches.
     *
     * @throws FcsHardwareException
     */
    public void open() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("openLatches")) {
            FcsUtils.parallelRun(() -> latchXminus.open(), () -> latchXplus.open());
        }
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
     * ____        _                      _     _ _           _   _                *
     * |  _ \  __ _| |_ __ _   _ __  _   _| |__ | (_) ___ __ _| |_(_) ___  _ __    *
     * | | | |/ _` | __/ _` | | '_ \| | | | '_ \| | |/ __/ _` | __| |/ _ \| '_ \   *
     * | |_| | (_| | || (_| | | |_) | |_| | |_) | | | (_| (_| | |_| | (_) | | | |  *
     * |____/ \__,_|\__\__,_| | .__/ \__,_|_.__/|_|_|\___\__,_|\__|_|\___/|_| |_|  *
     *                        |_|                                                  *
     * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    /**
     * Creates an object to be published on the STATUS bus by AutochangerTwoLatches.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerTwoLatches createStatusDataPublishedByTwoLatches() {
        StatusDataPublishedByAutochangerTwoLatches status = new StatusDataPublishedByAutochangerTwoLatches();
        status.setLockStatus(lockStatus);
        status.setFilterPresenceStatus(filterPresenceStatus);
        status.setFilterName(autochanger.getFilterOnTrucksName());
        status.setFilterId(autochanger.getFilterID());
        status.setFilterObservatoryName(autochanger.getFilterOnTrucksObservatoryName());
        status.setAvailable(autochanger.isAvailable());
        return status;
    }

    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByTwoLatches()));
    }
}
