
package org.lsst.ccs.subsystems.fcs.simulation;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Logger;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.CarouselClamp;
import org.lsst.ccs.subsystems.fcs.CarouselSocket;

import org.lsst.ccs.subsystems.fcs.drivers.CanOpenTTC580;

/**
 * Provides methods to simulate carousel status.
 *
 * @author virieux
 */
public class SimuTTC580 extends CanOpenTTC580 {
    private static final Logger FCSLOG = Logger.getLogger(SimuTTC580.class.getName());

    private static final long LOCKSENSORVALUEB = 9500;

    CarouselSocket socket1;
    CarouselSocket socket2;
    CarouselSocket socket3;
    CarouselSocket socket4;
    CarouselSocket socket5;
    SimuCarousel carousel;

    protected final Map<String, CarouselSocket> socketMapByID = new HashMap<>();

    @Override
    public void init() {
        super.init();
        ComponentLookup lookup = subs.getComponentLookup();
        carousel = (SimuCarousel) lookup.getComponentByPath("carousel");
        socket1 = (CarouselSocket) lookup.getComponentByPath("socket1");
        socket1 = (CarouselSocket) lookup.getComponentByPath("socket1");
        socket2 = (CarouselSocket) lookup.getComponentByPath("socket2");
        socket3 = (CarouselSocket) lookup.getComponentByPath("socket3");
        socket4 = (CarouselSocket) lookup.getComponentByPath("socket4");
        socket5 = (CarouselSocket) lookup.getComponentByPath("socket5");
        socketMapByID.put("1", socket1);
        socketMapByID.put("2", socket2);
        socketMapByID.put("3", socket3);
        socketMapByID.put("4", socket4);
        socketMapByID.put("5", socket5);
    }

    /**
     * Used to initialize simulation with a filter in each socket, and clamps
     * CLAMPEDONFILTER
     */
    public void simulateAllSocketCLAMPEDONFILTER() {
        FCSLOG.info("===> INITIALISATION OF CLAMPS STATE <===");
        simulateSocketClampedOnFilter(true, (short) 1);
        simulateSocketClampedOnFilter(false, (short) 2);
        simulateSocketClampedOnFilter(false, (short) 3);
        simulateSocketClampedOnFilter(false, (short) 4);
        simulateSocketClampedOnFilter(false, (short) 5);
    }

    @Override
    public void initializeAndCheckHardware() {
        super.initializeAndCheckHardware();
        updateFakePDOData(cobid1,0);
        /* initialisation of io status to 2*/
        updateFakePDOData(cobid2,(long)Math.pow(2, 59));
    }

    /**
     * Update PDOData in the simulated interface to CANopen.
     *
     * @param cobid
     * @param newPdo
     */
    public void updateFakePDOData(int cobid, long newPdo) {
        if (cobid == cobid1) {
            this.pdo1 = newPdo;
        } else if (cobid == cobid2) {
            this.pdo2 = newPdo;
        }
        ((SimuCanOpenInterface) tcpProxy.getCanInterface()).simulatePDOData(cobid, newPdo);
    }

    private static void checkID(short id) {
        if (id < 1 || id > 5) {
            throw new IllegalArgumentException(id + " not a valid socket number");
        }
    }

    /**
     * sensors are coded on 12 bits
     *
     * @param value
     */
    private static void checkSensorValue(int value) {
        if (value > 0xFFF) {
            throw new IllegalArgumentException(value + " value can't be > " + 0xFFF);
        }
    }

    @Override
    public long readLockSensorMinLocked() {
        return (long) LOCKSENSORVALUEB;
    }

    @Override
    public double readInternalTemperature() {
        return 30;
    }

    /**
     * Update pdo1 to simulate that socket socketID is at Standby, simulate also
     * IO Module Status in pdo1, update PDOData on the simulated interface to
     * CANopen. Update also carousel position to socket standby position in PDO
     * Data.
     *
     * socket ID is coded on the first 3 bits of pdo1.
     *
     * @param socketID socketID to simulate at Standby
     */
    public void simulateSocketAtStandby(short socketID) {
        checkID(socketID);
        long sock = (long) socketID << 61;
        long mask = 0x1FFFFFFFFFFFFFFFL;
        long newpdo = sock | (pdo1 & mask);
        updateFakePDOData(cobid1, newpdo);
        /* io module status: 1 = is socket in standby position */
        simulateIOStatus((short) 1);
        /* simulate carousel position is now socket position at standby*/
        CarouselSocket socket = socketMapByID.get(Integer.toString(socketID));
        carousel.simulateCarouselPosition(socket.getStandbyPosition());
    }

    /**
     * Update pdo2 to simulate that socket socketID is not at Standby, simulate
     * also IO Module Status in pdo2.
     *
     * socket ID is coded on the first 3 bits of pdo2.
     *
     * @param socketID socketID to simulate not at standby
     */
    public void simulateSocketNotAtStandby(short socketID) {
        FCSLOG.finer("simulate socket NOT at STANDBY: " + socketID);
        checkID(socketID);
        /* socketID*/
        long sock = (long) socketID << 61;
        long mask = 0x1FFFFFFFFFFFFFFFL;
        long newpdo = sock | (pdo2 & mask);
        updateFakePDOData(cobid2, newpdo);
        /* io module status: 2 = is ready, not in position */
        simulateIOStatusNotAtS((short) 2);
    }

    public void updatePdo1ForNoSocketInPosition() {
        long sock = (long) 0 << 61;
        long mask = 0x1FFFFFFFFFFFFFFFL;
        long newpdo = sock | (pdo1 & mask);
        updateFakePDOData(cobid1, newpdo);
        /* io module status: 2 = is ready */
        simulateIOStatus((short) 2);
    }

    /**
     * Update pdo1 without update State
     *
     * @param ioStatus
     */
    public void simulateIOStatus(short ioStatus) {
        FCSLOG.finer("simulate IOStatus: " + ioStatus);
        long status = (long) ioStatus << 58;
        long mask = 0xE3FFFFFFFFFFFFFFL;
        long newpdo = status | (pdo1 & mask);
        updateFakePDOData(cobid1, newpdo);
    }

    /**
     * Update pdo2 without update State
     *
     * @param ioStatus
     */
    public void simulateIOStatusNotAtS(short ioStatus) {
        FCSLOG.finer("simulate IOStatus: " + ioStatus);
        long status = (long) ioStatus << 58;
        long mask = 0xE3FFFFFFFFFFFFFFL;
        long newpdo = status | (pdo2 & mask);
        updateFakePDOData(cobid2, newpdo);
    }

    /**
     * Update pdo1 without update State
     *
     * @param id
     */
    @Command
    public void simuSocketAtSandbyIsLocked(short id) {
        simulateSocketAtStandby(id);
        boolean atStandby = true;
        simuSocketIsLocked(atStandby, id);
    }

    /**
     * Update pdo2 without update State
     *
     * @param id
     */
    public void simuSocketNotAtStandbyIsLocked(short id) {
        simulateSocketNotAtStandby(id);
        boolean atStandby = false;
        simuSocketIsLocked(atStandby, id);
    }

    /**
     * Update pdo1 or pdo2 depending on param atStandby without update State
     *
     * @param atStandby
     * @param id
     */
    public void simuSocketIsLocked(boolean atStandby, short id) {
        CarouselSocket socket = socketMapByID.get(Integer.toString(id));
        simulateClampIsLocked(atStandby, socket.getClampXminus());
        simulateClampIsLocked(atStandby, socket.getClampXplus());
    }

    /**
     * Update pdo1 or pdo2 depending on param atStandby without update State
     *
     * @param atStandby
     * @param id
     */
    public void simuSocketIsUnlocked(boolean atStandby, short id) {
        CarouselSocket socket = socketMapByID.get(Integer.toString(id));
        simulateClampIsUnlocked(atStandby, socket.getClampXminus());
        simulateClampIsUnlocked(atStandby, socket.getClampXplus());
    }

    /**
     * Update pdo1 or pdo2 depending on param atStandby without update State
     *
     * @param atStandby
     * @param id
     */
    public void simuSocketLockable(boolean atStandby, short id) {
        CarouselSocket socket = socketMapByID.get(Integer.toString(id));
        simulateClampFilterIsLockable(atStandby, socket.getClampXminus());
        simulateClampFilterIsLockable(atStandby, socket.getClampXplus());
    }

    /**
     * Update pdo1 or pdo2 depending on param atStandby without update State
     *
     * @param atStandby
     * @param id
     */
    public void simuSocketNoFilter(boolean atStandby, short id) {
        CarouselSocket socket = socketMapByID.get(Integer.toString(id));
        simulateClampNoFilter(atStandby, socket.getClampXminus());
        simulateClampNoFilter(atStandby, socket.getClampXplus());
    }

    /**
     * Update pdo1 or pdo2 depending on param atStandby and update carousel
     * clamps state.
     *
     * @param atStandby
     * @param id
     */
    public void simulateSocketClampedOnFilter(boolean atStandby, short id) {
        FCSLOG.info("==== simulateSocketClampedOnFilter for socket" + id);
        if (atStandby) {
            simulateSocketAtStandby(id);
        } else {
            simulateSocketNotAtStandby(id);
        }
        simuSocketLockable(atStandby, id);
        simuSocketIsLocked(atStandby, id);
        carousel.updateStateWithSensors();
    }

    /**
     * Update pdo1 or pdo2 depending on param atStandby and update carousel
     * clamps state.
     *
     * @param atStandby
     * @param id
     */
    public void simulateSocketUnclampedOnFilter(boolean atStandby, short id) {
        FCSLOG.info("==== simulateSocketUnclampedOnFilter for socket" + id);
        if (atStandby) {
            simulateSocketAtStandby(id);
        } else {
            simulateSocketNotAtStandby(id);
        }
        simuSocketLockable(atStandby, id);
        simuSocketIsUnlocked(atStandby, id);
        carousel.updateStateWithSensors();
    }

    /**
     * Update pdo1 or pdo2 depending on param atStandby and update
     * carousel clamps state.
     *
     * @param atStandby
     * @param id
     */
    public void simulateSocketUnclampedEmpty(boolean atStandby, short id) {
        FCSLOG.info("==== simulateSocketUnclampedEmpty for socket" + id);
        if (atStandby) {
            simulateSocketAtStandby(id);
        } else {
            simulateSocketNotAtStandby(id);
        }
        simuSocketNoFilter(atStandby, id);
        simuSocketIsUnlocked(atStandby, id);
        carousel.updateStateWithSensors();
    }

    /**
     * A socket is READYTOCLAMP when there is no filter, clampXminus is unlocked
     * and clampXplus is Locked.
     *
     * @param id
     */
    public void simulateSocketREADYTOCLAMP(boolean atStandby, short id) {
        simuSocketNoFilter(atStandby, id);
        if ( atStandby ) {
            simulateSocketAtStandby(id);
        }
        CarouselSocket socket = socketMapByID.get(Integer.toString(id));
        simulateClampIsUnlocked(atStandby, socket.getClampXminus());
        simulateClampIsLocked(atStandby, socket.getClampXplus());
        socket.getClampXminus().getController().goToSwitchOnDisabled();
        socket.getClampXplus().getController().goToSwitchOnDisabled();
        carousel.updateStateWithSensors();
    }

    protected void simulateLockSensorXminusValue(boolean atStandby, long lockValue) {
        long pdo = (long) (lockValue << 36);
        long mask = 0xFFFF000FFFFFFFFFL;
        long newpdo;
        if (atStandby) {
            newpdo = pdo | (pdo1 & mask);
            updateFakePDOData(cobid1, newpdo);
        } else {
            newpdo = pdo | (pdo2 & mask);
            updateFakePDOData(cobid2, newpdo);
        }
    }

    protected void simulateFilterPresenceXminusValue(boolean atStandby, long newValue) {
        long pdo = newValue << 24;
        long mask = 0xFFFFFFF000FFFFFFL;
        long newpdo;
        if (atStandby) {
            newpdo = pdo | (pdo1 & mask);
            updateFakePDOData(cobid1, newpdo);
        } else {
            newpdo = pdo | (pdo2 & mask);
            updateFakePDOData(cobid2, newpdo);
        }
    }

    protected void simulateLockSensorXplusValue(boolean atStandby, long newValue) {
        long pdo = newValue << 12;
        long mask = 0xFFFFFFFFFF000FFFL;
        long newpdo;
        if (atStandby) {
            newpdo = pdo | (pdo1 & mask);
            updateFakePDOData(cobid1, newpdo);
        } else {
            newpdo = pdo | (pdo2 & mask);
            updateFakePDOData(cobid2, newpdo);
        }
    }


    protected void simulateFilterPresenceXplusValue(boolean atStandby, long newValue) {
        long pdo = newValue & 0xFFF;
        long mask = 0xFFFFFFFFFFFFF000L;
        long newpdo;
        if (atStandby) {
            newpdo = pdo | (pdo1 & mask);
            updateFakePDOData(cobid1, newpdo);
        } else {
            newpdo = pdo | (pdo2 & mask);
            updateFakePDOData(cobid2, newpdo);
        }
    }

    public void simulateClampIsLocked(boolean atStandby, CarouselClamp clamp) {
        int milieu = (clamp.getLockSensorMaxLimit() - carousel.getMinLockedThreshold()) / 2;
        int lockVal = (carousel.getMinLockedThreshold() + milieu) / 4;
        simulateClampLockValue(atStandby, clamp, lockVal);
    }

    public void simulateClampIsUnlocked(boolean atStandby, CarouselClamp clamp) {
        int milieu = (clamp.getLockSensorOffset1() - clamp.getLockSensorMinLimit()) / 2;
        int lockVal = (clamp.getLockSensorMinLimit() + milieu) / 4;
        simulateClampLockValue(atStandby, clamp, lockVal);
    }

    private void simulateClampLockValue(boolean atStandby, CarouselClamp clamp, int lockVal) {
        checkSensorValue(lockVal);
        if (clamp.getName().contains("Xminus")) {
            simulateLockSensorXminusValue(atStandby, lockVal);
        } else if (clamp.getName().contains("Xplus")) {
            simulateLockSensorXplusValue(atStandby, lockVal);
        } else throw new IllegalArgumentException(clamp.getName() + " illegal value. "
                + "Should contain Xminus or Xplus");
    }


    /**
     * filter is engaged and lockable if filterPresence sensor returns a value between
     * FilterPresenceMinLimit and FilterPresenceOffset2
     *
     * @param atStandby
     * @param clamp
     */
    public void simulateClampFilterIsLockable(boolean atStandby, CarouselClamp clamp) {
        int val = (clamp.getFilterPresenceMinLimit() + clamp.getFilterPresenceOffset2()) / 8;
        simulateClampFilterPresenceValue(atStandby, clamp, val);
    }


    /**
     * filter is not loackable if filterPresence sensor returns a value between
     * FilterPresenceOffset2 and FilterPresenceMinNoFilter
     *
     * @param atStandby
     * @param clamp
     */
    public void simulateClampFilterIsNotLockable(boolean atStandby, CarouselClamp clamp) {
        int val = (clamp.getFilterPresenceOffset2() + carousel.getFilterPresenceMinNoFilter()) / 8;
        simulateClampFilterPresenceValue(atStandby, clamp, val);
    }

    /**
     * simulate there is no filter in socket socketID clampX-
     * @param atStandby
     * @param clamp
     */
    public void simulateClampNoFilter(boolean atStandby, CarouselClamp clamp) {
        int val = carousel.getFilterPresenceMinNoFilter() / 4 + 1;
        simulateClampFilterPresenceValue(atStandby, clamp, val);
    }


    public void simulateClampFilterIsInError(boolean atStandby, CarouselClamp clamp) {
        int val = clamp.getFilterPresenceMinLimit() / 4 - 1;
        simulateClampFilterPresenceValue(atStandby, clamp, val);
    }


    private void simulateClampFilterPresenceValue(boolean atStandby, CarouselClamp clamp, int val) {
        checkSensorValue(val);
        if (clamp.getName().contains("Xminus")) {
            simulateFilterPresenceXminusValue(atStandby, val);
        } else if (clamp.getName().contains("Xplus")) {
            simulateFilterPresenceXplusValue(atStandby, val);
        } else throw new IllegalArgumentException(clamp.getName() + " illegal value. "
                + "Should contain Xminus or Xplus");
    }

    public void simulateSocketAtStandbyOthersNotAtStandby(short socketID) {
        for (short i = 1; i < 6; i++) {
            /* pdo1 has to be changed first */
            simulateSocketAtStandby(socketID);
            if (i != socketID) {
                /* pdo2 is then changed for sockets not at Standby */
                simulateSocketNotAtStandby(i);
                carousel.updateStateWithSensors();
            }
        }
    }

    public void simulateNoSocketAtStandby() {
        /* pdo1 has to be changed first */
        updatePdo1ForNoSocketInPosition();
        for (short i = 1; i < 6; i++) {
            simulateSocketNotAtStandby(i);
            carousel.updateStateWithSensors();
        }
    }

    public void simulateSocketSensors(int pos) {
        FCSLOG.finer(name + " simulateSocketSensors with position = " + pos);
        /* find if a socket is at standby */
        short socketAtStandbyID = carousel.getSocketAtStandbyID(pos);
        if (socketAtStandbyID == 0) {
            simulateNoSocketAtStandby();
        } else {
            simulateSocketAtStandbyOthersNotAtStandby(socketAtStandbyID);
        }
    }

    /**
     * update socket at standby to simulate NOFILTER state.
     * Used by simulated
     * autochanger linear rail controller when updating its position.
     */
    public void updateDisengageFromAC() {
        CarouselSocket socketAtS = carousel.getSocketAtStandby();
        if (socketAtS == null) {
            FCSLOG.severe(name + " no socket at standby");
        } else {
            this.simuSocketNoFilter(true, (short) socketAtS.getId());
        }
    }

    /**
     * update socket at standby to simulate that autochanger stores filter on carousel so
     * simulate CLAMPEDONFILTER state at standby. Used by simulated autochanger linear
     * rail controller.
     * @param filterID
     */
    public void updateStoreFilterOnCarousel(int filterID) {
        CarouselSocket socketAtS = carousel.getSocketAtStandby();
        if (socketAtS == null) {
            FCSLOG.severe(name + " no socket at standby");
        } else {
            carousel.getSocketAtStandby().setFilterID(filterID);
            this.simulateSocketClampedOnFilter(true, (short) socketAtS.getId());
        }
        FCSLOG.info("carousel filter ID = " + carousel.getSocketAtStandby().getFilterID());
    }
}
