package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.ControlWord;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;

import java.util.Arrays;
import java.util.EnumMap;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * This is model for the clamps mechanism in the loader. 4 hooks are used for
 * the clamping of the filter in the loader. This hooks are moved all together
 * by the hooksController.
 *
 * Raises alert : - CAN_BUS_TIMEOUT in
 * updateStateWithSensorsToCheckIfActionIsCompleted
 *
 * @author virieux
 */
public class LoaderClamp extends MobileItem implements MovedByEPOSController, ControlledBySensors {
    private static final Logger FCSLOG = Logger.getLogger(LoaderClamp.class.getName());

    private final LoaderHook hook1;
    private final LoaderHook hook2;
    private final LoaderHook hook3;
    private final LoaderHook hook4;

    private final ForceSensor forceSensor0;
    private final ForceSensor forceSensor1;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter="unclampedStatusSensor")
    private DigitalSensor unclampedStatusSensor;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter="underClampedStatusSensor")
    private DigitalSensor underClampedStatusSensor;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter="clampedStatusSensor")
    private DigitalSensor clampedStatusSensor;

    @LookupField(strategy = LookupField.Strategy.CHILDREN, pathFilter="overClampedStatusSensor")
    private DigitalSensor overClampedStatusSensor;

    /* a list of hooks to make easier the computation of the LockStatus */
    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final LoaderHook[] hooks;

    @LookupField(strategy = LookupField.Strategy.TREE, pathFilter=".*\\/hooksController")
    private EPOSController hooksController;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private Loader loader;

    @LookupField(strategy = LookupField.Strategy.TREE, pathFilter = FCSCst.LOADER_TCPPROXY_NAME)
    private BridgeToHardware loaderTcpProxy;

    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;

    private LockStatus lockStatus = LockStatus.UNKNOWN;

    /**
     * computed from the 2 force sensors forceStatus.
     */
    private LockStatus forceStatus;

    @ConfigurationParameter(range = "0..50", description = "Trigger to know when action clamp is completed. "
            + "If force sensor voltage is over this limit then action clamp is completed.", units = "0.1 Volt", category = "loader")
    private volatile int minClampedVoltage = 25;

    @ConfigurationParameter(range = "0..100000", description = "Timeout : if closing the clamp last more than "
            + "this amount of time, then the subsystem goes in ERROR.", units = "millisecond", category = "loader")
    private volatile int timeoutForClosingHooks = 60000;

    @ConfigurationParameter(range = "0..100000", description = "Timeout : if closing strongly the clamp last "
            + "more than this amount of time, then the subsystem goes in ERROR.", units = "millisecond", category = "loader")
    private volatile int timeoutForClampingHooks = 30000;

    @ConfigurationParameter(range = "0..180000", description = "Timeout : if opening the clamp last more than"
            + " this amount of time, then the subsystem goes in ERROR.", units = "millisecond", category = "loader")
    private volatile int timeoutForOpeningHooks = 60000;

    private static final int TIMEOUT_FOR_MOVING_CLAMP = 60000;

    @ConfigurationParameter(range = "250000..400000", description = "Target encoder absolute value in qc when hooks are CLOSED", units = "unitless", category = "loader")
    private volatile int absolutePositionToClose = 325000;

    @ConfigurationParameter(range = "-40000..0", description = "Relative position in qc to unclamp when hooks are CLAMPED", units = "unitless", category = "loader")
    private volatile int relativePositionToUnclamp = -25000;

    @ConfigurationParameter(range = "0..1000", description = "Current to clamp hooks", units = "mA", category = "loader")
    private volatile int currentToClamp = 445;

    @ConfigurationParameter(range = "-500..0", description = "Current to open hooks", units = "mA", category = "loader")
    private volatile int currentToOpen = -200;

    @ConfigurationParameter(range = "0..500", description = "Current to open hooks in homing mode", units = "mA", category = "loader")
    private volatile int currentThreshold = 300;

    private int positionToReach = 0;

    private volatile boolean homingDone = false;

    private int position;

    /**
     * Build a new LoaderClampModule with 4 hooks and the parameters to configure
     * the EPOS controller in mode CURRENT and in mode HOMING.
     *
     * @param hook1
     * @param hook2
     * @param hook3
     * @param hook4
     * @param forceSensor0
     * @param forceSensor1
     */
    public LoaderClamp(LoaderHook hook1, LoaderHook hook2, LoaderHook hook3, LoaderHook hook4,
                       ForceSensor forceSensor0, ForceSensor forceSensor1) {
        this.hook1 = hook1;
        this.hook2 = hook2;
        this.hook3 = hook3;
        this.hook4 = hook4;
        this.hooks = new LoaderHook[] { hook1, hook2, hook3, hook4 };
        this.forceSensor0 = forceSensor0;
        this.forceSensor1 = forceSensor1;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByLoaderClamp.class, path);
        dataProviderDictionaryService.registerClass(StatusDataPublishedByLoaderHook.class, path + "/statusPublishedByHook1");
        dataProviderDictionaryService.registerClass(StatusDataPublishedByLoaderHook.class, path + "/statusPublishedByHook2");
        dataProviderDictionaryService.registerClass(StatusDataPublishedByLoaderHook.class, path + "/statusPublishedByHook3");
        dataProviderDictionaryService.registerClass(StatusDataPublishedByLoaderHook.class, path + "/statusPublishedByHook4");

        movementCounter = new EnumMap<>(MobileItemAction.class);
        for (MobileItemAction action : new MobileItemAction[]{
            MobileItemAction.CLOSE_LOADER_HOOKS,
            MobileItemAction.CLAMP_LOADER_HOOKS,
            MobileItemAction.OPEN_HOMING_LOADER_HOOKS,
            MobileItemAction.UNCLAMP_LOADER_HOOKS
        }) {
            action.registerDurationPerElement(dataProviderDictionaryService, path);
            movementCounter.put(action, PersistentCounter.newCounter(action.getCounterPath(path), subs, action.name()));
        }
    }

    @Override
    public void init() {
	    super.init();

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(FcsAlert.HARDWARE_ERROR.getAlert(), alwaysClear);
        alertService.registerAlert(FcsAlert.SDO_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlert.LO_SENSOR_ERROR.getAlert(), alwaysClear);
    }

    @Override
    public EPOSController getController() {
        return hooksController;
    }

    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
    // Private methods used by the simulator
    // ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

    public int getRelativePositionToClose() {
        return absolutePositionToClose;
    }

    public int getRelativePositionToUnclamp() {
        return relativePositionToUnclamp;
    }

    public int getCurrentToClamp() {
        return currentToClamp;
    }

    public int getCurrentToOpen() {
        return currentToOpen;
    }

    /**
     * Return position for end user. Do not read again controller. Used by the GUI.
     *
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Display position for end user."
            + "Do not read again controller.")
    public int getPosition() {
        return position;
    }

    /**
     * Returns true if loader clamp is homingDone.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if homing of loader clamp has been done.")
    public boolean isHomingDone() {
        return homingDone;
    }

    /**
     *
     * @return lockStatus
     */
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * Returns true if LockStatus=CLOSED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if Loader hooks are CLOSED")
    public boolean isClosed() {
        return lockStatus == LockStatus.CLOSED;
    }

    /**
     * Returns true if LockStatus=OPENED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if Loader hooks are OPENED")
    public boolean isOpened() {
        return lockStatus == LockStatus.OPENED;
    }

    /**
     * Returns true if LockStatus=CLAMPED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if the hooks are CLAMPED on filter")
    public boolean isClamped() {
        updateForceStatus();
        return forceStatus == LockStatus.CLAMPED;
    }

    /**
     * Returns true if clamp LockStatus is in ERROR. That
     * means that one hook is in ERROR status or one or more of the 4 hooks has a
     * different LockStatus than the other.
     *
     * @return true if clamp LockStatus is in ERROR
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if Loader hooks are in ERROR.")
    @Override
    public boolean isInError() {
        return lockStatus == LockStatus.ERROR;
    }

    /**
     * Returns true if loader CANopen devices are booted, identified and homingDone.
     *
     * @return
     */
    @Override
    public boolean myDevicesReady() {
        return loaderTcpProxy.allDevicesBooted();
    }

    @Override
    public void postStart() {
        FCSLOG.info(name + " postStart");
        if (hooksController.isBooted()) {
            initializeController();
        }
    }

    public void initializeController() {
        try {
            hooksController.initializeAndCheckHardware();
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(FcsAlert.HARDWARE_ERROR, " could not initialize loader clamp controller", ex);
        }
    }

    /**
     * Check if clamp controller is initialized.
     *
     * @throws RejectedCommandException if clamp is not homingDone
     */
    public void checkInitialized() {
        if (!hooksController.isInitialized()) {
            String msg = getName() + ": clamp is not initialized.";
            FCSLOG.severe(msg);
            throw new RejectedCommandException(msg);
        }
    }

    /**
     *
     */
    public void checkClamped() {
        if (!this.isClamped()) {
            throw new RejectedCommandException(name + " has to be CLAMPED");
        }
    }

    /**
     *
     */
    public void checkUnclamped() {
        if (this.isClamped()) {
            throw new RejectedCommandException(name + " has to be UNCLAMPED");
        }
    }

    /**
     *
     */
    public void checkClosed() {
        if (!this.isClosed()) {
            throw new RejectedCommandException(name + " has to be CLOSED");
        }
    }

    /**
     * Open clamp in order to release filter and do a homing.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Open loader hooks and do homing with method Current Threshold Negative Speed.",
            timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void open() {
        loader.updateStateAndCheckSensors();
        loader.checkConditionsForOpeningHooks();
        this.executeAction(MobileItemAction.OPEN_HOMING_LOADER_HOOKS, timeoutForOpeningHooks);
        homingDone = true;
    }

    /**
     * Closes clamp in order to hold the filter. This action closes softly the clamp
     * on the filter. The filter can't fall although it is not tight firmly. When
     * the clamp is closed, the filter is held by loader so autochanger can open its
     * latches and move trucks back without the filter. Closing hooks consists in
     * setting controller mode to PROFILE_POSITION, and going to relative position
     * absolutePositionToClose.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Close loader hooks. Do a homing BEFORE the close action when at HANDOFF and autochanger holding filter.",
            timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void close() {
        loader.updateStateAndCheckSensors();
        loader.checkLoaderNotEmpty();

        switch (lockStatus) {
            case CLOSED -> {
                FCSLOG.info(getName() + " already CLOSED. Nothing to do.");
            }
            case OPENED, INTRAVEL, UNKNOWN -> {
                doHomingIfPossible();
                if (!this.isHomingDone()) {
                    throw new RejectedCommandException(getName() + " abort close action because homing could not be done.");
                }
                positionToReach = absolutePositionToClose;
                this.executeAction(MobileItemAction.CLOSE_LOADER_HOOKS, timeoutForClosingHooks);
            }
            default -> {
                throw new RejectedCommandException(getName() + " has to be OPENED or IN_TRAVEL or UNKNOWN before a close action." +
                    " Current lockStatus: " + lockStatus.name());
            }
        }
    }

    /**
     * Do a homing (command open) when possible.
     *
     * at HANDOFF if filter is held by autochanger.
     *
     * See JIRA LSSTCCSFCS-339
     */
    private void doHomingIfPossible() {
        if (loader.isAtHandoff() && loader.isAutochangerHoldingFilter()) {
            FCSLOG.info(name + " loader at HANDOFF; filter held by autochanger; about to do a "
                    + "homing before a close action.");
            open();
        }
    }

    /**
     * Clamps hooks. When clamp is clamped, the filter is held tightly and carrier
     * can go to STORAGE position safely. Clamping consists in setting controller in
     * mode CURRENT and send currentToClamp to controller.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Clamp to hold tightly a filter.",
            timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void clamp() {
        loader.updateStateAndCheckSensors();

        switch (lockStatus) {
            case CLAMPED -> {
                FCSLOG.info(getName() + " is already CLAMPED. Nothing to do.");
            }
            case CLOSED -> {
                loader.checkLoaderNotEmpty();
                loader.checkConditionsForClampingHooks();
                this.executeAction(MobileItemAction.CLAMP_LOADER_HOOKS, timeoutForClampingHooks);
                afterActionClamp();
            }
            default -> {
                throw new RejectedCommandException(getName() + " has to be CLOSED before a clamp action."
                    + " Current lockStatus: " + lockStatus.name());
            }
        }
    }

    private void afterActionClamp() {
        if (overClampedStatusSensor.isOn()) {
            String msg = " force sensor value is higher than max force limit.";
            this.raiseAlarm(FcsAlert.HARDWARE_ERROR, msg);
            throw new FcsHardwareException(name + msg);
        }
        loader.updateFCSStateToReady();
        this.publishData();
    }

    /**
     * Recovery command to be executed to clamp a filter between STORAGE and
     * ENGAGED in current mode.
     * During NORMAL operations, when clamping at ENGAGED, use the clamp() method, not this one.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "When the loader has a filter between ENGAGED and STORAGE and the force status "
            + "of the hooks is UNCLAMPED, this recovery method should be used to get back to CLAMPED state "
            + "and continue with NORMAL commands.",
            timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void recoveryClamp() {
        loader.updateStateAndCheckSensors();

        if ( lockStatus == LockStatus.CLOSED && forceStatus == LockStatus.UNCLAMPED ) {
            loader.checkLoaderNotEmpty();
            // Loader carrier should be between storage and engaged
            loader.checkCarrierConditionsForRecoveryClampingHooks();

            this.executeAction(MobileItemAction.CLAMP_LOADER_HOOKS, timeoutForClampingHooks);

            // Raise an alarm if the hooks are over clamped
            afterActionClamp();

        } else {
            throw new RejectedCommandException(getName()
                + " to start the recoveryClamp the loader hooks should be CLOSED and the forceStatus UNCLAMPED."
                + " Currently lockStatus=" + lockStatus.name() + " forceStatus=" + forceStatus.name());
        }
    }

    /**
     * unclamp Loader clamp. At the end of this action, hooks are still CLOSED, but
     * doesn't hold tightly filter. the homing of the clamp is not required
     * before unclamp because unclamp is a relative motion.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Unclamp filter and return to CLOSED position.",
            timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void unclamp() {
        loader.updateStateAndCheckSensors();

        switch (lockStatus) {
            case CLOSED -> {
                FCSLOG.info(getName() + " is already CLOSED. Nothing to do.");
            }
            case CLAMPED -> {
                loader.checkLoaderNotEmpty();
                loader.checkConditionsForUnclampingHooks();
                positionToReach = position + relativePositionToUnclamp;
                String msg = String.format(
                    "Loader hooks unclamping info: position=%d, relativePositionToUnclamp=%d, positionToReach=%d",
                    position, relativePositionToUnclamp, positionToReach
                );
                FCSLOG.info(msg);
                this.executeAction(MobileItemAction.UNCLAMP_LOADER_HOOKS, timeoutForClampingHooks);
            }
            default -> {
                throw new RejectedCommandException(getName() + " has to be CLAMPED before an unclamp action." +
                    " Current lockStatus: " + lockStatus.name());
            }
        }
    }

    /**
     * Safely unclamp the loader hooks without homing between STORAGE and ENGAGED.
     * This is necessary when the force sensors are UNDER_CLAMPED or OVER_CLAMPED
     * after the loader was moved. It will unclamp the system so it can be clamped
     * again and the NORMAL commands can be used.
     * At the end of this action, hooks will be CLOSED and UNCLAMPED.
     * The homing of the clamp is not required here because it uses a relative motion.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "When the loader has a filter between ENGAGED and STORAGE and the force status "
            + "of the hooks is UNDER_CLAMPED or OVER_CLAMPED, this recovery method should be used to get back "
            + "to UNCLAMPED. It can be followed by recoveryClamp to get back to CLAMPED state.",
            timeout = TIMEOUT_FOR_MOVING_CLAMP)
    public void recoveryUnclamp() {
        loader.updateStateAndCheckSensors();

        if ( lockStatus == LockStatus.CLOSED && forceStatus != LockStatus.UNCLAMPED ) {
            loader.checkLoaderNotEmpty();
            // Loader carrier should be between storage and engaged
            loader.checkCarrierConditionsForRecoveryClampingHooks();

            positionToReach = position + relativePositionToUnclamp;
            String infomsg = String.format(
                "Starting loader recoveryUnclamping. Loader hooks position=%d | relativePositionToUnclamp=%d | positionToReach=%d",
                position, relativePositionToUnclamp, positionToReach);
            FCSLOG.info(infomsg);

            this.executeAction(MobileItemAction.UNCLAMP_LOADER_HOOKS, timeoutForClampingHooks);

        } else {
            throw new RejectedCommandException(getName()
                + " to start the recoveryUnclamp the loader hooks should be CLOSED and the forceStatus different from UNCLAMPED."
                + " Currently lockStatus=" + lockStatus.name() + " forceStatus=" + forceStatus.name());
        }
    }

    /**
     * Check if the action is completed. This method is called after
     * updateStateWithSensorsToCheckIfActionIsCompleted where the different new
     * values of current or position or sensors values have been updated.
     *
     * @param action
     * @return
     */
    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        /* A loader clamp motion is completed when the position is in a range of */
        /* 10 microns around the target position. */
        boolean actionCompleted = false;
        switch (action) {
            case OPEN_HOMING_LOADER_HOOKS -> {
                actionCompleted = hooksController.isTargetReached() && isOpened();
            }
            case CLOSE_LOADER_HOOKS -> {
                FCSLOG.info(name + " position = " + position);
                actionCompleted = (hooksController.isTargetReached()
                    && isPositionReached(this.positionToReach)
                    && isClosed());
            }
            case CLAMP_LOADER_HOOKS -> {
                actionCompleted = (clampedStatusSensor.isOn()
                    && forceSensor0.getVoltage() >= minClampedVoltage
                    && forceSensor1.getVoltage() >= minClampedVoltage);
            }
            case UNCLAMP_LOADER_HOOKS -> {
                actionCompleted = (isPositionReached(this.positionToReach)
                    && !clampedStatusSensor.isOn()
                    && hooksController.isTargetReached());
            }
            default -> {}
        }

        return actionCompleted;
    }

    private boolean isPositionReached(int positionToReach) {
        // The value below is very small but
        // - this is used with a relative position (the encoder does not need homing)
        // - the encoder always reaches the exact requested position while powered on
        int delta_position = 5;
        FCSLOG.finer(() -> "position=" + position + " positionToReach=" + positionToReach);
        return positionToReach - delta_position <= this.position && this.position <= positionToReach + delta_position;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        try {
            hooksController.checkFault();
            loader.updateStateWithSensors();
        } catch (SDORequestException ex) {
            this.raiseWarning(FcsAlert.SDO_ERROR, "error in updateStateWithSensorsToCheckIfActionIsCompleted: ", name, ex);
        }
    }

    @Override
    public void startAction(MobileItemAction action) {
        hooksController.checkFault();

        switch (action) {
            case OPEN_HOMING_LOADER_HOOKS -> {
                hooksController.changeMode(EposMode.HOMING);
                hooksController.writeParameter(Parameter.CurrentThresholdHomingMode, currentThreshold);
                hooksController.writeParameter(Parameter.HomingMethod, -4);
                hooksController.goToOperationEnable();
                /* start homing */
                hooksController.writeControlWord(ControlWord.ABSOLUTE_POSITION);
            }
            case CLOSE_LOADER_HOOKS -> {
                hooksController.changeMode(EposMode.PROFILE_POSITION);
                hooksController.enableAndWriteAbsolutePosition(this.absolutePositionToClose);
            }
            case CLAMP_LOADER_HOOKS -> {
                hooksController.changeMode(EposMode.CURRENT);
                hooksController.enableAndWriteCurrent((short) this.currentToClamp);
            }
            case UNCLAMP_LOADER_HOOKS -> {
                hooksController.changeMode(EposMode.PROFILE_POSITION);
                hooksController.enableAndWriteRelativePosition(this.relativePositionToUnclamp);
            }
            default -> {}
        }
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) {
        FCSLOG.finer(() -> name + " is ABORTING action " + action.toString() + " within delay " + delay);
        hooksController.quickStop();
        hooksController.goToSwitchOnDisabled();
    }

    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.finer(() -> name + " is ENDING action " + action.toString());
        /**
         *
         * Between STORAGE and HANDOFF, or HANDOFF and ENGAGED, the Local Protection System
         * shuts down the hooks (clamp) motor. If we don't disable the clamp controller during
         * carrier motion, it will go into Under Voltage Error (FAULT).
         *
         */
        this.hooksController.goToSwitchOnDisabled();
        this.publishData();
        loader.updateFCSStateToReady();
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        FCSLOG.finer(() -> name + " is STOPPING action " + action.toString() + " within delay " + delay);
        this.hooksController.quickStop(); //no test have been done
    }

    /**
     * reads sensors, computes and state and raises an ALERT if sensors are in
     * error.
     *
     */
    public void updateStateAndCheckSensors() {
        loader.updateStateWithSensors();
        checkSensors(FcsAlert.LO_SENSOR_ERROR, name);
    }

    /**
     * This methods updates the lockStatus and forceStatus.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    void updateState() {
        updatePosition();
        // forceStatus is based on the sensors
        updateForceStatus();
        // lockStatus is based on the forceStatus and the lockStatus of each hook
        updateLockStatus();

        this.publishData();
    }

    private boolean oneHookInError() {
        return Arrays.stream(hooks)
                    .anyMatch(hook -> hook.getLockStatus() == LockStatus.ERROR);
    }

    private boolean allHooksInState(LockStatus status) {
        return Arrays.stream(hooks)
                    .allMatch(hook -> hook.getLockStatus() == status);
    }

    private void updateLockStatus() {
        for (LoaderHook hook : hooks) {
            hook.updateState();
        }

        if (oneHookInError()) {
            this.lockStatus = LockStatus.ERROR;

        } else if (allHooksInState(LockStatus.OPENED)) {
            // needs the force status to be computed beforehand
            computeStatusHooksOpened();

        } else if (allHooksInState(LockStatus.CLOSED)) {
            this.lockStatus = clampedStatusSensor.isOn() ? LockStatus.CLAMPED : LockStatus.CLOSED;

        } else if (allHooksInState(LockStatus.INTRAVEL)) {
            this.lockStatus = LockStatus.INTRAVEL;

        } else {
            this.lockStatus = LockStatus.UNKNOWN;
        }
    }

    private void updateForceStatus() {
        if (unclampedStatusSensor.isOn()) {
            forceStatus = LockStatus.UNCLAMPED;

        } else if (underClampedStatusSensor.isOn()) {
            forceStatus = LockStatus.UNDER_CLAMPED;

        } else if (overClampedStatusSensor.isOn()) {
            forceStatus = LockStatus.OVER_CLAMPED;

        } else if (clampedStatusSensor.isOn()) {
            forceStatus = LockStatus.CLAMPED;

        } else {
            forceStatus = LockStatus.ERROR;
        }
    }

    private void computeStatusHooksOpened() {
        switch (forceStatus) {
            case UNCLAMPED -> this.lockStatus = LockStatus.OPENED;
            case CLAMPED, OVER_CLAMPED -> this.lockStatus = LockStatus.ERROR;
            default -> this.lockStatus = forceStatus;
        }
    }

    /**
     * Updates the position of the clamp in reading the CPU of the controller.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updatePosition() {
        try {
            this.position = hooksController.readPosition();
        } catch (SDORequestException ex) {
            FCSLOG.log(Level.WARNING,name + "=> ERROR IN READING CONTROLLER:", ex);
        }
    }

    /**
     * To display position for end user. Updates carrier position in reading
     * controller and returns it.
     *
     * @return position
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update and display clamp position from the controller.")
    public int readPosition() {
        updatePosition();
        return this.position;
    }

    /**
     * Return a printed list of hardware with initialization information. For debug
     * purpose.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return a printed list of hardware with initialization information.")
    public String printHardwareState() {
        StringBuilder sb = new StringBuilder(name);
        if (homingDone) {
            sb.append(" homing is DONE.");
        } else {
            sb.append(" homing is NOT DONE.");
        }
        return sb.toString();
    }

    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "List and display Loader clamp information.")
    public synchronized String toString() {
        updateState();
        updatePosition();
        StringBuilder sb = new StringBuilder("= " + name + " =");
        sb.append("Position: ").append(position).append("\n");
        sb.append("Homing done: ").append(homingDone).append("\n");
        sb.append("LockStatus: ").append(lockStatus.name()).append("\n");
        sb.append("+ hook1: ").append(hook1.getLockStatus().name()).append("\n");
        sb.append("+ hook2: ").append(hook2.getLockStatus().name()).append("\n");
        sb.append("+ hook3: ").append(hook3.getLockStatus().name()).append("\n");
        sb.append("+ hook4: ").append(hook4.getLockStatus().name()).append("\n");
        sb.append("~~~").append("\n");
        sb.append("ForceStatus: ").append(forceStatus.name()).append("\n");
        sb.append("+ force sensor 0 (mv): ").append(forceSensor0.getVoltage()).append("\n");
        sb.append("+ force sensor 1 (mv): ").append(forceSensor1.getVoltage()).append("\n");
        sb.append("~~~").append("\n");
        return sb.toString();
    }

    /**
     * Creates and returns the object to be published on the STATUS bus by the
     * LoaderClamp.
     *
     * @return
     */
    public StatusDataPublishedByLoaderClamp createStatusDataPublishedByLoaderClamp() {
        StatusDataPublishedByLoaderClamp status = new StatusDataPublishedByLoaderClamp();
        status.setClampState(lockStatus);
        status.setForceSensorVoltage0(forceSensor0.getVoltage());
        status.setForceSensorVoltage1(forceSensor1.getVoltage());
        status.setForceClampedStatus(clampedStatusSensor.isOn());
        status.setForceUnclampedStatus(unclampedStatusSensor.isOn());
        status.setForceUnderClampedStatus(underClampedStatusSensor.isOn());
        status.setForceOverClampedStatus(overClampedStatusSensor.isOn());
        status.setAllHooksInStateCLOSED(allHooksInState(LockStatus.CLOSED));
        status.setForceStatus(forceStatus);
        status.setHomingDone(homingDone);
        status.setControllerInError(hooksController.isInError());
        status.setStatusPublishedByHook1(hook1.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook2(hook2.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook3(hook3.createStatusDataPublishedByLoaderHook());
        status.setStatusPublishedByHook4(hook4.createStatusDataPublishedByLoaderHook());
        return status;
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     */
    @Override
    public void publishData() {
        KeyValueData kvd = new KeyValueData(path, this.createStatusDataPublishedByLoaderClamp());
        subs.publishSubsystemDataOnStatusBus(kvd);
    }

    @Override
    public void shutdown() {
        super.shutdown();
        if (hooksController.isBooted()) {
            hooksController.goToSwitchOnDisabled();
        }
        FCSLOG.info(name + " is shutdown.");
    }

}
