
package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

import java.util.logging.Logger;

/**
 *
 * Prototype online clamps have no encoder so they are opened and closed in
 * CURRENT mode. This class gathers methods and commands which are specific to
 * prototype.
 *
 * @author virieux
 */

// note using inheritance to modify behavior is breaking the meaning of
// inheritance
// better practice : factorize common behavior in abstract common parent
public class AutochangerThreeOnlineClampsProto extends AutochangerThreeOnlineClamps {
    private static final Logger FCSLOG = Logger.getLogger(AutochangerThreeOnlineClampsProto.class.getName());

    public AutochangerThreeOnlineClampsProto(AutochangerOnlineClamp onlineClampXminus,
            AutochangerOnlineClamp onlineClampXplus, AutochangerOnlineClamp onlineClampYminus) {
        super(onlineClampXminus, onlineClampXplus, onlineClampYminus);
    }

    // the modified behavior was in startAction
    // let's modify open and close

    @Override
    public void openClamps() {
        openClampsInCurrentMode();
    }

    @Override
    public void closeClamps() {
        closeClampsInCurrentMode();
    }

    /**
     * Closes the 3 clamps in mode CURRENT. initial state = OPENED final state =
     * CLOSED. for prototype only.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Close the 3 online clamps.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClampsInCurrentMode() {

        updateStateAndCheckSensors();
        if (isOpened()) {
            autochanger.checkConditionsForActioningOnlineClamps();
            onlineClampYminus.closeClampInCurrentMode();
            closeClampsXInCurrentMode();

        } else if (isClosed()) {
            FCSLOG.info(name + " clamps already CLOSED nothing to do");

        } else {
            throw new RejectedCommandException(name + " has to be unlocked before.");
        }
    }

    /**
     * Opens the 3 clamps with a small pressure. initial state = CLOSED final state
     * = OPENED. For prototype only.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Opens the 3 online clamps with a small pressure.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClampsInCurrentMode() {
        updateStateAndCheckSensors();
        if (isClosed()) {
            onlineClampYminus.checkReadyForAction();
            onlineClampYminus.openClampInCurrentMode();
            openClampsXInCurrentMode();

        } else if (isOpened()) {
            FCSLOG.info(name + " clamps already OPENED. Nothing to do.");
            throw new RejectedCommandException(name + " clamps already OPENED. Nothing to do.");

        } else {
            throw new RejectedCommandException(name + " has to be closed before.");
        }
    }

    /**
     * opens the 2 clampsX in mode CURRENT. for prototype only.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "open clampXs. for proto only.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClampsXInCurrentMode() {
        FcsUtils.parallelRun(() -> onlineClampXminus.openClampInCurrentMode(),
                () -> onlineClampXplus.openClampInCurrentMode());
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "close clamps X", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClampsXInCurrentMode() {
        FcsUtils.parallelRun(() -> onlineClampXminus.closeClampInCurrentMode(),
                () -> onlineClampXplus.closeClampInCurrentMode());
    }

    @Override
    public void postStart() {
        super.postStart();
        for (AutochangerOnlineClamp onlineClamp : clampsList) {
            onlineClamp.getController().setHomingDone();
        }

    }

}
