package org.lsst.ccs.subsystems.fcs;

import java.util.logging.Logger;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.IOModuleStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.GeneralAction;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;
import org.lsst.ccs.subsystems.fcs.utils.TTC580Utils;

/**
 * This is a socket on the carousel : there is 5 sockets on a carousel. When a
 * filter is on the carousel, it is attached at a socket. A socket has a clamp
 * on each side : - clampX- (clampXminus) - clampX+ (clampXplus) clampX+ and
 * clampX- are the same except for the method isLocked() See
 * FilterClampXminusModule and FilterClampXplusModule
 *
 * The main job of this class CarouselSocket is to synchronize the actions on
 * the 2 clamps. The state of the CarouselSocket is the state of the clamps when
 * both clamps state are identical.
 *
 * During an action on both clamps, we want to lock the CarouselSocket so no
 * other thread can try to access this object before the completion of action.
 * We do that with a Lock and Condition. (see java.util.concurrent).
 *
 * CarouselSocket extends MobileItem because we want to lock or release the
 * clamps when the carousel is at standby position and we want to wait until the
 * action is completed. We know if the action is completed in reading the clamps
 * sensors. So we use the general mechanism provided by the MobileItem.
 *
 *
 * @author virieux
 *
 */
public class CarouselSocket implements HasLifecycle, AlertRaiser {
    private static final Logger FCSLOG = Logger.getLogger(CarouselSocket.class.getName());

    private int id;

    @LookupName
    protected String name;

    @LookupPath
    protected String path;

    @LookupField(strategy = Strategy.ANCESTORS)
    private Subsystem subs;

    @LookupField(strategy = Strategy.TREE)
    protected Carousel carousel;

    private CarouselClamp clampXminus;

    private CarouselClamp clampXplus;

    @LookupField(strategy = Strategy.TREE, pathFilter=".*\\/clampXminusController")
    private EPOSController clampXminusController;

    @LookupField(strategy = Strategy.TREE, pathFilter=".*\\/clampXplusController")
    private EPOSController clampXplusController;

    @LookupField(strategy = Strategy.CHILDREN)
    protected SensorPluggedOnTTC580 ioModuleSensor;

    protected IOModuleStatus ioModuleStatus;

    /*To be able to know if the autochanger holds a filter. */
    /*This can't be a Autochanger because when the carousel is in Standalone mode
     /*there is not autochanger.
     /*FilterHolder is an Interface. */
    @LookupField(strategy = Strategy.TREE, pathFilter="autochanger")
    protected FilterHolder autochanger;

    @LookupField(strategy = Strategy.TREE)
    private FilterManager filterManager;

    @LookupField(strategy = Strategy.TREE)
    private AlertService alertService;

    @LookupField(strategy = Strategy.TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;


    /**
     * carousel positionOnCarousel (angle) when this socket is at standby
     * standbyPosition should be : 360 - positionOnCarousel, but could be different
     */
    @ConfigurationParameter(range = "-4500000..4500000", description = " carousel position when this socket is at standby [carousel encoder step]", units = "unitless", category = "carousel")
    private volatile int standbyPosition;

    /**
     * Used in filter exchange with autochanger. For some socket it's necessary
     * that autochanger pushes more, so deltaStandbyPosition will have a
     * positive value, for others it's necessary that autochanger pushes less,
     * so deltaStandbyPosition will have a negative value.
     *
     * See https://jira.slac.stanford.edu/browse/LSSTCCSFCS-507
     *
     */
    @ConfigurationParameter(range = "-2000..2000", description = "autochanger delta STANDBY position to be used to store a filter on carousel on this socket", units = "micron", category = "carousel")
    private volatile int deltaAutochangerStandbyPosition = 0;

    private int deltaPosition = 0;

    @Persist
    public volatile int filterID;

    private volatile FilterClampState clampsState = FilterClampState.UNDEFINED;

    /**
     * Flag to be set by the Camera team to indicate whether we can use this socket
     * If we do want to prohibit the use of this socket for some reason, use the
     * setAvailable command to switch this flag to false, or back again to true.
     */
    @Persist
    private volatile boolean available = true;

    // Counter for actions
    private PersistentCounter releaseClampsCounter;
    private PersistentCounter unlockClampsCounter;
    protected PersistentCounter storeFilterOnCarouselCounter;

    /**
     * Build a new CarouselSocket with a tickMillis of 5000.
     *
     * @param id
     * @param clampXminus
     * @param clampXplus
     */
    public CarouselSocket(int id, CarouselClamp clampXminus, CarouselClamp clampXplus) {
        this.id = id;
        this.clampXminus = clampXminus;
        this.clampXplus = clampXplus;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByCarouselSocket.class, path);

        // Counters
        unlockClampsCounter = PersistentCounter.newCounter(MobileItemAction.UNLOCK_CLAMPS.getCounterPath(path), subs, MobileItemAction.UNLOCK_CLAMPS.name());
        releaseClampsCounter = PersistentCounter.newCounter(MobileItemAction.RELEASE_CLAMPS.getCounterPath(path), subs, MobileItemAction.RELEASE_CLAMPS.name());
        MobileItemAction.UNLOCK_CLAMPS.registerDurationPerElement(dataProviderDictionaryService, path);
        MobileItemAction.RELEASE_CLAMPS.registerDurationPerElement(dataProviderDictionaryService, path);

        // High-level counters
        storeFilterOnCarouselCounter = PersistentCounter.newCounter(GeneralAction.STORE_FILTER_ON_CAROUSEL.getCounterPath(path),
                                                    subs, GeneralAction.STORE_FILTER_ON_CAROUSEL.name());
        GeneralAction.STORE_FILTER_ON_CAROUSEL.registerDurationPerElement(dataProviderDictionaryService, path);
    }

    public int getId() {
        return id;
    }

    public String getName() {
        return name;
    }

    public CarouselClamp getClampXminus() {
        return clampXminus;
    }

    public void setClampXminus(CarouselClamp clampXminus) {
        this.clampXminus = clampXminus;
    }

    public CarouselClamp getClampXplus() {
        return clampXplus;
    }

    public void setClampXplus(CarouselClamp clampXplus) {
        this.clampXplus = clampXplus;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns filterID if a filter is in the socket.")
    public int getFilterID() {
        return filterID;
    }

    /**
     *
     * @return NO_FILTER if no filter at standby or filter name
     */
    public String getFilterName() {
        return filterManager.getFilterNameByID(filterID);
    }

    public String getFilterObservatoryName() {
        return filterManager.getObservatoryNameByID(filterID);
    }

    /**
     * If the filter is different, then publish and persist it,
     * otherwise, do nothing.
     *
     */
    public void setFilterID(int filterID) {
        int formerFilterID = this.filterID;

        if ( formerFilterID == filterID )  {
            FCSLOG.fine(() -> name + " filter " + getFilterName() + " already on socket" + getId());
        } else {
            this.filterID = filterID;
            subs.getAgentPersistenceService().persistNow();
            publishData();
            FCSLOG.info(() -> name + " switching from filter " + filterManager.getFilterNameByID(formerFilterID) + " to filter " + this.getFilterName());
        }
    }

    public int getStandbyPosition() {
        return standbyPosition;
    }

    public int getDeltaPosition() {
        return deltaPosition;
    }

    public int getDeltaAutochangerStandbyPosition() {
        return deltaAutochangerStandbyPosition;
    }

    public IOModuleStatus getIOModuleStatus() {
        return ioModuleStatus;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    /**
     * Return true if this socket is at STANDBY position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if this socket is at STANDBY position on the carousel.")
    public boolean isAtStandby() {
        return carousel.getSocketAtStandbyID() == id;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if socket is ready to use or false if not.")
    public boolean isAvailable() {
        return available;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Set boolean available to true if socket is ready to use or false if not. "
            + "If available is false, this command should be used to return to 'available' only after FES engineers have been consulted.")
    public void setAvailable(boolean available) {
        this.available = available;
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
             description = "Flag the carousel socket as NOT READY TO USE. " +
             "If the socket holds a filter, it will prevent it from being used and will disappear from the list of available filters. " +
             "This is a protective command that can do no harm but will give some time to FES experts to review the status of the socket before deciding to reactivate it.")
    public void setUnavailable() {
        setAvailable(false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if socket is awake (ioModuleStatus different from NOT_POWERED_ON or BOOTING)")
    public boolean isAwake() {
        if ( !isAvailable() ) {
            /**
             * A non available socket should be considered "awaken" to prevent the powerOn
             * command from timing out
             * */
            FCSLOG.info(name + " socket not available, skipping test");
            return true;
        }
        FCSLOG.info(name + " ioModuleStatus = " + ioModuleStatus);
        return !ioModuleStatus.equals(IOModuleStatus.NOT_POWERED_ON) && !ioModuleStatus.equals(IOModuleStatus.BOOTING);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if socket clamps are responsive (filterPresence > 0)")
    public boolean areClampsResponsive() {
        if ( !isAvailable() ) {
            /**
             * A non available socket should be considered "awaken" to prevent the powerOn
             * command from  timing out
             */
            FCSLOG.info(name + " socket not available, skipping test");
            return true;
        }
        return clampXminus.isResponsive() && clampXplus.isResponsive();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if socket clamps are stabilised (providing normal clamp state)")
    public boolean areClampsStateStabilised() {
        if ( !isAvailable() ) {
            /**
             * A non available socket should be considered "awaken" to prevent the powerOn
             * command from  timing out
             */
            FCSLOG.info(name + " socket not available, skipping test");
            return true;
        }
        return clampXminus.isStabilised() && clampXplus.isStabilised();
    }

    /**
     * ***********************************************************************************************
     * END OF SETTERS AND GETTERS
     * ***********************************************************************************************
     */
    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void init() {
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(FcsAlert.CA_SENSOR_ERROR.getAlert(name), alwaysClear);

        this.clampXminus.setController(clampXminusController);
        this.clampXplus.setController(clampXplusController);
    }

    /**
     * Checking if autochanger is holding the filter is delegated to autochanger.
     *
     * @return true if autochanger is at STANDBY and holds a filter.
     */
    private boolean isAutochangerHoldingFilter() {
        return autochanger.isAtStandby() && autochanger.isHoldingFilter();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if hardware is connected and ready.")
    public boolean myDevicesReady() {
        return clampXminusController.isInitialized() && clampXplusController.isInitialized();
    }

    /**
     * Returns the state of the clamps. If the state is being updated and waiting
     * for a response from a sensor, this methods waits until the state is updated.
     * If the state is not being updated, it returns immediately the state.
     *
     * @return state
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return clamps state - doesn't read again sensors.")
    public FilterClampState getClampsState() {
        return clampsState;
    }

    /**
     * update state from sensors values.
     *
     */
    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-carouselSocket")) {

            if (ioModuleSensor.getValue() >= 1 && ioModuleSensor.getValue() <= 7) {
                ioModuleStatus = IOModuleStatus.getStatusByCode(ioModuleSensor.getValue());
            } else {
                ioModuleStatus = IOModuleStatus.UNKNOWN_STATUS;
            }
            if (ioModuleSensor.getValue() < 4 && ioModuleSensor.getValue() > 0) {
                updateClampsState();
            } else {
                clampsState = FilterClampState.UNDEFINED;
                /**
                 * Alex 04/25 : This should not raise an Alarm because it would otherwise limit
                 * solving issues by stopping all work every time. This is what we experienced today.
                 * Instead the degraded mode should be used to flag the socket as unavailable.
                 * and not trigger anything.
                 */
                if ( isAvailable() && !carousel.isAsleep() ) {
                    FCSLOG.info(name
                        + " IO module status = " + ioModuleSensor.getValue() + " => " + ioModuleStatus
                        + ", slip ring current = " + carousel.readSlipRingCurrent() + "mA");
                    raiseWarningOnlyIfNew(FcsAlert.CA_SENSOR_ERROR,
                        " IO module status = " + ioModuleSensor.getValue() + " => " + ioModuleStatus, name);
                }
            }
            updateDeltaPosition();
            publishData();
        }
    }

    public synchronized void updateDeltaPosition() {
        deltaPosition = carousel.getPosition() - this.standbyPosition % carousel.getFullTurn();
    }

    private void updateClampsState() {
        clampXminus.updateState();
        clampXplus.updateState();
        boolean inError = clampXminus.getClampState() == FilterClampState.ERROR
                || clampXplus.getClampState() == FilterClampState.ERROR;

        if (clampXminus.getClampState() == clampXplus.getClampState()) {
            this.clampsState = clampXminus.getClampState();
            FCSLOG.finer(() -> name + " : clamps are updated");

        } else if (clampXminus.getClampState() == FilterClampState.UNLOCKED_EMPTY && clampXplus.getClampState() == FilterClampState.READY_TO_LOCK) {
            if (!clampXminus.getController().isInState(EposState.OPERATION_ENABLE)
                    && !clampXplus.getController().isInState(EposState.OPERATION_ENABLE)) {
                this.clampsState = FilterClampState.READY_TO_LOCK;

            } else {
                this.clampsState = FilterClampState.UNLOCKED_EMPTY;
            }

        } else if (!inError) {
            FCSLOG.fine(() -> name + " clampXminus is " + clampXminus.getClampState() + " and clampXplus is "
                    + clampXplus.getClampState() + " clampState = " + FilterClampState.UNDEFINED);
            this.clampsState = FilterClampState.UNDEFINED;

        } else {
            this.clampsState = FilterClampState.ERROR;
            if ( isAvailable() ) {
                String msg = name + ": clampState at Xminus side is different from clampState at Xplus side. ";
                msg = msg + clampXminus.getName() + " state=" + clampXminus.getClampState() + " ";
                msg = msg + clampXplus.getName() + " state=" + clampXplus.getClampState();
                this.raiseWarningOnlyIfNew(FcsAlert.CA_SENSOR_ERROR, msg, name);
            }
        }
        if ( isAvailable() && !carousel.isAsleep() ) {
            /* no alarm is raised anymore in updateState so we have to raise alarm now */
            clampXminus.getSensorErrorCounter().forEach((msg) -> {
                raiseAlarm(FcsAlert.CA_SENSOR_ERROR, (String) msg, name);
            });
            clampXplus.getSensorErrorCounter().forEach((msg) -> {
                raiseAlarm(FcsAlert.CA_SENSOR_ERROR, (String) msg, name);
            });
        }
    }

    /**
     * update field filterID to be used only with whole FCS. In standalone mode,
     * filters can't be removed neither loaded into carousel so filterID never
     * changes.
     */
    public void updateFilterID() {
        if ( this.isEmpty() ) {
            setFilterID(0);
        } else if ( autochanger.isAtStandby() ) {
            setFilterID(autochanger.getFilterID());
        }
        FCSLOG.finer(() -> name + " filter ID = " + carousel.getSocketAtStandby().getFilterID());
    }

    /**
     * check and update lock sensor offset1 for the 2 clamps
     *
     * @param sdo
     */
    public void checkAndUpdateOffset1(long sdo) {
        clampXminus.checkAndUpdateOffset1(TTC580Utils.getOffset1Xminus(sdo));
        clampXplus.checkAndUpdateOffset1(TTC580Utils.getOffset1Xplus(sdo));
    }

    /**
     * check and update presence sensor offset2 for the 2 clamps
     *
     * @param sdo
     */
    public void checkAndUpdateOffset2(long sdo) {
        clampXminus.checkAndUpdateOffset2(TTC580Utils.getOffset2Xminus(sdo));
        clampXplus.checkAndUpdateOffset2(TTC580Utils.getOffset2Xplus(sdo));
    }

    /*
     * The carousel socket is empty if there is no filter in the socket. That
     * appends when clamps of both side of the socket are empty. A clamp is empty
     * when no filter is engaged in the clamp.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is no filter in the socket.")
    public boolean isEmpty() {
        return clampXminus.getFilterPresenceStatus() == FilterPresenceStatus.NOFILTER && clampXplus.getFilterPresenceStatus() == FilterPresenceStatus.NOFILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is a filter in the socket and the clamps"
            + "are LOCKED.")
    public boolean isClampedOnFilter() {
        return clampsState == FilterClampState.LOCKED_ON_FILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is a filter in the socket and the clamps"
            + "are UNLOCKED.")
    public boolean isUnclampedOnFilter() {
        return clampsState == FilterClampState.UNLOCKED_ON_FILTER;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if there is NO filter in the socket and the clamps"
            + "are UNLOCKED.")
    public boolean isUnclampedEmpty() {
        return clampsState == FilterClampState.UNLOCKED_EMPTY;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if this socket is ready to clamp a filter.")
    public boolean isReadyToClamp() {
        return clampsState == FilterClampState.READY_TO_LOCK;
    }

    /**
     * Releases the 2 clamps of the socket.
     *
     * Execute in parallel command release on the 2 clamps.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    private void releaseClamps(boolean forRecovery) {
        // for counter
        final long beginTime = System.currentTimeMillis();
        releaseClampsCounter.increment();

        try (AutoTimed at = new AutoTimed("socket-releaseClamps")) {
            carousel.updateStateWithSensors();
            FCSLOG.info("Checking conditions for release clamp " + name + " on socket " + "at standby position.");
            if (!this.isAtStandby()) {
                throw new RejectedCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");
            }
            if ( !forRecovery ) {
                if (clampXminus.isFilterEngaged() || clampXplus.isFilterEngaged()) {
                    throw new RejectedCommandException(
                            name + ": Can't release clamps because a filter is engaged in clamps.");
                }
                FCSLOG.info(name + ": Releasing clamps at standby position.");
            } else {
                FCSLOG.info(name + ": Releasing clamps with the filter still present, used only during recovery.");
            }
            FcsUtils.parallelRun(
                () -> clampXminus.release(),
                () -> clampXplus.release()
            );
        }
        MobileItemAction.RELEASE_CLAMPS.publishDurationPerElement(subs, System.currentTimeMillis() - beginTime, path);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Release the 2 clamps of this socket if the socket is at STANDBY position.")
    public void releaseClamps() {
        releaseClamps(false);
    }

    /**
     * This recovery method is to be used during a recovery of the clamps unlocking
     * to avoid checking if the filter is still inside the clamps when they are released
     */
    public void recoveryReleaseClamps() {
        releaseClamps(true);
        /**
         * In case the clamp release did not work correctly, we need to shake the system for clamping back X-
         * The testing of the clamp locking is done internally in the recovery locking method.
         */
        carousel.recoveryLockingXminus();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Unlock the 2 clamps of this socket if the socket is at STANDBY position.")
    public void unlockClamps() {
        // for counter
        final long beginTime = System.currentTimeMillis();
        unlockClampsCounter.increment();

        if (!carousel.isUnclampAllowedByPLC()) {
            throw new RejectedCommandException(name + " PLC does not allow to unlock clamps. Check PLCCarouselPanel.");
        }
        try (AutoTimed at = new AutoTimed("socket-unlockClamps")) {
            carousel.updateStateWithSensors();
            if (!this.isAtStandby()) {
                throw new RejectedCommandException(name + " is NOT AT STANDBY - can't unlock clamps.");
            }
            if (!this.isEmpty() && !this.isAutochangerHoldingFilter()) {
                throw new RejectedCommandException(
                        name + "cannot unlock clamps if NOT EMPTY and FILTER is NOT HELD by autochanger.");
            }
            FcsUtils.parallelRun(
                () -> clampXminus.unlock(),
                () -> clampXplus.unlock()
            );
            FCSLOG.info("Command unlockClamps completed");
            this.publishData();
        }
        MobileItemAction.UNLOCK_CLAMPS.publishDurationPerElement(subs, System.currentTimeMillis() - beginTime, path);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns a String representation of a CarouselSocket")
    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder("= " + name + " =");
        if ( !isAvailable() ) {
            sb.append("!!! NOT AVAILABLE !!!").append("\n");
        }
        String filterStr = isEmpty() ? "NONE" : getFilterName();
        sb.append("Filter: ").append(filterStr).append("\n");
        sb.append("Filter ID: ").append(filterID).append("\n");
        sb.append("At STANDBY: ").append(isAtStandby()).append("\n");
        sb.append("Clamp state: ").append(clampsState).append("\n");
        sb.append("IO Module status: ").append(ioModuleStatus).append("\n");
        sb.append("~~~").append("\n");
        return sb.toString();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns a string representation of a CarouselSocket and its clamps")
    public synchronized String printSocketInfo() {
        StringBuilder sb = new StringBuilder();
        sb.append(this.toString());
        sb.append(clampXminus.toString());
        sb.append(clampXplus.toString());
        return sb.toString();
    }

    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByCarouselSocket()));
    }

    /**
     * Create an object to be published on the STATUS bus by a CarouselSocket.
     *
     * @return
     */
    public StatusDataPublishedByCarouselSocket createStatusDataPublishedByCarouselSocket() {
        StatusDataPublishedByCarouselSocket status = new StatusDataPublishedByCarouselSocket();
        status.setAtStandby(this.isAtStandby());
        status.setIOStatus(ioModuleStatus);
        status.setClampsState(clampsState);
        status.setEmpty(this.isEmpty());
        status.setFilterID(filterID);
        status.setSocketID(id);
        status.setFilterObservatoryName(this.getFilterObservatoryName());
        status.setAvailable(available);
        /*  When there are no filter:
             + filterName is "NO FILTER"
             + filterObservatoryName is "NONE"
        */
        if (this.isEmpty()) {
            status.setFilterName(FCSCst.NO_FILTER);
        } else {
            status.setFilterName(getFilterName());
        }
        return status;
    }

}
