package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;
import java.util.EnumMap;
import java.util.logging.Logger;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.description.ComponentNode;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterPresenceStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.GeneralAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.errors.ActionTimeoutException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * A class to model a clamp that holds a filter on the carousel. Each clamp on
 * the carousel is coupled with 2 sensors which detect the presence of a filter,
 * or if the clamp is locked or unlocked. This class provides a method to lock,
 * unlock or release the clamps, and to compute the clampState of the clamp. The
 * clampState of the clamp is computed, updated and published on the status bus
 * each tick of the timer.
 *
 *
 * @author virieux
 */
public class CarouselClamp extends MobileItem implements MovedByEPOSController {
    private static final Logger FCSLOG = Logger.getLogger(CarouselClamp.class.getName());
    /**
     * to be able to known if autochanger hold a filter or not at STANDBY can't be
     * an instance of Autochanger because in standalone it's not an instance of
     * class Autochanger can't be strategy = TREE because in fcs__hardware
     * there are 2 FilterHolder : autochanger and loader
    */
    @LookupField(strategy = Strategy.TREE, pathFilter = "autochanger")
    private FilterHolder autochanger;

    @LookupField(strategy = Strategy.TREE)
    private Carousel carousel;

    @LookupField(strategy = Strategy.TREE, pathFilter = FCSCst.CHANGER_TCPPROXY_NAME)
    protected BridgeToHardware tcpProxy;

    @LookupField(strategy = Strategy.TREE)
    private AlertService alertService;

    private int id;

    /**
     * The motor controller which controls this clamp. This is initialized in the
     * initModule method of CarouselSocket.
     */
    protected EPOSController controller;

    /**
     * The filterPresenceSensor detects where if the filter is in the clamp.
     *
     * @See enum PresenceFilterStateOnClamp
     */
    private final SensorPluggedOnTTC580 filterPresenceSensor = new CarouselClampSensor();

    /**
     * The lock sensor detects if the clamp is locked or not.
     */
    private final SensorPluggedOnTTC580 lockSensor = new CarouselClampSensor();

    private FilterClampState clampState = FilterClampState.UNDEFINED;
    private FilterPresenceStatus filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;
    private LockStatus lockStatus = LockStatus.UNKNOWN;

    @ConfigurationParameter(description = "value of current to send to controller "
            + "to unlock carousel clamp", range = "-6000..6000", units = "mA", category = "carousel")
    protected volatile int currentToUnlock = -4660;

    /**
     * If lockSensor returns a value between lockSensorMinLimit and
     * lockSensorOffset1 the clamp is unlocked, If lockSensor returns a value
     * between lockSensorOffset1 and carousel.getLockSensorMinLocked(): we don't
     * know, If the sensor returns a value between
     * carousel.getLockSensorMinLocked() and lockSensorMaxValue: the clamp is
     * locked. If the sensor returns a value greater than lockSensorMaxValue the
     * sensor is in ERROR.
     *
     * offset1 is given by the electronic card inside each clamp
     *
     * offset1 is read on hyttc580 in Carousel.initializeAndCheckClampsOffset
     * offset1 is persisted and read at each startup.
     *
     */
    @Persist
    protected volatile Integer lockSensorOffset1 = 5000;

    /**
     * if filter presence sensor returns a value between 0 and filterPresenceOffset1
     * : sensor is in error.
     *
     * filterPresenceOffset1 has same value for all clamps.
     *
     */
    protected Integer filterPresenceMinLimit = 200;

    /**
     * if filter presence sensor returns a value between filterPresenceOffset1 and
     * filterPresenceOffset2 : the filter is engaged and lockable. This value is
     * read on hyttc580 when fcs starts. It is persisted. If the new value read is
     * different from the precedent value, it means that the clamp has been
     * replaced, and an ALERT must be launched.
     */
    @Persist
    protected volatile Integer filterPresenceOffset2 = 3000;

    @ConfigurationParameter(description = "An offset to account for a discrepancy in a given clamp filter presence sensor",
            range = "0..1000", units = "mV", category = "carousel")
    protected volatile Integer filterPresenceOffset3 = 0;

    @ConfigurationParameter(description = "Maximum value for the filter presence sensor on the clamp", range = "0..12000", units = "mV", category = "carousel")
    private volatile Integer filterPresenceMaxValue = 12000;

    @ConfigurationParameter(description = "Timeout for unlocking the clamp", range = "0..10000", units = "millisecond", category = "carousel")
    protected volatile long timeoutForUnlocking = 1500;

    @ConfigurationParameter(description = "Timeout for releasing the clamp", range = "0..10000", units = "millisecond", category = "carousel")
    protected volatile long timeoutForReleasing = 1500;

    /* updated from PDO of clamp controller */
    private int velocity = 0;

    private ArrayList<String> sensorErrorCounter;

    // Counter for socket usage
    protected PersistentCounter recoveryClampingXminusCounter;
    protected PersistentCounter recoveryUnclampingCounter;


    @SuppressWarnings("rawtypes")
    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByCarouselClamp.class, path);
        // TODO REFACTOR: Make this addition of counter the same across all classes if possible.
        // Movements counters
        movementCounter = new EnumMap<>(MobileItemAction.class);
        for (MobileItemAction action : new MobileItemAction[]{
            MobileItemAction.UNLOCK, MobileItemAction.RELEASE}) {
            action.registerDurationPerElement(dataProviderDictionaryService, path);
            movementCounter.put(action, PersistentCounter.newCounter(action.getCounterPath(path), subs, action.name()));
        }

        // High level action counter (store filter)
        if (isXminus()) {
            recoveryClampingXminusCounter = PersistentCounter.newCounter(GeneralAction.RECOVERY_CLAMP_XMINUS.getCounterPath(path),
                                                    subs, GeneralAction.RECOVERY_CLAMP_XMINUS.name());
            GeneralAction.RECOVERY_CLAMP_XMINUS.registerDurationPerElement(dataProviderDictionaryService, path);
        }
        recoveryUnclampingCounter = PersistentCounter.newCounter(GeneralAction.RECOVERY_UNCLAMP.getCounterPath(path),
                                                    subs, GeneralAction.RECOVERY_UNCLAMP.name());
        GeneralAction.RECOVERY_UNCLAMP.registerDurationPerElement(dataProviderDictionaryService, path);

        //We are adding here the filterPresenceSensor and the lockSensor to the lookup field.
        //For now we are using the same naming convention as was used in groovy:
        //  "filterPresenceX[minus|plus]{i}" and "lockSensorX[minus|plus]{i}"
        //where we extract X[minus|plus]{i} from this CarouselClampName
        String suffix = name.replace("clamp", "");
        ComponentLookup lookup = subs.getComponentLookup();
        ComponentNode thisNode = lookup.getComponentNodeForObject(this);

        lookup.addComponentNodeToLookup(thisNode, new ComponentNode("filterPresence"+suffix,filterPresenceSensor));
        lookup.addComponentNodeToLookup(thisNode, new ComponentNode("lockSensor"+suffix,lockSensor));
    }

    /**
     * Returns the amount of current needed to unlock this clamp. This is defined in
     * the ConfigurationSystem.
     *
     * @return
     */
    public short getCurrentToUnlock() {
        return (short) currentToUnlock;
    }

    /**
     *
     * @return a list of error messages when updating sensors
     */
    public ArrayList<String> getSensorErrorCounter() {
        return sensorErrorCounter;
    }

    /**
     * In the initialization phase, this method is used to initialize the
     * controller. cf initModule in CarouselSocket
     *
     * @param actuator
     */
    protected void setController(EPOSController actuator) {
        this.controller = actuator;
    }

    /**
     * @return the lockSensorOffset1 for simulation
     */
    public int getLockSensorOffset1() {
        return lockSensorOffset1;
    }

    public int getLockSensorMaxLimit() {
        if (isXminus()) {
            return carousel.getLockSensorMaxLimitXminus();
        } else {
            return carousel.getLockSensorMaxLimitXplus();
        }
    }

    public int getLockSensorMinLimit() {
        if (isXminus()) {
            return carousel.getLockSensorMinLimitXminus();
        } else {
            return carousel.getLockSensorMinLimitXplus();
        }
    }

    /**
     * @return the filterPresenceOffset1
     */
    public int getFilterPresenceMinLimit() {
        return filterPresenceMinLimit;
    }

    /**
     * For simulation
     *
     * @return the filterPresenceOffset2
     */
    public int getFilterPresenceOffset2() {
        return filterPresenceOffset2;
    }

    /**
     * For simulation.
     *
     * @return the filterPresenceSensor
     */
    public SensorPluggedOnTTC580 getFilterPresenceSensor() {
        return filterPresenceSensor;
    }

    /**
     * For simulation.
     *
     * @return the lockSensor
     */
    public SensorPluggedOnTTC580 getLockSensor() {
        return lockSensor;
    }

    /**
     * @return the lockStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "return the lockStatus")
    public LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     *
     * @return filterPresenceStatus
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "return filterPresenceStatus")
    public FilterPresenceStatus getFilterPresenceStatus() {
        return filterPresenceStatus;
    }

    public void setFilterPresenceOffset2(Integer value) {
        this.filterPresenceOffset2 = value;
    }

    public boolean isAvailable() {
        return carousel.getSocketByName("socket" + id).isAvailable();
    }

    /**
     * ***********************************************************************************************
     * * END OF SETTERS AND GETTERS
     * ***********************************************************************************************
     *
     */
    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void init() {
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(FcsAlert.CA_SENSOR_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlert.CA_LOCKING_RECOVERY_SUCCESS.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlert.CA_LOCKING_RECOVERY_FAILURE.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlert.CA_UNLOCKING_RECOVERY_SUCCESS.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlert.CA_UNLOCKING_RECOVERY_FAILURE.getAlert(name), alwaysClear);
    }

    /**
     * *** end of lifecycle methods
     * *************************************************
     */


    /**
     * *
     *
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if CANopen hardware is connected and ready.")
    @Override
    public boolean myDevicesReady() {
        return controller.isInitialized();
    }

    /**
     * Returns the clampState of the clamp. If the clampState is being updated and
     * waiting for a response from a sensor, this methods waits until the clampState
     * is updated. If the clampState is not being updated, it returns immediately the
     * clampState.
     *
     * @return clampState
     *
     */
    public FilterClampState getClampState() {
        return clampState;
    }

    /**
     * This methods returns true if the clamp is locked. In the simulator, it is
     * overridden.
     *
     * @return true if the clamp is locked
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if the clamp is locked")
    public boolean isLocked() {
        return this.lockStatus == LockStatus.LOCKED;
    }

    private boolean isAtStandby() {
        return carousel.getSocketAtStandbyID() == id;
    }

    /**
     * Test used to qualify the responsiveness of the clamp after carousel wake up.
     * This was implemented after we found out that the first filterPresence values read were
     * sometimes (<5%) 0 right after wake up. We therefore need to wait a bit longer to see
     * non zero values before considering the clamp is responsive.
     */

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if the clamp filter presence returns a value different from 0")
    public boolean isResponsive() {
        int fpresence = filterPresenceSensor.getValue();
        FCSLOG.info(name + " filterPresence value = " + fpresence);
        return fpresence > 0;
    }

    /**
     * Test used to qualify the state of the clamp after carousel wake up, it has no value outside
     * This was implemented after we found out that the lockStatus and filterPresence sensor values
     * had sometimes (<1%) a very hard time to reach their actual value right after wake up, probably
     * due to temperature averaging effects in the controller.
     * In normal operations, the clamp states should only be READY_TO_CLAMP, CLAMPED_ON_FILTER or
     * UNCLAMPED_EMPTY when waking up, otherwise we should already be in alarm.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if the clamp state after a wake up is expected")
    public boolean isStabilised() {
        switch (clampState) {
            case READY_TO_LOCK:
            case LOCKED_ON_FILTER:
            case UNLOCKED_EMPTY:
                FCSLOG.info(name + " clamp state = " + clampState);
                return true;

            default:
                FCSLOG.info(name + " clamp not stabilised. Current clamp state = " + clampState);
                return false;
        }
    }

    /**
     *
     * @return true if filter is engaged on the clamp
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if filter is engaged on the clamp : filter presence sensors sees it ")
    public boolean isFilterEngaged() {
        return this.filterPresenceStatus == FilterPresenceStatus.LOCKABLE;
    }

    /**
     * compare offset1 value given as argument with persisted value.
     *
     * @param newOffset1
     */
    public void checkAndUpdateOffset1(int newOffset1) {
        if (Math.abs(newOffset1 - lockSensorOffset1) > carousel.getMaxClampsOffsetDelta()) {
            lockSensorOffset1 = newOffset1;
            this.raiseAlarm(FcsAlert.CA_SENSOR_ERROR, " offset1 has changed; new value = " + newOffset1, name);
        }
    }

    /**
     * compare offset2 value given as argument with persisted value.
     *
     * @param newOffset2
     */
    public void checkAndUpdateOffset2(int newOffset2) {
        if (Math.abs(newOffset2 - filterPresenceOffset2) > carousel.getMaxClampsOffsetDelta()) {
            filterPresenceOffset2 = newOffset2;
            this.raiseAlarm(FcsAlert.CA_SENSOR_ERROR, " offset2 has changed; new value = " + newOffset2, name);
        }
    }

    /**
     *
     * @return true if this clamp is clampXminus.
     */
    public boolean isXminus() {
        return name.contains("Xminus");
    }

    /**
     * update filterPresenceStatus from value returned by filterPresenceSensor
     */
    public void updateFilterPresenceStatus() {
        String errorMsg;
        int newFilterPresenceSensorValue = filterPresenceSensor.getValue();
        if (newFilterPresenceSensorValue < filterPresenceMinLimit) {
            if (carousel.isRotating() || carousel.isAsleep()) {
                /*
                 * during rotation, communication with TTC30 is sometimes lost, it's a transient
                 * error, just log
                 */
                FCSLOG.info(name + String.format(name + " lost communication with TTC30 during rotation at position %d",
                        carousel.getPosition()));
            } else if ( this.isAvailable() ) {
                this.filterPresenceStatus = FilterPresenceStatus.ERROR;
                errorMsg = name + " ERROR new read value FOR FILTER POSITION SENSOR  = " + newFilterPresenceSensorValue
                        + " must be >=" + filterPresenceMinLimit + " lost communication with TTC30 ?"
                        + "slip ring current = " + carousel.readSlipRingCurrent() + "mA";
                sensorErrorCounter.add(errorMsg);
                this.raiseWarning(FcsAlert.CA_SENSOR_ERROR, errorMsg, name);
            }

        } else if (newFilterPresenceSensorValue < this.filterPresenceOffset2 + filterPresenceOffset3) {
            this.filterPresenceStatus = FilterPresenceStatus.LOCKABLE;

        } else if (newFilterPresenceSensorValue < carousel.getFilterPresenceMinNoFilter()) {
            this.filterPresenceStatus = FilterPresenceStatus.NOT_LOCKABLE;

        } else if (newFilterPresenceSensorValue < this.filterPresenceMaxValue) {
            this.filterPresenceStatus = FilterPresenceStatus.NOFILTER;

        } else if ( this.isAvailable() ) {
            this.filterPresenceStatus = FilterPresenceStatus.ERROR;
            errorMsg = name + " ERROR new read value FOR FILTER POSITION SENSOR  = " + newFilterPresenceSensorValue
                    + " must be <=" + this.filterPresenceMaxValue;
            sensorErrorCounter.add(errorMsg);
            this.raiseWarning(FcsAlert.CA_SENSOR_ERROR, errorMsg, name);
        }
    }

    /**
     * Update lockStatus from value returned by lockSensor.
     *
     * @throws FcsHardwareException
     */
    public void updateLockStatus() {
        String errorMsg;
        int mechanicalValue = lockSensor.getValue();
        if (mechanicalValue < getLockSensorMinLimit()) {
            if (carousel.isRotating() || carousel.isAsleep()) {
                /*
                 * during rotation, communication with TTC30 is sometimes lost, it's a transient
                 * error, just log
                 */
                FCSLOG.info(name + String.format(name + " lost communication with TTC30 during rotation at position %d",
                        carousel.getPosition()));
            } else if ( this.isAvailable() ) {
                this.lockStatus = LockStatus.ERROR;
                errorMsg = name + " ERROR new read value FOR LOCK SENSOR  = " + mechanicalValue + " should be >= "
                        + getLockSensorMinLimit() + " lost communication with TTC30 ?\n"
                        + "slip ring current = " + carousel.readSlipRingCurrent() + "mA";
                sensorErrorCounter.add(errorMsg);
                this.raiseWarning(FcsAlert.CA_SENSOR_ERROR, errorMsg, name);
            }

        } else if (mechanicalValue < lockSensorOffset1) {
            this.lockStatus = LockStatus.UNLOCKED;

            /*
             * for clampXminus, lockSensor can sent a value > lockSensorValueA but it is
             * opened
             */
        } else if (mechanicalValue <= lockSensorOffset1 + 500 && isXminus()) {
            this.lockStatus = LockStatus.RELAXED;

        } else if (mechanicalValue <= carousel.getMinLockedThreshold()) {
            this.lockStatus = LockStatus.UNKNOWN;

        } else if (mechanicalValue <= getLockSensorMaxLimit()) {
            this.lockStatus = LockStatus.LOCKED;

        } else if ( this.isAvailable() ) {
            this.lockStatus = LockStatus.ERROR;
            errorMsg = name + " ERROR new read value FOR LOCK SENSOR  = " + mechanicalValue + " should be <= "
                    + getLockSensorMaxLimit();
            sensorErrorCounter.add(errorMsg);
            this.raiseWarning(FcsAlert.CA_SENSOR_ERROR, errorMsg, name);
        }
    }

    /**
     * Update clampState from values sent by sensors. This method updates the clamp
     * clampState regarding the value returned by the filter presence sensor and the
     * value returned by the method isLocked(). When the update is completed, it
     * sends a signal to threads waiting to get the new value of clampState.
     *
     * @throws FcsHardwareException
     */
    // TODO: Why synchronized here ? It's not the case in other updateState
    public void updateState() {
        synchronized (this) {
            sensorErrorCounter = new ArrayList<String>();
            this.updateFilterPresenceStatus();
            this.updateLockStatus();
            this.computeClampState();
            this.updateVelocity();
        }
        this.publishData();
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        StatusDataPublishedByCarouselClamp status = this.createStatusDataPublishedByClamp();
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, status));
    }

    /**
     * Create an object to be published on the STATUS bus.
     *
     * @return
     */
    public StatusDataPublishedByCarouselClamp createStatusDataPublishedByClamp() {
        StatusDataPublishedByCarouselClamp status = new StatusDataPublishedByCarouselClamp();
        status.setClampState(clampState);
        status.setFilterPresenceStatus(filterPresenceStatus);
        status.setFilterPositionSensorValue(filterPresenceSensor.getValue());
        status.setLockSensorValue(lockSensor.getValue());
        status.setLockStatus(lockStatus);
        status.setLockSensorOffset1(lockSensorOffset1);
        status.setFilterPresenceMinLimit(filterPresenceMinLimit);
        status.setFilterPresenceOffset2(filterPresenceOffset2);
        return status;
    }

    /**
     * Compute the global state of the clamp given the lock sensor and the presence
     * filter sensor state. This has to be overridden for the clamp X-.
     *
     */
    private void computeClampState() {
        if ( lockStatus == LockStatus.ERROR ) {
            clampState = FilterClampState.ERROR;
            return;
        }

        switch (filterPresenceStatus) {
            case ERROR:
                clampState = FilterClampState.ERROR;
                break;

            case NOT_LOCKABLE:
                clampState = FilterClampState.UNLOCKABLE;
                break;

            case LOCKABLE:
                switch (lockStatus) {
                    case LOCKED:
                        clampState = FilterClampState.LOCKED_ON_FILTER;
                        break;

                    case UNLOCKED:
                        clampState = FilterClampState.UNLOCKED_ON_FILTER;
                        break;

                    default:
                        clampState = FilterClampState.UNDEFINED;
                }
                break;

            case NOFILTER:
                switch (lockStatus) {
                    case LOCKED:
                        clampState = FilterClampState.READY_TO_LOCK;
                        break;

                    case UNLOCKED:
                    case RELAXED:
                        clampState = FilterClampState.UNLOCKED_EMPTY;
                        break;

                    default:
                        clampState = FilterClampState.UNDEFINED;
                }
                break;

            default:
                clampState = FilterClampState.UNDEFINED;

        }
        FCSLOG.fine(() -> name + " is " + filterPresenceStatus + " and " + lockStatus + " lockValue = "
                    + lockSensor.getValue() + " clampState = " + clampState);
    }

    /**
     * The clamps on the carousel are locked automatically when the filter comes at
     * the standby position. To be able to lock automatically again, it has to be
     * released after each time it has been unlocked.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(level = Command.ENGINEERING_EXPERT, type = Command.CommandType.ACTION, description = "Release clamp in order to get ready to clamp a filter again")
    public void release() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("CarouselClamp-release")) {
            FCSLOG.info("Checking conditions for release clamp " + name + " on socket at standby position.");

            carousel.updateSocketAtStandbyReadSensorsNoPublication();
            FCSLOG.info("Releasing clamp " + name + " on socket at standby position.");
            this.executeAction(MobileItemAction.RELEASE, timeoutForUnlocking);
        }
        carousel.updateSocketAtStandbyWithSensors();
    }

    /**
     *
     * Check if controller has been disabled. For end user. Not to be used in other
     * commands because it read StatusWord by SDO.
     *
     * @return true if controller is not enabled.
     */
    @Command(level = Command.ENGINEERING_EXPERT, type = Command.CommandType.ACTION, description = "Check if controller has been disabled. For end user.")
    public boolean isReleased() {
        return !controller.isEnabled();
    }

    /**
     * Unlock the clamp when a filter is locked by the clamp. This is used only in
     * engineering mode.
     *
     * @throws RejectedCommandException
     * @throws FailedCommandException
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Unlock the clamp")
    public void unlock() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("CarouselClamp-unlock")) {
            FCSLOG.info(name + ": " + "UNLOCK State1 = " + clampState.toString());
            carousel.updateSocketAtStandbyReadSensorsNoPublication();

            if (isFilterEngaged() && autochanger.isAtStandby() && !autochanger.isHoldingFilter()) {
                throw new RejectedCommandException(
                        name + "can ONLY unlock a clamp if FILTER is HELD by autochanger.");
            }

            try {
                this.executeAction(MobileItemAction.UNLOCK, timeoutForUnlocking);
            } catch (ActionTimeoutException ex) {
                // Here the clamp is not completely opened so we need the recovery procedure
                recoveryUnlocking();
            }
        }
        carousel.updateSocketAtStandbyState();
    }

    /**
     * To recover locking when clampXplus is LOCKED and not clampXminus.
     * This is usually a tricky situation and we need to put a weird current value to "shake" the system
     * And then we need a relaxation time before testing if it worked.
     * After some tests on the real camera, we saw that we might need several attempts before it works
     *
     */
    public void recoveryLocking() {
        // for counter
        final long beginTime = System.currentTimeMillis();
        recoveryClampingXminusCounter.increment();

        final int maxRetry = 3;
        long duration = 0;
        int tryout = 0;
        boolean state_ok = false;
        while (tryout < maxRetry && !state_ok) {
            tryout += 1;
            this.controller.goToOperationEnable();
            this.controller.writeCurrent((short) carousel.getRecoveryLockingCurrent());
            try {
                waitForLocked(800, 250);
            } catch (RejectedCommandException ex) {
                FCSLOG.info(name + " recovery clamping tryout " + tryout + "/3 did not succeed"
                    + String.format(" filter presence = %d lock value = %d", filterPresenceSensor.getValue(), lockSensor.getValue()));
            } finally {
                this.controller.goToSwitchOnDisabled();
                /* relaxation time to let the clamp have a chance to complete locking
                see https://jira.slac.stanford.edu/browse/LSSTCCSFCS-503 */
                if ( !isLocked() ) {
                    FcsUtils.sleep(200, name);
                    tcpProxy.updatePDOData();
                    carousel.updateSocketAtStandbyState();
                    duration = System.currentTimeMillis() - beginTime;
                    updateLockStatus();
                }
            }
            state_ok = isLocked();
        }
        if (state_ok) {
            raiseWarning(FcsAlert.CA_LOCKING_RECOVERY_SUCCESS,
                String.format(
                    name + " clamp is finally LOCKED. filter presence = %d, lock value = %d",
                    filterPresenceSensor.getValue(),
                    lockSensor.getValue()
                ),
                name);
            FCSLOG.info(name + String.format(" go to state LOCKED duration = %d", duration));
        } else {
            raiseWarning(FcsAlert.CA_LOCKING_RECOVERY_FAILURE,
                String.format(
                    name + " could not be LOCKED during allocated time, stop waiting. filter presence = %d, lock value = %d",
                    filterPresenceSensor.getValue(),
                    lockSensor.getValue()
                ),
                name);
            String msg = name + " LOCKING RECOVERY FAILED ";
            String cause;
            if (velocity > carousel.getRecoveryMaxVelocity()) {
                cause = String.format(
                    "Cause = controller velocity reached %d which is over limit %d",
                    velocity,
                    carousel.getRecoveryMaxVelocity()
                );
            } else {
                cause = String.format("Cause = couldn't go to state LOCKED during time allocated of %d ms;",
                        duration);
            }
            String msg_log = msg + cause + String.format("; filter presence value = %d lock value = %d",
                    filterPresenceSensor.getValue(), lockSensor.getValue());
            raiseAlarm(FcsAlert.CA_LOCKING_RECOVERY_FAILURE, msg_log, name);
            FCSLOG.severe(msg_log);
            throw new FcsHardwareException(msg);
        }

        // for duration
        GeneralAction.RECOVERY_CLAMP_XMINUS.publishDurationPerElement(subs, System.currentTimeMillis() - beginTime, path);
    }

    private void waitForLocked(long timeoutMillis, int updateRate) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("CarouselClamp-waitForLocked")) {
            FcsUtils.checkAndWaitConditionWithTimeoutAndFixedDelay(
                () -> isLocked(),
                () -> {
                    tcpProxy.updatePDOData();
                    carousel.updateSocketAtStandbyState();
                    updateLockStatus();
                },
                "clampRecoveryLockingProcess",
                "carousel clamp recoveryLocking failed",
                timeoutMillis,
                updateRate
            );
        }
    }

    /**
     * To recover unlocking when a clamp (more often Xminus than Xplus) has failed to open.
     * This method starts with a cool down of the clamps during 2 seconds, followed by a manual
     * current ramp that pulls first slowly on the cable to build some tension, and ends with
     * a boost up to the current limit.
     * If this method does not manage to open the clamp, then we release the tension and disable
     * the clamp to avoid keeping it under voltage and heating up.
     * This method designed to be used for the automated recovery.
     * For the end user recovery, refer to recoveryUnlockingAndMoveFilterOnline.
     */
    public void recoveryUnlocking() {
        final long timeStart = System.currentTimeMillis();
        if ( getClampState() == FilterClampState.UNLOCKED_ON_FILTER ) {
            FCSLOG.info("The carousel " + name + " is already unlocked. Canceling recovery.");
            return;
        }

        // for counter
        recoveryUnclampingCounter.increment();

        /**
         * Manual current ramp for unlocking from 0 to the max current allowed
         * which should not exceed -5000 mA at the summit
         */
        final int[] currentRamp = {0, -150, -500, -1200, -1900, -2600, -3400, -4950};

        tcpProxy.updatePDOData();
        updateState();
        for (int current : currentRamp) {
            /*
                TODO TRENDING: the current value will be published two times,
                 + in enableAndWriteCurrent -> checkFault -> publishData,
                 + in updatePDOData -> updateFromPDO -> publishData.
                However, in the implementation, we cannot skip neither functions, because:
                 + enableAndWriteCurrent is writing new current to controller,
                 + updatePDOData is reading current (and all others PDOs) on the controller.
             */
            controller.enableAndWriteCurrent(current);
            tcpProxy.updatePDOData();
            updateState();
            if ( current == 0 ) {
                // Time to get the clamp motor to cool down
                FcsUtils.sleep(2000, name);
            } else {
                // Time to let the motor get to speed
                FcsUtils.sleep(500, name);
            }
        }

        carousel.waitForStateUnclampedOnFilter(5000);

        long duration = System.currentTimeMillis() - timeStart;

        String debug = String.format(
            "\nfilter presence value = %d, lock value = %d",
            filterPresenceSensor.getValue(), lockSensor.getValue()
        );

        if ( carousel.isUnclampedOnFilterAtStandby() ) {
            String successMsg = String.format("%s recoveryUnlocking reached state UNLOCKED in %d ms", name, duration);
            FCSLOG.info(successMsg);
            raiseWarning(FcsAlert.CA_UNLOCKING_RECOVERY_SUCCESS, successMsg + debug, name);
        } else {
            /* The recovery did not work, we need to release the clamps and raise the alarm */
            carousel.getSocketAtStandby().recoveryReleaseClamps();

            String msg = String.format(
                "Carousel socket %s could not be UNLOCKED after recoveryUnlocking during allocated time of %d ms.",
                name, duration
            );
            String help = String.format(
                "\nThis means filter %s cannot be extracted from the carousel %s currently.",
                carousel.getSocketAtStandby().getFilterObservatoryName(),
                carousel.getSocketAtStandby().getName()
            );
            help += "\nThis socket should probably be set unavailable until a FES expert can troubleshoot the issue with the unlocking.";
            help += "\nFirst get the autochanger out of the way, use 'fcs/autochanger moveEmptyFromStandbyToHandoff' (level 1)";
            help += "\nSecond go set the relevant socket as not available using 'fcs/carousel/socketX setUnavailable' (level 1)";

            FCSLOG.severe(msg + debug);
            /* The WARNING will stay visible when the ALARM will be cleared to solve the issue */
            raiseWarning(FcsAlert.CA_UNLOCKING_RECOVERY_FAILURE, msg + debug, name);
            raiseAlarm(FcsAlert.CA_UNLOCKING_RECOVERY_FAILURE, msg + help, name);
            throw new FcsHardwareException(msg);
        }
        // for duration
        GeneralAction.RECOVERY_UNCLAMP.publishDurationPerElement(subs, duration, path);
    }

    /**
     * General recovery method for unlocking the carousel clamps and crafted for use by end user.
     * It starts the recovery of the clamp and then checks the carousel clamp status before moving
     * the filter out of the way, release the clamp and then move and clamp ONLINE.
     * In case of the clamps not being opened at the end of the recovery, the clamps will be locked
     * again on the filter and the autochanger will unlatch and go empty to HANDOFF position.
     */
    public void recoveryUnlockingAndMoveFilterOnline() {
            if ( !(autochanger instanceof Autochanger) ) {
                throw new FcsHardwareException(name + " this recovery method should only be used on the real hardware, not the simulation.");
            }
            Autochanger ac = (Autochanger) autochanger;

            recoveryUnlocking();

            if ( carousel.isUnclampedOnFilterAtStandby() ) {
                FCSLOG.info("The carousel "+ name + " is now unlocked, recovery successful. " +
                    "Proceeding with disengaging the filter and releasing the clamps before moving the filter ONLINE.");
                ac.moveToApproachStandbyPositionWithLowVelocity();
                carousel.releaseClamps();
                ac.moveAndClampFilterOnline();
            } else {
                carousel.getSocketAtStandby().recoveryReleaseClamps();
                ac.moveEmptyFromStandbyToHandoff();
                String msg = "The carousel " + name + " unlocking recovery on " + carousel.getSocketAtStandby().getName() + " did not succeed. "
                    + "After recovery the clamp " + name + " ended in state " + getClampState().toString()
                    + " Both carousel clamps were released to avoid damaging the system and the autochanger was moved empty to HANDOFF position."
                    + " We recommend that the socket be set to 'unavailable' for the time being to avoid further damage.";
                throw new FcsHardwareException(name + msg);
            }
    }

    @Override
    public void startAction(MobileItemAction action) {
        switch (action) {
            case UNLOCK:
                this.controller.goToOperationEnable();
                // Following a request from Guillaume Daubard, we first send a small current
                // to the clamp to prepare it and prevent an unstable state when we send the
                // full current to unlock.
                int timeToPrepareUnlock = carousel.getTimeToPrepareUnlock();
                short currentToPrepareUnlock = (short) carousel.getCurrentToPrepareUnlock();

                this.controller.writeCurrent(currentToPrepareUnlock);

                // a little sleep to let time to the hardware to prepare
                FcsUtils.sleep(timeToPrepareUnlock, name);

                this.controller.writeCurrent((short) this.currentToUnlock);
                break;

            case RELEASE:
                controller.goToSwitchOnDisabled();
                break;

            default:
                throw new IllegalArgumentException("Action on clamp must be UNLOCK or RELEASE");
        }

    }

    @Override
    public boolean isActionCompleted(MobileItemAction action) {
        switch (action) {
            case UNLOCK:
                return this.lockStatus == LockStatus.UNLOCKED;

            case RELEASE:
                return !controller.isInState(EposState.OPERATION_ENABLE);

            default:
                throw new IllegalArgumentException("Action on clamp must be UNLOCK or RELEASE");
        }
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        carousel.updateSocketAtStandbyReadSensorsNoPublication();
        publishData();
    }

    @Override
    public void abortAction(MobileItemAction action, long delay) {
        FCSLOG.finer(() -> name + " is ABORTING action " + action.toString() + " within delay " + delay);
        FCSLOG.finer(() -> name + " NOTHING BEING DONE HERE");
    }

    @Override
    public void endAction(MobileItemAction action) {
        FCSLOG.finer(() -> name + " is ENDING action " + action.toString());
        carousel.updateStateWithSensors();
        FCSLOG.finer(() -> name + " NOTHING BEING DONE HERE");
    }

    public void checkVelocity() {
        FCSLOG.info(name + " checking velocity;" + " velocity = " + controller.getVelocity());
        FcsUtils.checkAndWaitConditionWithTimeoutAndFixedDelay(
                () -> Math.abs(controller.getVelocity()) < 20,
                () -> tcpProxy.updatePDOData(),
                name + " check controller velocity is around 0 after unlock ",
                name + ": controller velocity is too high after trying every 100ms during 500 ms",
                500,
                100);
    }

    @Override
    public void quickStopAction(MobileItemAction action, long delay) {
        throw new UnsupportedOperationException("Not supported yet. Not relevant to stop action clamps.");
    }

    @Override
    public EPOSController getController() {
        return controller;
    }


    public void updateVelocity() {
        if (isAtStandby()) {
            /* velocity is updated by PDO */
            velocity = controller.getVelocity();
        } else {
            velocity = 0;
        }
    }

    @Override
    public void postStart() {
        /* id is the last char of name */
        id = Integer.parseInt(name.substring(name.length() - 1));
        FCSLOG.info(name + " ID =" + id);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns a String representation of a CarouselClamp")
    @Override
    public synchronized String toString() {
        StringBuilder sb = new StringBuilder("+> " + name + " <+").append("\n");
        sb.append("+ clamp state: ").append(clampState).append("\n");
        sb.append("+ lock status: ").append(lockStatus).append("\n");
        sb.append("+ lock sensor value: ").append(lockSensor.getValue()).append("\n");
        sb.append("+ filter presence status: ").append(filterPresenceStatus).append("\n");
        sb.append("+ filter presence value: ").append(filterPresenceSensor.getValue()).append("\n");
        sb.append("~~~").append("\n");
        return sb.toString();
    }
}
