package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.ControlWord;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 *
 * @author virieux
 */
public interface EPOSControllerForCarousel extends EPOSController {

    int getPositionSensorType();

    void setPositionSensorType(int pst);

    void setHoldingBrakes(boolean holdingBrake);

    boolean isHoldingBrakes();

    /**
     * For carousel, status word is read by PDO so no need to read by SDO. During
     * actions, read by PDO are done every 250ms.
     *
     * @return status word
     */
    @Override
    default boolean isTargetReached() {
        /* target is reached when bit10 of status word is 1 */
        return ((getStatusWord() >> 10) & 1) == 1;
    }

    /**
     * Set PositionSensorType to Absolute encoder SSI (value=4)
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "GoToSwitchOnDisabled and set PositionSensorType to Absolute encoder SSI : 4")
    default void setPositionSensorTypeEncoderSSI() {
        FCSLOG.info(getName() + " setting PositionSensorType Absolute encoder SSI");
        goToSwitchOnDisabled();
        doSetEncoderSSI();
    }

    default void doSetEncoderSSI() {
        int controllerStructure = readControllerStructure();
        if (controllerStructure == 0) {
            writeParameter(Parameter.PositionSensorType, 4);
            /* check that position sensor type is SSI encoder. */
            checkPositionSensorType(4);
        } else {
            writeParameter(Parameter.PositionSensorType, 0x0204);
            /* check that position sensor type is SSI encoder. */
            checkPositionSensorType(0x0204);
        }

        int ssiPosition = readSSIPosition();
        if (Math.abs(ssiPosition) <= 1) {
            /* there is a bug in the controller in reading SSI position */
            /* switch back to Incremental Encoder */
            this.raiseWarning(FcsAlert.HARDWARE_ERROR, " after setPositionSensorTypeEncoderSSI, ssiPosition = " + ssiPosition
                    + " we go back to TypeSinusIncrementalEncoder", getName());
            doSetSinusIncrementalEncoder();
            /* and then back again to Encoder SSI */
            doSetEncoderSSI();
        }
    }

    /**
     * Set PositionSensorType to Sinus Incremental Encoder(value=8)
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "GoToSwitchOnDisabled "
            + "and set PositionSensorType to Sinus Incremental Encoder : 8")
    default void setPositionSensorTypeSinusIncrementalEncoder() {
        FCSLOG.info(getName() + " setting PositionSensorType Sinus Incremental Encoder");
        goToSwitchOnDisabled();
        doSetSinusIncrementalEncoder();
    }

    default void doSetSinusIncrementalEncoder() {
        int controllerStructure = readControllerStructure();
        if (controllerStructure == 0) {
            writeParameter(Parameter.PositionSensorType, 8);
            /* check that position sensor type is sinus incremental encoder. */
            checkPositionSensorType(8);
        } else {
            writeParameter(Parameter.PositionSensorType, 0x0208);
            /* check that position sensor type is sinus incremental encoder. */
            checkPositionSensorType(0x0208);
        }

    }

    default void checkPositionSensorType(int sensortype) {
        long timeoutMillis = 500;
        final long timeStart = System.currentTimeMillis();
        long duration = 0;
        boolean ok = false;
        while (!ok && duration <= timeoutMillis) {
            duration = System.currentTimeMillis() - timeStart;
            FcsUtils.sleep(2, getName());
            ok = (sensortype == readParameter(Parameter.PositionSensorType));
        }
        if (!ok) {
            String msg = getName() + " couldn't change position sensor type during time allocated of " + timeoutMillis
                    + " ms";
            FCSLOG.severe(msg);
            throw new FcsHardwareException(msg);
        }
        setPositionSensorType(sensortype);
    }

    /**
     *
     * @param position
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Define the actual position as position given as argument.")
    @Override
    default void defineAbsolutePosition(int position) {
        FCSLOG.finer(() -> getName() + " Defining Absolute Position:" + position);
        this.changeMode(EposMode.HOMING);

        // TODO REVIEW CAROUSEL Added at SLAC july 21 for tests. Should be removed eventually or be incorporated to a safer method
        if (EposMode.HOMING != readMode()) {
            // second chance
            FcsUtils.sleep(500, "Controller change mode");
            this.changeMode(EposMode.HOMING);
        }
        checkEposMode(EposMode.HOMING);
        this.writeParameter(Parameter.HomePosition, position);
        this.writeParameter(Parameter.HomingMethod, 35);
        this.goToOperationEnable();
        /* start homing */
        this.writeControlWord(ControlWord.ABSOLUTE_POSITION);
        checkHomingDone();
    }

    /**
     * Activate POWER SAVE in order to permit shutter to be enabled.
     */
    default void activatePowerSave() {
        int val = (int) readParameter(Parameter.DigitalOutputFonctionnalityState);
        writeParameter(Parameter.DigitalOutputFonctionnalityState, FcsUtils.force2one(val, 13));
        publishData();
        FCSLOG.info(getName() + ": POWER SAVE activated.");
    }

    /**
     * Deactivate POWER SAVE to be able to rotate carousel.
     */
    default void deactivatePowerSave() {
        int val = (int) readParameter(Parameter.DigitalOutputFonctionnalityState);
        writeParameter(Parameter.DigitalOutputFonctionnalityState, FcsUtils.force2zero(val, 13));
        publishData();
        FCSLOG.info(getName() + ": POWER SAVE deactivate command executed.");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read bit holding brake into DigitalOutputFonctionnalityState. For carousel. index:2078 subindex:1 bit:2")
    default boolean readHoldingBrakes() {
        int digitalOutput = (int) readParameter(Parameter.DigitalOutputFonctionnalityState);
        boolean holdingBrakes = ((digitalOutput >> 2) & 1) == 1;
        setHoldingBrakes(holdingBrakes);
        return holdingBrakes;
    }
}
