package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterFamily;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for an optical filter for the camera.
 *
 */
public class Filter {

    @LookupName
    private String name;

    @ConfigurationParameter(description = "Filter identification number as coded on its frame and read with 6 hall effect sensors", range = "0..63", units = "unitless", category = "filter")
    private volatile int filterID;

    @ConfigurationParameter(description = "Weight of the filter", range = "0..100", units = "kg", category = "filter")
    private volatile double weight;

    @ConfigurationParameter(description = "Filter family: Science | Dummy | French dummy | Test | Open | Unknown | None", units = "unitless", category = "filter")
    private volatile FilterFamily family;

    public Filter() {};
    /**
     * Build a filter with an ID and a weight and a family.
     *
     * @param filterID
     * @param weight
     * @param family
     */
    Filter(int filterID, double weight, FilterFamily family) {
        this.filterID = filterID;
        this.weight = weight;
        this.family = family;
    }

    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Filter identification number")
    public int getFilterID() {
        return filterID;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Filter weight")
    public double getWeight() {
        return weight;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Filter family")
    public FilterFamily getFamily() {
        return family;
    }

    /**
     * Filter name as accepted by the OCS Bridge
     *
     * @return filter name formatted as <name>_<filterID>
     */
    public String getObservatoryName() {
        if ( filterID == 0 ) {
            return "NONE";
        } else {
            return String.format("%s_%d", name, filterID);
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Filter: " + FcsUtils.padLeft(name, 4) + " | ");
        sb.append("ID: " + FcsUtils.padLeft(Integer.toString(filterID), 2) + " | ");
        sb.append("Weight: " + FcsUtils.padLeft(Double.toString(weight) + "kg", 6) + " | ");
        sb.append("Family: " + family.getFamilyName());
        return sb.toString();
    }

}
