
package org.lsst.ccs.subsystems.fcs.simulation;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.ComplementarySensors;
import org.lsst.ccs.subsystems.fcs.DigitalSensor;
import static org.lsst.ccs.subsystems.fcs.FCSCst.LOADER_CLAMP_NAME;
import org.lsst.ccs.subsystems.fcs.ForceSensor;
import org.lsst.ccs.subsystems.fcs.LoaderCarrier;
import org.lsst.ccs.subsystems.fcs.LoaderClamp;
import org.lsst.ccs.subsystems.fcs.RedondantSensors;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;
import static org.lsst.ccs.subsystems.fcs.simulation.SimuLoaderClampController.POSITION_OPENED;

/**
 * This class is for the simulation of a pluto gateway for the loader.
 * @author virieux
 */
public class SimuLoaderPlutoGateway extends SimuPlutoGateway {

    @LookupField(strategy = TREE, pathFilter = ".*\\/carrier")
    private LoaderCarrier carrier;
    @LookupField(strategy = TREE, pathFilter = ".*\\/" + LOADER_CLAMP_NAME)
    private LoaderClamp clamp;

    @LookupField(strategy = TREE, pathFilter = ".*\\/carrierController")
    private SimuLoaderCarrierController carrierController;

    @LookupField(strategy = TREE, pathFilter = ".*\\/hooksController")
    private SimuLoaderClampController hooksController;

    private DigitalSensor closeSensor1;
    private DigitalSensor openSensor1;
    private DigitalSensor closeSensor2;
    private DigitalSensor openSensor2;
    private DigitalSensor closeSensor3;
    private DigitalSensor openSensor3;
    private DigitalSensor closeSensor4;
    private DigitalSensor openSensor4;

    private DigitalSensor unclampedStatusSensor;
    private DigitalSensor underClampedStatusSensor;
    private DigitalSensor clampedStatusSensor;
    private DigitalSensor overClampedStatusSensor;
    private ForceSensor forceSensor0;
    private ForceSensor forceSensor1;

    private RedondantSensors filterPresenceSensors;
    private RedondantSensors loaderOnCameraSensors;
    private BinarySensor storagePositionSensors;
    private BinarySensor handoffPositionSensors;
    private BinarySensor engagedPositionSensors;

    /**
     * output signals from loaderPlutoGateway (read on loaderPlutoGateway)
     */
    @LookupField(strategy = TREE, pathFilter = ".*\\/Loader_LRH")
    private ComplementarySensors loader_LRH;

    @LookupField(strategy = TREE, pathFilter = ".*\\/Loader_LPS")
    private ComplementarySensors loader_LPS;

    @Override
    public void init() {
        super.init();
        ComponentLookup lookup = subs.getComponentLookup();
        this.closeSensor1 = (DigitalSensor) lookup.getComponentByPath("loaderCloseSensor1");
        this.closeSensor2 = (DigitalSensor) lookup.getComponentByPath("loaderCloseSensor2");
        this.closeSensor3 = (DigitalSensor) lookup.getComponentByPath("loaderCloseSensor3");
        this.closeSensor4 = (DigitalSensor) lookup.getComponentByPath("loaderCloseSensor4");
        this.openSensor1 = (DigitalSensor) lookup.getComponentByPath("loaderOpenSensor1");
        this.openSensor2 = (DigitalSensor) lookup.getComponentByPath("loaderOpenSensor2");
        this.openSensor3 = (DigitalSensor) lookup.getComponentByPath("loaderOpenSensor3");
        this.openSensor4 = (DigitalSensor) lookup.getComponentByPath("loaderOpenSensor4");
        this.clampedStatusSensor = (DigitalSensor) lookup.getComponentByPath("clampedStatusSensor");
        this.unclampedStatusSensor = (DigitalSensor) lookup.getComponentByPath("unclampedStatusSensor");
        this.underClampedStatusSensor = (DigitalSensor) lookup.getComponentByPath("underClampedStatusSensor");
        this.overClampedStatusSensor = (DigitalSensor) lookup.getComponentByPath("overClampedStatusSensor");
        this.forceSensor0 = (ForceSensor) lookup.getComponentByPath("forceSensor0");
        this.forceSensor1 = (ForceSensor) lookup.getComponentByPath("forceSensor1");

        this.loaderOnCameraSensors = (RedondantSensors) lookup.getComponentByPath("loaderOnCameraSensors");
        this.filterPresenceSensors = (RedondantSensors) lookup.getComponentByPath("loaderFilterPresenceSensors");
        this.storagePositionSensors = (RedondantSensors) lookup.getComponentByPath("loaderStorageSensors");
        this.handoffPositionSensors = (BinarySensor) lookup.getComponentByPath("loaderHandoffSensors");
        this.engagedPositionSensors = (BinarySensor) lookup.getComponentByPath("loaderEngagedSensors");

        this.loader_LRH = (ComplementarySensors) lookup.getComponentByPath("Loader_LRH");
        this.loader_LPS = (ComplementarySensors) lookup.getComponentByPath("Loader_LPS");
    }

    protected void simuBinarySensorOn(BinarySensor sensor) {
        if (sensor instanceof RedondantSensors) {
            replaceSensorValue(((RedondantSensors) sensor).getSensor(), true);
            replaceSensorValue(((RedondantSensors) sensor).getSensorR(), true);
        } else if (sensor instanceof ComplementarySensors) {
            replaceSensorValue(((ComplementarySensors) sensor).getSensor(), true);
            replaceSensorValue(((ComplementarySensors) sensor).getSensorC(), false);

        } else {
            replaceSensorValue((DigitalSensor) sensor, true);
        }
    }

    protected void simuBinarySensorOff(BinarySensor sensor) {
        if (sensor instanceof RedondantSensors) {
            replaceSensorValue(((RedondantSensors) sensor).getSensor(), false);
            replaceSensorValue(((RedondantSensors) sensor).getSensorR(), false);
        } else if (sensor instanceof ComplementarySensors) {
            replaceSensorValue(((ComplementarySensors) sensor).getSensor(), false);
            replaceSensorValue(((ComplementarySensors) sensor).getSensorC(), true);
        } else {
            replaceSensorValue((DigitalSensor) sensor, false);
        }
    }

    protected void simuRedondantSensorsInError(RedondantSensors sensors) {
        replaceSensorValue(sensors.getSensor(), true);
        replaceSensorValue(sensors.getSensorR(), false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderClampIsClosed() {
        simulateHooksAreClosed();
        simulateUnclampedStatusSensorIsOn();
        hooksController.setPosition(clamp.getRelativePositionToClose());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderClampIsClamped() {
        simulateHooksAreClosed();
        simulateClampedStatusSensorIsOn();
        hooksController.setPosition(SimuLoaderClampController.POSITION_CLAMPED);
    }

    private void simulateHooksAreClosed() {
        simulateHookIsClosed(closeSensor1, openSensor1);
        simulateHookIsClosed(closeSensor2, openSensor2);
        simulateHookIsClosed(closeSensor3, openSensor3);
        simulateHookIsClosed(closeSensor4, openSensor4);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderClampIsOpened() {
        simulateHookIsOpened(closeSensor1, openSensor1);
        simulateHookIsOpened(closeSensor2, openSensor2);
        simulateHookIsOpened(closeSensor3, openSensor3);
        simulateHookIsOpened(closeSensor4, openSensor4);
        simulateUnclampedStatusSensorIsOn();
        hooksController.setPosition(POSITION_OPENED);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderClampIsInTravel() {
        simulateHookIsInTravel(closeSensor1, openSensor1);
        simulateHookIsInTravel(closeSensor2, openSensor2);
        simulateHookIsInTravel(closeSensor3, openSensor3);
        simulateHookIsInTravel(closeSensor4, openSensor4);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateClampIsUnknown() {
        simulateHookIsOpened(closeSensor1, openSensor1);
        simulateHookIsOpened(closeSensor2, openSensor2);
        simulateHookIsOpened(closeSensor3, openSensor3);
        simulateHookIsClosed(closeSensor4, openSensor4);
    }

    /**
     * simulate that clamp is in error : if one hook is in error, clamp is in error.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateClampIsInError() {
        simulateHookIsInError(closeSensor1, openSensor1);
        simulateHookIsOpened(closeSensor2, openSensor2);
        simulateHookIsOpened(closeSensor3, openSensor3);
        simulateHookIsClosed(closeSensor4, openSensor4);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateHookIsClosed(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceSensorValue(closeSensor, true);
        replaceSensorValue(openSensor, false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateHookIsOpened(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceSensorValue(closeSensor, false);
        replaceSensorValue(openSensor, true);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateHookIsInError(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceSensorValue(closeSensor, true);
        replaceSensorValue(openSensor, true);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateHookIsInTravel(DigitalSensor closeSensor, DigitalSensor openSensor) {
        replaceSensorValue(closeSensor, false);
        replaceSensorValue(openSensor, false);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderIsConnectedOnCamera() {
        simuBinarySensorOn(loaderOnCameraSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderIsNotConnectedOnCamera() {
        simuBinarySensorOff(loaderOnCameraSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateFilterIsOnLoader() {
        simuBinarySensorOn(filterPresenceSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateFilterIsNotOnLoader() {
        simuBinarySensorOff(filterPresenceSensors);
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderCarrierIsAtStorage() {
        simuBinarySensorOn(storagePositionSensors);
        simuBinarySensorOff(handoffPositionSensors);
        simuBinarySensorOff(engagedPositionSensors);
        simuBinarySensorOn(loader_LPS);
        simuBinarySensorOff(loader_LRH);
        carrierController.setPosition(carrier.getStoragePosition());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderCarrierIsAtHandoff() {
        simuBinarySensorOff(storagePositionSensors);
        simuBinarySensorOn(handoffPositionSensors);
        simuBinarySensorOn(engagedPositionSensors);
        simuBinarySensorOff(loader_LPS);
        carrierController.setPosition(carrier.getHandoffPosition());
        carrier.updatePosition();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderCarrierIsAtEngaged() {
        simuBinarySensorOff(storagePositionSensors);
        simuBinarySensorOff(handoffPositionSensors);
        simuBinarySensorOn(engagedPositionSensors);
        carrierController.setPosition(carrier.getEngagedPosition());
        carrier.updatePosition();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateLoaderCarrierIsInTravel() {
        simuBinarySensorOff(storagePositionSensors);
        simuBinarySensorOff(handoffPositionSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateHandoffSensorsAreInError() {
        simuRedondantSensorsInError((RedondantSensors) handoffPositionSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateStorageSensorsAreInError() {
        simuRedondantSensorsInError((RedondantSensors) storagePositionSensors);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateClampedStatusSensorIsOn() {
        replaceSensorValue(clampedStatusSensor, true);
        replaceSensorValue(unclampedStatusSensor, false);
        replaceSensorValue(underClampedStatusSensor,false);
        replaceSensorValue(overClampedStatusSensor,false);
        simulateForceSensorsOn();
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateUnclampedStatusSensorIsOn() {
        replaceSensorValue(unclampedStatusSensor, true);
        replaceSensorValue(clampedStatusSensor, false);
        replaceSensorValue(underClampedStatusSensor,false);
        replaceSensorValue(overClampedStatusSensor,false);
    }


    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateForceSensorsOn() {
        simulateForceSensorClamped(forceSensor0);
        simulateForceSensorClamped(forceSensor1);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE)
    public void simulateForceSensorsOff() {
        simulateForceSensorUnclamped(forceSensor0);
        simulateForceSensorUnclamped(forceSensor1);
    }

    public void simulateForceSensorClamped(ForceSensor forceSensor) {
        this.updateForceSensor(forceSensor, 30);
    }

    public void simulateForceSensorUnclamped(ForceSensor forceSensor) {
        this.updateForceSensor(forceSensor, 10);
    }

    private void updateForceSensor(ForceSensor forceSensor, int voltage) {
        this.readValues[forceSensor.getByteNumero()] = voltage;
    }

    @Override
    public void postStart() {
        //Filter is closed && clamped and carrier is at storage position
        simulateLoaderClampIsClosed();
        simulateLoaderIsConnectedOnCamera();
        simulateClampedStatusSensorIsOn();
        simulateFilterIsOnLoader();
        simulateLoaderCarrierIsAtStorage();
    }
}
