package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;

import java.lang.Math;
import java.time.Duration;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.Predicate;
import java.util.logging.Logger;
import java.util.stream.Collectors;

import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.HasDataProviderInfos;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutochangerInclinationState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.CarouselPowerMode;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.CarouselPowerState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.GeneralAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.McmFilterState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.TelescopeMountRotatorState;
import org.lsst.ccs.subsystems.fcs.common.BridgeToLoader;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.ActionGuard;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;
import org.lsst.ccs.subsystem.ocsbridge.states.CameraMotionState;

/**
 * The main module in FCS. Commands can be sent to the FcsMain from the CCS
 * Master Control Module (MCM) in normal operation mode or from a Console.
 *
 * <P/>
 * Example of commands that can be sent to the FcsMain : <BR/>
 * <TT>
 * Console> invoke fcs-test MoveFilterToOnline <i>filtername</i>
 * </TT>
 * <P/>
 * The main goals of the FcsMain is :
 * <UL>
 * <LI>Receive commands from MCM or ccs-console for the whole subsystem.</LI>
 * <LI>Execute commands received from MCM or ccs-console.</LI>
 * <LI>Handle the logic of the whole subsystem.</LI>
 * </UL>
 * <LI>Publish on the status bus the status of the whole subsystem.</LI>
 *
 * @author virieux
 *
 */
public class FcsMain extends MainModule implements HasDataProviderInfos {

    /**
     *
     */
    @SuppressWarnings("unused")
    private static final long serialVersionUID = 7669526660659959402L;
    private static final Logger FCSLOG = Logger.getLogger(FcsMain.class.getName());

    private static final Duration SETFILTER_MAX_DURATION = Duration.ofSeconds(90);
    private static final Duration LOAD_UNLOAD_FILTER_MAX_DURATION = Duration.ofSeconds(480);

    private static final String PUBLICATION_KEY_FOR_MCM = "fcs/mcm";

    @LookupField(strategy = LookupField.Strategy.TREE)
    private BridgeToLoader bridgeToLoader;

    @LookupField(strategy = CHILDREN)
    private Carousel carousel;

    @LookupField(strategy = CHILDREN)
    private Autochanger autochanger;

    @LookupField(strategy = CHILDREN)
    private Loader loader;

    @LookupField(strategy = CHILDREN)
    private FilterManager filterManager;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    // Initialize the state of the software lock between TMA / rotator and FCS
    private volatile TelescopeMountRotatorState tmaAndRotatorMotionStatus = TelescopeMountRotatorState.UNKNOWN;

    // Predicate used to check the presence of the OCS-Bridge on the buses
    private final Predicate<AgentInfo> ocsBridgePredicate = (ai) ->  {
            return ai.getAgentProperty("agentType").equals(AgentType.OCS_BRIDGE.name());
    };

    /**
     * Average duration of commands issued by the observatory
     * These are constants because they are not supposed to change much
     * TODO: find a better way of handling these values and verify the duplication of use with SETFILTER_MAX_DURATION above used as a timeout
     */
    private int carouselWakeUpDuration = 20;
    private int setFilterFastDuration = 95;
    private int setFilterSlowDuration = 120;

    /* percentage of command setFilter already done */
    private int percentage = 0;

    /* previous socket ID of the filter which is on the autochanger. */
    @Persist
    private volatile int previousSocketID = -1;

    private boolean firstMCMpublication = true;

    private Map<GeneralAction, PersistentCounter> movementCounter;

    /* Store last time a function was executed to check for a minimal interval between execution */
    private AtomicLong lastUpdateStateWithSensors = new AtomicLong(0);

    @Override
    public void build() {
        if (bridgeToLoader == null) {
            throw new RuntimeException("The FcsMain expected a bridge to the Loader, but none was found.");
        }
        if (bridgeToLoader.equals(bridge)) {
            throw new RuntimeException(
                    "In FcsMain both bridges are the same instance. We expected two different bridges.");
        }
        super.build();
        //Registering the KeyValueData published below

        KeyValueData data = getDataForMcm();
        dataProviderDictionaryService.registerData(data);
        String mainPath = data.getKey();
        KeyValueDataList dataList = (KeyValueDataList) data.getValue();
        for ( KeyValueData d : dataList ) {
            String path = mainPath + "/" + d.getKey();
            DataProviderInfo info = dataProviderDictionaryService.getDataProviderDictionary().getDataProviderInfoForPath(path);
            String units = "unitless";
            String description = "";
            switch (d.getKey()) {
                case "filter_on_autochanger":
                    description = "OCS name of the filter currently on the autochanger";
                    break;
                case "filter_previous_socketID":
                    description = "ID of the socket on which the current filter used to sit";
                    break;
                case "autochanger_trucks_position":
                    description = "Absolute position of the autochanger trucks";
                    units = "micron";
                    break;
                case "autochanger_trucks_state":
                    description = "Known autochanger position (STANDBY, HANDOFF, ONLINE) or IN_TRAVEL";
                    break;
                case "setFilter_percentage":
                    description = "Progress of the setFilter command [%]";
                    break;
                case "proximity":
                    description = "Value of the autochanger proximity sensor";
                    break;
                default:
                    units = "unitless";
                    description = "";
            }
            info.addAttribute(DataProviderInfo.Attribute.UNITS, units);
            info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, description);
        }
        agentStateService.updateAgentState(McmFilterState.UNKNOWN);
    }

    @Override
    public void init() {
        super.init();
        /*
         * define a role for my subsystem in order to make FcsGUI listen to my subsystem
         */
        subs.getAgentService(AgentPropertiesService.class).setAgentProperty("org.lsst.ccs.subsystem.fcs.wholefcs",
                "fcs");

        // register counters and durations for high level actions
        movementCounter = new EnumMap<>(GeneralAction.class);
        for (GeneralAction action : new GeneralAction[]{
            GeneralAction.SET_FILTER,
            GeneralAction.SET_FILTER_AT_HANDOFF_FOR_LOADER,
            GeneralAction.STORE_FILTER_ON_CAROUSEL,
            GeneralAction.SET_NO_FILTER,
            GeneralAction.DISENGAGE_FILTER_FROM_CAROUSEL,
            GeneralAction.LOAD_FILTER,
            GeneralAction.UNLOAD_FILTER,
            GeneralAction.CONNECT_LOADER,
            GeneralAction.DISCONNECT_LOADER}) {
            movementCounter.put(action, PersistentCounter.newCounter(action.getCounterPath(), subs, action.name()));
            registerActionDuration(action.name());
        }

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(FcsAlert.HARDWARE_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlert.CA_LOCKING_ISSUE.getAlert(name), alwaysClear);
        alertService.registerAlert(FcsAlert.HARDWARE_ERROR.getAlert("carousel"), alwaysClear);
    }

    private void updateTmaAndRotatorStatus(TelescopeMountRotatorState tmaRotatorState) {
        TelescopeMountRotatorState currentMotionStatus = TelescopeMountRotatorState.valueOf(tmaAndRotatorMotionStatus.name());
        tmaAndRotatorMotionStatus = tmaRotatorState;
        if ( !currentMotionStatus.equals(tmaAndRotatorMotionStatus) ) {
            FCSLOG.info(name
                    + String.format(
                            " Registering a Camera Motion State (software lock) change from %s to %s",
                            currentMotionStatus.name(),
                            tmaAndRotatorMotionStatus.name()
                    )
            );
        }
    }
    private final StatusMessageListener cameraMotionStatusMessageListener = new StatusMessageListener() {

        private final AtomicBoolean isFirstStatusStateChangeNotification = new AtomicBoolean(true);

        @SuppressWarnings("rawtypes")
        @Override
        public void onStatusMessage(StatusMessage msg) {
            StateBundle stateToProcess = null;
            if (isFirstStatusStateChangeNotification.getAndSet(false)) {
                stateToProcess = msg.getState();
            } else if (msg instanceof StatusStateChangeNotification) {
                StatusStateChangeNotification stateChange = (StatusStateChangeNotification) msg;
                stateToProcess = stateChange.getNewState().diffState(stateChange.getOldState());
            }
            if (stateToProcess != null) {
                CameraMotionState cameraMotionState = stateToProcess.getState(CameraMotionState.class);
                if (cameraMotionState != null) {
                    FCSLOG.finer("Camera Motion State received from OCS-Bridge = " + cameraMotionState.name());
                    switch (cameraMotionState) {
                        case LOCKED:
                            updateTmaAndRotatorStatus(TelescopeMountRotatorState.LOCKED);
                            break;
                        case UNLOCKED:
                            updateTmaAndRotatorStatus(TelescopeMountRotatorState.UNLOCKED);
                            break;
                        default:
                            FCSLOG.warning(name + "Unknown Camera Motion (software lock) state: " + cameraMotionState.name());
                            updateTmaAndRotatorStatus(TelescopeMountRotatorState.UNKNOWN);
                    }
                }
            }
        }
    };

    @Override
    public void postInit() {

        getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(new AgentPresenceListener() {
            @Override
            public void connecting(AgentInfo... agents) {
                for (AgentInfo agent : agents) {
                    if (ocsBridgePredicate.test(agent)) {
                        getMessagingAccess().addStatusMessageListener(
                                cameraMotionStatusMessageListener,
                                (m) -> {
                                    return ocsBridgePredicate.test(m.getOriginAgentInfo());
                                }
                        );
                        break;
                    }
                }
            }

            @Override
            public void disconnecting(AgentInfo agent) {
                if ( ocsBridgePredicate.test(agent) ) {
                    getMessagingAccess().removeStatusMessageListener(cameraMotionStatusMessageListener);
                    updateTmaAndRotatorStatus(TelescopeMountRotatorState.UNKNOWN);
                }
            }
        });
    }

    @Override
    public void postStart() {
        // Request the OCS-Bridge to publish its states when FCS joins the buses
        for (AgentInfo agent : getMessagingAccess().getAgentPresenceManager().listConnectedAgents()) {
            if (ocsBridgePredicate.test(agent)) {
                ConcurrentMessagingUtils messagingUtils = new ConcurrentMessagingUtils(getMessagingAccess());
                CommandRequest cmd = new CommandRequest(agent.getName(), "publishState");
                FCSLOG.info("Requesting the OCS-Bridge to publish its states");
                messagingUtils.sendAsynchronousCommand(cmd);
            }
            break;
        }

        if ( FcsUtils.isProto() ) {
            // If we are running on prototype (in Paris) then bypass the software lock on telescope
            updateTmaAndRotatorStatus(TelescopeMountRotatorState.PROTOTYPE);
        }
    }

    @Override
    public void shutdown() {
        // Stop listening to the buses
        getMessagingAccess().removeStatusMessageListener(cameraMotionStatusMessageListener);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Print the current software lock status on the telescope and rotator motion")
    public String getTelescopeSoftwareLockStatus() {
        return tmaAndRotatorMotionStatus.name();
    }

    private void registerActionDuration(String actionName) {
        String path = getDurationPath(actionName);
        dataProviderDictionaryService.registerData(new KeyValueData(path, 0));
        DataProviderInfo info = dataProviderDictionaryService.getDataProviderDictionary().getDataProviderInfoForPath(path);
        info.addAttribute(DataProviderInfo.Attribute.UNITS, "millisecond");
        info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "Duration of " + actionName);
        info.addAttribute(DataProviderInfo.Attribute.TYPE, "long");
    }

    private String getDurationPath(String actionName) {
        return "duration/" + actionName;
    }

    @Override
    public void updateFCSStateToReady() {
        if (carousel.isInitialized() && autochanger.isInitialized() && loader.isInitialized()) {
            /* The initialization has been done, so now the hardware is ready */
            updateAgentState(FilterState.READY);
            updateAgentState(FilterReadinessState.READY);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Sets the filters state to READY even if some sensors are still missing in autochanger.")
    public void forceFilterReadinessStateToReady() {
        // because of some sensors still missing, autochanger is never initialized
        // but we need the Filter to be ready for filter exchanges
        // TODO GLOBAL REVIEW should be removed  - was still used in Nov 2023 !!
        updateAgentState(FilterState.READY);
        updateAgentState(FilterReadinessState.READY);
    }

    /**
     * Return true if the changer is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the changer is connected.")
    public boolean isChangerConnected() {
        return this.bridge.isReady();
    }

    /**
     * Return true if the loader is connected.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected.")
    public boolean isLoaderConnected() {
        return autochanger.isLoaderConnected();
    }

    /**
     * Return true if the hardware of the changer is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the changer is ready.")
    public boolean isChangerReady() {
        return this.bridge.allDevicesBooted();
    }

    /**
     * Return true if the hardware of the loader is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the hardware of the loader is ready.")
    public boolean isLoaderReady() {
        return this.bridgeToLoader.allDevicesBooted();
    }

    /**
     * Disconnect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Disconnect the loader hardware.")
    public void disconnectLoaderCANbus() {
        movementCounter.get(GeneralAction.DISCONNECT_LOADER).increment();
        long beginTime = System.currentTimeMillis();
        loader.disconnectLoaderCANbus();
        publishActionDuration(GeneralAction.DISCONNECT_LOADER, System.currentTimeMillis() - beginTime);
        /* after loader disconnection update state for GUI*/
        updateStateWithSensors();
    }

    /**
     * Connect the loader hardware.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Connect the loader hardware.")
    public void connectLoaderCANbus() {
        movementCounter.get(GeneralAction.CONNECT_LOADER).increment();
        long beginTime = System.currentTimeMillis();
        loader.connectLoaderCANbus();
        publishActionDuration(GeneralAction.CONNECT_LOADER, System.currentTimeMillis() - beginTime);
        /* after loader connection update state for GUI*/
        updateStateWithSensors();
    }

    /**
     * For Whole FCS GUI. Has to be overridden in FcsMain.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the list of LOADER CANopen hardware that this subsystem manages.")
    @Override
    public List<String> listLoaderHardwareNames() {
        return this.bridgeToLoader.listHardwareNames();
    }

    /**
     * For whole FCS GUI.
     * Override listLoSensorsNames of MainModule
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the list of names of sensors plugged on Loader pluto gateway.")
    @Override
    public List<String> listLoSensorsNames() {
        return bridgeToLoader.listLoSensorsNames();
    }

    public boolean isFilterInCamera(int filterID) {
        return carousel.isFilterOnCarousel(filterID) || autochanger.isFilterOnAC(filterID);
    }

    public boolean isFilterAvailable(int filterID) {
        return getAvailableFilterMap().entrySet().stream().anyMatch(e -> e.getKey() == filterID);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Store the filter on a given socket of the carousel and move the empty autochanger to HANDOFF position.", autoAck = false, timeout = 60000)
    public void storeFilterOnSocket(String socketName) {
        storeFilterOnSocket(socketName, true);
    }

    private void storeFilterOnSocket(String socketName, boolean withPrecision) {
        try (AutoTimed at = new AutoTimed("storeFilterOnSocket")) {
            updateStateWithSensors();
            if ( withPrecision ) {
                FCSLOG.info(name + " === About to store filter on carousel and to handoff for loader ===");
            } else {
                FCSLOG.info(name + " === About to store filter on carousel ===");
            }
            // increment the action in the database
            movementCounter.get(GeneralAction.STORE_FILTER_ON_CAROUSEL).increment();
            final long beginTime = System.currentTimeMillis();
            if (!FilterReadinessState.READY.equals(this.getFilterReadinessState())) {
                throw new RejectedCommandException(name + " FilterReadinessState must be READY");
            }

            if ( !autochanger.isAvailable() ) {
                throw new RejectedCommandException(name +" the autochanger is currently unavailable");
            }

            CarouselSocket desiredSocket = carousel.getSocketByName(socketName);

            if (!desiredSocket.isAvailable()) {
                throw new RejectedCommandException(name + " " + socketName + " is unavailable");
            }

            if ( desiredSocket != carousel.getSocketAtStandby() && !carousel.isAvailable() ) {
                throw new RejectedCommandException(name + " the carousel rotation is currently unavailable");
            }

            if (!autochanger.isAtStandby()) {
                // carousel must be empty if the autochanger has a filter
                if (autochanger.isHoldingFilter() && !desiredSocket.isEmpty()) {
                    throw new RejectedCommandException(name + " the target carousel socket is not empty");
                }
            }

            FCSLOG.info(name + " === About to store filter on carousel ===");

            // Make sure the autochanger is free to move
            if (autochanger.isAtOnline()) {
                AutochangerThreeOnlineClamps onlineClamps = autochanger.getOnlineClamps();
                if (onlineClamps.isLocked()) {
                    onlineClamps.unlockClamps();
                }
                if (onlineClamps.isClosed()) {
                    if (onlineClamps.isHomingDone()) {
                        onlineClamps.openClampsNoXBrakes();
                    } else {
                        onlineClamps.homingFusion();
                    }
                }
                updateStateWithSensors();

                if (onlineClamps.isOpened()) {
                    FCSLOG.info(name + " === autochanger is free to move ===");
                } else {
                    throw new FcsHardwareException(name + " autochanger online clamps should be opened");
                }
            }

            carousel.rotateSocketToStandby(socketName);
            carousel.checkDeltaPosition();
            FCSLOG.info(name + " === carousel is ready to receive a filter at standby ===");

            if (!autochanger.isEmpty()) {
                autochanger.moveFilterToStandbyPlusDelta(carousel.getSocketAtStandby().getDeltaAutochangerStandbyPosition());
                /* because both subsystems sensors have changed we have to update again */
                updateStateWithSensors();
            }

            if (carousel.isHoldingFilterAtStandby()) {
                FCSLOG.info("=== carousel is CLAMPED_ON_FILTER === ");
                carousel.getSocketAtStandby().updateFilterID();
                FCSLOG.info("filter ID on CAROUSEL at STANDBY " + carousel.getSocketAtStandby().getFilterID());
                FCSLOG.info("filter ID on autochanger = " + autochanger.getFilterID());
                FCSLOG.info("==================================");

            } else {
                recoveryLockingProcess();
                if (carousel.isHoldingFilterAtStandby()) {
                    carousel.getSocketAtStandby().updateFilterID();
                } else {
                    String msg = name + ": carousel should be CLAMPED_ON_FILTER when autochanger is "
                            + "at STANDBY with a filter; recovery process didn't work";
                    this.raiseAlarm(FcsAlert.CA_LOCKING_ISSUE, msg, name);
                    throw new FcsHardwareException(msg);
                }
            }
            if (autochanger.isAtStandby()) {
                // wait update local protection system
                autochanger.waitForProtectionSystemUpdate();
                FCSLOG.info(name + ": is going to moveEmptyFromStandbyToHandoff");
                autochanger.moveEmptyFromStandbyToHandoff(withPrecision);
            } else {
                autochanger.goToHandOff();
            }
            publishActionDuration(GeneralAction.STORE_FILTER_ON_CAROUSEL, System.currentTimeMillis() - beginTime);

        }
    }

    private void storeFilterOnCarouselRelaxed() {
        storeFilterOnCarousel(false);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Store the filter in the carousel and move the empty autochanger to HANDOFF position for loader."
            + " Initial State for autochanger: a filter at HANDOFF or ONLINE. Final State for autochanger : empty at HANDOFF for loader.", autoAck = false, timeout = 60000)
    public void storeFilterOnCarousel() {
        storeFilterOnCarousel(true);
    }

    private void storeFilterOnCarousel(boolean withPrecision) {
        try (AutoTimed at = new AutoTimed("storeFilterOnCarousel")) {
            updateStateWithSensors();

            if ( withPrecision ) {
                FCSLOG.info(name + " === About to store filter on carousel and go to handoff for loader===");
            } else {
                FCSLOG.info(name + " === About to store filter on carousel and go to handoff ===");
            }
            final long beginTime = System.currentTimeMillis();
            movementCounter.get(GeneralAction.STORE_FILTER_ON_CAROUSEL).increment();
            if (!FilterReadinessState.READY.equals(this.getFilterReadinessState())) {
                throw new RejectedCommandException(name + " FilterReadinessState must be READY");
            }

            // carousel must be empty and available at STANDBY
            if (!carousel.isReadyToGrabAFilterAtStandby()) {
                throw new RejectedCommandException(name + " carousel should be at standby with an empty and available socket");
            }
            // autochanger should be at HANDOFF or ONLINE
            if (!(autochanger.isAtHandoff() || autochanger.isAtOnline())) {
                throw new RejectedCommandException(name + " autochanger must be at HANDOFF or at ONLINE");
            }

            // autochanger should be available
            if ( !autochanger.isAvailable() ) {
                throw new RejectedCommandException(name + " autochanger must be available");
            }

            if (!autochanger.isHoldingFilter()) {
                throw new RejectedCommandException(name + " autochanger must hold the filter");
            }

            // Make sure the carousel is aligned
            // change made on July 18 to clean the sequence of operations Carousel first then Autochanger
            carousel.checkDeltaPosition();

            // Make sure the autochanger is free to move
            if (autochanger.isAtOnline()) {
                if (autochanger.getOnlineClamps().isLocked()) {
                    autochanger.getOnlineClamps().unlockClamps();
                }
                if (autochanger.getOnlineClamps().isClosed()) {
                    if (autochanger.getOnlineClamps().isHomingDone()) {
                        autochanger.getOnlineClamps().openClampsNoXBrakes();
                    } else {
                        autochanger.getOnlineClamps().homingFusion();
                    }
                }
            }
            updateStateWithSensors();

            if (autochanger.getOnlineClamps().isOpened()) {
                FCSLOG.info(name + " === autochanger is free to move ===");
            } else {
                throw new FcsHardwareException(name + " autochanger online clamps should be opened");
            }
            FCSLOG.info(name + " === carousel is ready to receive a filter at standby ===");
            autochanger.moveFilterToStandbyPlusDelta(carousel.getSocketAtStandby().getDeltaAutochangerStandbyPosition());
            /* because both subsystems sensors have changed we have to update again */
            updateStateWithSensors();

            if (carousel.isHoldingFilterAtStandby()) {
                FCSLOG.info("=== carousel is CLAMPED_ON_FILTER === ");
                carousel.getSocketAtStandby().updateFilterID();
                FCSLOG.info("filter ID on CAROUSEL at STANDBY " + carousel.getSocketAtStandby().getFilterID());
                FCSLOG.info("filter ID on autochanger = " + autochanger.getFilterID());
                FCSLOG.info("==================================");

            } else {
                recoveryLockingProcess();
                if (carousel.isHoldingFilterAtStandby()) {
                    carousel.getSocketAtStandby().updateFilterID();
                } else {
                    String msg = name + ": carousel should be CLAMPED_ON_FILTER when autochanger is "
                            + "at STANDBY with a filter; recovery process didn't work";
                    this.raiseAlarm(FcsAlert.CA_LOCKING_ISSUE, msg, name);
                    throw new FcsHardwareException(msg);
                }
            }
            if (autochanger.isAtStandby()) {
                // wait update local protection system
                autochanger.waitForProtectionSystemUpdate();
                FCSLOG.info(name + ": is going to moveEmptyFromStandbyToHandoff");
                autochanger.moveEmptyFromStandbyToHandoff(withPrecision);
                // for MCM
                previousSocketID = -1;
                publishDataForMcm();
            } else {
                throw new FcsHardwareException(
                        name + ": autochanger should be at STANDBY after " + "moveFilterToStandby() command");
            }
            publishActionDuration(GeneralAction.STORE_FILTER_ON_CAROUSEL, System.currentTimeMillis() - beginTime);
        }
    }

    private void recoveryLockingProcess() {
        try (AutoTimed at = new AutoTimed("recoveryLockingProcess")) {
            if (carousel.getClampXplus().isLocked() && !carousel.getClampXminus().isLocked()) {
                FcsUtils.sleep(200, name);
                carousel.recoveryLockingXminus();
            }
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, autoAck = false, description = "Unclamp filter from carousel, move autochanger to approachStandby position and release the carousel clamps", timeout = 15000)
    public void disengageFilterFromCarousel() {
        try (AutoTimed at = new AutoTimed("disengageFilterFromCarousel")) {
            final long beginTime = System.currentTimeMillis();
            movementCounter.get(GeneralAction.DISENGAGE_FILTER_FROM_CAROUSEL).increment();
            updateStateWithSensors();
            if (!FilterReadinessState.READY.equals(getFilterReadinessState())) {
                throw new RejectedCommandException(name + " FilterReadinessState must be READY");
            }
            if (!carousel.isAtStandby()) {
                throw new RejectedCommandException(name + "Carousel should be at Standby");
            }
            if (!(autochanger.isAtStandby())) {
                throw new RejectedCommandException("autochanger should be at Standby Position");
            }
            if (!autochanger.isAvailable()) {
                throw new RejectedCommandException("autochanger should be available");
            }
            // Unclamp the filter from the carousel
            carousel.unlockClamps();
            carousel.updateStateWithSensors();
            carousel.waitForProtectionSystemUpdate();

            if (!carousel.isUnclampedOnFilterAtStandby()) {
                String msg = name + "Carousel clamps still locked. Aborting autochanger movement because "
                        + "carousel is still holding the filter. bad state for socketAtStandby: "
                        + carousel.getClampsStateAtStandby() + " should be UNCLAMPED_ON_FILTER";
                raiseAlarm(FcsAlert.HARDWARE_ERROR, msg, name);
                throw new FcsHardwareException(name + msg);
            }
            autochanger.waitForProtectionSystemUpdate();
            // for MCM
            previousSocketID = carousel.getSocketAtStandbyID();
            // autochanger begins to leave standby position : Move autochanger to approach standby position
            autochanger.getAutochangerTrucks().moveToApproachStandbyPositionWithLowVelocity();
            publishDataForMcm();

            updateStateWithSensors();
            if (!carousel.isEmptyAtStandby()) {
                raiseAlarm(FcsAlert.HARDWARE_ERROR, "Carousel is still seeing the filter", name);
                throw new FcsHardwareException(
                        name + " aborting autochanger movement because carousel is still seeing filter.");
            }

            // Release the clamps
            FcsUtils.asyncRun(() -> {
                try {
                    carousel.releaseClamps();
                } catch (Exception e) {
                    this.raiseWarning(FcsAlert.HARDWARE_ERROR, "cannot release carousel clamps", "carousel");
                }
            });
            updateStateWithSensors();
            publishActionDuration(GeneralAction.DISENGAGE_FILTER_FROM_CAROUSEL, System.currentTimeMillis() - beginTime);
        }
    }

    // SET FILTER PRECONDITIONS
    public boolean autochangerNotInTravel() {
        return autochanger.isAtHandoff() || autochanger.isAtOnline() || autochanger.isAtStandby();
    }

    public boolean latchesOpenOrClosed() {
        return autochanger.getLatches().isClosed() || autochanger.getLatches().isOpened();
    }

    public boolean onlineClampsOpenOrLocked() {
        return autochanger.getOnlineClamps().isLocked() || autochanger.getOnlineClamps().isOpened();
    }

    public boolean filterAtOnlineMustBeLocked() {
        return !autochanger.isAtOnline() || autochanger.isEmpty() || autochanger.getOnlineClamps().isLocked();
    }

    public boolean filterAtStandbyMustBeHeld() {
        return !autochanger.isAtStandby() || autochanger.isEmpty() || autochanger.isHoldingFilter()
                || carousel.isHoldingFilterAtStandby();
    }

    public boolean carouselHoldingFilterOrReadyToGrab() {
        return carousel.isHoldingFilterAtStandby() || carousel.isReadyToGrabAFilterAtStandby();
    }

    public boolean carouselReadyToClampAtStandby() {
        return autochanger.isAtStandby() || autochanger.isEmpty() || carousel.isReadyToGrabAFilterAtStandby();
    }

    // Are we in a situation for which a filter exchange is safe from the telescope point of view
    public boolean areTmaAndRotatorLocked() {
        return tmaAndRotatorMotionStatus.equals(TelescopeMountRotatorState.LOCKED) ||
                tmaAndRotatorMotionStatus.equals(TelescopeMountRotatorState.PROTOTYPE) ||
                isLoaderConnected() ||
                FcsUtils.isSimu();
    }

    // Are the trucks in a position that allows for the moves ?
    public boolean trucksMustBeStraight() {
        return autochanger.getAutochangerInclinationState().equals(AutochangerInclinationState.STRAIGHT);
    }

    /**
     * move filter to online. command to be executed by mcm. can be executed
     * only when FilterReadinessState is READY, loader not connected, no
     * controller is error, carousel stopped at STANDBY, autochanger stopped at
     * STANDBY, ONLINE or HANDOFF, no sensors in error, if a filter is on
     * autochanger at HANDOFF or ONLINE => carousel socket at STANDBY must be
     * READY_TO_CLAMP && autochanger latches have to be CLOSED, if a filter is
     * on autochanger at STANDBY => carousel or autochanger must hold the
     * filter, if a filter is on autochanger at ONLINE => online clamps must be
     * LOCKED.
     *
     * @param filterID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Move filter to ONLINE position.", timeout = 180000, autoAck = false)
    public void setFilter(int filterID) {
        if ( filterID == 0 ) {
            setNoFilterAtHandoffOrOnline(true);
        } else {
            setFilterAtHandoffOrOnline(filterID, true, false);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Move filter to ONLINE position.", timeout = 180000, autoAck = false)
    public void setFilterByName(String filterName) {
        subs.helper()
                .precondition(filterManager.containsFilterName(filterName), "%s: Unknown filter name : %s", name, filterName)
                .precondition(FilterReadinessState.READY).action(() -> {
                        /* filterName is correct*/
            setFilter(filterManager.getFilterID(filterName));
//Just for MCM tests on PROTO.
//            setFilterAtHandoffOrOnline(filterManager.getFilterID(filterName), false);
                });
    }

    /**
     * move filter to HANDOFF. command to be executed by mcm. can be executed
     * only when FilterReadinessState is READY, loader not connected, no
     * controller is error, carousel stopped at STANDBY, autochanger stopped at
     * STANDBY, ONLINE or HANDOFF, no sensors in error, if a filter is on
     * autochanger at HANDOFF or ONLINE => carousel socket at STANDBY must be
     * READY_TO_CLAMP && autochanger latches have to be CLOSED, if a filter is
     * on autochanger at STANDBY => carousel or autochanger must hold the
     * filter, if a filter is on autochanger at ONLINE => online clamps must be
     * LOCKED.
     *
     * @param filterID of the filter to move
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Select filter and move to HANDOFF position without going to ONLINE.", timeout = 180000, autoAck = false)
    public void setFilterAtHandoff(int filterID) {
        if ( filterID == 0 ) {
            setNoFilterAtHandoffOrOnline(false);
        } else {
            setFilterAtHandoffOrOnline(filterID, false, false);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Select filter and move to HANDOFF position without going to ONLINE.", timeout = 180000, autoAck = false)
    public void setFilterAtHandoffForLoader(int filterID) {
        setFilterAtHandoffOrOnline(filterID, false, true);
    }

    @Deprecated
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Store all filters on carousel and move autochanger trucks to ONLINE.", timeout = 180000, autoAck = false)
    public void setNoFilter() {
        setNoFilterAtHandoffOrOnline(true);
    }

    @Deprecated
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Store all filters on carousel and move autochanger trucks to HANDOFF.", timeout = 180000, autoAck = false)
    public void setNoFilterAtHandoff() {
        setNoFilterAtHandoffOrOnline(false);
    }

    @SuppressWarnings("unchecked")
    private void setFilterAtHandoffOrOnline(int filterID, boolean toOnline, boolean forLoader) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setFilter")) {

            updateStateWithSensors();
            if (carousel.isPowerSaveActivated()) {
                carousel.powerOn(carouselWakeUpDuration);
                autochanger.waitForProtectionSystemUpdate();
            }
            percentage = 0;
            publishDataForMcm();

            /* if a controller is in fault, FCS goes in ALARM state */
            checkControllers();
            FCSLOG.info(name + ": filter to set online:" + filterID);
            subs.helper()
                    .precondition(!agentStateService.isInState(AlertState.ALARM),
                            "can't execute commands in ALARM state.")
                    .precondition(areTmaAndRotatorLocked(),
                            "The telescope motion and rotator need to be locked before performing a filter exchange. Current status of the lock is %s", getTelescopeSoftwareLockStatus())
                    .precondition(trucksMustBeStraight(),
                            "The FCS does not allow a filter exchange to be executed because the autochanger/camera is currently tilted. Ensure the rotator is set back to 0 before trying again.")
                    // CHECK it should be valid now
                    // loaderPresenceSensor is invalid
                    // .precondition(!isLoaderConnected(), "loader is connected - can't continue
                    // setFilter")
                    .precondition(filterManager.containsFilterID(filterID), "%s: Unknown filter id : %s", name,
                            filterID)
                    .precondition(isFilterInCamera(filterID), "%s: filter %s / filter name %s is out of camera", name, filterID, filterManager.getFilterNameByID(filterID))
                    .precondition(isFilterAvailable(filterID), "%s: filter %s / filter name %s is not available", name, filterID, filterManager.getFilterNameByID(filterID))
                    .precondition(carousel.isAtStandby(), "carousel not stopped at STANDBY position")
                    .precondition(autochangerNotInTravel(), "autochanger trucks are not at a HANDOFF or ONLINE or STANDBY")
                    .precondition(autochanger.isAvailable(), "autochanger is not available")
                    .precondition(latchesOpenOrClosed(),
                            "%s: bad state for autochanger latches - have to be OPENED or CLOSED",
                            autochanger.getLatches().getLockStatus())
                    .precondition(onlineClampsOpenOrLocked(),
                            "%s: bad state for autochanger ONLINE clamps - have to be OPENED or LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtOnlineMustBeLocked(),
                            "%s: bad state for autochanger ONLINE clamps - at ONLINE with a filter should be LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtStandbyMustBeHeld(),
                            "When a filter is at STANDBY it must be held by autochanger or by carousel.")
                    .precondition(carouselHoldingFilterOrReadyToGrab(),
                            "%s: bad state for carousel : should be holding a filter or ready to receive a filter",
                            carousel.getClampsStateAtStandby())
                    .precondition(carouselReadyToClampAtStandby(),
                            "%s: bad state for carousel when a filter is on autochanger at HANDOFF or ONLINE: should be READYTOCLAMP",
                            carousel.getClampsStateAtStandby())
                    .precondition(FilterReadinessState.READY)
                    .duration(SETFILTER_MAX_DURATION)
                    .enterFaultOnException(true)
                    .action(() -> {
                        GeneralAction action = forLoader? GeneralAction.SET_FILTER_AT_HANDOFF_FOR_LOADER : GeneralAction.SET_FILTER;
                        movementCounter.get(action).increment();

                        final long beginTime = System.currentTimeMillis();
                        if (autochanger.isHoldingFilter()) {
                            //LOADED
                            agentStateService.updateAgentState(McmFilterState.LOADED);
                            FCSLOG.info(name + ": autochanger is holding a filter, filter name= "
                                    + autochanger.getFilterOnTrucksName());
                        } else {
                            agentStateService.updateAgentState(McmFilterState.NOFILTER);
                            FCSLOG.info(name + ": autochanger is not holding filter, filter name on autochanger"
                                    + autochanger.getFilterOnTrucksName());
                            FCSLOG.info(name + ": filterID: " + filterID + " is on socket:"
                                    + carousel.getFilterSocket(filterID).getId());
                        }
                        /**
                         * /* If the filter we want to move ONLINE is NOT on autochanger : /* - if autochanger is at
                         * STANDBY it has to be moved empty to HANDOFF /* - if autochanger is at HANDOFF or
                         * ONLINE, and another filter is on autochanger this filter has to be store on carousel,
                         * /* then autochanger can be moved empty at HANDOFF.
                         */
                        carousel.setControllerPositionSensorTypeEncoderSSI();
                        if (!autochanger.isFilterOnAC(filterID)) {
                            //UNLOADING
                            agentStateService.updateAgentState(McmFilterState.UNLOADING);
                            FCSLOG.info(name + ": filterID: " + filterID + " is NOT on autochanger");
                            if (autochanger.isAtStandby()) {
                                FCSLOG.info(name + " autochanger is at STANDBY, it has to be moved empty at HANDOFF");
                                autochanger.moveEmptyFromStandbyToHandoff();
                            } else {
                                /* autochanger can be at HANDOFF or ONLINE with a filter or empty */
                                /* if a filter is on autochanger */
                                if (autochanger.isHoldingFilter()) {
                                    FCSLOG.info(
                                            name + ": autochanger is holding filter: " + autochanger.getFilterOnTrucksName());
                                    FCSLOG.info(name + ": is going to store a filter on carousel");
                                    // No need here to use precision at handoff since the autochanger
                                    // is going back to the carousel next
                                    storeFilterOnCarouselRelaxed();
                                }
                            }
                            percentage = 20;
                            publishDataForMcm();

                            /* Now autochanger is empty at Handoff or at ONLINE*/
                            if (!(autochanger.isEmpty() && (autochanger.isAtHandoff() || autochanger.isAtOnline()))) {
                                throw new FcsHardwareException(name + ": autochanger is not empty at handoff or online");
                            }
                            // Rotate desired filter to standby
                            //ROTATING
                            agentStateService.updateAgentState(McmFilterState.ROTATING);
                            carousel.rotateSocketToStandby(carousel.getFilterSocket(filterID).getName());
                            FcsUtils.sleep(100, name);
                            percentage = 40;
                            publishDataForMcm();
                            updateStateWithSensors();
                            if (carousel.isAtStandby()) {
                                FCSLOG.info(name + ": carousel is at standby");
                            } else {
                                throw new FcsHardwareException(name + ": carousel should be at standby after "
                                        + "rotateSocketToStandby command.");
                            }

                            /* At this point filterID should be on carousel at STANDBY */
                            if (carousel.getFilterIDatStandby() == filterID) {
                                // Carousel could have continued rotation due to unbalanced state
                                carousel.checkDeltaPosition();
                                // Go and grab filter on autochanger
                                // wait for autochanger PLC to update after carousel rotation
                                // otherwise trucks controller is in UNDER_VOLTAGE fault.
                                autochanger.waitForProtectionSystemUpdate();
                                //LOADING
                                agentStateService.updateAgentState(McmFilterState.LOADING);
                                autochanger.grabFilterAtStandby();
                                updateStateWithSensors();
                                // for MCM
                                percentage = 60;
                                previousSocketID = carousel.getSocketAtStandbyID();
                                publishDataForMcm();

                            } else {
                                throw new FcsHardwareException(name + " filterID at standby is "
                                        + carousel.getFilterIDatStandby() + " should be: " + filterID);
                            }
                            updateStateWithSensors();
                            if (!autochanger.isFilterOnAC(filterID)) {
                                throw new FcsHardwareException(name + " filter: " + filterID + " should be now on autochanger");
                            }
                        }
                        /* now filterID is on autochanger. autochanger is at STANDBY or at HANDOFF or ONLINE. */
                        if (autochanger.isAtStandby()) {
                            try (ActionGuard g = autochanger.getAutochangerTrucks().new ReleasedACBrakes()) {

                                // Unclamp filter from carousel and move to approach position
                                disengageFilterFromCarousel();
                                percentage = 80;
                                publishDataForMcm();
                                updateStateWithSensors();

                                if (toOnline) {
                                    // Move the filter to online position
                                    autochanger.moveAndClampFilterOnline();
                                } else {
                                    // Move the filter to handoff position
                                    moveToHandoffWithHighVelocity(forLoader);
                                }
                            }
                        } else if (toOnline) {
                            if (autochanger.isAtHandoff()) {
                                autochanger.moveAndClampFilterOnline();

                            } else if (autochanger.isAtOnline()) {
                                autochanger.lockFilterAtOnline();
                            }
                        } else {
                            if (autochanger.isAtOnline() && autochanger.getOnlineClamps().isOpened()) {
                                try (ActionGuard g = autochanger.getAutochangerTrucks().new ReleasedACBrakes()) {
                                    autochanger.getAutochangerTrucks().moveToApproachOnlinePositionWithLowVelocity();
                                    moveToHandoffWithHighVelocity(forLoader);
                                }
                            }
                        }
                String filterName = filterManager.getFilterNameByID(filterID);
                updateAgentState(FilterState.valueOf("ONLINE_" + filterName.toUpperCase()));
                this.publishData();
                publishActionDuration(action, System.currentTimeMillis() - beginTime);
                //LOADED
                agentStateService.updateAgentState(McmFilterState.LOADED);
                percentage = 100;
                publishDataForMcm();
                // setFilterAtHandoff should never go back to powerSave
                if ( toOnline && carousel.isPowerSaveAllowed() ) {
                    carousel.powerSave();
                }
                // set the value back to zero after last publication
                percentage = 0;
            }); // END of action
        }
    }


    // TODO: This is a temporary solution to avoid the problem of the autochanger online clamps being activated. To be removed.
    @SuppressWarnings("unchecked")
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Set filter online but do not close the clamps. This is used for Winter 2023/2024 focal plane tests to avoid activating the autochanger online clamps.", timeout = 180000, autoAck = false)
    public void setFilterNoClamping(int filterID) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setFilterNoClamping")) {

            updateStateWithSensors();
            publishDataForMcm();
            /* if a controller is in fault, FCS goes in ALARM state */
            checkControllers();
            FCSLOG.info(name + ": filter to set online without clamping:" + filterID);
            subs.helper()
                    .precondition(!agentStateService.isInState(AlertState.ALARM),
                            "can't execute commands in ALARM state.")
                    // CHECK it should be valid now
                    // loaderPresenceSensor is invalid
                    // .precondition(!isLoaderConnected(), "loader is connected - can't continue
                    // setFilter")
                    .precondition(filterManager.containsFilterID(filterID), "%s: Unknown filter id : %s", name,
                            filterID)
                    .precondition(isFilterInCamera(filterID), "%s: filter %s / filter name %s is out of camera", name, filterID, filterManager.getFilterNameByID(filterID))
                    .precondition(carousel.isAtStandby(), "carousel not stopped at STANDBY position")
                    .precondition(autochangerNotInTravel(), "AC trucks are not at a HANDOFF or ONLINE or STANDBY")
                    .precondition(latchesOpenOrClosed(),
                            "%s: bad state for autochanger latches - have to be OPENED or CLOSED",
                            autochanger.getLatches().getLockStatus())
                    .precondition(onlineClampsOpenOrLocked(),
                            "%s: bad state for AC ONLINE clamps - have to be OPENED or LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtOnlineMustBeLocked(),
                            "%s: bad state for AC ONLINE clamps - at ONLINE with a filter should be LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtStandbyMustBeHeld(),
                            "When a filter is at STANDBY it must be held by AC or by carousel.")
                    .precondition(carouselHoldingFilterOrReadyToGrab(),
                            "%s: bad state for carousel : should be holding a filter or ready to receive a filter",
                            carousel.getClampsStateAtStandby())
                    .precondition(carouselReadyToClampAtStandby(),
                            "%s: bad state for carousel when a filter is on AC at HANDOFF or ONLINE: should be READYTOCLAMP",
                            carousel.getClampsStateAtStandby())
                    .precondition(FilterReadinessState.READY)
                    .duration(SETFILTER_MAX_DURATION)
                    .enterFaultOnException(true)
                    .action(() -> {
                        movementCounter.get(GeneralAction.SET_FILTER).increment();

                        final long beginTime = System.currentTimeMillis();
                        if (autochanger.isHoldingFilter()) {
                            //LOADED
                            agentStateService.updateAgentState(McmFilterState.LOADED);
                            FCSLOG.info(name + ": AC is holding a filter, filter name= "
                                    + autochanger.getFilterOnTrucksName());
                        } else {
                            agentStateService.updateAgentState(McmFilterState.NOFILTER);
                            FCSLOG.info(name + ": AC is not holding filter, filter name on AC"
                                    + autochanger.getFilterOnTrucksName());
                            FCSLOG.info(name + ": filterID: " + filterID + " is on socket:"
                                    + carousel.getFilterSocket(filterID).getId());
                        }
                        /**
                         * /* If the filter we want to move ONLINE is NOT on AC : /* - if AC is at
                         * STANDBY it has to be moved empty to HANDOFF /* - if AC is at HANDOFF or
                         * ONLINE, and another filter is on AC this filter has to be store on carousel,
                         * /* then AC can be moved empty at HANDOFF.
                         */
                        carousel.setControllerPositionSensorTypeEncoderSSI();
                        if (!autochanger.isFilterOnAC(filterID)) {
                            //UNLOADING
                            agentStateService.updateAgentState(McmFilterState.UNLOADING);
                            FCSLOG.info(name + ": filterID: " + filterID + " is NOT on AC");
                            if (autochanger.isAtStandby()) {
                                FCSLOG.info(name + " autochanger is at STANDBY, it has to be moved empty at HANDOFF");
                                autochanger.moveEmptyFromStandbyToHandoff();
                            } else {
                                /* autochanger can be at HANDOFF or ONLINE with a filter or empty */
                                /* if a filter is on AC */
                                if (autochanger.isHoldingFilter()) {
                                    FCSLOG.info(
                                            name + ": AC is holding filter: " + autochanger.getFilterOnTrucksName());
                                    FCSLOG.info(name + ": is going to store a filter on carousel");
                                    storeFilterOnCarouselRelaxed();
                                }
                            }

                            /* Now autochanger is empty at Handoff or at ONLINE*/
                            if (!(autochanger.isEmpty() && (autochanger.isAtHandoff() || autochanger.isAtOnline()))) {
                                throw new FcsHardwareException(name + ": autochanger is not empty at handoff or online");
                            }
                            // Rotate desired filter to standby
                            //ROTATING
                            agentStateService.updateAgentState(McmFilterState.ROTATING);
                            carousel.rotateSocketToStandby(carousel.getFilterSocket(filterID).getName());
                            FcsUtils.sleep(100, name);
                            updateStateWithSensors();
                            if (carousel.isAtStandby()) {
                                FCSLOG.info(name + ": carousel is at standby");
                            } else {
                                throw new FcsHardwareException(name + ": carousel should be at standby after "
                                        + "rotateSocketToStandby command.");
                            }

                            /* At this point filterID should be on carousel at STANDBY */
                            if (carousel.getFilterIDatStandby() == filterID) {
                                // Carousel could have continued rotation due to unbalanced state
                                carousel.checkDeltaPosition();
                                // Go and grab filter on AC
                                // wait for autochanger PLC to update after carousel rotation
                                // otherwise trucks controller is in UNDER_VOLTAGE fault.
                                autochanger.waitForProtectionSystemUpdate();
                                //LOADING
                                agentStateService.updateAgentState(McmFilterState.LOADING);
                                autochanger.grabFilterAtStandby();

                            } else {
                                throw new FcsHardwareException(name + " filterID at standby is "
                                        + carousel.getFilterIDatStandby() + " should be: " + filterID);
                            }
                            updateStateWithSensors();
                            if (!autochanger.isFilterOnAC(filterID)) {
                                throw new FcsHardwareException(name + " filter: " + filterID + " should be now on AC");
                            }
                        }
                        /* now filterID is on AC. AC is at STANDBY or at HANDOFF or ONLINE. */
                        if (!autochanger.isAtOnline()) {
                            if (autochanger.isAtStandby()) {
                                try (ActionGuard g = autochanger.getAutochangerTrucks().new ReleasedACBrakes()) {

                                    // Unclamp filter from carousel and move to approach position
                                    disengageFilterFromCarousel();
                                    updateStateWithSensors();

                                    // Move the filter to online position
                                    autochanger.moveFilterOnlineNoClamping();
                                }
                            } else if (autochanger.isAtHandoff()) {
                                autochanger.moveFilterOnlineNoClamping();
                            }
                        }

                String filterName = filterManager.getFilterNameByID(filterID);
                updateAgentState(FilterState.valueOf("ONLINE_" + filterName.toUpperCase()));
                this.publishData();
                publishActionDuration(GeneralAction.SET_FILTER, System.currentTimeMillis() - beginTime);
                //LOADED
                agentStateService.updateAgentState(McmFilterState.LOADED);
                publishDataForMcm();
                    }); // END of action
        }
    }


    private void moveToHandoffWithHighVelocity(boolean forLoader) {
        if (forLoader) {
            autochanger.getAutochangerTrucks().fastProfile();
            autochanger.getAutochangerTrucks().goToHandOffForLoader();
        } else {
            autochanger.getAutochangerTrucks().moveToHandoffWithHighVelocity();
        }
    }

    @SuppressWarnings("unchecked")
    private void setNoFilterAtHandoffOrOnline(boolean toOnline) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("setNoFilter")) {
            updateStateWithSensors();
            if (carousel.isPowerSaveActivated()) {
                carousel.powerOn(carouselWakeUpDuration);
            }
            publishDataForMcm();
            /* if a controller is in fault, FCS goes in ALARM state */
            checkControllers();
            subs.helper()
                    .precondition(!agentStateService.isInState(AlertState.ALARM),
                            "can't execute commands in ALARM state.")
                    .precondition(areTmaAndRotatorLocked(),
                            "The telescope motion and rotator need to be locked before performing a filter exchange. Current status of the lock is %s", getTelescopeSoftwareLockStatus())
                    .precondition(trucksMustBeStraight(),
                            "The FCS does not allow a filter exchange to be executed because the autochanger/camera is currently tilted. Ensure the rotator is set back to 0 before trying again.")
                    .precondition(carousel.isAtStandby(), "carousel not stopped at STANDBY position")
                    .precondition(autochangerNotInTravel(), "autochanger trucks are not at a HANDOFF or ONLINE or STANDBY")
                    .precondition(autochanger.isAvailable(), "autochanger is not available")
                    .precondition(latchesOpenOrClosed(),
                            "%s: bad state for autochanger latches - have to be OPENED or CLOSED",
                            autochanger.getLatches().getLockStatus())
                    .precondition(onlineClampsOpenOrLocked(),
                            "%s: bad state for autochanger ONLINE clamps - have to be OPENED or LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtOnlineMustBeLocked(),
                            "%s: bad state for autochanger ONLINE clamps - at ONLINE with a filter should be LOCKED",
                            autochanger.getOnlineClamps().getLockStatus())
                    .precondition(filterAtStandbyMustBeHeld(),
                            "When a filter is at STANDBY it must be held by autochanger or by carousel.")
                    .precondition(carouselHoldingFilterOrReadyToGrab(),
                            "%s: bad state for carousel : should be holding a filter or ready to receive a filter",
                            carousel.getClampsStateAtStandby())
                    .precondition(carouselReadyToClampAtStandby(),
                            "%s: bad state for carousel when a filter is on autochanger at HANDOFF or ONLINE: should be READYTOCLAMP",
                            carousel.getClampsStateAtStandby())
                    .precondition(FilterReadinessState.READY)
                    .duration(SETFILTER_MAX_DURATION)
                    .enterFaultOnException(true)
                    .action(() -> {
                        movementCounter.get(GeneralAction.SET_NO_FILTER).increment();
                        final long beginTime = System.currentTimeMillis();
                        if (autochanger.isHoldingFilter()) {
                    FCSLOG.info(
                            name + ": autochanger is holding filter: " + autochanger.getFilterOnTrucksName());
                    FCSLOG.info(name + ": is going to store a filter on carousel");
                    //UNLOADING
                    agentStateService.updateAgentState(McmFilterState.UNLOADING);
                    storeFilterOnCarousel();
                } else {
                    FCSLOG.info(name + " no filter to store on carousel");
                }
                /* Now autochanger is empty at Handoff or Online */
                if (!(autochanger.isEmpty() && (autochanger.isAtHandoff() || autochanger.isAtOnline()))) {
                    throw new FcsHardwareException(name + ": autochanger is not empty at handoff");
                }
                if (toOnline) {
                    autochanger.goToOnline();
                }
                updateAgentState(FilterState.ONLINE_NONE);
                updateAgentState(McmFilterState.NOFILTER);
                if (carousel.isPowerSaveAllowed()) {
                    carousel.powerSave();
                }
                publishDataForMcm();
                this.publishData();
                publishActionDuration(GeneralAction.SET_NO_FILTER, System.currentTimeMillis() - beginTime);
                    });
        }
    }

    /**
     * just for mcm testing
     *
     * @param state
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "change subsystem state.")
    public void changeState(String state) {
        updateAgentState(FilterState.valueOf(state));
        FCSLOG.fine(() -> "SUBSYSTEM STATE=" + isInState(FilterState.valueOf(state)));
    }

    @Deprecated
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "list filters in Filter Changer.")
    public Map<Integer, String> listFiltersOnChanger() {
        Map<Integer, String> lf = new HashMap<Integer, String>();
        if (autochanger.getFilterID() != 0) {
            lf.put(autochanger.getFilterID(), autochanger.getFilterOnTrucksName());
        }
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty()) {
                lf.put(socket.getFilterID(), socket.getFilterName());
            }
        });
        return lf;
    }

    @Deprecated
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "list filters in Filter Changer.")
    public List<String> listAllFilters() {
        return listAllFilterNames();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "List the possible filters names as defined in the filter manager.")
    public List<String> listAllFilterNames() {
        return filterManager.getFilterNames();
    }

    /**
     * Produce a synthetic map of the filters in the FES organised by position (either autochanger or a socket)
     *
     * @return Map of FES position and associated filter
     */
    public Map<String, Filter> getFcsFilterMap() {
        Map<String, Filter> lf = new TreeMap<String, Filter>();
        if (autochanger.getFilterID() != 0) {
            lf.put("autochanger", filterManager.getFilterByID(autochanger.getFilterID()));
        }
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty() && socket.getFilterID() != 0) {
                lf.put(socket.getName(), filterManager.getFilterByID(socket.getFilterID()));
            }
        });
        return lf;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Print information on the filters physically in the LSSTCam.")
    public String printFilterMap() {
        StringBuilder sb = new StringBuilder("Filters on FES: \n");
        getFcsFilterMap().entrySet().stream().forEach(e -> {
            String socketName = e.getKey();
            if ( socketName != "autochanger" && !carousel.getSocketByName(socketName).isAvailable() ) {
                sb.append("* ");
            }
            sb.append(socketName);
            sb.append("\n\t");
            sb.append(e.getValue().toString());
            sb.append("\n");
        });
        sb.append("\n'*' marks the currently unavailable sockets with filters.\n");
        return sb.toString();
    }

    /**
     * This methods prints information on a given filter
     *
     * @param filterName
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Print general information on a filter.")
    public String printFilterInfo(String filterName) {
        return filterManager.getFilterByName(filterName).toString();
    }

    /**
     * This method lists the available filters installed in the FES
     * It starts with the list of filters installed in the camera and can be affected by the availability
     * of several elements of the Filter Exchange System such as
     * - the carousel unable to rotate
     * - the autochanger trucks unable to move
     * - one or several carousel sockets malfunctioning
     *
     * Such elements will be marked as unavailable during daytime or at night if an issue is encountered and the
     * system will thus switch to a degraded mode where the availability of filters is reduced.
     *
     * @return the list of available filters in a HashMap { filterID : filtername }
     *
     */
    public Map<Integer, String> getAvailableFilterMap() {
        Map<Integer, String> availableFilterMap = new TreeMap<Integer, String>();

        // First scenario, the autochanger is flagged as unavailable
        if (!autochanger.isAvailable()) {
            if (autochanger.isAtOnline()) {
                // either a filter is at ONLINE or it's empty and the filter will be NONE
                availableFilterMap.put(autochanger.getFilterID(), autochanger.getFilterOnTrucksName());
            }
            return availableFilterMap;
        }

        // Second scenario, the carousel rotation is not available but the autochanger is working fine
        //   - 1) a filter is on the current socket at STANDBY and the socket is available it can be used
        //   - 2) a filter is on the autochanger so it can be used
        //   - in both cases, if the socket is available, setNoFilter can also be used
        if (!carousel.isAvailable()) {
            if ( !carousel.socketAtStandby.isEmpty() ) {
                if ( carousel.socketAtStandby.isAvailable() ) {
                    availableFilterMap.put(carousel.socketAtStandby.getFilterID(), carousel.socketAtStandby.getFilterName());
                }
                availableFilterMap.put(0, "NONE");
            } else if ( !autochanger.isEmpty() ) {
                availableFilterMap.put(autochanger.getFilterID(), autochanger.getFilterOnTrucksName());
                if ( carousel.socketAtStandby.isAvailable() ) {
                   availableFilterMap.put(0, "NONE");
                }
            }
            return availableFilterMap;
        }

        // Third scenario, the carousel rotation is available and the autochanger is working fine, but some socket might be unavailable.
        // - start with a filter in autochanger
        if (!autochanger.isEmpty() && !autochanger.isAtStandby()) {
            availableFilterMap.put(autochanger.getFilterID(), autochanger.getFilterOnTrucksName());
            // If there is no free and available socket on the carousel to drop the filter in, we are stuck with the filter on autochanger
            if (!carousel.socketsMap.values().stream().anyMatch((socket) -> (socket.isEmpty() && socket.isAvailable()))) {
                return availableFilterMap;
            }
        };
        // - continue with available filters on the carousel
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if (!socket.isEmpty() && socket.getFilterID() != 0 && socket.isAvailable()) {
                Filter f = filterManager.getFilterByID(socket.getFilterID());
                availableFilterMap.put(f.getFilterID(), f.getName());
            }
        });
        // - end with the "NONE" filter
        availableFilterMap.put(0, "NONE");

        return availableFilterMap;
    }

    /***************************************************************
     *  __  __  ____ __  __      __                                *
     * |  \/  |/ ___|  \/  |    / /                                *
     * | |\/| | |   | |\/| |   / /                                 *
     * | |  | | |___| |  | |  / /                                  *
     * |_|  |_|\____|_|  |_| /_/                                   *
     *                                                             *
     *   ___   ____ ____        ____  ____  ___ ____   ____ _____  *
     *  / _ \ / ___/ ___|      | __ )|  _ \|_ _|  _ \ / ___| ____| *
     * | | | | |   \___ \ _____|  _ \| |_) || || | | | |  _|  _|   *
     * | |_| | |___ ___) |_____| |_) |  _ < | || |_| | |_| | |___  *
     *  \___/ \____|____/      |____/|_| \_\___|____/ \____|_____| *
     *                                                             *
     ***************************************************************/

    /**
     * This command is used by scripts to run sequences of filter exchanges
     *
     * @return List of filter IDs
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the list of available filters ID in the Filter Exchange System")
    public List<Integer> getAvailableFiltersID() {
        return getAvailableFilterMap().keySet().stream()
            .filter(filterID -> filterID != 0)
            .toList();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide a mapping carousel sockets to available filters ID in the Filter Exchange System")
    public Map<String, Integer> getSocketsToAvailableFiltersID() {
        List<Integer> availableFiltersID = getAvailableFiltersID();
        Map<String, Integer> fm = new TreeMap<String, Integer>();
        carousel.getSocketsMap().values().stream().forEach(socket -> {
            if ( socket.isEmpty() ) {
                if ( socket.isAtStandby() && availableFiltersID.contains(autochanger.getFilterID()) ) {
                    fm.put(socket.getName(), autochanger.getFilterID());
                }
            } else if ( availableFiltersID.contains(socket.getFilterID()) ) {
                fm.put(socket.getName(), socket.getFilterID());
            }
        });
        return fm;
    }

    /**
     * This command is used on a daily basis by the OCS-bridge to request the list of available filters
     * in order to produce an observing schedule for the night.
     *
     * @return List of filter names
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the list of available filters in the Filter Exchange System formatted as wanted by the OCS Bridge")
    public List<String> getAvailableFilters() {
        return getAvailableFilterMap().entrySet().stream()
            .map(entry -> filterManager.getOCSFilterNameByID(entry.getKey()))
            .collect(Collectors.toList());
    }

    /**
     * This command is called by the OCS-bridge to know the installed filters in the camera
     *
     * @return List of filter names
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the list of installed filters in the Filter Exchange System formatted as wanted by the OCS Bridge.")
    public List<String> getInstalledFilters() {
        return getFcsFilterMap().entrySet().stream()
            .map(entry -> filterManager.getOCSFilterNameByID(entry.getValue().getFilterID()))
            .collect(Collectors.toList());
    }

    /**
     * This command is called by the OCS-bridge to know the name of the filter in operation
     *
     * @return The filter name
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the name of the filter currently online, formatted as wanted by the OCS Bridge")
    public String getOnlineFilterName() {
        int filterId = 0;  // No filter
        if ( autochanger.isAtOnline() ) {
            filterId = autochanger.getFilterID();
        }

        return filterManager.getOCSFilterNameByID(filterId);
    }

    /**
     * This command is called by the OCS-bridge to know the maximum angle (degrees) from vertical at which a filter exchange can be executed
     *
     * @return angle in degrees
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the maximum angle from vertical at which a filter exchange can be executed as wanted by the OCS Bridge.")
    public double getMaxAngleForFilterChange() {
        // TODO Alex: currently a placeholder, update this value
        return 90;
    }

    /**
     * This command is called by the OCS-bridge to know the maximum angle (degrees) from vertical at which a filter exchange can be executed in normal speed
     *
     * @return duration
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the maximum angle from vertical at which a filter exchange can be executed in normal speed (fast) as wanted by the OCS Bridge.")
    public double getMaxAngleForFastFilterChange() {
        /* The carousel is set to use the slow velocity for an airmass > 3
        zenith = arccos(1/airmass) => elevation ~ 70.5 degrees */
        int airmass = 3;
        return Math.acos(1. / airmass) * 180 / Math.PI;
    }

    /**
     * This command is called by the OCS-bridge to know the maximum time of a slow filter exchange during operations
     *
     * @return duration
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the maximum duration of a filter exchange in slow-mode conditions as wanted by the OCS Bridge.")
    public Duration getDurationForSlowFilterChange(String filterName) {
        return Duration.ofSeconds(setFilterSlowDuration);
    }

    /**
     * This command is called by the OCS-bridge to know the maximum time of a normal filter exchange during operations
     *
     * @return duration
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Provide the maximum duration of a filter exchange in normal conditions as wanted by the OCS Bridge.")
    public Duration getDurationForFastFilterChange(String filterName) {
        return Duration.ofSeconds(setFilterFastDuration);
    }

    /**
     * This command is called by the OCS-bridge to control the power state of the carousel and get the system ready for a filter change in advance.
     *
     * @param mode is the wake up mode decided by the observatory 0: go to sleep, 1: wake up and go back to sleep after setFilter, 2: wake up and stay up
     */
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Control the power state of the filter changer carousel. "
        + "Options are 0: go to sleep, 1: wake up and go back to sleep after setFilter, 2: wake up and stay up")
    public void wakeFilterChanger(int mode) {
        try {
            CarouselPowerMode wakeUpMode = CarouselPowerMode.values()[mode];
            FCSLOG.info("wakeFilterChanger was issued with mode : " + wakeUpMode);
            wakeFilterChanger(wakeUpMode);
        } catch (ArrayIndexOutOfBoundsException ex) {
            throw new RejectedCommandException("The only available modes for wakeFilterChanger are 0: go to sleep, 1: wake up, 2: wake up and stay up");
        }
    }

    private void wakeFilterChanger(CarouselPowerMode mode) {
        switch (mode) {
            case GO_TO_SLEEP:
                if ( agentStateService.isInState(CarouselPowerState.REGULAR) ) {
                    carousel.powerSave();
                } else {
                    FCSLOG.info("filter changer already in power save");
                }
                break;
            case WAKE_UP:
                carousel.setPowerSaveAllowed(true);
                if ( agentStateService.isInState(CarouselPowerState.LOW_POWER) ) {
                    carousel.powerOn(carouselWakeUpDuration);
                } else {
                    FCSLOG.info("filter changer already waken up, switching power save allowed to true");
                }
                break;
            case STAY_UP:
                carousel.setPowerSaveAllowed(false);
                if ( agentStateService.isInState(CarouselPowerState.LOW_POWER) ) {
                    carousel.powerOn(carouselWakeUpDuration);
                } else {
                    FCSLOG.info("filter changer already waken up, switching power save allowed to false");
                }
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "")
    public Duration getDurationForWakeUp(int mode) {
        return Duration.ofSeconds(carouselWakeUpDuration);
    }

    // END OF MCM / OCS-BRIDGE COMMANDS

    /**
     * Load a filter from the loader to the camera. The loader must hold a filter at
     * STORAGE position. The autochanger must be empty at HANDOFF, latches open. At
     * the end of this command the filter is held by autochanger at Handoff
     * position, and the loader carrier is empty at STORAGE.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @SuppressWarnings("unchecked")
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Load a filter from the loader to the camera. The loader must hold a filter at STORAGE position."
            + "The autochanger must be empty at HANDOFF, latches open. At the end of this command the"
            + "filter is held by autochanger at Handoff position, and the loader carrier is empty at STORAGE.",
            timeout = 480000, autoAck = false)
    public void loadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("loadFilter")) {
            //
        // TODO GLOBAL REVIEW check that the camera is in the horizontal position
        // to test if carousel is empty at standby is over engineering : it has to be
        // tested by autochanger before command StoreFilterOnCarousel
        updateStateWithSensors();
        subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM),
                        "can't execute commands in ALARM state.")
                // autochanger trucks must be empty at "Hand-Off" position
                .precondition(autochanger.isAtHandoff() && autochanger.isEmpty(),
                        " autochanger is not empty at HANDOFF position; can't load a filter.")
                // autochanger latches must be open.
                .precondition(autochanger.getLatches().isOpened(),
                        "Autochanger latches must be open before loadFilter command.")
                // The Loader carrier is retracted at "Storage" position and the filter is clamped
                .precondition(loader.isClampedOnFilter() && loader.isAtStorage(),
                        name + " can't load filter because loader is not holding a filter at storage position.")
                .duration(LOAD_UNLOAD_FILTER_MAX_DURATION)
                .enterFaultOnException(true)
                .action(() -> {
                    movementCounter.get(GeneralAction.LOAD_FILTER).increment();
                    final long beginTime = System.currentTimeMillis();

            if (!autochanger.isPositionCorrectForExchangeWithLoader()) {
                autochanger.goToHandOffForLoader();
            }

            loader.moveFilterToHandoff();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            autochanger.closeLatches();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            loader.openClampAndMoveEmptyToStorage();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();
            publishActionDuration(GeneralAction.LOAD_FILTER, System.currentTimeMillis() - beginTime);
                    }); // END of action
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Load a filter from the loader to the camera"
            + "and store it on socket which name is given as argument. The loader must hold a filter at STORAGE position."
            + "The autochanger must be empty at HANDOFF, latches open. At the end of this command the"
            + "filter is store on carousel socket which name is given as argument, autochanger is empty at Handoff position, "
            + "and the loader carrier is empty at STORAGE.")
    public void loadFilterOnSocket(String socketName) {
        // note: this is a composite action LOAD_FILTER --> ROTATE_SOCKET_TO_STANDBY --> STORE_FILTER_ON_CAROUSEL
        // with 3 counters/durations
        updateStateWithSensors();
        FcsUtils.checkSocketName(socketName);
        CarouselSocket socket = carousel.getSocketsMap().get(socketName);
        if (!socket.isEmpty()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not empty. "
                    + "Select an empty socket.", socketName));
        }
        if (!socket.isAvailable()) {
            throw new RejectedCommandException(String.format("Can't execute loadFilterOnSocket because %s is not available. "
                    + "Select an available socket.", socketName));
        }
        loadFilter();
        carousel.rotateSocketToStandby(socketName);
        storeFilterOnCarousel();
    }

    /**
     * Unload a filter from the camera to the loader. The camera has to be at
     * horizontal position. The filter has to be on Autochanger at HANDOFF. Loader
     * carrier must be empty at STORAGE. At the end of this command, autochanger is
     * empty at HANDOFF, latches are open, and loader carrier is holding the filter
     * at STORAGE (hooks are clamped).
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @SuppressWarnings("unchecked")
    @Command(type = Command.CommandType.ACTION, level = Command.NORMAL, description = "Unload a filter from the camera to the loader. The camera has to be at horizontal position. "
            + "The filter has to be on Autochanger at HANDOFF. Loader carrier must be empty at STORAGE. "
            + "\nAt the end of this command, autochanger is empty at HANDOFF, latches are open, "
            + "and loader carrier is holding the filter at STORAGE (hooks are clamped)",
            timeout = 480000, autoAck = false)
    public void unloadFilter() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("unloadFilter")) {
            // TODO GLOBAL REVIEW check that the camera is in the horizontal position
        updateStateWithSensors();
        subs.helper()
                .precondition(!agentStateService.isInState(AlertState.ALARM),
                        "can't execute commands in ALARM state.")
                // the auto changer trucks must hold a filter at "Hand-Off" position
                .precondition(autochanger.isAtHandoff() && autochanger.isHoldingFilter(),
                        " autochanger is not holding a filter at HANDOFF position; can't unload a filter.")
                .precondition(autochanger.isAvailable(), "autochanger is not available")
                .duration(LOAD_UNLOAD_FILTER_MAX_DURATION)
                .enterFaultOnException(true)
                .action(() -> {
                    movementCounter.get(GeneralAction.UNLOAD_FILTER).increment();
                    final long beginTime = System.currentTimeMillis();

            if (!autochanger.isPositionCorrectForExchangeWithLoader()) {
                if (autochanger.isLinearRailMotionAllowed()) {
                    autochanger.goToHandOffForLoader();
                } else {
                    throw new RejectedCommandException("Autochanger trucks position is not correct for exchange with loader "
                            + " but command goToHandOffForLoader can't be executed because PLC doesn't allow to move trucks.");
                }
            }

            loader.moveEmptyToHandoffAndClose();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            autochanger.openLatches();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            loader.moveFilterToStorage();
            FcsUtils.sleep(50, name);
            updateStateWithSensors();

            publishActionDuration(GeneralAction.UNLOAD_FILTER, System.currentTimeMillis() - beginTime);
                    }); // END of action
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Recover from a timeout or an exception raised during the carousel clamping process. "
                + "This command moves the filter back to handoff position and releases both carousel clamps.")
    public void recoveryCarouselClamping() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("recoveryCarouselClamping")) {
            carousel.unlockClamps();
            autochanger.moveToApproachStandbyPositionWithLowVelocity();
            carousel.releaseClamps();
            autochanger.moveToHandoffWithHighVelocity();
        }
    }

    private void recoveryCarouselUnclamping(boolean forXplus) {
        CarouselClamp clamp;
        String clampName;
        EPOSController controller;

        if ( forXplus ) {
            clamp = carousel.getClampXplus();
            clampName = "Xplus";
            controller = carousel.getClampXplusController();
        } else {
            clamp = carousel.getClampXminus();
            clampName = "Xminus";
            controller = carousel.getClampXminusController();
        }

        if ( clamp.getClampState() == FilterClampState.UNCLAMPED_ON_FILTER ) {
            FCSLOG.info("The " + clampName + " clamp is already unclamped. Canceling recovery.");
            return;
        }

        int finalCurrent = clamp.getCurrentToUnlock();

        // Manual current ramp - the last value should exceed finalCurrent
        // but the factor 14 was chosen so that the maximum current value
        // sent to the controller does not exceed 5000 mA at the summit
        int currentIncrement = (int) Math.round(finalCurrent / 14);
        int current = 0;

        while ( Math.abs(current) < Math.abs(finalCurrent) ) {
            controller.enableAndWriteCurrent(current);
            if ( current == 0 ) {
                // Time to get the clamp motor to cool down
                FcsUtils.sleep(2000, name);
            } else {
                // Time to let the motor get to speed
                FcsUtils.sleep(500, name);
            }
            current += currentIncrement;
            currentIncrement *= 2;
        }

        carousel.waitForStateUnclampedOnFilter(2000);

        if (carousel.isUnclampedOnFilterAtStandby()) {
            FCSLOG.info("The "+ clampName + " is now unclamped. Recovery successful. Proceeding with disengaging the filter and releasing the clamps.");
            autochanger.moveToApproachStandbyPositionWithLowVelocity();
            carousel.releaseClamps();
            autochanger.fastTrucksProfile();
            autochanger.goToHandOffForLoader();
        } else {
            carousel.releaseClamps();
            autochanger.moveEmptyFromStandbyToHandoff();
            String msg = "The carousel " + clampName + " unlocking recovery on " + carousel.getSocketAtStandby().getName() + " did not succeed. "
                + "After recovery the clamp " + clampName + " ended in state " + clamp.getClampState().toString()
                + " Both carousel clamps were released to avoid damaging the system and the autochanger was moved empty to HANDOFF position."
                + " We recommend that the socket be set to 'unavailable' for the time being to avoid further damage.";
            throw new FcsHardwareException(name + msg);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
    description = "Recover from the Xplus clamp getting stuck during the carousel unclamping process. "
        + "This command works specifically on the Xplus clamp to attempt to open it to release the filter. "
        + "The socket should be monitored on the GUI while executing this command and the recovery can be "
        + "considered successful if the Xplus clamp state switches to UNCLAMPED_ON_FILTER.")
    public void recoveryCarouselXplusUnclamping() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("recoveryCarouselXplusUnclamping")) {
            recoveryCarouselUnclamping(true);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
        description = "Recover from the Xminus clamp getting stuck during the carousel unclamping process. "
            + "This command works specifically on the Xminus clamp to attempt to open it to release the filter. "
            + "The socket should be monitored on the GUI while executing this command and the recovery can be "
            + "considered successful if the Xminus clamp state switches to UNCLAMPED_ON_FILTER.")
    public void recoveryCarouselXminusUnclamping() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("recoveryCarouselXminusUnclamping")) {
            recoveryCarouselUnclamping(false);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "check if controllers are in fault")
    @Override
    public void checkControllers() {
        this.bridge.checkControllers();
    }

    /**
     * Update state in reading sensors for all components of Filter Changer :
     * carousel, autochanger and loader. Read also autochanger trucks and ONLINE
     * clamps position.
     *
     * @throws FcsHardwareException
     */
    @Override
    public synchronized void updateStateWithSensors() {

        long beginTime = System.currentTimeMillis();

        // Check if the system is in simulation mode
        if (!FcsUtils.isSimu()) {
            // Check the time elapsed since the last sync was sent on the bus
            // If this amounts to less that 100ms, use the current values and do not send a sync
            // The 100 ms rate was decided so that it would be useful for the trending and still
            // be higher than the rate at which the system is able to receive consecutive syncs
            // which is ~ 50 ms according to Guillaume Daubard
            if (beginTime - lastUpdateStateWithSensors.get() < 100) {
                FCSLOG.info(name + " do not repeat updateStateWithSensors; last one was executed less than 100ms ago.");
                return;
            }
        }
        lastUpdateStateWithSensors.set(beginTime);

        // synchronized : parallel operations should do it just once and n-1 will just
        // wait.

        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-fcs")) {
            // Carousel
            carousel.updateStateWithSensors();
            if (carousel.isAtStandby()) {
                carousel.getSocketAtStandby().updateFilterID();
            }
            // Autochanger
            autochanger.updateStateWithSensors();
            if (autochanger.getAutochangerTrucks().isAtOnline() && autochanger.isHoldingFilter()
                    && autochanger.getOnlineClamps().isLocked()) {
                String filterName = filterManager.getFilterNameByID(autochanger.getFilterID());
                updateAgentState(FilterState.valueOf("ONLINE_" + filterName.toUpperCase()));
            } else {
                updateAgentState(FilterState.valueOf("ONLINE_NONE"));
            }
            // update loader sensors is loader is connected on camera
            if (loader.isCANbusConnected()) {
                loader.updateStateWithSensors();
            }
        }
        /* publish for MCM for the first time */
        if (firstMCMpublication) {
            this.publishDataForMcm();
            firstMCMpublication = false;
        }
    }

    @Override
    public void initializeHardware() {
        carousel.initializeHardware();
        autochanger.initializeHardware();
        loader.initializeHardware();
        postStart();
        FCSLOG.info("initializeHardware done");
    }

    @Override
    public void publishData() {
        super.publishData();
        bridgeToLoader.publishData();
    }

    public void publishDataForMcm() {
        // The following is used at startup to make sure the system is awake
        if (autochanger.getAutochangerTrucks().getPosition() == 0) {
            FcsUtils.sleep(100, "main");
            autochanger.updateStateWithSensors();
        }
        updateAgentState(autochanger.getAutochangerInclinationState());
        subs.publishSubsystemDataOnStatusBus(getDataForMcm());
    }

    /**
     * publishes data needed by MCM when MCM connects.
     * @param agents
     */
    @Override
    public void publishDataProviderCurrentData(AgentInfo... agents) {
        for (AgentInfo agent : agents) {
            if (AgentInfo.AgentType.MCM.equals(agent.getType())) {
                FCSLOG.info("FCS published for MCM that just joined the buses.");
                publishDataForMcm();
                break;
            }
        }
    }

    /**
     * publish action duration for trending db
     * @param action the action to be recorded
     * @param duration duration of the action
     */
    public void publishActionDuration(GeneralAction action, long duration) {
        subs.publishSubsystemDataOnStatusBus(
            new KeyValueData(getDurationPath(action.name()), duration));
    }

    private KeyValueData getDataForMcm() {
        KeyValueDataList kvdl = new KeyValueDataList();
        kvdl.addData("filter_on_autochanger", filterManager.getOCSFilterNameByID(autochanger.getFilterID()));
        kvdl.addData("filter_previous_socketID", previousSocketID);
        kvdl.addData("autochanger_trucks_position", autochanger.getAutochangerTrucks().getPosition());
        kvdl.addData("autochanger_trucks_state", autochanger.getAutochangerTrucksState());
        kvdl.addData("setFilter_percentage", percentage);
        kvdl.addData("proximity", autochanger.getOnlineProximityDistance());
        return new KeyValueData(PUBLICATION_KEY_FOR_MCM, kvdl);
    }

    @Override
    public void postShutdown() {
        FCSLOG.info(name + " is shutting down.");
        bridge.doShutdown();
        if (loader.isCANbusConnected()) {
            bridgeToLoader.doShutdown();
        }
        FCSLOG.info(name + " is shutdown.");
    }

    @Override
    public String vetoTransitionToNormalMode() {
        if (this.isChangerReady()) {
            publishDataForMcm();
            if (autochanger.isHoldingFilter() && autochanger.isAtOnline()) {
                updateAgentState(McmFilterState.LOADED);
            } else {
                updateAgentState(McmFilterState.NOFILTER);
            }
            return super.vetoTransitionToNormalMode();
        } else {
            return "FilterChanger not ready";
        }
    }


}
