package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForOnlineClamp;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * This class is the driver for the EPOS controller which controls the
 * Autochanger ONLINE clamps. The model is EPOS2 25/5.
 *
 * Online Clamp controllers values (statusWord, position, averageCurrent, followingError and velocity)
 * are read by PDO so CanOpenEPOSOnlineClamp extends CanOpenEPOSWithPDO.
 *
 *
 * @author virieux
 */
public class CanOpenEPOSOnlineClamp extends CanOpenEPOSWithPDO implements EPOSControllerForOnlineClamp {

    private boolean homingDone = false;
    /**
     * A boolean to be set each time brake is activated or released. This
     * boolean avoid to send a command to read on the controller each time we
     * publish controller status.
     */
    private boolean brakeActivatedPub = false;

    @Override
    public void init() {
        super.init();
        hasEncoder = true;
    }

    @Override
    public void resetHoming() {
        this.homingDone = false;
    }

    @Override
    public boolean isBrakeActivatedPub() {
        return brakeActivatedPub;
    }

    @Override
    public void setBrakeActivatedPub(boolean brakeActivatedToPublish) {
        this.brakeActivatedPub = brakeActivatedToPublish;
    }

    /**
     * Release brake to be able to open or close clamp. Only for tests for end
     * users.
     *
     * releaseBrake for an ONlineClamp controller consists in forcing to ZERO bit 15
     * of parameter DigitalOutputFonctionnalityState.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Release brake to be able to move clamp.")
    public void releaseBrake() {
        // If the controller is not enabled and we release brakes it could lead to a
        // lost of a filter.
        if (!isEnabled()) {
            throw new RejectedCommandException(name + " can't releaseBrake if not enabled.");
        }
        doReleaseBrake();
    }

    @Override
    public void shutdown() {
        super.shutdown();
        if (isBooted()) {
            activateBrakeAndDisable();
        }
    }

    /**
     * return true if homing is done: the controller knows its absolute
     * position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE,
             description = "return true if homing is done: the controller knows its absolute position.")
    @Override
    public boolean isHomingDone() {
        return homingDone;
    }

    /**
     * homing of a ONLINE clamp in final products AC1 and AC2, the ONLINE clamps
     * have encoders. So we open and close them in PROFILE_POSITION mode. This
     * can be done when the homing of the controller has been done before. The
     * homing has to be done when fcs starts or each time the controller has
     * been shutdown. The homing is to be done when the clamp has been opened
     * in CURRENT mode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT,
             description = "Perform homing procedure of the clamp.")
    @Override
    public void homing() {
        homingDone = false;
        this.defineAbsolutePosition(0);
        this.checkHomingDone();
        position = readPosition();
        if (position != 0) {
            throw new FcsHardwareException(name + " could not do homing.");
        }
        homingDone = true;
        activateBrakeAndDisable();
    }

    @Override
    public void setHomingDone(){
        this.homingDone = true;
    }

    // homingStatus is read from the controller and set accordingly
    @Override
    public void updateHomingFromControllerStatusWord() {
        homingDone = checkHomingStatusOnController();
    }

}



