package org.lsst.ccs.subsystems.fcs.drivers;

import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_TRUCKS_ERROR;

import java.util.logging.Logger;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.description.ComponentLookup;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.ControlWord;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.Parameter;

import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForLinearRail;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class is the driver of the EPOS controller EPOS24 which controls motion
 * of Autochanger Trucks on the linear rails.
 *
 * @author virieux
 */
public class CanOpenEPOSLinearRailTruck extends CanOpenEPOSWithPDO implements EPOSControllerForLinearRail {
    static final Logger FCSLOG = Logger.getLogger(CanOpenEPOSLinearRailTruck.class.getName());

    @ConfigurationParameter(range = "0..10000000", units = "unitless", description = "Minimal position of the encoder ribbon.", category = "controller")
    private volatile int encoderRibbonMinValue;

    private boolean homingDone;

    private CanOpenEPOSLinearRailTruck otherAcTruckController;

    /**
     * A boolean to be set each time brake is activated or released. This
     * boolean avoid to send a command to read on the controller each time we
     * publish controller status.
     */
    private boolean brakeActivatedPub = false;

    private boolean configDigitalInputOK = true;
    private boolean configDigitalOutputOK = true;
    private StringBuilder configDigitalInputSb;
    private StringBuilder configDigitalOutputSb;

    @Override
    public boolean isBrakeActivatedPub() {
        return brakeActivatedPub;
    }

    @Override
    public void setBrakeActivatedPub(boolean brakeActivatedToPublish) {
        this.brakeActivatedPub = brakeActivatedToPublish;
    }

    @Override
    public void init() {
        super.init();
        hasEncoder = true;
    }

    @Override
    public void build() {
        super.build();
        ComponentLookup lookup = subs.getComponentLookup();
        if (getName().contains("Xminus")) {
            otherAcTruckController = (CanOpenEPOSLinearRailTruck) lookup.getComponentByPath("acTruckXplusController");
        } else {
            otherAcTruckController = (CanOpenEPOSLinearRailTruck) lookup.getComponentByPath("acTruckXminusController");
        }
    }

    /**
     * return true if homing is done: the controller knows its absolute position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE,
             description = "return true if homing is done: the controller knows its absolute position.")
    @Override
    public boolean isHomingDone() {
        return homingDone;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Do homing of the controller")
    @Override
    public void homing() {
        int absolutePosition = readSSIPosition() - encoderRibbonMinValue;
        doHoming(absolutePosition);
        publishData();
    }

    /**
     * Check if the controller position is correctly calibrated on the absolute position
     * and set homingDone flag accordingly.
     *
     * The procedure is used at FCS startup to check wether homing of the trucks needs to be done.
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check whether homing of the truck needs to be done.")
    public void checkHomingNeeded() {
        checkInitialized();
        int ssiPosition = readSSIPosition();
        int absolutePosition = ssiPosition - encoderRibbonMinValue;
        int position = readPosition();
        String msg = (
            getName() +
            " | encoderRibbonMinValue (configuration param)=" + encoderRibbonMinValue +
            ", SSI position=" + ssiPosition +
            ", Absolute position (ssi - ribbon value)=" + absolutePosition +
            ", Controller position=" + position
        );
        FCSLOG.finer(() -> msg);
        if (Math.abs(absolutePosition - position) < 10) {
            FCSLOG.info(getName() + " autochanger truck position is well calibrated with the SSIencoder.");
            homingDone = true;
        } else {
            FCSLOG.info(getName() + " homing should be done before moving.");
            homingDone = false;
            String error_msg = (
                "Homing of the autochanger " + getName() + " truck is wrong or has been lost.\n" +
                "This is an abnormal situation, we should not have to redo the homing of the autochanger trucks.\n" +
                "The most probable cause is a wrong value of the encoderRibbonMinValue in the configuration. \n" +
                "Contact a FES expert with the following information\n" +
                msg
            );
            raiseAlarm(AC_TRUCKS_ERROR, error_msg);
        }
    }

    private void doHoming(int position) {
        FCSLOG.finer(() -> getName() + " ==> BEGIN homing");
        EposMode myMode = readMode();
        goToOperationEnable();
        defineAbsolutePosition(position);
        try {
            checkHomingDone(position);
            homingDone = true;
            FCSLOG.finer(() -> getName() + " ==> END homing");
        } catch ( FcsHardwareException e ) {
            homingDone = false;
            FCSLOG.warning(getName() + " ==> END homing with error: \n" + e.getMessage());
            throw e;
        } finally {
            changeMode(myMode);
            activateBrakeAndDisable();
        }

    }

    private void checkHomingDone(int position) {
        FcsUtils.waitCondition(
            () -> isTargetReached(),
            null,
            "checkTrucksHomingCompleted",
            500
        );
        if (isTargetReached() && position == readPosition()) {
            // Consider homing completed
            writeControlWord(ControlWord.SWITCH_ON);
        } else {
            throw new FcsHardwareException(getName() + " homing procedure could not be completed: target is not reached after 500 ms");
        }
    }

    @Override
    public void faultReset() {
        super.faultReset();
        otherAcTruckController.updateEposState();
        if (otherAcTruckController.getEposState() != EPOSEnumerations.EposState.FAULT) {
            otherAcTruckController.activateBrakeAndDisable();
            otherAcTruckController.publishData();
        }
    }

    /**
     * Configure this controller to be used as an autochanger linear rail truck
     * controller. This method is to be executed during initialization phase.
     *
     * This configures the digital inputs of the EPOS controller to work with a
     * holding brake.
     *
     * Digital Inputs activated for Linear Rails :
     * **************************************************************************
     * Drive Enable Input=01 Mask=Enabled Polarity=HighActive
     * QuickStop Input=02 Mask=Enabled Polarity=HighActive
     * **************************************************************************
     * 1 - Configure the Digital Input Functionality index 0x2070 subindex 01 to 04
     * Functionality tab Value = 15 General Purpose A Value = 14 General Purpose B
     * Value = 13 General Purpose C Value = 12 General Purpose D Value = 11 General
     * Purpose E Value = 10 General Purpose F Value = 9 General Purpose G Value = 8
     * General Purpose H Value = 7 General Purpose I Value = 6 General Purpose J
     * Value = 5 QuickStop 1 Value = 4 Drive enable 1 Value = 3 Position marker 0
     * Value = 2 Home switch 0 Value = 1 Positive Limit switch 0 Value = 0 Negative
     * Limit switch 0
     * **************************************************************************
     * 2 - Configure the Digital Input Polarity index 0x2071 subindex 03 Value=0
     * HighActive Value=1 Low Active
     * **************************************************************************
     * 3 - Configure the Digital Input Mask index 0x2071 subindex 02 Value=0
     * HighActive Value=1 Low Active
     * **************************************************************************
     * 4 - Configure the Digital Input Execution Mask index 0x2071 subindex 04
     * Used to suppress execution of the digital input functionalities.
     * The function (Negative or Positive Limit Switch Error Routine) will be
     * executed when the associated bit in functionalities state register goes high
     * and the bit in this execution mask is set.
     * Functionality tab Value = 15 General Purpose A 0 Value = 14
     * General Purpose B 0 Value = 13 General Purpose C 0 Value = 12 General Purpose
     * D 0 Value = 11 General Purpose E 0 Value = 10 General Purpose F 0 Value = 9
     * General Purpose G 0 Value = 8 General Purpose H 0 Value = 7 General Purpose I
     * 0 Value = 6 General Purpose J 0 Value = 5 QuickStop 1 Value = 4 Drive enable
     * 1 Value = 3 Position marker 0 Value = 2 Home switch 0 Value = 1 Positive
     * Limit switch 0 Value = 0 Negative Limit switch 0
     *
     *
     *
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "To configure the autochanger linear rails controllers.")
    @Override
    public void configureDigitalInputOfLinearRails() {
        // Drive Enable
        writeParameter(Parameter.ConfigurationOfDigitalInput1, 0x4);

        // QuickStop
        writeParameter(Parameter.ConfigurationOfDigitalInput2, 0x5);

        // DigitalInputPolarity
        // Value=0 HighActive
        // Value=1 Low Active
        writeParameter(Parameter.DigitalInputFonctionnalityPolarity, 0x0);

        // DigitalInputFonctionnalityMask
        // Used to display the state of the digital input functionalities.
        // If a bit is set to “1”, the functionality state will be displayed.
        writeParameter(Parameter.DigitalInputFonctionnalityMask, 0x30);

        // DigitalInputFonctionnalityExecutionMask
        writeParameter(Parameter.DigitalInputFonctionnalityExecutionMask, 0x30);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "check configuration of digital input of linear rail controller.")
    public void checkConfigDigitalInputOfLinearRails() {
        configDigitalInputSb = new StringBuilder();
        configDigitalInputSb.append(" Configuration of Digital Input:");
        checkParamInput(Parameter.ConfigurationOfDigitalInput1, 4);
        checkParamInput(Parameter.ConfigurationOfDigitalInput2, 5);
        checkParamInput(Parameter.DigitalInputFonctionnalityPolarity, 0);
        checkParamInput(Parameter.DigitalInputFonctionnalityMask, 0x30);
        checkParamInput(Parameter.DigitalInputFonctionnalityExecutionMask, 0x30);
        if (configDigitalInputOK) {
            FCSLOG.info(configDigitalInputSb.toString());
        } else {
            FCSLOG.severe(configDigitalInputSb.toString());
        }
    }

    private void checkParamInput(Parameter param, int configValue) {
        int paramValue = (int) this.readParameter(param);
        if (configValue != paramValue) {
            configDigitalInputOK = false;
        }
        configDigitalInputSb.append(param).append("=");
        configDigitalInputSb.append(paramValue).append(";");
    }

    /**
     * Configuration of an EPOS controller to work with a holding brake. This is
     * used for controllers of the Autochanger Linear Rails.
     *
     * Digital Outputs activated for Linear Rails are: General_A
     * Output=01 Mask=Enabled Polarity=HighActive Ready/Fault
     * Output=02 Mask=Enabled Polarity=HighActive Holding Brake
     * Output=03 Mask=Enabled Polarity=LowActive
     * ********************************************************************************
     * 1- First of all, mask all outputs index 0x2079 subindex 01
     * ********************************************************************************
     * 2- Configure the Digital Output Functionalities index 0x2079 subindex 01 to 03
     * Functionality tab Value = 15 General Purpose Out_A Value = 14 General
     * Purpose Out_B Value = 13 General Purpose Out_C Value = 12 General Purpose
     * Out_D Value = 11 General Purpose Out_E Value = 10..8 not used Value = 7..3
     * reserved Value = 2 Holding Brake Value = 1 Position compare Value = 0
     * Ready/Fault
     * ********************************************************************************
     * 3 - Configure the Digital Output Functionality Polarity index 0x2078
     * subindex 0x03 Value = 0 associated output not change => HighActive Value = 1
     * associated output inverted => LowActive Tab Digital Output Functionality
     * Mask Value Bit_15 General Purpose Out_A 0 Bit_14 General Purpose Out_B 0
     * Bit_13 General Purpose Out_C 0 Bit_12 General Purpose Out_D 0 Bit_11 General
     * Purpose Out_E 0 Bit_10 General Purpose Out_F 0 Bit_9 General Purpose Out_G 0
     * Bit_8 General Purpose Out_H 0 Bit_7 reserved 0 Bit_6 reserved 0 Bit_5
     * reserved 0 Bit_4 reserved 0 Bit_3 reserved 0 Bit_2 Holding Brake 1 (read
     * only) Bit_1 Position Compare 0 (read only) Bit_0 Ready/Fault 0 (read only)
     * ********************************************************************************
     * 4 - Configure the Digital Output Functionality Mask index 0x2078 subindex
     * 0x02 Value = 0 functionality not activated Value = 1 functionality activated
     * ********************************************************************************
     * Digital Output Functionality Mask Value Bit_15 General Purpose Out_A 1
     * Bit_14 General Purpose Out_B 0 Bit_13 General Purpose Out_C 0 Bit_12 General
     * Purpose Out_D 0 Bit_11 General Purpose Out_E 0 Bit_10 General Purpose Out_F 0
     * Bit_9 General Purpose Out_G 0 Bit_8 General Purpose Out_H 0 Bit_7 reserved 0
     * Bit_6 reserved 0 Bit_5 reserved 0 Bit_4 reserved 0 Bit_3 reserved 0 Bit_2
     * Holding Brake 1 Bit_1 Position Compare 0 Bit_0 Ready/Fault 1
     * ********************************************************************************
     * To configure the autochanger linear rails controllers. This method is to be
     * executed during initialization phase.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "To configure the autochanger linear rails controllers.")
    // tested on CPPM test bench in september 2015
    @Override
    public void configureDigitalOutputOfLinearRails() {
        // 1-Mask all outputs
        writeParameter(Parameter.DigitalOutputFonctionnalityMask, 0);

        // 2-Configuration of Digital Output Functionalities
        // General_A
        writeParameter(Parameter.ConfigurationOfDigitalOutput1, 0xF);
        // Ready/Fault
        writeParameter(Parameter.ConfigurationOfDigitalOutput2, 0);
        // Holding Brake
        writeParameter(Parameter.ConfigurationOfDigitalOutput3, 0x2);

        // 3-Configuration of Digital Output Fonctionnality Polarity
        writeParameter(Parameter.DigitalOutputFonctionnalityPolarity, 0x4);

        // 4-Configuration of Digital Output Fonctionnality Mask
        writeParameter(Parameter.DigitalOutputFonctionnalityMask, 0x8005);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "check configuration of digital output of linear rail controller.")
    public void checkConfigDigitalOutputOfLinearRails() {
        configDigitalOutputSb = new StringBuilder();
        configDigitalOutputSb.append(" Configuration of Digital Output:");
        checkParamOutput(Parameter.ConfigurationOfDigitalOutput1, 0xf);
        checkParamOutput(Parameter.ConfigurationOfDigitalOutput2, 0);
        checkParamOutput(Parameter.ConfigurationOfDigitalOutput3, 2);
        checkParamOutput(Parameter.DigitalOutputFonctionnalityPolarity, 4);
        checkParamOutput(Parameter.DigitalOutputFonctionnalityMask, 0x8005);
        if (configDigitalOutputOK) {
            FCSLOG.info(configDigitalOutputSb.toString());
        } else {
            FCSLOG.severe(configDigitalOutputSb.toString());
        }
    }

    private void checkParamOutput(Parameter param, int configValue) {
        int paramValue = (int) this.readParameter(param);
        if (configValue != paramValue) {
            configDigitalOutputOK = false;
        }
        configDigitalOutputSb.append(param).append("=");
        configDigitalOutputSb.append(paramValue).append(";");
    }

    @Override
    public void shutdown() {
        // shutdown of linear rail controllers is done in AutochangerTwoTrucks
    }


}
