package org.lsst.ccs.subsystems.fcs;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.subsystems.fcs.common.BinarySensor;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.HasLifecycle;

/**
 * Represents a couple of DigitalSensor which are complementary : when one has
 * the value 1 the other must have value 0 and vice versa. If both sensors have
 * the same value, the couple of sensors is in error.
 *
 * This sort of sensors are used in autochanger. The goal is to be able to
 * detect errors in sensors (capteur défectueux, error de câblage)
 *
 * @author virieux
 */
public class ComplementarySensors implements BinarySensor, HasLifecycle {

    @LookupPath
    private String path;

    @LookupField(strategy = LookupField.Strategy.CHILDREN)
    private final List<DigitalSensor> sensors = new ArrayList<>();

    /**
     * first sensor in the couple of complementary sensors.
     */
    private DigitalSensor sensor;

    /**
     * complementary sensor.
     */
    private DigitalSensor sensorC;

    //Public constructor for groovy
    public ComplementarySensors() {

    }

    //Constructor for tests
    ComplementarySensors(DigitalSensor s1, DigitalSensor s2) {
        sensors.add(s1);
        sensors.add(s2);
        loadSensors(sensors);
    }

    @Override
    public void build() {
        loadSensors(sensors);
    }

    final void loadSensors(List<DigitalSensor> sensors) {
        if ( sensors.size() != 2 ) {
            throw new RuntimeException("A ComplementarySensors instance requires two DigitalSensors as its children. For "+path+" there are "+sensors.size());
        }
        sensor = sensors.get(0);
        sensorC = sensors.get(1);
    }

    /**
     * for simulation
     *
     * @return
     */
    public DigitalSensor getSensor() {
        return sensor;
    }

    /**
     * for simulation
     *
     * @return
     */
    public DigitalSensor getSensorC() {
        return sensorC;
    }

    /**
     *
     * @return true if this couple of sensors are in error.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if this couple of sensors are in error.")
    @Override
    public boolean isInError() {
        return sensor.isOn() == sensorC.isOn();
    }

    /**
     *
     * @return true if sensor's value is 1 and sensorC's value is 0
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if sensor's value is 1 and sensorC's value is 0.")
    @Override
    public boolean isOn() {
        return sensor.isOn() && !sensorC.isOn();
    }

}
