package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.ANCESTORS;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CAN_BUS_READING_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_ONLINE_CLAMPS_CLOSED_AT_STARTUP;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_ONLINE_CLAMPS_STAY_ENABLED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.CLOSED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.INTRAVEL;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.LOCKED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.OPENED;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.LockStatus.UNKNOWN;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.checkAndWaitConditionWithTimeoutAndFixedDelay;

import java.time.Duration;
import java.util.Arrays;
import java.util.concurrent.ScheduledFuture;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.ConfigurationParameterChanger;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.LookupPath;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.common.ControlledBySensors;
import org.lsst.ccs.subsystems.fcs.common.StrainGauge;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenSeneca4RTD;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils.AutoTimed;


/**
 * The Three Online clamps which holds a filter when it is at ONLINE position
 * can ve viewed as a single object. It's the goal of this class to represent
 * this object. Opening or closing the 3 clamps at ONLINE is not the same action
 * as opening or closing online clamps one by one.
 *
 * @author virieux
 */
/**
 * Represents the set of the 3 onlineClamps of autochanger. This class gathers
 * the method to send command to the 3 onlineClamps all together.
 *
 * It is no more a MobileItem since we use AutochangerOnlineClamp commands.
 *
 * @author virieux
 */
public class AutochangerThreeOnlineClamps implements ControlledBySensors, SignalHandler, HasLifecycle {
    private static final Logger FCSLOG = Logger.getLogger(AutochangerThreeOnlineClamps.class.getName());

    @LookupName
    protected String name;

    @LookupPath
    protected String path;

    @LookupField(strategy = ANCESTORS)
    private Subsystem subs;

    @LookupField(strategy = TREE)
    protected Autochanger autochanger;

    @LookupField(strategy = TREE, pathFilter = ".*\\/onlineStrainGauge")
    private StrainGauge onlineStrainGauge;

    @LookupField(strategy = TREE, pathFilter = ".*\\/tempSensorsDevice2")
    private CanOpenSeneca4RTD tempSensorsDevice2;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    protected final AutochangerOnlineClamp onlineClampXminus;
    protected final AutochangerOnlineClamp onlineClampXplus;
    protected final AutochangerOnlineClamp onlineClampYminus;
    protected final AutochangerOnlineClamp[] clampsList;

    private FcsEnumerations.LockStatus lockStatus = UNKNOWN;

    /* Tools for current ramp */
    protected ScheduledFuture<?> currentRampHandle;

    @LookupField(strategy = TREE)
    private AgentPeriodicTaskService periodicTaskService;

    @ConfigurationParameter(range = "0..30000", description = "Timeout in milliseconds : if closing the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.", units = "millisecond", category = "autochanger")
    protected volatile int timeoutForLockingClamps = 15000;

    @ConfigurationParameter(range = "0..30000", description = "Timeout in milliseconds : if unlocking the clamps last more "
            + "than this amount of time, then the subsystem goes in ERROR.", units = "millisecond", category = "autochanger")
    protected volatile int timeoutForUnlockingClamps = 20000;

    @ConfigurationParameter(range = "0..5000", description = "Maximum time in milliseconds to lock the 3 clamps", units = "millisecond", category = "autochanger")
    protected volatile int maxTimeToLockAllClamps = 3000;

    @ConfigurationParameter(range = "0..5000", description = "Maximum time in milliseconds to unlock the 3 clamps", units = "millisecond", category = "autochanger")
    protected volatile int maxTimeToUnlockAllClamps = 3000;

    @ConfigurationParameter(range = "0..5000", description = "Maximum time in milliseconds to open the 3 clamps", units = "millisecond", category = "autochanger")
    protected volatile int maxTimeToOpenClampsX = 2000;

    @ConfigurationParameter(range = "0..5000", description = "Maximum time in milliseconds to close the 3 clamps", units = "millisecond", category = "autochanger")
    protected volatile int maxTimeToCloseClampsX = 2000;

    @ConfigurationParameter(range = "50..1000", description = "Minimal period for current ramps. Should be > 50", units = "millisecond", category = "autochanger")
    protected volatile int minPeriod = 200;

    /**
     * **************************************************
     */
    /* configurable parameters for onlineStrainGauge */
    /* if maxLockedStrain <= strain < maxClosedStrain, clamps are CLOSED */
    /* if strain >= maxClosedStrain, clamps are OPENED */
    @ConfigurationParameter(range = "-10000..10000", units = "mV", description = "If strain superior to maxClosedStrain, clamps are OPENED", category = "autochanger")
    private volatile short maxClosedStrain = 1572;

    /* If strain < maxLockedStrain, clamps are LOCKED */
    @ConfigurationParameter(range = "-10000..10000", units = "mV", description = "If strain strictly inferior to maxLockedStrain, clamps are LOCKED", category = "autochanger")
    private volatile short maxLockedStrain = 1484;

    /* If strain < minLockedStrain, clamps may be in ERROR */
    @ConfigurationParameter(range = "-10000..10000", units = "mV", description = "If strain strictly inferior to minLockedStrain, clamps may be in ERROR", category = "autochanger")
    private volatile short minLockedStrain = 500;


    @ConfigurationParameter(range = "0..5", units = "unitless", description = "Gain to convert from raw strain to temperature-aware strain", category = "autochanger")
    private volatile double strainGain = 2.978;

    /*
     * strain read on strainGauge as a voltage. It measures the deformation of
     * onlineClampXplus on PROTO and onlineClampXminus on AC1 and AC2. Units = mV
     */
    private short rawStrain = 0;

    /* deformation computed from rawStrain and temperature */
    private short normalizedStrain = 0;

    /**
     * Used to know if lockStatus has been initialized from gauge strain. At startup
     * this value is false. At first updateState this value is set to true.
     */
    private boolean lockStatusInitialized = false;

    /**
     * **************************************************
     */
    /**
     * Create a AutochangerThreeOnlineClamps with 3 AutochangerOnlineClamp.
     *
     * @param onlineClampXminus
     * @param onlineClampXplus
     * @param onlineClampYminus
     */
    public AutochangerThreeOnlineClamps(AutochangerOnlineClamp onlineClampXminus,
            AutochangerOnlineClamp onlineClampXplus, AutochangerOnlineClamp onlineClampYminus) {
        this.onlineClampXminus = onlineClampXminus;
        this.onlineClampXplus = onlineClampXplus;
        this.onlineClampYminus = onlineClampYminus;
        clampsList = new AutochangerOnlineClamp[] { onlineClampYminus, onlineClampXminus, onlineClampXplus };
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerThreeClamps.class, path);
    }

    @Override
    public void init() {

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(AC_SENSOR_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(CAN_BUS_READING_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(AC_SENSOR_ERROR.getAlert(), alwaysClear);
        alertService.registerAlert(AC_ONLINE_CLAMPS_CLOSED_AT_STARTUP.getAlert(), alwaysClear);
        alertService.registerAlert(AC_ONLINE_CLAMPS_STAY_ENABLED.getAlert(), alwaysClear);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if lockStatus has been first initialized from strain.")
    public boolean isLockStatusInitialized() {
        return lockStatusInitialized;
    }

    /**
     *
     * @return true if homing has been done for the 3 controllers of clamps.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if homing has been done for the 3 ONLINE clamp controllers.")
    public boolean isHomingDone() {
        return onlineClampXminus.isHomingDone() && onlineClampXplus.isHomingDone() && onlineClampYminus.isHomingDone();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Reads the homing status on each controller of the online clamps. Returns true if all status are true.")
    public boolean checkHomingStatusOnController() {
        return onlineClampXminus.checkHomingStatusOnController() && onlineClampXplus.checkHomingStatusOnController() && onlineClampYminus.checkHomingStatusOnController();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
        description = "Reads the homing status on each controller of the online clamps. Update homingDone in FCS according " +
            "to the results. If all are true, homingDone is set to true, otherwise it is set to false.")
    public String updateHomingStatusAccordingToController() {
        String message = "Homing status";
        boolean fcsHomingDoneStatus = isHomingDone();
        boolean controllerHomingDoneStatus = checkHomingStatusOnController();
        if (fcsHomingDoneStatus == controllerHomingDoneStatus) {
            message += fcsHomingDoneStatus ? ": done":"undone" + " (already up to date).";
        } else {
            onlineClampXminus.updateHomingFromControllerStatusWord();
            onlineClampXplus.updateHomingFromControllerStatusWord();
            onlineClampYminus.updateHomingFromControllerStatusWord();
            message += controllerHomingDoneStatus ? " is set to done.":" is set to undone.";
        }
        return message;
    }


        /**
         * set a minimal value for current ramps period
         *
         * @param minPeriod
         */
    @ConfigurationParameterChanger
    public void setMinPeriod(int minPeriod) {
        if (minPeriod < 50) {
            throw new IllegalArgumentException(minPeriod + " bad value for minPeriod. should be > 50");
        } else {
            this.minPeriod = minPeriod;
        }
    }

    /**
     *
     * @return lockStatus
     */
    public FcsEnumerations.LockStatus getLockStatus() {
        return lockStatus;
    }

    /**
     * Returns true if LockStatus=LOCKED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if the 3 clamps are LOCKED.")
    public boolean isLocked() {
        return onlineClampXminus.isLocked() && onlineClampXplus.isLocked() && onlineClampYminus.isLocked();
    }

    /**
     * Returns true if LockStatus=CLOSED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if the 3 clamps are CLOSED.")
    public boolean isClosed() {
        return onlineClampXminus.isClosed() && onlineClampXplus.isClosed() && onlineClampYminus.isClosed();
    }

    /**
     * Returns true if any clamp's LockStatus=CLOSED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if any of the 3 clamps is CLOSED.")
    public boolean isAnyClosed() {
        return Arrays.stream(clampsList).anyMatch(c -> c.isClosed());
    }

    /**
     * Returns true if all clamp controller are disabled.
     *
     * @return
     */
    public boolean areAllDisabled() {
        return Arrays.stream(clampsList).allMatch(c -> !c.getController().isEnabled());
    }

    /**
     * Returns true if LockStatus=OPENED
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if the 3 clamps are OPENED.")
    public boolean isOpened() {
        return onlineClampXminus.isOpened() && onlineClampXplus.isOpened() && onlineClampYminus.isOpened();
    }

    /**
     * @return true if the 3 clamps are in Travel between opened position and closed
     *         position.
     */
    private boolean isInTravel() {
        return onlineClampXminus.isInTravel() || onlineClampXplus.isInTravel() || onlineClampYminus.isInTravel();
    }

    /**
     * Returns true if LockStatus=ERROR
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns true if one of the clamp is in error.")
    @Override
    public boolean isInError() {
        return onlineClampXminus.isInError() || onlineClampXplus.isInError() || onlineClampYminus.isInError();
    }

    /**
     * Return true if the 3 onlineClamps hardware is lockStatusInitialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the 3 onlineClamps hardware is ready : controllers ready, "
            + "controllers parameters checked and controllers configured.")
    public boolean isInitialized() {
        return onlineClampXminus.isInitialized() && onlineClampXplus.isInitialized()
                && onlineClampYminus.isInitialized();
    }

    /**
     * Perform homing of the 3 online clamps
     * The process requires the opening of the clamps and needs therefore
     * to distinguish between the cases where the clamps are already opened or not.
     * When the clamps are not opened, the opening cannot be done simultaneously and
     * Y- clamp should be opened first.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Do homing of the 3 ONLINE clamps : open in CURRENT mode and homing of controller", timeout = 6000)
    public void homing() {
        try (AutoTimed at = new AutoTimed("online clamps homing")) {
            if ( isClosed() || isInTravel() ) {
                onlineClampYminus.openInCurrentModeAndHoming();
                FcsUtils.parallelRun(
                    () -> onlineClampXminus.openInCurrentModeAndHoming(),
                    () -> onlineClampXplus.openInCurrentModeAndHoming()
                );
            } else if ( isOpened() ) {
                FcsUtils.AsyncTasks asyncRun = FcsUtils.asyncRun();
                for (AutochangerOnlineClamp clamp : clampsList) {
                    asyncRun.asyncRun(() -> clamp.openInCurrentModeAndHoming());
                }
                asyncRun.await();
            } else {
                throw new RejectedCommandException(name + " online clamps should be OPENED or CLOSED before attempting homing.");
            }
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Perform homing of the three online clamps : open and current threshold homing", timeout = 10000)
    public void homingCurrentThreshold() {
        try (AutoTimed at = new AutoTimed("Three online clamps homing with current threshold")) {
            if ( isClosed() || isInTravel() ) {
                onlineClampYminus.openAndHoming();
                FcsUtils.parallelRun(
                    () -> onlineClampXminus.openAndHoming(),
                    () -> onlineClampXplus.openAndHoming()
                );
            } else if ( isOpened() ) {
                FcsUtils.AsyncTasks asyncRun = FcsUtils.asyncRun();
                for (AutochangerOnlineClamp clamp : clampsList) {
                    asyncRun.asyncRun(() -> clamp.openAndHoming());
                }
                asyncRun.await();
            } else {
                throw new RejectedCommandException(name + " online clamps should be OPENED or CLOSED before attempting homing.");
            }
        }
    }

    /**
     * test periodicTask.
     */
    public void testPeriodTask() {
        periodicTaskService.scheduleAgentPeriodicTask(
                new AgentPeriodicTask(name + "-updateStateAndCheckSensors", this::updateStateAndCheckSensors)
                        .withIsFixedRate(true).withLogLevel(Level.WARNING).withPeriod(Duration.ofMillis(5000)));
    }

    /**
     * close clamps in mode PROFILE_POSITION. for AC1 and AC2
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Close the three online clamps in profile position mode. Homing of the clamps is required", timeout = 10000)
    public void closeClamps() {
        try (AutoTimed at = new AutoTimed("closeClamps")) {
            if (!isHomingDone()) {
                throw new RejectedCommandException(name + " homing not done yet. Can't close.");
            }
            updateStateAndCheckSensors();

            if (isOpened()) {
                autochanger.checkConditionsForActioningOnlineClamps();
                onlineClampYminus.close();
                FcsUtils.parallelRun(() -> onlineClampXminus.close(), () -> onlineClampXplus.close());

            } else if (isClosed()) {
                FCSLOG.info(name + " clamps already CLOSED nothing to do");

            } else {
                // TODO rajouter le cas inTravel()
                throw new RejectedCommandException(name + " has to be unlocked before.");
            }
        }
    }

    /**
     * Opens the 3 clamps. initial state = CLOSED final state = OPENED. For final
     * products AC1 and AC2. For prototype see openClampsInCurrentMode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Open the three online clamps in profile position mode. Homing of the clamps is required", timeout = 10000)
    public void openClamps() {
        try (AutoTimed at = new AutoTimed("openClamps")) {
            if (!isHomingDone()) {
                throw new RejectedCommandException(name + " homing not done yet for the 3 clamps. Can't open.");
            }
            updateStateAndCheckSensors();
            if (isClosed()) {
                onlineClampYminus.checkReadyForAction();
                // First open clampY
                onlineClampYminus.open();
                // Then open clampX in parallel.
                FcsUtils.parallelRun(() -> onlineClampXminus.open(), () -> onlineClampXplus.open());

                // we brake + disable on clamp lock and clamp open

            } else if (isOpened()) {
                FCSLOG.info(name + " clamps already OPENED. Nothing to do.");

            } else {
                throw new RejectedCommandException(name + " has to be closed before.");
            }
        }
    }

    /**
     * Locks clamps: closed with a strong pressure (high current). The clamps have
     * to be CLOSED. At the end of this action, the clamps are CLOSED but a strong
     * pressure to hold safely the clamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Lock the three online clamps for good positioning of the filter", timeout = 10000)
    public void lockClamps() {
        try (AutoTimed at = new AutoTimed("lockClamps")) {
            updateStateAndCheckSensors();
            if (isClosed()) {
                // Here again not possible to do the 3 in parallel for the repeatability of the positioning
                onlineClampYminus.lock();
                FcsUtils.parallelRun(() -> onlineClampXminus.lock(), () -> onlineClampXplus.lock());
                // Brakes and goToSwitchOnDisabled are now handled by the individual .lock()

                // We check all controllers are disabled
                try {
                    checkAndWaitConditionWithTimeoutAndFixedDelay(
                        () -> areAllDisabled(),
                        () -> this.updateStateAndCheckSensors(),
                        "waitForCurrentDisabled",
                        "At least one online clamp controller is still enabled after lockClamps().",
                        250L,
                        100L
                    );
                } catch (RejectedCommandException e) {
                    String msg = "Autochanger Online Clamps Controllers in an unexpected state:\n"
                        + "They should have been disabled after lock(), but at least one has been found enabled.\n"
                        + "This state need to be reviewed by an expert.\n"
                        + "You should contact a FES expert with the debug information below to confirm the state of the Autochanger Online Clamps Controllers, and decide what to do next.\n"
                        + "Debug info:\n"
                        + "Yminus: " + (onlineClampYminus.getController().isEnabled() ? "enabled":"disabled") + ", "
                        + "Xminus: " + (onlineClampXminus.getController().isEnabled() ? "enabled":"disabled") + ", "
                        + "Xplus: "  + (onlineClampXplus.getController().isEnabled()  ? "enabled":"disabled");
                    raiseAlarm(AC_ONLINE_CLAMPS_STAY_ENABLED, msg);
                    throw e;
                }
            } else if (isLocked()) {
                FCSLOG.info(name + " is already LOCKED. Nothing to do.");
            } else {
                throw new RejectedCommandException(name + " have to be CLOSED before being locked.");
            }
        }
    }

    /**
     * Unlocks clamps : slows down current sent to controller in order to decrease
     * pressure on the clamps. The clamps have to be LOCKED. At the end of this
     * action, the clamps are CLOSED with a small pressure of the clamp hardware on
     * the filter frame.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Unlock the three online clamps to put them in closed position and disabled", timeout = 10000)
    public void unlockClamps() {
        try (AutoTimed at = new AutoTimed("unlockClamps")) {
            updateStateAndCheckSensors();
            if (isLocked()) {
                onlineClampYminus.unlock();
                FcsUtils.parallelRun(() -> onlineClampXminus.unlock(), () -> onlineClampXplus.unlock());

                FcsUtils.parallelRun(() -> onlineClampYminus.getController().goToSwitchOnDisabled(),
                () -> onlineClampXminus.getController().goToSwitchOnDisabled(),
                () -> onlineClampXplus.getController().goToSwitchOnDisabled());

            } else if (isClosed()) {
                FCSLOG.info(name + " is already CLOSED. Nothing to do.");

            } else {
                throw new RejectedCommandException(name + " have to be LOCKED before being unlocked.");
            }
        }
    }

    /**
     * to be used by setFilter
     *
     */
    public void lockFilterAtOnline() {
        try (AutoTimed at = new AutoTimed("lockFilterAtOnline")) {
            if (isLocked()) {
                FCSLOG.info(name + " onlineClamps LOCKED : nothing to do");
            } else if (isClosed()) {
                lockClamps();
            } else if (isOpened()) {
                if (!isHomingDone()) {
                    homing();
                }
                closeClamps();
                lockClamps();
            }
        }
    }

    /**
     * Use by command unclampAndMoveFilterToHandoff. For tests only.
     */
    public void unlockAndOpen() {
        if (isLocked()) {
            unlockClamps();
        }
        if (isClosed()) {
            // TODO : do a checkHomingAtStartup here
            if (isHomingDone()) {
                openClamps();
            } else {
                /* command homing open clamps in CURRENT mode*/
                homing();
            }
        }
    }

    /**
     * Return true if the 3 onlineClamps hardware is ready.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the 3 onlineClamps hardware is ready.")
    public boolean myDevicesReady() {
        return onlineClampXminus.myDevicesReady() && onlineClampXplus.myDevicesReady()
                && onlineClampYminus.myDevicesReady();
    }

    /**
     * Return temperature-aware strain value from the online clamps strain gauge
     *
     * For the PROTO, the strain gauge and temperature from the Xplus clamp is used.
     * For AC1 and AC2, the strain gauge and temperature from the Xminus clamp is used.
     *
     * Because computeNormalizedStrain is called in postStart we can't use
     * temp = tempClampMotorXplus.getValue() / 10
     *
     * @return
     */
    private short computeNormalizedStrain() {
        /*
         */
        // TODO: made it work with the AC1 and AC2 but the PROTO won't work with this code
        double temp = 20;
        /* give a chance to tempSensorsDevice2 to be alive */
        if (!tempSensorsDevice2.isBooted()) {
            tempSensorsDevice2.updateDeviceInfo();
        }
        if (tempSensorsDevice2.isBooted()) {
            temp = (double) tempSensorsDevice2.readChannel(1);
        } else {
            this.raiseWarningOnlyIfNew(AC_SENSOR_ERROR,
                    " can't read temperature because tempSensorsDevice2 is not booted,"
                            + " temperature is supposed to be 20°C",
                    name);
        }
        /* give a chance to onlineStrainGauge to be alive */
        if (!onlineStrainGauge.isBooted()) {
            onlineStrainGauge.updateDeviceInfo();
        }
        if (onlineStrainGauge.isBooted()) {
            rawStrain = onlineStrainGauge.readStrain();
        } else {
            this.raiseWarningOnlyIfNew(AC_SENSOR_ERROR,
                    " can't read strainGain because onlineStrainGauge is not booted", name);
        }
        normalizedStrain = (short) ((double) rawStrain - (strainGain * (temp - 20)));
        FCSLOG.finer("temperature=" + temp + " / ONLINE read strain=" + rawStrain + " / normalized strain="
                + normalizedStrain);
        return normalizedStrain;
    }

    /**
     * Creates an object to be published on the status bus.
     *
     * @return
     */
    public StatusDataPublishedByAutochangerThreeClamps createStatusDataPublishedByThreeClamps() {
        StatusDataPublishedByAutochangerThreeClamps status = new StatusDataPublishedByAutochangerThreeClamps();
        status.setLockStatus(lockStatus);
        status.setOnlineClampStrain(normalizedStrain);
        status.setOnlineClampRawStrain(rawStrain);
        status.setHomingDone(isHomingDone());
        return status;
    }

    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(path, createStatusDataPublishedByThreeClamps()));
    }

    protected void updateStateAndCheckSensors() {
        autochanger.updateStateWithSensors();
        checkSensors(AC_SENSOR_ERROR, name);
    }

    /**
     * This methods updates lockStatus from the values return by the sensors.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Update state from sensors values.")
    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-ac3clamps")) {
            for (AutochangerOnlineClamp clamp : clampsList) {
                clamp.updateState();
            }
            normalizedStrain = computeNormalizedStrain();
            computeLockStatus();
            this.publishData();
        }
    }

    /**
     * compute lockStatus from the 3 ONLINE clamps lockStatus.
     */
    private void computeLockStatus() {
        if (isInError()) {
            this.lockStatus = ERROR;

        } else if (isOpened()) {
            this.lockStatus = OPENED;

        } else if (isClosed()) {
            this.lockStatus = CLOSED;

        } else if (isLocked()) {
            this.lockStatus = LOCKED;

        } else if (isInTravel()) {
            this.lockStatus = INTRAVEL;

        } else {
            this.lockStatus = UNKNOWN;
        }

    }

    /**
     * return a lock status computed from strain read on onlineStrainGauge.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return a lock status computed from the normalizedStrain value")
    public FcsEnumerations.LockStatus computeStrainGaugeLockStatus() {
        normalizedStrain = computeNormalizedStrain();
        FCSLOG.info(name + " ONLINE strain = " + normalizedStrain);

        if (normalizedStrain > maxClosedStrain) {
            return OPENED;

        } else if (normalizedStrain > maxLockedStrain) {
            return CLOSED;

        } else if (normalizedStrain > minLockedStrain) {
            return LOCKED;

        } else {
            return ERROR;
        }
    }

    /**
     * Compute lockStatus in reading gauge strain. This is done only the first time
     * updateState is called. After computeLockStatus is called.
     */
    private void computeLockStatusFromStrain() {

        FcsEnumerations.LockStatus strainLockStatus = computeStrainGaugeLockStatus();
        if (isInError()) {
            this.lockStatus = ERROR;

        } else if (isOpened()) {
            this.lockStatus = OPENED;

        } else if (isInTravel()) {
            this.lockStatus = INTRAVEL;

        } else if (isClosed()) {
            if (strainLockStatus == LOCKED) {
                this.lockStatus = LOCKED;

            } else if (strainLockStatus == CLOSED) {
                this.lockStatus = CLOSED;

            } else if (strainLockStatus == OPENED || strainLockStatus == ERROR) {
                // The strainLockStatus is either OPENED or ERROR
                this.lockStatus = INTRAVEL;

            } else {
                this.lockStatus = ERROR;
            }

        } else {
            /*
             * if lockStatus is not the same for the 3 clamps we are in this case. It can
             * happen during the closing (or the opening) of the 3 clamps when a clamp is
             * already closed but not the 2 others.
             */
            this.lockStatus = UNKNOWN;
        }
    }

    /**
     * initialize autochanger online clamps hardware after initialization. to be executed if
     * during boot process some hardware is missing.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Initialize autochanger online clamps hardware after initialization. To be executed if during boot process some hardware is missing.")
    public void initializeHardware() {
        this.postStart();
        for (AutochangerOnlineClamp clamp : clampsList) {
            clamp.postStart();
        }
    }

    @Override
    public void postStart() {
        FCSLOG.fine(() -> name + " BEGIN postStart.");
        if (onlineStrainGauge.isBooted()) {
            onlineStrainGauge.initializeAndCheckHardware();
        } else {
            onlineStrainGauge.raiseAlarmIfMissing();
        }
        if (tempSensorsDevice2.isBooted()) {
            try {
                computeLockStatusFromStrain();
                lockStatusInitialized = true;
                this.publishData();
            } catch (SDORequestException ex) {
                this.raiseWarning(CAN_BUS_READING_ERROR, "could not computeLockStatusFromStrain ", name, ex);
            }
        } else {
            tempSensorsDevice2.raiseWarningIfMissing();
        }

        // Check that onlineClamps are in CLOSED state at startup
        updateState();
        if (isAnyClosed()) {
            // Look which online CLAMP is CLOSED
            StringBuilder debug = new StringBuilder("Debug info:\n");
            for (AutochangerOnlineClamp clamp : clampsList) {
                debug.append(clamp.getName() + "'s lockStatus=" + clamp.getLockStatus().name() + ", ");
                debug.append("position=" + clamp.getController().getPosition() + ", ");
                debug.append("targetPositionToClose=" + clamp.getTargetPositionToClose() + ",\n");
            }
            // TODO BUG: This method seems to have a flawed logic, fixes it. The way strain limits for Closed and Locked and Open are set does not fit actual tests, especially on AC2
            //debug.append("strainGaugeLockStatus=" + computeStrainGaugeLockStatus().name() + ", ");
            debug.append("normalizedStrain=" + normalizedStrain + ".\n");

            // /!\ Warning, we do not check general constistency (e.g. every clamp is in the same status)
            String msg = "Autochanger Online Clamps are in an unexpected state at startup:\n"
                + "They should all be OPENED or LOCKED at startup but at least one has been found CLOSED.\n"
                + "Since there is no lock sensor for these clamps, this state needs to be reviewed by an expert."
                + "You should contact a FES expert with the debug information below to confirm the state of the Autochanger Online Clamps, and decide what to do next.\n"
                + debug.toString();
            raiseAlarm(AC_ONLINE_CLAMPS_CLOSED_AT_STARTUP, msg);
        }

        FCSLOG.fine(() -> name + " END postStart.");
    }
}
