package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.SIBLINGS;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposMode.CURRENT;
import static org.lsst.ccs.subsystems.fcs.EPOSEnumerations.EposState.SWITCH_ON_DISABLED;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CHANGER_TCPPROXY_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import static org.lsst.ccs.subsystems.fcs.FCSCst.SOCKET_NAME;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.BrakeState.NO_BRAKE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.BrakeState.NO_SENSOR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_ROTATION_RECOVERY;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_ROTATION_RECOVERY_FAILURE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.CA_ROTATION_RECOVERY_SUCCESS;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.IN_FAULT;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState.CLAMPED_ON_FILTER;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.MobileItemAction.ROTATE_CAROUSEL_TO_RELATIVE_POSITION;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.drivers.canopenjni.PDOData;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations.ControlWord;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.BrakeState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterClampState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.GeneralAction;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.HoldingBrakesState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.CarouselPowerState;
import org.lsst.ccs.subsystems.fcs.common.ADCInterface;
import org.lsst.ccs.subsystems.fcs.common.AcceleroInterface;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.EPOSControllerForCarousel;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.MobileItem;
import org.lsst.ccs.subsystems.fcs.common.PT100Interface;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.common.SensorPluggedOnTTC580;
import org.lsst.ccs.subsystems.fcs.common.TTC580Interface;
import org.lsst.ccs.subsystems.fcs.errors.ControllerFaultException;
import org.lsst.ccs.subsystems.fcs.errors.FailedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.errors.SDORequestException;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a representation of the hardware of the carousel. It receives
 * commands from the FCSMainModule and send back an acknowledge. It publishes
 * data on the status bus. In engineering mode it can receive commands from the
 * engineering console.
 *
 *
 * @author virieux
 *
 */
public class Carousel extends MobileItem implements FilterHolder {

    @SuppressWarnings("unused")
    private static final long serialVersionUID = -2376279469784152348L;
    private static final Logger FCSLOG = Logger.getLogger(Carousel.class.getName());
    @LookupField(strategy = TREE)
    private MainModule main;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    private int rotationTimeout;

    /**
     * carousel position given by carouselControl.
     */
    protected int position = 0;
    private int relativeTargetPosition;

    /**
     * an absolute target position to reach in a rotation is modified each time a
     * rotation command has been launched.
     */
    protected int absoluteTargetPosition;


    /**
     * startPosition is the absolute position at the beginning of a rotation.
     * used to compute estimated position during rotation : during rotation, the carousel controller is set
     * in encoder type incremental so the position read on controller is an incremental position
     * not an absolute position.
     *
     *
     */
    protected int startPosition;

    @ConfigurationParameter(description = "If the value of deltaPosition exceeds this threshold after the recovery procedure, cancel any rotation action.", units = "unitless", range = "0..2181120", category = "carousel")
    private volatile int maxStandbyDeltaPosition = 130;

    @ConfigurationParameter(description = "When deltaPosition exceeds this threshold, a rotation recovery procedure is initiated", units = "unitless", range = "0..2181120", category = "carousel")
    private volatile int recoveryStandbyDeltaPosition = 110;

    /**
     * 0.1 tour = 436224 pas (2Pi = 4260 X 1024) 1 socket = 0.2 tour = 1/5 tour =
     * 872448 pas
     */
    private final int fullTurn = 4362240;
    private final int halfTurn = fullTurn / 2;

    @ConfigurationParameter(description = "Velocity in slow mode.", units = "rpm", range = "0..5000", category = "carousel")
    public volatile int slowVelocity = 500;

    @ConfigurationParameter(description = "Acceleration in slow mode.", units = "rpm/s", range = "0..5000", category = "carousel")
    public volatile int slowAcceleration = 200;

    @ConfigurationParameter(description = "Deceleration in slow mode.", units = "rpm/s", range = "0..5000", category = "carousel")
    public volatile int slowDeceleration = 200;

    @ConfigurationParameter(description = "Timeout for the rotation in slow mode.", units = "millisecond", range = "0..300000", category = "carousel")
    public volatile int slowRotationTimeout = 100000;

    @ConfigurationParameter(description = "Velocity in fast mode.", units = "rpm", range = "0..5000", category = "carousel")
    public volatile int fastVelocity = 3400;

    @ConfigurationParameter(description = "Acceleration in fast mode.", units = "rpm/s", range = "0..5000", category = "carousel")
    public volatile int fastAcceleration = 2000;

    @ConfigurationParameter(description = "Deceleration in fast mode.", units = "rpm/s", range = "0..5000", category = "carousel")
    public volatile int fastDeceleration = 1000;

    @ConfigurationParameter(description = "Timeout for the rotation in fast mode.", units = "millisecond", range = "0..300000", category = "carousel")
    public volatile int fastRotationTimeout = 20000;

    @ConfigurationParameter(description = "Time to wait for the protection system to update its status after motion events.", units = "millisecond", range = "0..5000", category = "carousel")
    public volatile long timeToUpdateProtectionSystem = 2000;

    @ConfigurationParameter(description = "For command unlock of carousel clamp : current to send to prepare unlock", range = "-3200..3200", units = "mA", category = "carousel")
    protected volatile int currentToPrepareUnlock = -100;

    @ConfigurationParameter(description = "For command unlock of carousel clamp : time between little current to "
            + "prepare hardware and currentToLock ", range = "0..500", units = "millisecond", category = "carousel")
    protected volatile int timeToPrepareUnlock = 200;

    @ConfigurationParameter(description = "A current to be sent to clampXminus controller during locking recovery.", units = "mA", range = "0..1000", category = "carousel")
    public volatile int recoveryLockingCurrent = 400;

    @ConfigurationParameter(description = "If the velocity in clampXminus controller goes over this value during locking recovery, "
            + "it means that the recovery process has failed. [rpm]", units = "unitless", range = "0..100", category = "carousel")
    public volatile int recoveryMaxVelocity = 60;

    @ConfigurationParameter(description = "A number of steps to go back if after rotation carousel "
            + "position has exceeded standbyPosition by more than maxStandbyDeltaPosition.", units = "micron", range = "0..2181120", category = "carousel")
    public volatile int recoveryBackwardStep = 10000;

    @ConfigurationParameter(description = "Number of steps to go forward if after rotation carousel "
            + "position has exceeded standbyPosition by more than maxStandbyDeltaPosition.", units = "micron", range = "0..2181120", category = "carousel")
    public volatile int recoveryForwardStep = 150000;

    @ConfigurationParameter(range = "0..30000", description = "Upper brake1 (bay I) limit for state CLOSED", units = "mV", category = "carousel")
    private volatile int brake1Limit = 19033;

    @ConfigurationParameter(range = "0..30000", description = "Upper brake2 (bay N) limit for state CLOSED", units = "mV", category = "carousel")
    private volatile int brake2Limit = 16959;

    @ConfigurationParameter(range = "0..30000", description = "Upper brake3 (bay X) limit for state CLOSED", units = "mV", category = "carousel")
    private volatile int brake3Limit = 21063;

    /* brake in bay I*/
    private BrakeState brakeState1;
    /* brake in bay N*/
    private BrakeState brakeState2;
    /* brake in bay X*/
    private BrakeState brakeState3;

    private static int BRAKE_NO_SENSOR_LIMIT = 500;

    /*This actuator opens the clamp Xminus when the carousel is halted at STANDBY filterPosition.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/clampXminusController")
    private EPOSController clampXminusController;

    /*This actuator opens the clamp Xplus when the carousel is halted at STANDBY filterPosition.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/clampXplusController")
    private EPOSController clampXplusController;

    /*Controls carousel rotation.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/carouselController")
    protected EPOSControllerForCarousel carouselController;

    /*CANOpen devices to read the values of the clamps sensors.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/hyttc580")
    private TTC580Interface hyttc580;

    /*CANOpen devices to read the values of the brakes sensors and temperatures.*/
    @LookupField(strategy = TREE, pathFilter = ".*\\/ai814")
    protected ADCInterface ai814;

    /* To read motor and brakes temperatures */
    @LookupField(strategy = TREE, pathFilter = ".*\\/pt100")
    private PT100Interface pt100;

    /*CANOpen accelerometer */
    @LookupField(strategy = TREE, pathFilter = ".*\\/accelerobf")
    public AcceleroInterface accelerobf;

    /* To be able to know if the autochanger holds a filter. */
    @LookupField(strategy = SIBLINGS, pathFilter = "autochanger")
    private FilterHolder autochanger;

    /**
     * A map to store the sockets by their names. The key of this map is the socket
     * name. This map is built by Toolkit during INITIALISATION.
     *
     */
    @LookupField(strategy = CHILDREN)
    protected final Map<String, CarouselSocket> socketsMap = new TreeMap<>();

    /**
     * map of sensors attached to this device.
     */
    @LookupField(strategy = TREE)
    protected Map<String, SensorPluggedOnTTC580> sensorsMap = new HashMap<>();

    @LookupField(strategy = SIBLINGS, pathFilter = CHANGER_TCPPROXY_NAME)
    protected BridgeToHardware tcpProxy;

    private boolean initialized = false;
    protected boolean clampsStateInitialized = false;

    /**
     * Flag to be set by the Camera team to indicate whether we can rotate the carousel
     * If we do want to prohibit the carousel from rotating for some reason, use the setAvailable
     * command to switch this flag to false, or back again to true.
     */
    @Persist
    private volatile boolean available = true;

    /**
     * value of field socketAtStandbyID is read on httc580
     */
    private int socketAtStandbyID;
    protected CarouselSocket socketAtStandby;

    /**
     * Threshold that gives the 'locked' status to the clamps. The values of the
     * Baumer sensors, which are used to determine the locking status of the
     * carousel clamps, depend strongly on ambient temperature. The TTC580
     * processes temperature sensor values from the TTC30 modules and uses this
     * information to correct the threshold that gives the 'locked' status to
     * the clamps. This value (in mV) is read through the CAN bus from an SDO,
     * by sending a read request at index 0x6404, subindex 5.
     */
    protected volatile int minLockedThreshold = 9500;

    @ConfigurationParameter(description = "If lockSensor side X- returns a value < lockSensorMinLimitXminus, "
            + "sensor is in ERROR.", range = "0..12000", units = "mV", category = "carousel")
    private volatile int lockSensorMinLimitXminus = 201;

    @ConfigurationParameter(description = "If lockSensor side X+ returns a value < lockSensorMinLimitXplus, "
            + "sensor is in ERROR.", range = "0..12000", units = "mV", category = "carousel")
    private volatile int lockSensorMinLimitXplus = 202;

    @ConfigurationParameter(description = "If lockSensor side X- returns a value above lockSensorMaxLimitXminus, "
            + "the sensor is in error.", range = "0..12000", units = "mV", category = "carousel")
    private volatile int lockSensorMaxLimitXminus = 10001;

    @ConfigurationParameter(description = "If lockSensor side X+ returns a value above lockSensorMaxLimitXplus, "
            + "the sensor is in error.", range = "0..12000", units = "mV", category = "carousel")
    private volatile int lockSensorMaxLimitXplus = 11002;

    @ConfigurationParameter(description = "Minimum value of filter presence sensor without filter. Generic value used by all carousel clamps", range = "0..12000", units = "mV", category = "carousel")
    private volatile int filterPresenceMinNoFilter = 9200;

    /* if difference between offset read on hyttc580 and persisted offset is over this limit an ALARM is launched */
    @ConfigurationParameter(description = "If difference between offset read on hyttc580 and persisted offset is over this limit an ALARM is launched",
            range = "0..1000", units = "mV", category = "carousel")
    private volatile int maxClampsOffsetDelta = 100;

    @ConfigurationParameter(description = "If true powerSaveActivate is activated at the end of setFilter command.", units = "unitless", category = "carousel")
    private volatile boolean powerSaveAllowed = true;

    private double meanClampsTemperature;

    private PersistentCounter rotateCounter; // High level counter called from GUI


    /* Store last time a function was executed to check for a minimal interval between execution */
    private AtomicLong lastUpdateLockSensorMinLocked = new AtomicLong(0);
    private AtomicLong lastUpdateStateWithSensors = new AtomicLong(0);
    private AtomicLong lastUpdateTemperatures = new AtomicLong(0);

    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByCarousel.class, name);
        dataProviderDictionaryService.registerClass(StatusDataPublishedByCarouselBrakes.class, name+ "/brakes");

        movementCounter = new EnumMap<>(FcsEnumerations.MobileItemAction.class);
        for (FcsEnumerations.MobileItemAction action : new FcsEnumerations.MobileItemAction[]{
            ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION,
            ROTATE_CAROUSEL_TO_RELATIVE_POSITION}) {
            registerActionDuration(action);
            movementCounter.put(action, PersistentCounter.newCounter(action.getActionCounterPath(path), subs, action.name()));
        }
        rotateCounter = PersistentCounter.newCounter(GeneralAction.ROTATE_SOCKET_TO_STANDBY.getCounterPath(),
                                                    subs, GeneralAction.ROTATE_SOCKET_TO_STANDBY.name());
        registerHighLevelActionDuration(GeneralAction.ROTATE_SOCKET_TO_STANDBY);


        //register Component State HoldingBrakesState
        agentStateService.registerState(HoldingBrakesState.class, "Carousel Holding Brakes", this);
        agentStateService.updateAgentComponentState(this, HoldingBrakesState.FALSE);
    }

    @Override
    public void init() {
	super.init();
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(HARDWARE_ERROR.getAlert(carouselController.getName()), alwaysClear);
        alertService.registerAlert(HARDWARE_ERROR.getAlert(name), alwaysClear);
        alertService.registerAlert(IN_FAULT.getAlert(name), alwaysClear);
        alertService.registerAlert(CA_ROTATION_RECOVERY.getAlert(name), alwaysClear);
        alertService.registerAlert(CA_ROTATION_RECOVERY_FAILURE.getAlert(name), alwaysClear);
        alertService.registerAlert(CA_ROTATION_RECOVERY_SUCCESS.getAlert(name), alwaysClear);
    }

    /**
     *
     * @return true if CANopen devices are booted and initialized and homing has
     *         been done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if CANopen devices are booted and initialized and homing has been done.")
    boolean isInitialized() {
        return this.myDevicesReady();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if power save can be activated at the end of setFilter command.")
    public boolean isPowerSaveAllowed() {
        return powerSaveAllowed;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Set allowPoserSave.")
    public void setPowerSaveAllowed(boolean powerSaveAllowed) {
        this.powerSaveAllowed = powerSaveAllowed;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carousel is able to rotate or false if not.")
    public boolean isAvailable() {
        return this.available;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ADVANCED, description = "Flag carousel rotation as unavailable if false or available if true. "
            + "If available is false, this command should be used to return to 'available' only after FES engineers have been consulted.")
    public void setAvailable(boolean available) {
        this.available = available;
    }

    /**
     *
     * @return true if rotation is allowed by PLC.
     */
    public boolean isRotationAllowedByPLC() {
        return sensorsMap.get("carousel/plc/caEnableRotation").getValue() == 1;
    }

    /**
     *
     * @return true if rotation is allowed by PLC.
     */
    public boolean isUnclampAllowedByPLC() {
        return sensorsMap.get("carousel/plc/caEnableUnclamp").getValue() == 1;
    }

    /**
     *
     * @return true if clamp state is initialized for all clamps
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if clamp state is initialized for all clamps.")
    public boolean isClampsStateInitialized() {
        return clampsStateInitialized;
    }

    /* used by carousel clamp */
    public int getCurrentToPrepareUnlock() {
        return currentToPrepareUnlock;
    }

    /* used by carousel clamp */
    public int getTimeToPrepareUnlock() {
        return timeToPrepareUnlock;
    }

    /* used by carousel clamp */
    public Integer getMinLockedThreshold() {
        return minLockedThreshold;
    }

    public int getLockSensorMinLimitXminus() {
        return lockSensorMinLimitXminus;
    }

    public int getLockSensorMinLimitXplus() {
        return lockSensorMinLimitXplus;
    }

    public int getLockSensorMaxLimitXminus() {
        return lockSensorMaxLimitXminus;
    }

    public int getLockSensorMaxLimitXplus() {
        return lockSensorMaxLimitXplus;
    }

    public int getFilterPresenceMinNoFilter() {
        return filterPresenceMinNoFilter;
    }

    /* used by carousel clamp */
    public int getRecoveryLockingCurrent() {
        return recoveryLockingCurrent;
    }

    /* used by carousel clamp */
    public int getRecoveryMaxVelocity() {
        return recoveryMaxVelocity;
    }

    /* used by carousel clamp */
    public int getMaxClampsOffsetDelta() {
        return maxClampsOffsetDelta;
    }

    /**
     * used by tests.
     *
     * @return
     */
    public int getFullTurn() {
        return fullTurn;
    }

    public Map<String, CarouselSocket> getSocketsMap() {
        return socketsMap;
    }

    /**
     * Return a CarouselSocket which name is given as parameter.
     *
     * @param socketName
     * @return
     */
    public CarouselSocket getSocketByName(String socketName) {
        if (socketsMap.containsKey(socketName)) {
            return socketsMap.get(socketName);
        } else {
            throw new IllegalArgumentException(name + ": no such name for socket:" + socketName);
        }
    }

    /**
     * return carousel position.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return carousel position.", alias = "printPosition")
    public int getPosition() {
        return position;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXminusController() {
        return clampXminusController;
    }

    /**
     * @return the clampXminusController
     */
    public EPOSController getClampXplusController() {
        return clampXplusController;
    }

    /**
     * This method returns the clampX- which is at STANDBY filterPosition. It can
     * returns null if there is no socketAtStandby halted at STANDBY filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXminus() {
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXminus();
        }
    }

    /**
     * This method returns the clampX+ which is at STANDBY filterPosition. It can
     * returns null if there is no socketAtStandby halted at STANDBY filterPosition.
     *
     * @return
     */
    public CarouselClamp getClampXplus() {
        if (socketAtStandby == null) {
            return null;
        } else {
            return socketAtStandby.getClampXplus();
        }
    }

    /**
     * Used to publish on the STATUS bus for the GUI. Returns
     *
     * @return true if a socket is HALTED at STANDBY filterPosition, false
     *         otherwise.
     */
    @Override
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if a socket is HALTED at STANDBY position, false otherwise.")
    public boolean isAtStandby() {
        return socketAtStandbyID >= 1 && socketAtStandbyID <= 5;
    }

    /**
     * Return the socket HALTED at STANDBY filterPosition if there is one. Otherwise
     * return null.
     *
     * @return
     */
    public CarouselSocket getSocketAtStandby() {
        return socketAtStandby;
    }

    /**
     *
     * @return ID of socket at STANDBY
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return ID of socket at STANDBY, 0 if carousel is not stopped at STANDBY")
    public int getSocketAtStandbyID() {
        return socketAtStandbyID;
    }

    /**
     * Returns name of filter which is in the socket at STANDBY position or
     * NO_FILTER if there is no filter at STANDBY.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns name of filter at STANDBY position  or"
            + " NO_FILTER if carousel is not at STANDBY or there is no filter at STANDBY.")
    public String getFilterAtStandbyName() {
        if (socketAtStandby == null || this.isEmptyAtStandby()) {
            return NO_FILTER;
        } else {
            return socketAtStandby.getFilterName();
        }
    }

    /**
     * Returns ID of filter which is in the socket at STANDBY position or 0 if there
     * is no filter at STANDBY.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns ID of filter at STANDBY position  or"
            + " 0 if there is no filter at STANDBY.")
    public int getFilterIDatStandby() {
        if (socketAtStandby == null) {
            return 0;
        } else {
            return socketAtStandby.getFilterID();
        }
    }

    /**
     * return true if filterID in on carousel.
     *
     * @param filterID
     * @return
     */
    public boolean isFilterOnCarousel(int filterID) {
        return socketsMap.values().stream().anyMatch((socket) -> (socket.getFilterID() == filterID));
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "To change filterID on socket which ID is given as argument.")
    public void changeFilterID(int filterID, int socketID) {
        if (socketID < 1 || socketID > 5) {
            throw new IllegalArgumentException(socketID + ": bad value - enter a digit between 1 and 5");
        }
        if (this.isFilterOnCarousel(filterID)) {
            int sockID = this.getFilterSocket(filterID).getId();
            throw new IllegalArgumentException(filterID + " filter already on carousel on socket" + sockID);
        }
        String socketName = SOCKET_NAME + socketID;
        socketsMap.get(socketName).setFilterID(filterID);
    }



    /**
     * initialize carousel hardware after initialization. to be executed if during
     * boot process some hardware is missing.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Initialize carousel hardware after initialization. To be executed if during boot process some hardware is missing.")
    public void initializeHardware() {
        tcpProxy.bootProcess();
        this.postStart();
    }

    /**
     * Executed when all components HasLifeCycle of subsystem have been checked.
     */
    @Override
    public void postStart() {
        FCSLOG.info(name + " BEGIN postStart.");
        if (carouselController.isBooted()) {
            initializeRotationController();
            long profileVelocity = carouselController.readProfileVelocity();
            //to initialize data published by carouselController - could go to postStart in CanOpenEPOS
            carouselController.readProfileAcceleration();
            carouselController.readProfileDeceleration();
            if (profileVelocity == slowVelocity) {
                rotationTimeout = slowRotationTimeout;
            } else {
                rotationTimeout = fastRotationTimeout;
            }
            /**
             * Try to update carousel position and if carouselController is in
             * fault catch exception to let subsystem starts nevertheless.
             */
            try {
                updatePosition();
            } catch (ControllerFaultException ex) {
                FCSLOG.log(Level.SEVERE, "Carousel controller failed to start", ex);
                this.raiseAlarm(IN_FAULT, ex.toString(), name);
            }
            updateHoldingBrakesState();
        }
        if (clampXminusController.isBooted()) {
            initializeClampController(clampXminusController);
        }
        if (clampXplusController.isBooted()) {
            initializeClampController(clampXplusController);
        }
        if (hyttc580.isBooted()) {
            initializeClampsState();
            initializeAndCheckClampsOffset();
        }

        hyttc580.checkTTC580Temperatures(); // first occurrence of the periodic task

        if ( isPowerSaveActivated() ) {
            agentStateService.updateAgentState(CarouselPowerState.LOW_POWER);
        } else {
            agentStateService.updateAgentState(CarouselPowerState.REGULAR);
        }

        FCSLOG.info(name + " END postStart.");
    }

    private void initializeRotationController() {
        try {
            carouselController.initializeAndCheckHardware();
            this.initialized = true;

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize controller", carouselController.getName(), ex);
        }
    }

    /**
     * check that controller is correctly configured. This command can't be executed
     * in CarouselClamp.postStart because the 5 clamps Xminus share the same
     * controller and idem for Xplus.
     */
    private void initializeClampController(EPOSController controller) {
        try {
            /*
             * check that parameters on CPU are those on configuration
             */
            controller.initializeAndCheckHardware();
            if (!controller.isInMode(CURRENT)) {
                this.raiseAlarm(HARDWARE_ERROR, " is not in CURRENT mode.", controller.getName());
            }

        } catch (FcsHardwareException | FailedCommandException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize controller ", controller.getName(), ex);
        }
    }


    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Disable carousel controller, set the position sensor to TypeEncoderSSI, and check that the ssi position is correct.")
    public void setControllerPositionSensorTypeEncoderSSI() {
        carouselController.setPositionSensorTypeEncoderSSI();
        position = carouselController.readPosition();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "slow down profile velocity, acceleration and deceleration in carousel controller.")
    public void setSlowMode() {
        carouselController.changeProfileVelocity(slowVelocity);
        carouselController.changeProfileAcceleration(slowAcceleration);
        carouselController.changeProfileDeceleration(slowDeceleration);
        rotationTimeout = slowRotationTimeout;
        publishData();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "raise profile velocity, acceleration and deceleration in carousel controller.")
    public void setFastMode() {
        carouselController.changeProfileVelocity(fastVelocity);
        carouselController.changeProfileAcceleration(fastAcceleration);
        carouselController.changeProfileDeceleration(fastDeceleration);
        rotationTimeout = fastRotationTimeout;
        publishData();
    }

    public void waitForProtectionSystemUpdate() {
        waitForStateUnclampedOnFilter(timeToUpdateProtectionSystem);
    }

    /**
     * wait until carousel is unclamped and empty at STANDBY
     *
     * @param timeout after this delay don't wait anymore.
     */
    public void waitForStateUnclampedOnFilter(long timeout) {

        FcsUtils.waitCondition(() -> isUnclampedOnFilterAtStandby(), () -> updateSocketAtStandbyReadSensorsNoPublication(),
                "waitForStateUnclampedOnFilter", timeout);

    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Wait until socket at STANDBY is awake")
    public void waitUntilCarouselIsAwake() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("waitUntilCarouselIsAwake")) {
            FcsUtils.checkAndWaitConditionWithTimeoutAndFixedDelay(
                    () -> isAwake(),
                    () -> updateCarouselIOStatus(),
                    "checkCarouselIsAwake",
                    name + ": Carousel is still not awake after trying every 100ms during 5s",
                    5000,
                    100
            );
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Power save deactivate and wait until all ttc30 are awake")
    public void powerWakeupTest() {
        powerOn();
        waitUntilCarouselIsAwake();
    }

    private void updateCarouselIOStatus() {
        tcpProxy.updatePDOData();
        updateState();
    }

    /**
     * This command is used to know if carousel is awake in order to be able to execute command setFilter.
     *
     * If a socket is at STANDBY and is awake, we can go on because carousel protection system takes into account only the socket at STANDBY.
     * In this case, no need to test if the other sockets are awake.
     * If there is no socket at STANDBY we have to test that all sockets are awake.
     *
     * @return true if a socket is at STANDBY and is awake or if all sockets are awake
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "return true if a socket is at STANDBY and is awake or if all sockets are awake")
    public boolean isAwake() {
        return isSocketAtStandbyAwake() || allSocketsAwake();
    }

    /**
     * Used by jython scripts
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if a socket is at STANDBY and available")
    public boolean isSocketAtStandbyAvailable() {
        if ( socketAtStandby != null ) {
            return socketAtStandby.isAvailable();
        }
        return false;
    }

    /**
     *
     * @return true if the socket at STANDBY is awake.
     */
    public boolean isSocketAtStandbyAwake() {
        return socketsMap.values().stream().anyMatch((socket) -> (socket.isAwake() && socket.isAtStandby()));
    }

    /**
     * This command tests that all sockets are awake.
     * After a powerSaveDeactivate it takes more time to return true than method isSocketAtStandbyAwake.
     * @return true if all sockets are awake, without taking into account that there is a socket at STANDBY.
     */
    public boolean allSocketsAwake() {
        return socketsMap.values().stream().allMatch((socket) -> (socket.isAwake()));
    }


    public void checkSensorTypeIncremental() {
        int sensorType = carouselController.readPositionSensorType();
        if (sensorType != 8) {
            throw new FcsHardwareException(
                    name + " PositionSensorType has to be set to Incremental Encoder before motion.");
        }
    }

    /**
     * Publish Data on status bus for trending data base and GUIs.
     *
     */
    @Override
    public void publishData() {
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name, createStatusDataPublishedByCarousel()));
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name + "/brakes", createStatusDataPublishedByCarouselBrakes()));
    }

    /**
     * Create an object StatusDataPublishedByCarousel to be published on the STATUS
     * bus.
     *
     * @return status
     */
    public StatusDataPublishedByCarousel createStatusDataPublishedByCarousel() {
        StatusDataPublishedByCarousel status = new StatusDataPublishedByCarousel();
        status.setPosition(position);
        status.setPositionSensorType(carouselController.getPositionSensorType());
        // during rotation positionSensorType is incremental so the position read on the controller is the incrementalPosition.
        status.setEstimatedPosition((startPosition + position) % fullTurn);
        status.setAtStandby(isAtStandby());
        status.setMoving(moving);
        status.setSocketAtStandbyID(socketAtStandbyID);
        // if carousel is empty at STANDBY getFilterAtStandbyName() returns NO_FILTER
        status.setFilterAtStandbyName(getFilterAtStandbyName());
        status.setFilterAtStandbyId(getFilterIDatStandby());
        if (isAtStandby()) {
            status.setSocketAtStandbyName(socketAtStandby.getName());
            status.setEmptyAtStandby(this.isEmptyAtStandby());
            status.setDeltaPositionAtStandby(socketAtStandby.getDeltaPosition());
            status.setClampsStateAtStandby(socketAtStandby.getClampsState());
            status.setIOStatusAtStandby(socketAtStandby.getIOModuleStatus());
        } else if (isPowerSaveActivated()) {
            /*for FcsGeneralView*/
            status.setSocketAtStandbyName("POWER_SAVE");
        } else if (socketAtStandbyID == 0) {
            status.setSocketAtStandbyName("NO_SOCKET_AT_STANDBY");
        } else if (socketAtStandbyID == 7) {
            status.setSocketAtStandbyName("ERROR_READING_ID");
        }
        status.setRotationTimeout(rotationTimeout);
        status.setMinLocked(minLockedThreshold);
        status.setMeanClampsTemperature(meanClampsTemperature);
        status.setAvailable(available);
        return status;
    }

    /**
     * Create an object StatusDataPublishedByCarouselBrakes to be published on the STATUS
     * bus.
     *
     * @return status
     */
    public StatusDataPublishedByCarouselBrakes createStatusDataPublishedByCarouselBrakes() {
        StatusDataPublishedByCarouselBrakes status = new StatusDataPublishedByCarouselBrakes();
        status.setSensor1(ai814.getInput(0));
        status.setSensor2(ai814.getInput(1));
        status.setSensor3(ai814.getInput(2));
        status.setBrakeState1(computeBrakeState(brake1Limit, ai814.getInput(0)));
        status.setBrakeState2(computeBrakeState(brake2Limit, ai814.getInput(1)));
        status.setBrakeState3(computeBrakeState(brake3Limit, ai814.getInput(2)));
        status.setTemperature1(pt100.getTemperature(1));
        status.setTemperature2(pt100.getTemperature(2));
        status.setTemperature3(pt100.getTemperature(3));
        status.setTemperature4(pt100.getTemperature(4));
        return status;
    }

    private static BrakeState computeBrakeState(int limit, long sensorValue) {
        if (sensorValue > limit) {
            return BrakeState.CLOSED;
        } else if (sensorValue > BRAKE_NO_SENSOR_LIMIT) {
            return BrakeState.NO_BRAKE;
        } else {
            return BrakeState.NO_SENSOR;
        }
    }

    /**
     * This method let us know if the carousel is ready to receive a filter at
     * STANDBY filterPosition : - the carousel must not rotate - an empty
     * socketAtStandby is at STANDBY filterPosition.
     *
     * @return true if the filterPosition of the carousel matches the filterPosition
     *         when one of its sockets is at STANDBY filterPosition and this
     *         socketAtStandby is empty. false
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if Carousel is stopped and no filter is at STANDBY position")
    public boolean isReadyToGrabAFilterAtStandby() {

        if (this.isMoving()) {
            return false;
        }
        if (socketAtStandby == null) {
            return false;
        }
        if (!socketAtStandby.isAvailable()) {
            return false;
        }
        return socketAtStandby.isEmpty() && socketAtStandby.isReadyToClamp();
    }

    /**
     *
     * @return true if a filter is clamped at STANDBY position and carousel is
     *         stopped
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns true if a filter is clamped at STANDBY position")
    public boolean isHoldingFilterAtStandby() {
        if (this.isMoving()) {
            return false;
        }
        if (socketAtStandby == null || socketAtStandby.isEmpty()) {
            return false;
        }
        return socketAtStandby.isClampedOnFilter();
    }

    /**
     * Returns true if carousel is rotating
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Returns false if carousel controller is SWITCH_ON_DISABLED.")
    public boolean isRotating() {
        return !carouselController.isInState(SWITCH_ON_DISABLED);
    }

    /**
     *
     * @return true if hyttc580 says that brakes are activated.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "PLC value for carousel brakes activated")
    public boolean isStatusBrakesActivated() {
        return sensorsMap.get("carousel/plc/caBrakesActivated").getValue() == 1;
    }

    /**
     *
     * @return true if hyttc580 says that power save is activated.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "PLC value for carousel power save activated")
    public boolean isPowerSaveActivated() {
        return sensorsMap.get("carousel/plc/powerSave").getValue() == 1;
    }

    /**
     * A carousel brake is disabled if it is equipped and live (not NO_SENSOR)
     * and in state NO_BRAKE.
     *
     * In the carousel-PROTO, brake in bay N and brake in bay X are not
     * equipped.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "NO_BRAKE signal on all 3 AI814 sensors")
    public boolean areBrakesDisabledFromAI814() {
        boolean brake1Disabled = brakeState1 == NO_SENSOR || brakeState1 == NO_BRAKE;
        boolean brake2Disabled = brakeState2 == NO_SENSOR || brakeState2 == NO_BRAKE;
        boolean brake3Disabled = brakeState3 == NO_SENSOR || brakeState3 == NO_BRAKE;
        return brake1Disabled && brake2Disabled && brake3Disabled;
    }

    /**
     *
     * @return true is brakes are allowing rotation.
     *
     */
    protected boolean areBrakesAllowingRotation() {
        return !carouselController.isHoldingBrakes()
                && !isStatusBrakesActivated()
                && areBrakesDisabledFromAI814();
    }

    private void updateBrakesStatesFromAI814() {
        brakeState1 = computeBrakeState(brake1Limit, ai814.getInput(0));
        brakeState2 = computeBrakeState(brake2Limit, ai814.getInput(1));
        brakeState3 = computeBrakeState(brake3Limit, ai814.getInput(2));
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Update brakes status from all sensors")
    public void updateBrakesStatus() {
        //to update holdingBrakes
        updateHoldingBrakesState();
        //to update PDO to read hyttc580 and AI814 data.
        tcpProxy.updatePDOData();
        updateBrakesStatesFromAI814();
    }

    protected void updateHoldingBrakesState() {
        boolean hb = carouselController.readHoldingBrakes();
        if (hb) {
            agentStateService.updateAgentComponentState(this, HoldingBrakesState.TRUE);
        } else {
            agentStateService.updateAgentComponentState(this, HoldingBrakesState.FALSE);
        }
    }

    /**
     * Updates the filterPosition of the carousel in reading the CPU of the
     * controller. Used at startup and by end user for debug.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.SDORequestException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Set position sensor type SSI and update carousel position in reading controller.")
    public void updatePosition() {
        this.setControllerPositionSensorTypeEncoderSSI();
        try {
            this.position = carouselController.readPosition();
        } catch (SDORequestException ex) {
            FCSLOG.log(Level.WARNING,name + "=> ERROR IN READING CONTROLLER:", ex);
        }
        this.publishData();
    }


    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one time.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    public void updateClampsStateWithSensorsFromPDO() {

        this.tcpProxy.updatePDOData();
        PDOData pdoStore = tcpProxy.getPDOData();

        FCSLOG.finest(() -> name + ":pdoStore=" + pdoStore.toString());
        socketsMap.values().stream().forEach(socket -> {
            socket.updateState();
        });
    }

    /**
     * In carousel, sensors are updated from PDOs. PDOs are received 2 by 2 : 0x180
     * + ttc580 nodeID for clamp at STANDBY. (ttc580 pdo1) 0x280 + ttc580 nodeID for
     * a clamp not at STANDBY. (ttc580 pdo2) PDO2 contains socketID not at STANDBY
     * in turns. Exemple : if socket1 is at STANDBY - first sync returns pdo1 with
     * socket1 values and pdo2 with socket2 values - second sync returns pdo1 with
     * socket1 values and pdo2 with socket3 values - third sync returns pdo1 with
     * socket1 values and pdo2 with socket4 values - fourth sync returns pdo1 with
     * socket1 values and pdo2 with socket5 values
     *
     * socketID is coded in each PDO. After FCS start we need to send 4 sync to know
     * all clamps state.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update clamps state in sending 5 sync.")
    public void initializeClampsState() {
        FCSLOG.info(name + " Initializing clamps state....");
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        byte count = 1;
        while (count <= 5) {
            FCSLOG.finer(name + " sync no " + count);
            tcpProxy.updatePDOData();
            hyttc580.updateFromPDO(tcpProxy.getPDOData());
            updateState();
            count++;
        }
        clampsStateInitialized = true;
        publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE,
            description = "Update carousel clamps offset1 and offset2 in reading hyttc580 and launch "
            + "an ALARM if values read and different from previous values. "
            + "The new values are saved in persistence file.")
    public void initializeAndCheckClampsOffset() {
        FCSLOG.info(name + " Initializing clamps offsets....");
        hyttc580.checkBooted();
        hyttc580.checkInitialized();
        socketsMap.values().stream().forEach(socket -> {
            byte socketId = (byte) socket.getId();
            long offset2 = hyttc580.readOffset2SDO(socketId);
            socket.checkAndUpdateOffset2(offset2);
            long offset1 = hyttc580.readOffset1SDO(socketId);
            socket.checkAndUpdateOffset1(offset1);
            socket.getClampXminus().publishData();
            socket.getClampXplus().publishData();
        });
        subs.getAgentPersistenceService().persistNow();
    }

    /**
     *
     * @param filterID
     * @return null if filter is not on carousel, else return socket where filter is
     *         stored.
     */
    public CarouselSocket getFilterSocket(int filterID) {
        CarouselSocket socket = null;
        for (CarouselSocket sock : socketsMap.values()) {
            if (sock.getFilterID() == filterID) {
                return sock;
            }
        }
        return socket;
    }

    /**
     * Release clamps at STANDBY position to get ready to clamp again.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Release clamps at STANDBY position to get ready to clamp again")
    public void releaseClamps() {
        updateStateWithSensors();
        if (socketAtStandby == null) {
            throw new RejectedCommandException(
                    name + " can't release clamps when no socket is halted at STANDBY position.");
        } else {
            socketAtStandby.releaseClamps();
        }
    }

    /**
     * Unlocks the clamps at STANDBY.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws FailedCommandException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Unlock the clamps at STANDBY.")
    public void unlockClamps() {
        updateSocketAtStandbyReadSensorsNoPublication();
        if (socketAtStandby == null) {
            throw new RejectedCommandException(
                    name + " can't unlock clamps while a socket is not halted at STANDBY position.");
        } else {
            FCSLOG.info("Unlocking clamps at STANDBY.");
            socketAtStandby.unlockClamps();
        }
    }

    /**
     * This is a recovery command when storing a carousel at STANDBY : sometimes,
     * clampXplus is LOCKED but clampXminus is not LOCKED. This command sends a
     * little current in clampXminus controller to complete the locking.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Lock clampXminus when clampXplus is already locked.")
    public void recoveryLockingXminus() {
        updateStateWithSensors();
        if (!isAtStandby()) {
            throw new RejectedCommandException(name + " is NOT AT STANDBY - can't use command recoveryLockingXminus.");
        }
        if (socketAtStandby.getClampXplus().getClampState() == CLAMPED_ON_FILTER
                && socketAtStandby.getClampXminus().getClampState() != CLAMPED_ON_FILTER) {
            socketAtStandby.getClampXminus().recoveryLocking();
        }
    }

    /**
     * Return true if the filterID given as argument is at STANDBY position.
     *
     * @param filterID
     * @return
     */
    @Deprecated
    public boolean isAtStandby(int filterID) {
        return socketAtStandby.getFilterID() != 0 && socketAtStandby.getFilterID() == filterID;
    }

    @Override
    public boolean myDevicesReady() {
        return carouselController.isBooted() && carouselController.isInitialized();
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ ROTATION COMMANDS *********************************
     */
    /**
     * ****************************************************************
     */
    /**
     * Check if carousel rotation is permitted.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException (RuntimeException)
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if carousel rotation is permitted.")
    public void checkConditionsForRotation() {
        String message;
        if (!initialized) {
            throw new FcsHardwareException("Carousel hardware is not initialized. Can't rotate.");
        }

        // commented out in July 2017 because no TTC-580
        // updateState();
        if (this.isAtStandby() && socketAtStandby.isUnclampedOnFilter()) {
            message = "Filter at STANDBY position is not held by clamps. Can't rotate carousel.";
            FCSLOG.severe(message);
            throw new RejectedCommandException(message);
        }

        if (!autochanger.isAtHandoff()) {
            throw new RejectedCommandException(name + " can't rotate if autochanger is not at HANDOFF position.");
        }
    }


    /**
     * Rotate to position newPos. Condition LPM to rotate : AP2 & AP3. This command
     * doesn't do homing but : 1) set sensor type SSI encoder and read position
     * given by SSI encoder : positionAbsolue 2) compute diffPos == the diff between
     * newPos and positionAbsolue 3) set sensor type incremental 4)set
     * absoluteTargetPosition to diffPos. 5) Then rotate to absoluteTargetPosition %
     * fullTurn.
     *
     * @param newPos
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Rotate carousel to a new absolute position.", timeout = 60000)
    public void rotateToAbsolutePosition(int newPos) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("rotateToAbsolutePosition")) {
            /* read temperatures before rotation */
            if (pt100.isBooted()) {
                pt100.updateTemperatures();
            }

            /* subsystem must not be in ALARM state */
            checkReadyForAction();

            /* check that power save is not activated */
            if (isPowerSaveActivated()) {
                throw new RejectedCommandException(
                        name + " can't rotate because power save is activated");
            }

            /* carouselController should be initialized */
            carouselController.checkInitialized();

            /* carouselController should not be in fault */
            carouselController.checkFault();

            /* autochanger must be at ONLINE or HANDOFF */
            if (!autochanger.isAtHandoff() && !autochanger.isAtOnline()) {
                throw new RejectedCommandException(
                        name + " can rotate only if autochanger is at Handoff or at Online  ");
            }

            double airmass = accelerobf.getAirmass();
            if (airmass > 3) {
                setSlowMode();
            } else {
                setFastMode();
            }
            FcsUtils.sleep(10, name);

            /* read position given by SSI encoder */
            carouselController.setPositionSensorTypeEncoderSSI();

            int absolutePosition = carouselController.readPosition();
            startPosition = absolutePosition;

            /* set sensor type incremental */
            carouselController.setPositionSensorTypeSinusIncrementalEncoder();
            int incrementalPosition = carouselController.readPosition();
            // incrementalPosition should be 0 or very near 0
            FCSLOG.info(
                    name + " absolutePosition = " + absolutePosition + " incrementalPosition = " + incrementalPosition);

            int diffPos = newPos - absolutePosition;
            if (Math.abs(diffPos) <= halfTurn) {
                absoluteTargetPosition = diffPos + incrementalPosition;
            } else {
                if (diffPos < 0) {
                    absoluteTargetPosition = diffPos + incrementalPosition + fullTurn;
                } else {
                    absoluteTargetPosition = diffPos + incrementalPosition - fullTurn;
                }
            }

            FCSLOG.info(name + " is at incremental position: " + incrementalPosition
                    + "; about to rotate to incremental position" + absoluteTargetPosition);

            // rotate the carousel
            this.executeAction(ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION, rotationTimeout);
            /* in endAction a setControllerPositionSensorTypeEncoderSSI() is done */
        }
    }

    /**
     * rotate carousel within a relative position given as argument. If carousel is
     * at initial position initPos, if the argument given to this method is
     * relativePos, at the end of the motion, carousel position should be initPos +
     * relativePos.
     *
     * @param relativePos relative position.
     * @param timeout
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Rotate carousel to a relative position.", timeout = 60000)
    public void rotateToRelativePosition(int relativePos, long timeout) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("rotateToRelativePosition")) {
            /* carouselController should be initialized */
            carouselController.checkInitialized();

            /* carouselController should not be in fault */
            carouselController.checkFault();

            /* autochanger must be at ONLINE or HANDOFF */
            if (!autochanger.isAtHandoff() && !autochanger.isAtOnline()) {
                throw new RejectedCommandException(
                        name + " can rotate only if autochanger is at Handoff or at Online.");
            }

            /* set sensor type incremental */
            carouselController.setPositionSensorTypeSinusIncrementalEncoder();
            int incrementalPosition = carouselController.readPosition();

            relativeTargetPosition = relativePos + incrementalPosition;
            // rotate the carousel
            this.executeAction(ROTATE_CAROUSEL_TO_RELATIVE_POSITION, timeout);
            /*
             * switch back controller to PositionSensorTypeEncoderSSI is done is endAction
             */
        }
    }

    /**
     * Rotate carousel to move a socket which name is given as argument to STANDBY
     * position. This methods computes the shortest way to go to STANDBY position.
     *
     * @param socketName
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move a socket which name is given as argument to STANDBY position.", alias = "moveSocketToStandby", timeout = 50000)
    public void rotateSocketToStandby(String socketName) {
        FcsUtils.checkSocketName(socketName);
        CarouselSocket socketToMove = this.socketsMap.get(socketName);
        if (!socketToMove.isAtStandby() || socketAtStandbyNeedsRotation(socketToMove)) {
            long beginTime = System.currentTimeMillis();
            rotateCounter.increment();
            int requiredPosition = socketToMove.getStandbyPosition();
            FCSLOG.info(name + " is at position: " + position + "; about to rotate to position: " + requiredPosition);
            rotateToAbsolutePosition(requiredPosition);
            checkDeltaPosition();
            subs.publishSubsystemDataOnStatusBus(
                new KeyValueData(
                    GeneralAction.ROTATE_SOCKET_TO_STANDBY.getDurationPath(),
                    System.currentTimeMillis() - beginTime));
        }
        FCSLOG.info(name + ":" + socketName + " is at STANDBY position on carousel.");
    }

    public void checkDeltaPosition() {
        if (socketAtStandby != null) {
            socketAtStandby.updateDeltaPosition();
            if (socketAtStandbyNeedsRotation(socketAtStandby)) {
                recoveryRotation();
            }
        }
    }

    private boolean socketAtStandbyNeedsRotation(CarouselSocket socket) {
        /* socket is already at STANDBY*/
        socket.updateDeltaPosition();
        return Math.abs(socket.getDeltaPosition()) > recoveryStandbyDeltaPosition;
    }

    /**
     * Rotate carousel full turn. A number of turns can be given as argument. This
     * turns carousel in relative position.
     *
     * @param nbTurn number of turns to rotate
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Rotate carousel full turn. A number of turns can be given as argument.")
    public void rotateFullTurn(int nbTurn) {
        if (Math.abs(nbTurn) > 3) {
            throw new IllegalArgumentException("nbTurn=" + nbTurn + " should not be more than 3");
        }
        // slowRotationTimeout is the time maximum to rotate 2 sockets at slow speed;
        // for a full turn (5 sockets), 5/2 should be the right factor, but experience
        // showed that 2 is enough.
        long timeout = Math.abs(nbTurn) * 2 * slowRotationTimeout;
        int relativePosition = fullTurn * nbTurn;
        rotateToRelativePosition(relativePosition, timeout);
    }

    private void recoveryRotation() {
        if (!autochanger.isAtOnline() && !autochanger.isAtHandoff()) {
            FCSLOG.info(name + " can rotate only if autochanger is at HANDOFF or ONLINE. Can't execute recoveryRotation.");
            return;
        }
        double airmass = accelerobf.getAirmass();
        if (airmass > 2) {
            /* in this case recoveryRotationStraight will not work in a first time */
            recoveryRotationBackward();
            /* then try more time recoveryStraight */
            if (Math.abs(socketAtStandby.getDeltaPosition()) > recoveryStandbyDeltaPosition) {
                recoveryRotationStraight();
            }
        } else {
            /* try first fast and easy recovery*/
            recoveryRotationStraight();
            socketAtStandby.updateDeltaPosition();
            /* then try more time consuming recovery if the first one was not enough*/
            if (Math.abs(socketAtStandby.getDeltaPosition()) > recoveryStandbyDeltaPosition) {
                recoveryRotationBackward();
            }
        }
        socketAtStandby.updateDeltaPosition();
        if (Math.abs(socketAtStandby.getDeltaPosition()) > maxStandbyDeltaPosition) {
            raiseAlarm(CA_ROTATION_RECOVERY_FAILURE,
                String.format("After 2 recovery tries delta position at STANDBY is %s is still over %s.\n",
                 socketAtStandby.getDeltaPosition(), maxStandbyDeltaPosition)
                 + "The operator should use recoveryRotationForward to align the carousel to the STANDBY position.", name);
        } else {
            raiseWarning(CA_ROTATION_RECOVERY_SUCCESS, "After recovery delta position at STANDBY is "
                    + socketAtStandby.getDeltaPosition(), name);
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Triggered when the carousel's position exceeds the STANDBY position by more than deltaPositionMax (typically around 200).\n"
            + "Explanation: if the selected carousel socket for a filter exchange between the autochanger and the carousel "
            + "has a significant offset from the STANDBY position, the operation will be halted or prevented. "
            + "To resolve this issue, the carousel socket must be aligned with the STANDBY position using a dedicated recovery rotation command.\n"
            + "This command attempts to align the carousel socket with the STANDBY position by reversing the previous movement "
            + "and then rotating back to the STANDBY position while compensating for any offset (to account for play in the brakes).\n"
            + "This method is the most precise and reliable but is slower (approximately 7 seconds). "
            + "It is recommended for use when recoveryRotationStraight fails or when dealing with large imbalances in filter configuration "
            + "(such as during filter loading/unloading from the camera).\n"
            + "The FCS will automatically try a recovery if needed. "
            + "If it fails, the operator should use recoveryRotationForward to align the carousel to the STANDBY Position.")
    public void recoveryRotationBackward() {
        if (!isAtStandby()) {
            throw new RejectedCommandException(name
                + " it seems that no Carousel socket is at STANDBY position, meaning that recoveryRotationBackward is not relevant here."
                + " It might be that the socket IOModule is not responding correctly. Some ideas to consider: first check the IO Status on the "
                + " carousel socket panel and verify the STANDBY position and actual positions. If IO Status is unknown but"
                + " the carousel socket seems at STANDBY, then the only way out is to perform a restart of the FCS along with"
                + " a powercycle of the carousel 24V and 48V and the FES PLC. Instructions for this can be found on the Camera Confluence.");
        }
        int deltaPositionAtStandby = socketAtStandby.getDeltaPosition();
        this.raiseWarning(CA_ROTATION_RECOVERY, String.format(" |delta position at STANDBY| = %d is over %d "
                + "about to try recoveryRotationBackward", socketAtStandby.getDeltaPosition(), recoveryStandbyDeltaPosition), name);
        int delta_sign = deltaPositionAtStandby >= 0 ? 1 : -1;
        int relativePosition = delta_sign * recoveryBackwardStep;
        /* relative rotation : go back for recoveryBackwardStep steps */
        FCSLOG.info(name + " about to rotateToRelativePosition " + relativePosition);
        rotateToRelativePosition(relativePosition, 7000);
        /*absolute rotation : go to target position - deltaPositionAtStandby */
        int newPos = socketAtStandby.getStandbyPosition() - deltaPositionAtStandby;
        FCSLOG.info(name + " about to rotateToAbsolutePosition to " + newPos);
        this.rotateToAbsolutePosition(newPos);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Triggered when the carousel's position exceeds the STANDBY position by more than deltaPositionMax (typically around 200).\n"
            + "Explanation: if the selected carousel socket for a filter exchange between the autochanger and the carousel "
            + "has a significant offset from the STANDBY position, the operation will be halted or prevented. "
            + "To resolve this issue, the carousel socket must be aligned with the STANDBY position using a dedicated recovery rotation command.\n"
            + "This command attempts to align the carousel socket with the STANDBY position by reversing the previous movement "
            + "and then rotating back to the STANDBY position while compensating for any offset (to account for play in the brakes).\n"
            + "This method is less precise than the other recovery methods.\n"
            + "It is recommended for use when recoveryRotationStraight and recoveryRotationBackward have both failed.\n"
            + "The FCS will automatically try a recovery if needed. "
            + "If it fails, the operator should use recoveryRotationForward to align the carousel to the STANDBY Position.")
    public void recoveryRotationForward() {
        if (!isAtStandby()) {
            throw new RejectedCommandException(name + " no socket at STANDBY method recoveryRotationForward is not relevant.");
        }
        int deltaPositionAtStandby = socketAtStandby.getDeltaPosition();
        int delta_sign = deltaPositionAtStandby >= 0 ? 1 : -1;
        int relativePosition = -delta_sign * recoveryForwardStep;
        FCSLOG.info(name + " about to rotateToRelativePosition " + relativePosition);
        rotateToRelativePosition(relativePosition, 7000);
        int newPos = socketAtStandby.getStandbyPosition();
        FCSLOG.info(name + " about to rotateToAbsolutePosition to " + newPos);
        this.rotateToAbsolutePosition(newPos);
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE,
            description = "Triggered when the carousel's position exceeds the STANDBY position by more than deltaPositionMax (typically around 200).\n"
            + "Explanation: if the selected carousel socket for a filter exchange between the autochanger and the carousel "
            + "has a significant offset from the STANDBY position, the operation will be halted or prevented. "
            + "To resolve this issue, the carousel socket must be aligned with the STANDBY position using a dedicated recovery rotation command.\n"
            + "This command attempts to align the carousel socket with the STANDBY position "
            + "by executing a straightforward rotation to correct the measured misalignment. "
            + "This method is the fastest (~3s) and is generally effective, especially when the camera is not "
            + "positioned horizontally and the filter weight imbalance is minimal.\n"
            + "It is recommended to use that one first.\n"
            + "The FCS will automatically try a recovery if needed. "
            + "If it fails, the operator should use recoveryRotationForward to align the carousel to the STANDBY Position.")
    public void recoveryRotationStraight() {
        if (!isAtStandby()) {
            throw new RejectedCommandException(name + " no socket at STANDBY method recoveryRotationForward is not relevant.");
        }
        int deltaPositionAtStandby = socketAtStandby.getDeltaPosition();
        this.raiseWarning(CA_ROTATION_RECOVERY, String.format(" |delta position at STANDBY| = %d is over %d "
                + "about to try recoveryRotationStraight", socketAtStandby.getDeltaPosition(), recoveryStandbyDeltaPosition), name);
        int newPos = socketAtStandby.getStandbyPosition() - deltaPositionAtStandby;
        FCSLOG.info(name + " about to rotateToAbsolutePosition to " + newPos);
        this.rotateToAbsolutePosition(newPos);
    }

    @Override
    public boolean isActionCompleted(FcsEnumerations.MobileItemAction action) {
        if (ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            return carouselController.isTargetReached();

        } else if (ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION.equals(action)) {
            return carouselController.isTargetReached();
        }
        return false;
    }

    @Override
    public void updateStateWithSensorsToCheckIfActionIsCompleted() {
        // updateStateWithSensors updates also position, readCurrent, readVelocity,
        // etc...
        updateStateWithSensorsDuringMotion();
    }

    /**
     * Starts action ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION or
     * ROTATE_CAROUSEL_TO_RELATIVE_POSITION. This : - enable controller, - make
     * controller go to required position previously written in field
     * absoluteTargetPosition or relativeTargetPosition, - writeControlWord "3F" or
     * "7F" on controller depending if we want to go to a relative position or to an
     * absolute position.
     *
     * @param action
     */
    @Override
    public void startAction(FcsEnumerations.MobileItemAction action) {

        main.updateAgentState(FcsEnumerations.FilterState.valueOf("ROTATING"));

        if (ROTATE_CAROUSEL_TO_RELATIVE_POSITION.equals(action)) {
            carouselController.enableAndWriteRelativePosition(this.relativeTargetPosition);

        } else if (ROTATE_CAROUSEL_TO_ABSOLUTE_POSITION.equals(action)) {
            carouselController.writeTargetPosition(this.absoluteTargetPosition);

            FcsUtils.checkAndWaitConditionWithTimeoutAndFixedDelay(
                () -> isRotationAllowedByPLC(),
                () -> tcpProxy.updatePDOData(),
                "checkRotationAllowedByPLC",
                name + ": PLC did not allow motion after trying every 100ms during 500 ms",
                500,
                100);

            // Waiting for relay to send correct info
            FcsUtils.sleep(100, name);

            carouselController.goToOperationEnable();

            // TODO REVIEW CAROUSEL:
            // ask Pierre to check that it is still necessary
            updateBrakesStatus();
            if (!areBrakesAllowingRotation()) {
                FcsUtils.checkAndWaitConditionWithTimeout(
                    () -> areBrakesAllowingRotation(),
                    () -> updateBrakesStatus(),
                    "checkCarouselBrakesReleased",
                    name + ": carousel brakes are still activated.",
                    500);
            }

            carouselController.writeControlWord(ControlWord.ABSOLUTE_POSITION_AND_MOVE);
        }
    }


    @Override
    public void abortAction(FcsEnumerations.MobileItemAction action, long delay) {
        FCSLOG.finer(() -> name + " is ABORTING action " + action.toString() + " within delay " + delay);
        this.carouselController.stopPosition();
        this.carouselController.goToSwitchOnDisabled();
    }

    @Override
    public void endAction(FcsEnumerations.MobileItemAction action) {
        FCSLOG.finer(() -> name + " is ENDING action " + action.toString());
        int deltaPosBeforeDisable = position - absoluteTargetPosition;
        FCSLOG.info(name + " delta position before disable = " + deltaPosBeforeDisable);
        this.carouselController.goToSwitchOnDisabled();
        main.updateAgentState(FcsEnumerations.FilterState.valueOf("CAROUSEL_STOPPED"));
        /**
         * switch back controller to PositionSensorTypeEncoderSSI LSSTCCSFCS-223 this
         * has to be done here in endAction, because it has to be executed whatever
         * happened before (even if controller is in default)
         */
        setControllerPositionSensorTypeEncoderSSI();
        /*
         * Read position can be done now because controller sensor type is EncoderSSI.
         */
        position = carouselController.readPosition();
        startPosition = 0;
        FCSLOG.info(name + " position after disable = " + position + " delta " + (position - absoluteTargetPosition));
        initializeClampsState();
        updateStateWithSensors();
    }

    @Override
    public void quickStopAction(FcsEnumerations.MobileItemAction action, long delay) {
        this.carouselController.stopPosition();
        this.carouselController.goToSwitchOnDisabled();
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ END of ROTATION COMMANDS **************************
     */
    /**
     * ****************************************************************
     */
    /**
     * ****************************************************************
     */
    /**
     * ************ methods which override FilterHolder *** ***********
     */
    /**
     * ****************************************************************
     */
    /**
     * Return true if carousel is holding a filter at STANDBY position.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Override
    public boolean isHoldingFilter() {
        return this.isHoldingFilterAtStandby();
    }

    // TODO REVIEW FILTER HOLDER find something more clever to do here.
    @Override
    public boolean isNotHoldingFilter() {
        return !this.isHoldingFilterAtStandby();
    }

    @Override
    public boolean isAtHandoff() {
        return false;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return the ID of filter at STANDBY")
    @Override
    public int getFilterID() {
        return socketAtStandby.getFilterID();
    }

    /**
     * Read the clamps state from PDO : all the clamp sensors are read at one time.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read sensors and update state", timeout = 2000)
    @Override
    public synchronized void updateStateWithSensors() {

        long beginTime = System.currentTimeMillis();

        // Check if the system is in simulation mode
        if (!FcsUtils.isSimu()) {
            // Check the time elapsed since the last sync was sent on the bus
            // If this amounts to less that 100ms, use the current values and do not send a sync
            // The 100 ms rate was decided so that it would be useful for the trending and still
            // be higher than the rate at which the system is able to receive consecutive syncs
            // which is ~ 50 ms according to Guillaume Daubard
            if (beginTime - lastUpdateStateWithSensors.get() < 100) {
               return;
            }
        }
        lastUpdateStateWithSensors.set(beginTime);

        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-carousel")) {
            hyttc580.checkBooted();
            if (!clampsStateInitialized) {
                throw new FcsHardwareException(
                        name + ": clamps state not initialized. " + " Please launch command initializeClampsState.");
            }
            tcpProxy.updatePDOData();
            updateState();
            updateMinLockedThreshold();
            updateTemperatures();
            updateHoldingBrakesState();
            publishData();
        }
    }

    /**
     * This is similar to updateStateWithSensors but without
     * readAndUpdateOutputInterlocks() because there is no need to refresh the
     * output interlocks (signals coming from autochanger when the rotation has begun).
     * updateStateWithSensorsDuringMotion updates state only from PDO. But
     * readAndUpdateOutputInterlocks reads SDO.
     */
    private void updateStateWithSensorsDuringMotion() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensorsDuringMotion-carousel")) {
            hyttc580.checkBooted();
            if (!clampsStateInitialized) {
                throw new FcsHardwareException(
                        name + ": clamps state not initialized. " + " Please launch command initializeClampsState.");
            }
            tcpProxy.updatePDOData();
            updateState();
            updateHoldingBrakesState();
            publishData();
            carouselController.publishDataDuringMotion();
        }
    }

    /**
     * read lockSensorMinLocked on hyttc580.     * don't read more than one time by minute.
     */
    public void updateMinLockedThreshold() {
        // do not execute more than one time by minute (60000ms)
        if (System.currentTimeMillis() - lastUpdateLockSensorMinLocked.get() < 60000) {
            return;
        }
        minLockedThreshold = (int) readLockSensorMinLocked();
        lastUpdateLockSensorMinLocked.set(System.currentTimeMillis());
    }

    public long readLockSensorMinLocked() {
        return hyttc580.readLockSensorMinLocked();
    }

    /**
     * Overridden method from FilterHolder
     *
     * @return false
     */
    @Override
    public boolean isAtOnline() {
        // carousel is never at ONLINE
        return false;
    }

    /**
     * ****************************************************************
     */
    /**
     * ************ END of methods which override FilterHolder ********
     */
    /**
     * ****************************************************************
     */
    public void updateSocketAtStandbyReadSensorsNoPublication() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateSocketAtStandbyReadSensorsNoPublication-carousel")) {
            hyttc580.checkBooted();
            hyttc580.checkInitialized();
            tcpProxy.updatePDOData();
            updateSocketAtStandbyState();
        }
    }

    public void updateSocketAtStandbyState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateSocketAtStandbyState-carousel")) {
            this.socketAtStandbyID = hyttc580.getSocketId(hyttc580.getPdo1());
            //if TTC30 is sleeping (POWER SAVE ACTIVATED), PDO1 is in binary format: 101 + all zeros.
            if (socketAtStandbyID >= 1 && socketAtStandbyID <= 5) {
                socketAtStandby = socketsMap.get("socket" + socketAtStandbyID);
                socketAtStandby.updateState();
                socketAtStandby.updateFilterID();
            }
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update socket at STANDBY state from hyttc580 data.", timeout = 2000)
    public void updateSocketAtStandbyWithSensors() {
        updateSocketAtStandbyReadSensorsNoPublication();
        publishData();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update state from hyttc580 data.", timeout = 2000)
    public void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-carousel")) {
            updateSocketAtStandbyState();
            // doesn't read again controller CPU because the following values have been
            // refreshed by PDO.
            position = carouselController.getPosition();
            // The use of the scheduler is required in order to leave readCurrent thread as
            // soon as possible. Since we use the scheduler and not FcsUtils async functions
            // we need to use startAsync to get proper timing indentation/markers
            int level = FcsUtils.getTimingLevel();
            Level logLevel = FcsUtils.getLogLevel();
            subs.getScheduler().schedule(() -> {
                FcsUtils.startAsync(level, logLevel);
                updateSocketNotAtStandby();
                FcsUtils.endAsync();
            }, 0, TimeUnit.SECONDS);
        }
    }

    public void updateSocketNotAtStandby() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateSocketNotAtStandby")) {
            int socketNotAtStandbyID = hyttc580.getSocketId(hyttc580.getPdo2());
            //if TTC30 is sleeping (POWER SAVE ACTIVATED) PDO2 is in binary format: 101 + all zeros.
            if (socketNotAtStandbyID >= 1 && socketNotAtStandbyID <= 5) {
                CarouselSocket socketToUpdate = socketsMap.get("socket" + socketNotAtStandbyID);
                socketToUpdate.updateState();
            }
        }
    }

    public FilterClampState getClampsStateAtStandby() {
        if (socketAtStandby == null) {
            return FilterClampState.UNDEFINED;
        }

        return socketAtStandby.getClampsState();
    }

    public boolean isEmptyAtStandby() {
        if (socketAtStandby == null) {
            return false;
        }

        return socketAtStandby.isEmpty();
    }

    public boolean isUnclampedOnFilterAtStandby() {
        if (socketAtStandby == null) {
            return false;
        }

        return socketAtStandby.isUnclampedOnFilter();
    }

    /**
     * read Temperatures on pt100.
     *
     * don't read more than one time by minute.
     */
    public void updateTemperatures() {
        // do not execute more than one time by minute (60000ms)
        if (System.currentTimeMillis() - lastUpdateTemperatures.get() < 60000) {
            return;
        }
        readBrakesAndMotorTemperatures();
        meanClampsTemperature = readMeanClampsTemperature();
        lastUpdateTemperatures.set(System.currentTimeMillis());
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read on pt100 the temperatures", timeout = 2000)
    public void readBrakesAndMotorTemperatures() {
        pt100.updateTemperatures();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Read on hyttc580 the mean temperature for all the clamps in Celsius", timeout = 2000)
    public double readMeanClampsTemperature() {
        return hyttc580.readMeanClampsTemperature();
    }


    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Activate POWER SAVE mode.", alias = "activatePowerSave")
    public void powerSave() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("powerSave")) {
            carouselController.activatePowerSave();
            agentStateService.updateAgentState(CarouselPowerState.LOW_POWER);
            FCSLOG.info(getName() + ": POWER SAVE activated.");
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ADVANCED, description = "Deactivate POWER SAVE mode.")
    public void powerOn() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("powerSaveDeactivate")) {
            carouselController.deactivatePowerSave();
            agentStateService.updateAgentState(CarouselPowerState.REGULAR);
            FCSLOG.info(getName() + ": POWER SAVE deactivated.");
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT,
            description = "Disable interlock shutter. Do it only if you understand what you are doing.")
    public void interlockShutterDisable() {
        hyttc580.interlockShutterDisable();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT,
            description = "Enable interlock shutter. Do it only if you understand what you are doing.")
    public void interlockShutterEnable() {
        hyttc580.interlockShutterEnable();
    }

    public void registerHighLevelActionDuration(GeneralAction action) {
        String path = "duration/" + action.name();
        dataProviderDictionaryService.registerData(new KeyValueData(path, 0.0));
        DataProviderInfo info = dataProviderDictionaryService.getDataProviderDictionary().getDataProviderInfoForPath(path);
        info.addAttribute(DataProviderInfo.Attribute.UNITS, "millisecond");
        info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "Duration of the action " + action.name());
    }
}
