package org.lsst.ccs.subsystems.fcs.common;

import org.lsst.ccs.drivers.canopenjni.PDOData;

/**
 * To store the values returned by a PDO. The PDO request is a way to read all
 * the values returned by a CANopen devices in the same request.
 *
 * For example, if we have an ADC which has 8 physical entries, 8 sensors can be
 * physically connected to this ADC. So to read the value sent by the sensor we
 * need to know the nodeID of the ADC and on which entry the sensor is connected
 * to (inputNB).
 *
 * A PDO request is a CANopen command "sync".
 *
 * The response to a PDO request is a String which is formatted like :
 * "ADC1_1=10,ADC1_2=F53B, ...." That means that the sensor connected to the
 * entry 1 of ADC which nodeID is 1, has the value 10. And that the sensor
 * connected to the entry 2 of ADC which nodeID is 1, has the value F53B.
 *
 * The goal of this class is to provide a method (updatePDOs) to extract the
 * values sent by sensors from this response and to store this values in a hasp
 * map.
 *
 */
public class PDOStorage extends PDOData {

    private static final long serialVersionUID = 2487036147845829605L;

    /**
     * Returns the value stored for a CANopen device identified by its nodeID and
     * for an input number on this CANopen device.
     *
     * @param nodeID
     * @param inputNB
     * @return
     */
    public long getVarValue(int nodeID, int inputNB) {
        return pdos.get(nodeID + inputNB);
    }

    public void merge(PDOData d) {
        pdos.putAll(d.getPDOs());
    }

    /**
     * For the simulation ONLY. This set a new value in the PDO Storage in order to
     * simulate a sensor.
     *
     * @param nodeID
     * @param inputNB
     * @param varValue
     */
    public void setVarValue(int nodeID, int inputNB, int varValue) {
        pdos.put(nodeID + inputNB, Long.valueOf(varValue));
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("PDO STORAGE:");
        sb.append("(SIZE=").append(pdos.size()).append(")");
        sb.append(pdos.toString());
        return sb.toString();
    }

}
