package org.lsst.ccs.subsystems.fcs;

import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterFamily;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is a model for an optical filter for the camera.
 *
 */
public class Filter {

    @LookupName
    private String name;

    @ConfigurationParameter(description = "weight of this filter", range = "0..100", units = "kg", category = "filter")
    private volatile double weight;

    @ConfigurationParameter(description = "Filter identification number as coded on its frame and read with 6 hall effect sensors.", range = "0..63", units = "unitless", category = "filter")
    private volatile int filterID;

    @ConfigurationParameter(description = "Filter family: S ; D ; F ; T ; O; U; N", units = "unitless", category = "filter")
    private volatile FilterFamily family;

    /**
     * Default constructor.
     */
    public Filter() {};
    /**
     * Build a filter with an ID and a weight and a socketName.
     *
     * @param filterID
     * @param weight
     */
    Filter(int filterID, double weight) {
        this.filterID = filterID;
        this.weight = weight;
    }

    /**
     * Returns filterID as Integer
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Filter identification number.")
    public int getFilterID() {
        return filterID;
    }

    public String getName() {
        return name;
    }

    /**
     * return filter weight
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Filter weight.")
    public double getWeight() {
        return weight;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Filter family.")
    public FilterFamily getFamily() {
        return family;
    }

    public void setName(String name) {
        this.name = name;
    }

    public void setFamily(FilterFamily family) {
        this.family = family;
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("Filter: " + FcsUtils.padLeft(name, 4) + " | ");
        sb.append("ID: " + FcsUtils.padLeft(Integer.toString(filterID), 2) + " | ");
        sb.append("Weight: " + FcsUtils.padLeft(Double.toString(weight) + "kg", 6) + " | ");
        sb.append("Family: " + family.getFamilyName());
        return sb.toString();
    }

}
