package org.lsst.ccs.subsystems.fcs.utils;

import static org.lsst.ccs.subsystems.fcs.FCSCst.SOCKET_NAME;

import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedDeque;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.subsystems.fcs.EPOSEnumerations;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;

/**
 * This class gathers static final variables and static methods.
 *
 * By example, to creates Objects for GUI and Status Bus.
 *
 * @author virieux
 */
public final class FcsUtils {
    private static final Logger FCSLOG = Logger.getLogger(FcsUtils.class.getName());
    /**
     * The maximum value which a number coded on 2 bytes can reach.
     */
    public static final int MAX_VALUE_2BYTES = 65535;

    /**
     * The maximum number of CANopen devices which can be on a CANbus. Also the
     * maximum CANopen node ID a device can have.
     */
    public static final int MAX_NODE_ID = 127;

    /**
     * A value for all fcs Modules tick millis.
     */
    public static final int TICK_MILLIS = 3000;

    private static final String[] CAN_OPEN_COMMANDS = new String[] { "sync", "scan", "rsdo", "wsdo", "info", "quit",
            "srtr", "reset" };

    /**
     *
     * @param name
     */
    public static void checkSocketName(String name) {
        String errorMsg = ": invalid socket name - Socket name must be \"" + SOCKET_NAME + 'X' + "\" where 0 " + '<'
                + "= X " + '<' + "= 5";

        /* Socket Name can be an empty String or */
        /* Socket Name can be "AC" if filter is on AUTOCHANGER. */
        if (name.isEmpty() || name.contentEquals("AC"))
            return;

        /* Socket Name must start with "socket". */
        if (!name.startsWith(SOCKET_NAME))
            throw new IllegalArgumentException(name + errorMsg);

        /* Socket Name must end with 1, 2, 3, 4 or 5. */
        String end = name.substring(6);
        try {
            int ix = Integer.parseInt(end);
            if (ix < 1 || ix > 5)
                throw new IllegalArgumentException(name + errorMsg);

        } catch (NumberFormatException ex) {
            throw new IllegalArgumentException(name + errorMsg, ex);
        }
    }

    /**
     * Computes a new value given as argument in replacing bit which is at position
     * given as argument by 0.
     *
     * @param val value to transform
     * @param bit bit number to be replaced
     * @return new value
     * @throws IllegalArgumentException if bit <0 or bit >15
     */
    public static int force2zero(int val, int bit) {
        if (bit < 0 || bit > 15) {
            throw new IllegalArgumentException(String.format("bad value for bit=%s - must be : >0 and <15", bit));
        }
        return val & ~(1 << bit);
    }

    /**
     * Return a new value where the digit (bit number given as argument) has been
     * replaced by 1.
     *
     * @param val
     * @param bit
     * @return
     */
    public static int force2one(int val, int bit) {
        if (bit < 0 || bit > 15) {
            throw new IllegalArgumentException(String.format("bad value for bit=%s - must be : >0 and <15", bit));
        }
        return val | (1 << bit);
    }

    /**
     * Sets the nth bit
     *
     * @param val     the value to modify
     * @param bit     the bit number to set
     * @param zeroOne if zero forces to zero. Forces to one otherwise
     * @return
     */
    public static int forceBit(int val, int bit, int zeroOne) {
        if (zeroOne == 0) {
            return force2zero(val, bit);
        } else {
            return force2one(val, bit);
        }
    }

    /**
     * build a Can Open wsdo Command that can understand the C wrapper.
     *
     * exemple : wsdo,2,6411,01,2,3000
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @param size
     * @param data
     * @return
     */
    public static String buildWsdoCommand(int nodeID, int index, int subindex, int size, int data) {
        char sep = ',';
        StringBuilder sb = new StringBuilder("wsdo").append(sep);
        sb.append(Integer.toHexString(nodeID)).append(sep);
        sb.append(Integer.toHexString(index)).append(sep);
        sb.append(Integer.toHexString(subindex)).append(sep);
        sb.append(Integer.toHexString(size)).append(sep);
        sb.append(Integer.toHexString(data));
        return sb.toString();
    }

    /**
     * build a Can Open rsdo Command that can understand the C wrapper. exemple:
     * rsdo,1,1018,0
     *
     * @param nodeID
     * @param index
     * @param subindex
     * @return
     */
    public static String buildRsdoCommand(int nodeID, int index, int subindex) {
        char sep = ',';
        StringBuilder sb = new StringBuilder("rsdo").append(sep);
        sb.append(Integer.toHexString(nodeID)).append(sep);
        sb.append(Integer.toHexString(index)).append(sep);
        sb.append(Integer.toHexString(subindex));
        return sb.toString();
    }

    /**
     * returns true if the command given as argument is an available CANopen
     * command.
     *
     * @param command
     * @return
     */
    public static boolean isValidCommandWord(String command) {
        for (String s : CAN_OPEN_COMMANDS) {
            if (s.equals(command))
                return true;
        }
        return false;
    }

    /**
     * @param command
     * @throws IllegalArgumentException if command is not a valid command.
     */
    public static void checkCommand(String command) {
        if (command == null)
            throw new IllegalArgumentException(" null command");
        String[] words = command.split(",");
        if (words.length == 0 || !isValidCommandWord(words[0])) {
            throw new IllegalArgumentException(command + " invalid command");
        }
    }

    /**
     * Convert a number coded on 8 bits which represents a signed number (INTEGER8)
     * to a byte.
     *
     * @param integer8
     * @return
     */
    public static byte convertInteger8(long integer8) {
        if (integer8 >= Math.pow(2, 8)) {
            throw new IllegalArgumentException(integer8 + " too big. Should be coded on 8 bits, so < Math.pow(2,8)");
        }
        return (byte) integer8;
    }

    /**
     * Convert a number coded on 16 bits which represents a signed number
     * (INTEGER16) to a short. Used to interpret numbers coming from CANopen
     * devices where we can have signed or unsigned numbers depending of which
     * parameter we read.
     *
     * INTEGER16=0 => value=0 INTEGER16=2356 => value=2356 INTEGER16=65342 =>
     * value=-194 INTEGER16=32767 => value=32767 INTEGER16=32768 => value=-32768
     * INTEGER16=65535 => value=-1
     *
     * @param integer16 a long because readSDO command returns a long.
     * @return
     */
    public static short convertInteger16(long integer16) {
        if (integer16 >= Math.pow(2, 16)) {
            throw new IllegalArgumentException(integer16 + " too big. Should be coded on 16 bits, so < Math.pow(2,16)");
        }
        return (short) integer16;
    }

    /**
     * Convert a number coded on 32 bits which represents a signed number
     * (INTEGER32) to an int.
     *
     * @param integer32
     * @return
     */
    public static int convertInteger32(long integer32) {
        if (integer32 >= Math.pow(2, 32)) {
            throw new IllegalArgumentException(integer32 + " too big. Should be coded on 32 bits, so < Math.pow(2,32)");
        }
        return (int) integer32;
    }

    /**
     * Displays the list of parameters that we have to define for a given mode.
     *
     * @param modeInString
     * @return
     */
    public static String displayListParameters(String modeInString) {
        StringBuilder sb = new StringBuilder("List of parameters for mode: ");
        sb.append(modeInString);
        sb.append("\n");
        EPOSEnumerations.Parameter[] params = EPOSEnumerations.EposMode.valueOf(modeInString).getParameters();
        for (EPOSEnumerations.Parameter param : params) {
            sb.append(param.display());
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * Displays the list of parameters.
     *
     * @return
     */
    public static String displayListParameters() {
        StringBuilder sb = new StringBuilder("List of parameters : ");
        sb.append("\n");
        EPOSEnumerations.Parameter[] params = EPOSEnumerations.Parameter.values();
        for (EPOSEnumerations.Parameter param : params) {
            sb.append(param.display());
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * For online clamps.
     *
     * @param initialValue
     * @param finalValue
     * @param increment
     * @return
     */
    public static int getSignedStepHeight(int initialValue, int finalValue, int increment) {
        if (finalValue - initialValue > 0) {
            return increment;
        } else {
            return -increment;
        }
    }

    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) {
        System.out.println(FcsUtils.displayListParameters("HOMING"));
        System.out.println(FcsUtils.displayListParameters("CURRENT"));
        System.out.println(FcsUtils.displayListParameters("PROFILE_POSITION"));
        System.out.println(FcsUtils.displayListParameters());
    }

    public static int computeNewCurrentValue(int stepHeight, int currentValue, int finalValue) {
        if (stepHeight > 0) {
            return Math.min(currentValue + stepHeight, finalValue);
        } else {
            return Math.max(currentValue + stepHeight, finalValue);
        }
    }

    public static void sleep(int duration, String itemName) {
        try {
            FCSLOG.finest(() -> itemName + " BEGIN SLEEP for " + duration + " milliseconds");
            reportSleep(duration);
            Thread.sleep(duration);
            FCSLOG.finest(() -> itemName + " END SLEEP for " + duration + " milliseconds");
        } catch (InterruptedException ex) {
            String msg = itemName + " interrupted while sleeping";
            FCSLOG.log(Level.SEVERE, itemName + msg, ex);
        }
    }

    public static void checkPositive(int arg) {
        if (arg < 0) {
            throw new IllegalArgumentException(arg + ":illegal value. Must be > 0");
        }
    }

    /**
     * Compute index to read a byte on a CanOpenPlutoGateway
     *
     * @param byteNumero
     * @return
     */
    public static int returnPlutoIndex(int byteNumero) {
        int index = 0x6000 + byteNumero / 4;
        return index;
    }

    /**
     * Compute subindex to read a byte on a CanOpenPlutoGateway
     *
     * @param byteNumero
     * @return
     */
    public static int returnPlutoSubindex(int byteNumero) {
        int subindex;
        subindex = 1 + byteNumero % 4;
        return subindex;
    }

    /**
     * Wait for a condition to be true, with increasing wait time from 10ms to 200ms
     * Will run the update function if not null while waiting
     * Will exit ok if condition is met before timeout
     * Otherwise will raise exception
     *
     * @param condition
     * @param update
     * @param name
     * @param error
     * @param timeout
     */
    public static void checkAndWaitConditionWithTimeout(Callable<Boolean> condition, Runnable update, String name, String error, long timeout) {
        try (AutoTimed at = new AutoTimed(name)) {
            long startTime = System.currentTimeMillis();
            long delay = 10;
            while (System.currentTimeMillis() - startTime < timeout) {
                try {
                    if (condition.call())
                        return;
                } catch (Exception e) {
                    throw new RejectedCommandException("failed predicate test for " + error + " " + e.getMessage());
                }
                if (update != null)
                    update.run();
                try {
                    Thread.sleep(delay);
                } catch (InterruptedException e) {
                }
                delay *= 2;
                if (delay > 200)
                    delay = 200;
            }
            throw new RejectedCommandException(error);
        }
    }

    /**
     * Wait for a condition to be true, with increasing wait time from 10ms to 200ms
     * Will run the update function if not null while waiting
     * Will exit ok if condition is met before timeout (1000 ms)
     * Otherwise will raise exception
     *
     * @param condition
     * @param update
     * @param name
     * @param error
     */
    public static void checkAndWaitCondition(Callable<Boolean> condition, Runnable update, String name, String error) {
        checkAndWaitConditionWithTimeout(condition, update, name, error, 1000);
    }

    /**
     * Wait for a condition to be true, with fixed delay
     * Will run the update function if not null while waiting
     * Will exit ok if condition is met before timeout
     * Otherwise will raise exception
     *
     * @param condition
     * @param update
     * @param name
     * @param error
     * @param timeout
     * @param delay
     */
    public static void checkAndWaitConditionWithTimeoutAndFixedDelay(Callable<Boolean> condition, Runnable update, String name, String error, long timeout, long delay) {
        try (AutoTimed at = new AutoTimed(name)) {
            long startTime = System.currentTimeMillis();
            while (System.currentTimeMillis() - startTime < timeout) {
                try {
                    if (condition.call())
                        return;
                } catch (Exception e) {
                    throw new RejectedCommandException("failed predicate test for " + error + " " + e.getMessage());
                }
                if (update != null)
                    update.run();
                try {
                    Thread.sleep(delay);
                } catch (InterruptedException e) {
                }
            }
            throw new RejectedCommandException(error);
        }
    }


    /**
     * Wait for a condition to be true, with increasing wait time from 10ms to 200ms
     * Will run the update function if not null while waiting
     * Will exit ok if condition is met or upon timeout
     *
     * @param condition
     * @param update
     * @param name
     * @param timeout
     */
    public static void waitCondition(Callable<Boolean> condition, Runnable update, String name, long timeout) {
        try (AutoTimed at = new AutoTimed(name)) {
            long startTime = System.currentTimeMillis();
            long delay = 10;
            while (System.currentTimeMillis() - startTime < timeout) {
                try {
                    if (condition.call())
                        return;
                } catch (Exception e) {
                }
                try {
                    Thread.sleep(delay);
                } catch (InterruptedException e) {
                }
                if (update != null)
                    update.run();
                delay *= 2;
                if (delay > 200)
                    delay = 200;
            }
        }
    }
    /*
     * code timing
     */

    static class TimingEntry {
        /**
         * TimingEntry is used to time function and output the duration on the logs. Default is to output at
         * defaultTimedLoggerLevel level (default level for the application is INFO).
         */

        public TimingEntry(String name, int indentationLevel, boolean async, Level logLevel) {
            this.name = name;
            this.indentationLevel = indentationLevel;
            this.async = async;
            this.logLevel = logLevel;
        }

        private String name;
        private long start = System.currentTimeMillis();
        private long sleep = 0;
        private int indentationLevel;
        private boolean async;
        private Level logLevel;
        public long getDuration() {
            return System.currentTimeMillis() - start;
        }

        public long getSleep() {
            return sleep;
        }

        public synchronized void reportSleep(long s) {
            sleep += s;
        }

        public String getName() {
            return name;
        }

        public boolean isAsync() {
            return async;
        }

        public int getIndentationLevel() {
            return indentationLevel;
        }

        public Level getLogLevel() {
            return logLevel;
        }
    }

    static ThreadLocal<ConcurrentLinkedDeque<TimingEntry>> localStack = new ThreadLocal<ConcurrentLinkedDeque<TimingEntry>>() {
        protected ConcurrentLinkedDeque<TimingEntry> initialValue() {
            return new ConcurrentLinkedDeque<>();
        }
    };

    public static int getTimingLevel() {
        return localStack.get().peek() == null ? 0 : localStack.get().peek().getIndentationLevel();
    }

    public static Level getLogLevel() {
        return getLogLevel(Level.parse(defaultTimedLoggerLevelAsString));
    }

    public static Level getLogLevel(Level defaultLevel) {
        return localStack.get().peek() == null ? defaultLevel : localStack.get().peek().getLogLevel();
    }

    static boolean isAsync() {
        return localStack.get().peek() != null && localStack.get().peek().isAsync();
    }

    static void startTimedAction(String actionName, Level logLevel) {
        Level parentLevel = getLogLevel(logLevel); // if a parent Timing Entry exists, then Logger.Level will match its value
        localStack.get().push(new TimingEntry(actionName, getTimingLevel() + 1, isAsync(), parentLevel));
        int space = getTimingLevel() * 2;
        String out = String.format("%1s%" + space + "s start timed %s", isAsync() ? "x" : ">", " ", actionName);
        FCSLOG.log(parentLevel, out);
    }

    static void endTimedAction(String actionName) {
        int space = getTimingLevel() * 2;
        TimingEntry te = localStack.get().pop();
        if (te == null) {
            FCSLOG.warning("endTimedAction not balanced for " + actionName);
            return;
        }

        String out = String.format("%1s%" + space + "s end timed   %s duration %d ms incl sleep %d ms",
                te.isAsync() ? "x" : ">", " ", actionName, te.getDuration(), te.getSleep());
        FCSLOG.log(te.getLogLevel(), out);

        // let's propagate the sleep above
        reportSleep(te.getSleep());

    }

    public static void startAsync(int level, Level logLevel) {
        Level parentLevel = getLogLevel(logLevel);
        localStack.get().push(new TimingEntry("async", level + 1, true, parentLevel));
        int space = (level * 2 + 2);
        String out = String.format("x%" + space + "s start timed %s", " ", "async");
        FCSLOG.log(parentLevel, out);
    }

    public static void endAsync() {
        endTimedAction("async");
    }

    /**
     * report that we slept through the FcsUtil sleep method
     *
     * @param ms time we slept
     */
    static void reportSleep(long ms) {
        TimingEntry te = localStack.get().peek();
        if (te != null)
            te.reportSleep(ms);
    }

    /**
     * class to be used in a try-with-resource block to report the time spent in the
     * block. Handles a hierarchy of calls Will report also the time spent sleeping
     * through the FcsUtil sleep method (fixed time sleep)
     */
    public static class AutoTimed implements AutoCloseable {
        String name;
        Level logLevel;

        public AutoTimed(String name) {
            this(name, Level.parse(defaultTimedLoggerLevelAsString));
        }

        public AutoTimed(String name, Level logLevel) {
            this.name = name;
            this.logLevel= logLevel;
            startTimedAction(name, logLevel);
        }

        @Override
        public void close() {
            endTimedAction(name);
        }
    }

    static ExecutorService execSvc = Executors.newCachedThreadPool(new ThreadFactory() {
        public Thread newThread(Runnable r) {
            Thread t = Executors.defaultThreadFactory().newThread(r);
            t.setDaemon(true);
            return t;
        }
    });

    /**
     * execute in parallel a list of tasks will wait for all tasks to terminate if a
     * task throws an exception it will be propagated
     *
     * @param tasks
     */
    public static void parallelRun(Runnable... tasks) {
        AsyncTasks t = new AsyncTasks();

        for (Runnable task : tasks) {
            t.asyncRun(task);
        }

        t.await();
    }

    public static class AsyncTasks {
        List<Future<?>> f = new ArrayList<>();

        int level = getTimingLevel();
        Level logLevel = getLogLevel(Level.parse(defaultTimedLoggerLevelAsString));
        public AsyncTasks asyncRun(Runnable task) {
            Runnable r = () -> {
                startAsync(level, logLevel);
                task.run();
                endAsync();
            };

            // f.add(execSvc.submit(task));
            f.add(execSvc.submit(r));
            return this;
        }

        public void await() {
            // wait and propagate any exception
            for (Future<?> ff : f) {
                try {
                    ff.get();
                } catch (InterruptedException | ExecutionException e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    public static AsyncTasks asyncRun(Runnable task) {
        return new AsyncTasks().asyncRun(task);
    }

    public static AsyncTasks asyncRun() {
        return new AsyncTasks();
    }

    // some timing tests are different with simu/test environments

    static boolean simuTested = false;
    static boolean simu = false;

    public static boolean isJUnitTest() {
        for (StackTraceElement element : Thread.currentThread().getStackTrace()) {
            if (element.getClassName().startsWith("org.junit.")) {
                return true;
            }
        }
        return false;
    }

    public static boolean isSimu() {
        if (!simuTested) {
            Properties props = BootstrapResourceUtils.getBootstrapSystemProperties();
            simu = "simulation".equals(props.getProperty("org.lsst.ccs.run.mode"));
            if (simu == false)
                simu = isJUnitTest();
            simuTested = true;
        }
        return simu;
    }

    private static String defaultTimedLoggerLevelAsString = "FINE";

    public static void setDefaultTimedLoggerLevel(String levelName) {
        defaultTimedLoggerLevelAsString= levelName;
    }
    public static String getDefaultTimedLoggerLevel() {
        return defaultTimedLoggerLevelAsString;
    }

}
