package org.lsst.ccs.subsystems.fcs;

import static org.lsst.ccs.commons.annotations.LookupField.Strategy.CHILDREN;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.SIBLINGS;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.AC_SENSOR_ERROR;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.HARDWARE_ERROR;

import java.time.Duration;
import java.util.EnumMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;

import org.lsst.ccs.ConfigurationService;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.DESCENDANTS;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.commons.annotations.Persist;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.subsystems.fcs.FCSCst.CHANGER_TCPPROXY_NAME;
import static org.lsst.ccs.subsystems.fcs.FCSCst.NO_FILTER;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutochangerInclination;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutochangerTrucksLocation;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.GeneralAction;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.FilterHolder;
import org.lsst.ccs.subsystems.fcs.common.PersistentCounter;
import org.lsst.ccs.subsystems.fcs.common.PlutoGatewayInterface;
import org.lsst.ccs.subsystems.fcs.drivers.CanOpenSeneca4RTD;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;
import org.lsst.ccs.subsystems.fcs.utils.ActionGuard;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This class is used with the final hardware and prototype in which we have
 * online clamps to hold the filter at online position. It's not used for Single
 * Filter Test.
 *
 * CAMERA PROTECTION SYSTEM Signal coming from carousel : CS : Carousel stopped
 * on socket CF0: No filter in carousel socket CF1: Filer in carousel socket
 * CFC: Filter clamped on carousel socket
 *
 * Signal coming from loader : LPS: Loader at Storage position (loader is
 * connected) LRH: Loader at Handoff position, closed but not clamped
 *
 * Local Protection Module Signals emitted by autochanger: enableRailLin1 enable driver
 * linear rail (X+) enableRailLin2 enable follower linear rails (X-) enableClamps
 * enableLatches
 *
 * Sensors in AC1 and AC2 and not in PROTO :
 *
 * loaderPresenceSensors * lockOutSensors
 *
 *
 * @author virieux
 */
public class Autochanger implements FilterHolder, AlertRaiser, HasLifecycle {
    private static final Logger FCSLOG = Logger.getLogger(Autochanger.class.getName());
    @LookupField(strategy = TOP)
    private Subsystem subs;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    @LookupField(strategy = TREE)
    private ConfigurationService configurationService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupName
    protected String name;

    @LookupField(strategy = SIBLINGS, pathFilter = CHANGER_TCPPROXY_NAME)
    private BridgeToHardware tcpProxy;

    private final PlutoGatewayInterface plutoGateway;

    @LookupField(strategy = SIBLINGS, pathFilter = "filterManager")
    private FilterManager filterManager;

    @LookupField(strategy = SIBLINGS, pathFilter = "filterIdentificator")
    protected FilterIdentificator filterIdentificator;

    @LookupField(strategy = CHILDREN, pathFilter = "autochangerTrucks")
    private AutochangerTwoTrucks autochangerTrucks;

    @LookupField(strategy = CHILDREN, pathFilter = "latches")
    protected AutochangerTwoLatches latches;

    @LookupField(strategy = CHILDREN, pathFilter = "onlineClamps")
    private AutochangerThreeOnlineClamps onlineClamps;

    @LookupField(strategy = TREE)
    private MainModule main;

    @LookupField(strategy = TREE, pathFilter = ".*\\/tempSensorsDevice1")
    private CanOpenSeneca4RTD tempSensorsDevice1;

    @LookupField(strategy = TREE, pathFilter = ".*\\/tempSensorsDevice2")
    private CanOpenSeneca4RTD tempSensorsDevice2;

    @LookupField(strategy = TREE)
    protected AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = SIBLINGS, pathFilter = ".*carousel")
    private FilterHolder carousel;

    @LookupField(strategy = SIBLINGS, pathFilter = ".*loader")
    private FilterHolder loader;

    @ConfigurationParameter(range = "0..5000", description = "Time to wait until protection system signals are updated", units = "millisecond", category = "autochanger")
    public volatile long timeToUpdateProtectionSystem = 2000;

    @ConfigurationParameter(range = "0..100", description = "Time to wait between activateBrake and disableOperation for online clamps", units = "millisecond", category = "autochanger")
    private volatile int waitTimeForBrakeOC = 20;

    @ConfigurationParameter(range = "0..100", description = "Time to wait between activateBrake and disableOperation for linear rails", units = "millisecond", category = "autochanger")
    private volatile int waitTimeForBrakeLR = 20;

    @Persist
    protected volatile int filterOnTrucksID;


    /**
     * CPS signal : carousel_CS Carousel stop on socket
     */
    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/carouselStoppedAtStandbySensors")
    private ComplementarySensors carouselStoppedAtStandbySensors;


    /**
     * Loader Presence Sensor : true if loader is connected on autochanger
     */
    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/loaderPresenceSensors")
    private ComplementarySensors loaderPresenceSensors;

    /**
     * Local Protection Module signals
     */
    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/lpmLinearRail1Status")
    private DigitalSensor lpmLinearRail1Status;

    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/lpmLinearRail2Status")
    private DigitalSensor lpmLinearRail2Status;

    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/lpmOnlineClampsStatus")
    private DigitalSensor lpmOnlineClampsStatus;

    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/lpmLatchesStatus")
    private DigitalSensor lpmLatchesStatus;

    //AutochangerInclination for MCM
    //OUT_AIN (DigitalSensor)
    @LookupField(strategy = DESCENDANTS, pathFilter = ".*\\/OUT_AIN")
    private DigitalSensor AIN;

    // Counter for actions
    protected Map<GeneralAction, PersistentCounter> actionCounter;


    public Autochanger(PlutoGatewayInterface plutoGateway) {
        this.plutoGateway = plutoGateway;
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    public BridgeToHardware getTcpProxy() {
        return tcpProxy;
    }

    public int getWaitTimeForBrakeOC() {
        return waitTimeForBrakeOC;
    }

    public int getWaitTimeForBrakeLR() {
        return waitTimeForBrakeLR;
    }

    /**
     * For MCM
     *
     * @return
     */
    public AutochangerInclination getAutochangerInclinationState() {
        if (AIN.isOn()) {
            return AutochangerInclination.STRAIGHT;
        } else {
            return AutochangerInclination.TILTED;
        }
    }

    /**
     * For MCM
     *
     * @return trucks location
     */
    public AutochangerTrucksLocation getAutochangerTrucksLocation() {
        if (isAtHandoff()) {
            return AutochangerTrucksLocation.HANDOFF;
        } else if (isAtOnline()) {
            return AutochangerTrucksLocation.ONLINE;
        } else if (isAtStandby()) {
            return AutochangerTrucksLocation.STANDBY;
        } else {
            return AutochangerTrucksLocation.IN_TRAVEL;
        }
    }

    /**
     * For MCM
     *
     *
     * @return online proximity distance read on proximity sensor
     */
    public int getOnlineProximityDistance() {
        return autochangerTrucks.getOnlineProximityDistance();
    }

    @Override
    public void build() {
        dataProviderDictionaryService.registerClass(StatusDataPublishedByAutochangerSeneca.class, name + "/temperatures");
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("PublishAutochangerTemperatures", this::checkSenecaTemperatures)
            .withIsFixedRate(true).withPeriod(Duration.ofMinutes(10)));
        plutoGateway.registerSubsystemIdentifier(name);
        actionCounter = new EnumMap<>(GeneralAction.class);
        for (GeneralAction action : new GeneralAction[]{
            GeneralAction.GRAB_FILTER_AT_STANDBY,
            GeneralAction.MOVE_AND_CLAMP_FILTER_ONLINE}) {
            actionCounter.put(action, PersistentCounter.newCounter(action.getCounterPath(), getSubsystem(), action.name()));
            registerActionDuration(action);
        }
    }

    @Override
    public void init() {

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };

        alertService.registerAlert(HARDWARE_ERROR.getAlert(), alwaysClear);
        alertService.registerAlert(HARDWARE_ERROR.getAlert(plutoGateway.getName()), alwaysClear);
        alertService.registerAlert(AC_SENSOR_ERROR.getAlert(), alwaysClear);
        alertService.registerAlert(AC_SENSOR_ERROR.getAlert(name), alwaysClear);
    }
    /**
     * Return true if an error has been detected on sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if an error has been detected on sensors.")
    public boolean isSensorsInError() {
        return latches.isInError() || onlineClamps.isInError() || autochangerTrucks.isPositionSensorsInError();
    }

    /**
     * returns trucks to be able to send commands to trucks in fcs subsystem.
     *
     * @return trucks
     */
    public AutochangerTwoTrucks getAutochangerTrucks() {
        return autochangerTrucks;
    }

    /**
     * returns onlineClamps to be able to send commands to onlineClamps in fcs
     * subsystem.
     *
     * @return onlineClamps
     */
    public AutochangerThreeOnlineClamps getOnlineClamps() {
        return onlineClamps;
    }

    /**
     * return latches
     *
     * @return
     */
    public AutochangerTwoLatches getLatches() {
        return latches;
    }

    /**
     * If autochanger holds a filter, returns filterID, else returns 0.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "If autochanger holds a filter, return filterID, else return 0.")
    @Override
    public int getFilterID() {
        return filterOnTrucksID;
    }

    /**
     * Used by simulator only.
     *
     * @param filterOnTrucksID
     */
    public void setFilterOnTrucksID(int filterOnTrucksID) {
        int formerFilterOnTrucksID = this.filterOnTrucksID;

        if (formerFilterOnTrucksID != filterOnTrucksID) {
            this.filterOnTrucksID = filterOnTrucksID;
            subs.getAgentPersistenceService().persistNow();
        }
    }

    /**
     *
     * @param filterID
     * @return return true if filter with filterID is on autochanger
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "return true if filter with filterID is on autochanger")
    public boolean isFilterOnAC(int filterID) {
        return filterOnTrucksID == filterID && filterOnTrucksID != 0;
    }

    /**
     *
     * @param filterID
     * @return true if filterID is on Autochanger and trucks are ONLINE and latches
     *         are closed.
     */
    public boolean isFilterONLINE(int filterID) {
        return isFilterOnAC(filterID) && this.isAtOnline() && this.isHoldingFilter();
    }

    /**
     * If autochanger holds a filter, return filter name else return null.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "If a filter is in autochanger, return filter name else return NO_FILTER.")
    public String getFilterOnTrucksName() {
        if (filterOnTrucksID != 0) {
            return filterManager.getFilterNameByID(filterOnTrucksID);
        } else {
            return NO_FILTER;
        }
    }

    /**
     * Returns the boolean field loaderConnected. If the loaderConnected boolean is
     * being updated and waits for a response from a sensor, this methods waits
     * until loaderConnected is updated. If the field loaderConnected is not being
     * updated, it returns immediately the field loaderConnected.
     *
     * @return atHandoff
     *
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the loader is connected to the camera. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderConnected() {
        return loaderPresenceSensors.isOn() && !loaderPresenceSensors.isInError();
    }

    /**
     *
     * @return
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if the carousel is holding the filter at STANDBY position.")
    public boolean isCarouselHoldingFilterAtStandby() {
        return this.carousel.isAtStandby() && this.carousel.isHoldingFilter();
    }

    /**
     * Return true if loader is holding a filter. This command doesn't
     * read again the sensors.
     *
     * @return
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if loader is holding a filter. "
            + "This command doesn't read again the sensors.")
    public boolean isLoaderHoldingFilterAtHandoff() {
        return loader.isHoldingFilter();
    }

    /**
     * @return true if all autochanger hardware is initialized and bridge is
     *         hardwareReady.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all autochanger CANopen devices are booted, identified and initialized.")
    public boolean isCANDevicesReady() {
        return tcpProxy.allDevicesBooted();
    }

    /**
     * @return if all autochanger CANopen devices are booted, identified and
     *         initialized and homing of the controllers is done.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all autochanger CANopen devices are booted, identified and "
            + "initialized and homing of the controllers is done.")
    public boolean isHardwareReady() {
        return tcpProxy.allDevicesBooted() && isInitialized();
    }

    /**
     * Return true if plutoGateway, trucks, latches and onlineClamps are
     * initialized.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if all autochanger hardware is initialized.")
    public boolean isInitialized() {
        return plutoGateway.isInitialized() && autochangerTrucks.isInitialized() && latches.isInitialized()
                && onlineClamps.isInitialized();
    }

    public boolean isLinearRailMotionAllowed() {
        return this.lpmLinearRail1Status.isOn() && this.lpmLinearRail2Status.isOn();
    }

    public void waitForProtectionSystemUpdate() {
        waitForRailMotionAllowed(timeToUpdateProtectionSystem);
    }

    /**
     * wait until linear rail motion is authorized by local protection system.
     * because there are temporisation in local protection system, there is a delay
     * between sensors authorize us to move and local protection system authorizes.
     *
     * @param timeout after this delay don't wait anymore.
     */
    private void waitForRailMotionAllowed(long timeout) {
        final String methodName = "waitForRailMotionAllowed";
        final String className = Autochanger.class.getName();
        FCSLOG.entering(className, methodName, timeout);
        FcsUtils.waitCondition(() -> isLinearRailMotionAllowed(), () -> plutoGateway.updateValues(),
                className, timeout);

        // we don't trust the first read
        FcsUtils.sleep(10, className);
        if (!isLinearRailMotionAllowed()) {
            FCSLOG.finer("Attention: isLinearRailMotionAllowed glitch 10ms , waiting again");
        }
        FcsUtils.sleep(50, className);
        if (!isLinearRailMotionAllowed()) {
            FCSLOG.finer("Attention: isLinearRailMotionAllowed glitch 60ms, waiting again");
        }
        FcsUtils.waitCondition(() -> isLinearRailMotionAllowed(), () -> plutoGateway.updateValues(),
                "waitForRailMotionAllowed", timeout);
        FCSLOG.exiting(Autochanger.class.getName(), methodName);
    }

    @Override
    public void postStart() {
        final String methodName = "postStart";
        FCSLOG.entering(Autochanger.class.getName(), methodName);
        if (plutoGateway.isBooted()) {
            initializeGateway();
            try {
                // In postStart if a RuntimeException is thrown, fcs cannot starts. To avoid
                // this we have to catch the exception and raise an ALERT of level ALARM.
                updateStateWithSensors();
            } catch (Exception ex) {
                this.raiseAlarm(HARDWARE_ERROR, name + " couldn't updateStateWithSensors in postStart ", ex);
            }
        } else {
            plutoGateway.raiseAlarmIfMissing();
        }
        if (tempSensorsDevice1.isBooted()) { // if device1 is booted, it is expected that device2 is as well
            checkSenecaTemperatures(); // first occurrence of the periodic task
        }
        plutoGateway.publishSubsystemIdentifierData();
        FCSLOG.exiting(Autochanger.class.getName(), methodName);
    }

    private void initializeGateway() {
        try {
            this.plutoGateway.initializeAndCheckHardware();
        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, name + " couldn't initialize gateway", plutoGateway.getName(), ex);
        }
    }

    /**
     * For end users, for tests and in engineering mode. This command can be used to
     * recover after a missing hardware during fcs startup. For example when fcs was
     * started before hardware power up. Check all hardware and publish data.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "For end users, for tests and in engineering mode. "
            + "This command can be used to recover after a missing hardware "
            + "during fcs startup. For example, when fcs was started before hardware "
            + "power up. Check all hardware and publish data.", timeout = 5000)
    public void initializeHardware() {
        FCSLOG.info(name + " BEGIN initializeHardware");

        // check that all hardware is booted and identified.
        tcpProxy.bootProcess();

        try {
            this.postStart();
            autochangerTrucks.postStart();
            onlineClamps.postStart();
            latches.postStart();

        } catch (FcsHardwareException ex) {
            this.raiseAlarm(HARDWARE_ERROR, " couldn't initialize autochanger", ex);
        }
        FCSLOG.info(name + " END initializeHardware");
    }

    /**
     * Check if Local Protection Module allows linear rails motions.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if Local Protection Module allows linear rails motion.")
    public void checkLinearRailMotionAllowed() {
        FcsUtils.checkAndWaitCondition(() -> lpmLinearRail1Status.isOn() && lpmLinearRail2Status.isOn(),
                () -> plutoGateway.updateValues(), "checkLinearRailMotionAllowed",
                name + ": linear rails motion NOT allowed by Local Protection Module.");
    }

    /**
     * Check if Local Protection Module allows latches motion.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if Local Protection Module allows latches motion.")
    public void checkLatchMotionAllowed() {
        FcsUtils.checkAndWaitCondition(() -> lpmLatchesStatus.isOn(), () -> plutoGateway.updateValues(),
                "checkLatchMotionAllowed", name + ": latches open or close NOT allowed by Local Protection Module.");
    }

    /**
     * Check if Local Protection Module allows online clamps motion.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if Local Protection Module allows online clamps motion.")
    public void checkOnlineClampMotionAllowed() {
        FcsUtils.checkAndWaitCondition(() -> lpmOnlineClampsStatus.isOn(), () -> plutoGateway.updateValues(),
                "checkOnlineClampMotionAllowed",
                name + ": Online clamps open or close NOT allowed by Local Protection Module.");
    }

    /**
     * Check if latches can be opened. If latches can't be opened because the
     * conditions for opening are not filled, this command throws a
     * RejectedCommandException. If latches are in error, it throws a
     * FcsHardwareException.
     *
     * @param bypassSensorError
     * @throws FcsHardwareException,RejectedCommandException
     */
    public void checkConditionsForOpeningLatches(boolean bypassSensorError) {
        FCSLOG.info(name + " checking pre-conditions for opening latches");

        updateStateWithSensors();

        checkLatchesInitialized();

        if (latches.isInError() && !bypassSensorError) {
            throw new FcsHardwareException(name + ": latches are in ERROR state - can't open latches.");

        } else if (isEmpty()) {
            throw new RejectedCommandException(name + ": no filter in autochanger - can't open latches.");

        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter but is not "
                    + "at handoff position neither at standby - can't open latches.");

        } else if (autochangerTrucks.isAtStandby()) {
            checkConditionsForOpeningLatchesAtStandby();

        } else if (autochangerTrucks.isAtHandoff()) {
            checkConditionsForOpeningLatchesAtHandoff();
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if the latches can be opened.")
    public void checkConditionsForOpeningLatches() {
        checkConditionsForOpeningLatches(false);
    }

    private void checkLatchesInitialized() {
        if (!latches.isInitialized()) {
            throw new RejectedCommandException(
                    name + ": latches are not initialized. " + "Please initialize hardware first.");
        }
    }

    private void checkConditionsForOpeningLatchesAtStandby() {
        /* autochanger holds a filter at STANDBY position */
        if (isCarouselHoldingFilterAtStandby()) {
            FCSLOG.info(() -> name + " carousel is holding filter at STANDBY => latches can be open safely.");

        } else {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter and is  "
                    + "at STANDBY position but carousel doesn't hold the filter " + "- can't open latches.");
        }
    }

    private void checkConditionsForOpeningLatchesAtHandoff() {
        /* autochanger holds a filter at HANDOFF position */
        if (isLoaderHoldingFilterAtHandoff()) {
            FCSLOG.info(() -> name + " loader is holding filter at HANDOFF => latches can be open safely.");

        } else {
            throw new RejectedCommandException(name + ": autochanger is loaded with a filter and is  "
                    + "at HANDOFF position but loader doesn't hold the filter " + "- can't open latches.");
        }
    }

    /**
     * Check if Autochanger latches can be closed.
     *
     * @param bypassSensorError
     * @throws RejectedCommandException
     */
    public void checkConditionsForClosingLatches(boolean bypassSensorError) {
        FCSLOG.info(() -> name + " checking conditions for closing latches.");

        updateStateWithSensors();
        checkLatchesInitialized();

        if (latches.isInError() && !bypassSensorError) {
            throw new RejectedCommandException(name + ": latches are in ERROR state - can't close latches.");

        } else if (isEmpty()) {
            throw new RejectedCommandException(name + ": no filter in autochanger - can't close latches.");

        } else if (!autochangerTrucks.isAtStandby() && !autochangerTrucks.isAtHandoff()) {
            throw new RejectedCommandException(
                    name + ": autochanger is not " + "at handoff position neither at standby - can't close latches.");
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if Autochanger latches can be closed.")
    public void checkConditionsForClosingLatches() {
        checkConditionsForClosingLatches(false);
    }

    /**
     * log if actions on online clamps are allowed, throws an exception otherwise.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Check and log if actions on online clamps are allowed,"
            + "throws an exception otherwise.")
    public void checkConditionsForActioningOnlineClamps() {
        if (!isAtOnline()) {
            throw new RejectedCommandException(
                    name + " actions not allowed on ONLINE clamps " + "when trucks are not at ONLINE.");
        }
        if (isEmpty()) {
            throw new RejectedCommandException(
                    name + " actions not allowed on ONLINE clamps " + "when no filter on trucks.");
        }
        FCSLOG.info(() -> name + " autochangerTrucks are at ONLINE, a filter is there : actions are allowed.");
    }

    /**
     * This methods checks that filter can be moved by trucks. Returns immediately
     * if autochanger trucks are empty. Throws RejectedCommandException if a filter
     * is in trucks AND held both by autochanger and (loader or carousel).
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     * @throws RejectedCommandException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Check if its safe for filters to move autochanger trucks.")
    public void checkFilterSafetyBeforeMotion() {

        String msg = name + " can't move trucks because ";
        this.updateStateWithSensors();

        if (this.isSensorsInError()) {
            handleSensorsError();

        } else if (isEmpty()) {
            FCSLOG.info(() -> name + " trucks are empty - can move");

        } else if (!this.onlineClamps.isOpened()) {
            throw new RejectedCommandException(
                    name + " : a filter " + "is in trucks and ONLINE clamps are NOT OPENED.");

        } else if (isAtStandby()) {
            checkFilterSafetyAtStandby(msg);

        } else if (autochangerTrucks.isAtHandoff()) {
            FCSLOG.info(() -> name + " : trucks are AT HANDOFF ");
            checkFilterSafetyAtHandoff(msg);

        } else if (this.autochangerTrucks.isAtOnline()) {
            FCSLOG.info(() -> name + " : trucks are at ONLINE ");
            checkFilterSafetyAtOnline(msg);

        } else if (!latches.isHoldingFilter() && !carousel.isHoldingFilter()) {
            throw new RejectedCommandException(msg + " neither autochanger neither carousel "
                    + "is holding filter. Close latches or carousel clamps.");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at STANDBY.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtStandby(String message) {
        if (isHoldingFilter() && isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(message + "both carousel and autochanger are holding filter at STANDBY");

        } else if (!isHoldingFilter() && !isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    message + "neither carousel nor autochanger are holding filter at STANDBY");

        } else if (!(latches.isClosed() || latches.isOpened())) {
            throw new RejectedCommandException(message + " latches must be opened or closed.");

        } else {
            FCSLOG.info(() -> name + " filter safe at STANDBY - can move");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at HANDOFF.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtHandoff(String message) {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    message + "autochanger is holding filter at HANDOFF but another filter is at STANDBY "
                            + "- can't move trucks");
        }
        if (isHoldingFilter() && isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(message + "both loader and autochanger are holding filter at HANDOFF");
        } else if (!isHoldingFilter() && !isLoaderHoldingFilterAtHandoff()) {
            throw new RejectedCommandException(
                    message + "neither loader nor autochanger are holding filter at HANDOFF");
        } else {
            FCSLOG.info(() -> name + " filter safe at HANDOFF - can move");
        }
    }

    /**
     * Check if trucks can be moved when loaded with a filter at ONLINE. It can be
     * move : - if no filter is at STANDBY (otherwise risk of collision) - AND
     * ONLINE clamps are opened, - AND latches are CLOSED.
     *
     * @param message
     * @throws FcsHardwareException
     */
    private void checkFilterSafetyAtOnline(String message) {
        if (isHoldingFilter() && this.isCarouselHoldingFilterAtStandby()) {
            throw new RejectedCommandException(
                    message + "autochanger is holding filter at ONLINE but another filter is at STANDBY "
                            + "- can't move trucks");
        }

        if (isHoldingFilter() && !onlineClamps.isOpened()) {
            throw new RejectedCommandException(
                    message + "onlineClamps have to be opened on filter at ONLINE - can't move trucks.");
        }

        if (!isHoldingFilter() && !onlineClamps.isLocked()) {
            throw new RejectedCommandException(
                    message + "neither latches nor onlineClamps are holding filter at ONLINE");
        } else {
            FCSLOG.info(() -> name + " filter safe at ONLINE - can move");
        }
    }

    /**
     * Test in carousel position at STANDBY is correct to go and grab a filter
     * or store a filter on carousel.
     */
    public void checkCarouselDeltaPosition() {
        if (carousel instanceof Carousel) {
            ((Carousel) carousel).checkDeltaPosition();
        }
    }

    /**
     * Used in AutochangerTwoTrucks
     */
    public void carouselUpdateStandbyState() {
        if (carousel instanceof Carousel) {
            ((Carousel) carousel).updateSocketAtStandbyState();
        }
    }

    /**
     * Raise ALERT and throw FcsHardwareException with a detailed message when error
     * is detected in sensors.
     *
     * @throws FcsHardwareException
     */
    private void handleSensorsError() {
        boolean transientError = false;
        String msg = name + " error detected in sensors :";
        if (this.autochangerTrucks.isPositionSensorsInError()) {
            msg += " trucks position sensors";
            transientError = autochangerTrucks.isPositionSensorErrorsTransient();
        }
        if (this.latches.isFilterEngagedInError()) {
            transientError = transientError || isLatchesErrorTransient();
        }
        if (this.latches.isInError()) {
            msg += " latches sensors";
        }
        if (this.onlineClamps.isInError()) {
            msg += " onlineClamps sensors";
        }
        if (transientError) {
            this.raiseWarning(AC_SENSOR_ERROR, msg + " - can be a transient error. ", name);
        } else {
            this.raiseAlarm(AC_SENSOR_ERROR, msg);
            throw new FcsHardwareException(msg);
        }
    }

    private boolean isLatchesErrorTransient() {
        return Math.abs(autochangerTrucks.getPosition() - 984000) < 5000;
    }

    /**
     * This method reads all the sensors and updates the trucks, latches and online
     * clamps state.
     *
     * @throws org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Update clamp state in reading sensors.")
    @Override
    public synchronized void updateStateWithSensors() {
        // do not execute if last finished less than 100 ms ago.
        // we keep it at the end of actions but don't repeat it
        if (System.currentTimeMillis() - lastUpdateStateWithSensors.get() < 100 && !FcsUtils.isSimu())
            return;
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateStateWithSensors-ac")) {
            plutoGateway.checkInitialized();
            this.plutoGateway.updateValues();
            /* tcpProxy.updatePDOData() updates all devices values that can be updated by PDO */
            tcpProxy.updatePDOData();
            updateState();
            updateFilterOnTrucksID();
            lastUpdateStateWithSensors.set(System.currentTimeMillis());
        }
    }

    AtomicLong lastUpdateStateWithSensors = new AtomicLong(0);

    /**
     * update filterOnTrucksID
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "update filterOnTrucksID.")
    public void updateFilterOnTrucksID() {
        if (!latches.isEmpty()) {
            setFilterOnTrucksID(filterIdentificator.getFilterId());
        } else {
            setFilterOnTrucksID(0);
        }
    }

    private void updateState() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("updateState-ac")) {
            this.autochangerTrucks.updateState();
            this.latches.updateState();
            this.onlineClamps.updateState();
            this.filterIdentificator.retrieveFilterId();
            if (this.isAtStandby() && carouselStoppedAtStandbySensors.isOn()) {
                carousel.updateStateWithSensors();
            }
        }
    }

    /**
     * Update FCS state and FCS readiness state and publish on the status bus. Check
     * that Autochanger hardware is ready to be operated and moved. This means that
     * : - all CAN open devices are booted, identified and initialized, -
     * initialization has been done on the controllers.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Force FCS state and FCS readiness state to be ready and publishes on the status bus. TO BE USED ONLY IN EXTREME CASES.")
    public void updateFCSStateToReady() {
        if (this.isCANDevicesReady() && latches.isInitialized() && autochangerTrucks.isInitialized()
                && onlineClamps.isInitialized()) {
            main.updateFCSStateToReady();
        }
    }

    /****************************************************************************
     * Methods from FilterHolder
     ***************************************************************************
     */

    /**
     * Return true if a filter is in trucks and latches are LOCKED.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if a filter is in trucks and latches are CLOSED.")
    @Override
    public boolean isHoldingFilter() {
        return latches.isHoldingFilter();
    }

    @Override
    public boolean isNotHoldingFilter() {
        return latches.isOpened();
    }

    /**
     * Return true if autochanger trucks are at HANDOFF. This command doesn't read
     * again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks are at HANDOFF. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtHandoff() {
        return autochangerTrucks.isAtHandoff();
    }

    /**
     * Return true if autochanger trucks are at STANDBY. This command doesn't read
     * again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks are at STANDBY. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtStandby() {
        return autochangerTrucks.isAtStandby();
    }

    /***************************************************************************
     * end of Methods from FilterHolder
     ***************************************************************************
     */

    /**
     * return true if autochanger trucks are at ONLINE.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks are at ONLINE. "
            + "This command doesn't read again the sensors.")
    @Override
    public boolean isAtOnline() {
        return autochangerTrucks.isAtOnline();
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if autochanger trucks position is around approachStandbyPosition. "
            + "This command doesn't read again the sensors.")
    public boolean isAtApproachStandbyPosition() {
        return Math.abs(autochangerTrucks.getPosition() - autochangerTrucks.getApproachStandbyPosition()) < 1000;
    }

    public boolean isPositionCorrectForExchangeWithLoader() {
        return autochangerTrucks.isPositionCorrectForExchangeWithLoader();
    }

    /****************************************************************************
     * AutochangerTrucks ENGINEERING_ROUTINE ACTION commands
     ***************************************************************************
     */

    /**
     * Move Autochanger trucks to the Handoff position. If Autochanger trucks are
     * already at HANDOFF position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Handoff position.")
    public void goToHandOff() {
        autochangerTrucks.goToHandOff();
    }

    /**
     * Move Autochanger trucks to the Handoff position with a better accuracy.
     *
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Handoff "
            + "position to prepare an exchange with the loader. The position accuracy requirement is tighter than for command goToHandoff (see positionRangeAtHandoffForLoader and positionRangeAtHandoff).")
    public void goToHandOffForLoader() {
        autochangerTrucks.goToHandOffForLoader();
    }

    /**
     * Move Autochanger trucks to the Online position. If Autochanger trucks are
     * already at ONLINE position, it does nothing.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Online position.")
    public void goToOnline() {
        autochangerTrucks.goToOnline();
    }

    /**
     * Move Autochanger trucks to the Standby position. If Autochanger trucks are
     * already at STANDBY position, it does nothing.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Standby position.")
    public void goToStandby() {
        checkCarouselDeltaPosition();
        autochangerTrucks.goToStandby();
    }

    /**
     * For end users at the console.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Move Autochanger trucks to the Approach Standby position. "
            + "Move fast if position > approachStandbyPosition, slow otherwise.")
    public void moveToApproachStandby() {
        checkCarouselDeltaPosition();
        /* trucks are between standby and approachStandby we have to move slowly*/
        if (autochangerTrucks.getPosition() > autochangerTrucks.getApproachStandbyPosition()) {
            autochangerTrucks.moveToApproachStandbyPositionWithLowVelocity();
        } else {
            autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "goToOnline with filter then adjust position then close and finally lock clamps", autoAck = false, timeout = 20000)
    public void moveAndClampFilterOnline() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveAndClampFilterOnline")) {
            actionCounter.get(GeneralAction.MOVE_AND_CLAMP_FILTER_ONLINE).increment();
            final long beginTime = System.currentTimeMillis();
            autochangerTrucks.moveAndClampFilterOnline();
            subs.publishSubsystemDataOnStatusBus(
                new KeyValueData(getDurationPath(GeneralAction.MOVE_AND_CLAMP_FILTER_ONLINE),
                    System.currentTimeMillis() - beginTime));
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "goToOnline with filter then adjust position but DO NOT CLOSE or LOCK the clamps", autoAck = false, timeout = 20000)
    public void moveFilterOnlineNoClamping() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveFilterOnlineNoClamping")) {
            autochangerTrucks.moveFilterOnlineNoClamping();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "unlock and open clamps in doing homing if needed, "
            + "then move slowly to approach online position, then move fast to handoff", autoAck = false, timeout = 20000)
    public void unclampAndMoveFilterToHandoff() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("unclampAndMoveFilterToHandoff")) {
            autochangerTrucks.unclampAndMoveFilterToHandoff();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go slowly to approachStandbyPosition ")
    public void moveToApproachStandbyPositionWithLowVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachStandbyPositionWithLowVelocity")) {
            autochangerTrucks.moveToApproachStandbyPositionWithLowVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go fast to approachStandbyPosition ")
    public void moveToApproachStandbyPositionWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachStandbyPositionWithHighVelocity")) {
            autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go fast to approachOnlinePosition ")
    public void moveToApproachOnlinePositionWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachOnlinePositionWithHighVelocity")) {
            autochangerTrucks.moveToApproachOnlinePositionWithHighVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go slowly to slow motion position ")
    public void moveToSlowMotionPositionWithLowVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToSlowMotionPositionWithLowVelocity")) {
            autochangerTrucks.moveToSlowMotionPositionWithLowVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go fast to slow motion position ")
    public void moveToSlowMotionPositionWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToSlowMotionPositionWithHighVelocity")) {
            autochangerTrucks.moveToSlowMotionPositionWithHighVelocity();
        }
    }


    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "go slowly to approachOnlinePosition ")
    public void moveToApproachOnlinePositionWithLowVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToApproachOnlinePositionWithLowVelocity")) {
            autochangerTrucks.moveToApproachOnlinePositionWithLowVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "align follower and move empty from approachStandby to Handoff")
    public void alignFollowerAndMoveEmptyFromApproachToHandoff() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("alignFollowerAndMoveEmptyFromApproachToHandoff")) {
            autochangerTrucks.alignFollowerAndMoveEmptyFromApproachToHandoff();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move slowly to standby position")
    public void moveToStandbyWithLowVelocity() {
        if (carousel instanceof Carousel) {
            ((Carousel) carousel).checkDeltaPosition();
        }
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToStandbyWithLowVelocity")) {
            autochangerTrucks.moveToStandbyWithLowVelocity();
        }
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move fast to handoff position")
    public void moveToHandoffWithHighVelocity() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveToHandoffWithHighVelocity")) {
            autochangerTrucks.moveToHandoffWithHighVelocity();
        }
    }

    /**
     * Move a filter to standby position step by step. A first step with a high
     * speed to the approachStandbyPosition. a second step to standby position with
     * low speed.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move filter to approachPosition with high speed and move to STANDBY with lowSpeed.", autoAck = false, timeout = 30000)
    public void moveFilterToStandby() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveFilterToStandby")) {
            autochangerTrucks.moveFilterToStandby();
        }
    }

    /**
     * Move a filter to standby position plus deltaStandbyPosition step by
     * step.
     *
     * A first step with a high speed to the approachStandbyPosition. A second
     * step to standby position with low speed.
     *
     * @param deltaStandbyPosition
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move filter to approachPosition with high speed and move to STANDBY + deltaStandbyPosition with lowSpeed.", autoAck = false, timeout = 30000)
    public void moveFilterToStandbyPlusDelta(int deltaStandbyPosition) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveFilterToStandbyPlusDelta")) {
            autochangerTrucks.moveFilterToStandbyPlusDelta(deltaStandbyPosition);
        }
    }

    /**
     * move follower truck to masterPosition. When moving along the rails with follower
     * controller in MASTER_ENCODER mode, at the end of motion there is a small
     * misalignment. This consists in : setting follower Controller mode to
     * PROFILE_POSITION, go to absolute masterPosition pos, and then set
     * follower Controller mode back to MASTER_ENCODER mode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Align follower controller position to driver controller position.", timeout = 20000)
    public void alignFollower() {
        autochangerTrucks.alignFollowerStrict();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Align follower controller position to driver controller position to prepare interaction with loader (see configuration parameter in category autochanger : deltaPositionForAlignStrictLoader).", timeout = 20000)
    public void alignFollowerForLoader() {
        autochangerTrucks.alignFollowerForLoader();
    }

    /**
     * change speed & acceleration profile to slow
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "change ProfileVelocity parameter to lowSpeed and slowProfileAcceleration and slowProfileDeceleration")
    public void slowTrucksProfile() {
        autochangerTrucks.slowProfile();
    }

    /**
     * change speed and acceleration profile to fast
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "change ProfileVelocity and ProfileAcceleration and ProfileDeceleration parameters to highSpeed")
    public void fastTrucksProfile() {
        autochangerTrucks.fastProfile();
    }

    /**
     * Do homing of both trucks controllers.
     *
     * @throws RejectedCommandException
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Do homing for both  controllers.")
    public void homingTrucks() {
        autochangerTrucks.homing();
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "move to approach position with high velocity then move to STANDBY with low velocity")
    public void dockingAtStandbyPositionWithoutFilter() {
        try (ActionGuard g = autochangerTrucks.new ReleasedACBrakes()) {
            autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
            autochangerTrucks.moveToStandbyWithLowVelocity();
        }
    }

    /****************************************************************************
     * END OF AutochangerTrucks ENGINEERING_ROUTINE commands
     ***************************************************************************
     */

    /****************************************************************************
     * AutochangerLatches ENGINEERING_ROUTINE ACTION commands
     ***************************************************************************
     */

    /**
     * Open latches
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Open latches.")
    public void openLatches() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("openLatches")) {
            latches.open();
        }
    }

    /**
     * Close latches
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Close latches.")
    public void closeLatches() {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("closeLatches")) {
            latches.close();
        }
    }

    /****************************************************************************
     * END OF AutochangerLatches ENGINEERING_ROUTINE commands
     ***************************************************************************
     */

    /****************************************************************************
     * AutochangerOnlineClamps ENGINEERING_ROUTINE ACTION commands
     ***************************************************************************
     */

    /**
     * Opens the 3 clamps. initial state = CLOSED final state = OPENED. For final
     * products AC1 and AC2. For prototype see openClampsInCurrentMode.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Opens the 3 online clamps in mode PROFILE_POSITION.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void openClamps() {
        onlineClamps.openClamps();
    }

    /**
     * close clamps in mode PROFILE_POSITION. for AC1 and AC2
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = " close clamps in mode PROFILE_POSITION for AC1 and AC2", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void closeClamps() {
        onlineClamps.closeClamps();
    }

    /**
     * Locks clamps : closed with a strong pressure (high current). The clamps have
     * to be CLOSED. At the end of this action, the clamps are CLOSED but a strong
     * pressure to hold safely the clamps.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "Locks clamps : closed with a strong pressure (high current).", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_CLOSING)
    public void lockClamps() {
        onlineClamps.lockClamps();
    }

    /**
     * Unlocks clamps : slows down current sent to controller in order to decrease
     * pressure on the clamps. The clamps have to be LOCKED. At the end of this
     * action, the clamps are CLOSED with a small pressure of the clamp hardware on
     * the filter frame.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "unlocks clamps : decreases current in controller to decrease pressure.", timeout = AutochangerOnlineClamp.TIMEOUT_FOR_OPENING)
    public void unlockClamps() {
        onlineClamps.unlockClamps();
    }

    /**
     * do homing of the 3 clamps
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "do homing of the 3 ONLINE clamps : open in CURRENT mode and homing of controller", timeout = 6000)
    public void homingClamps() {
        onlineClamps.homing();
    }

    /****************************************************************************
     * END OF AutochangerOnlineClamps ENGINEERING_ROUTINE commands
     ***************************************************************************
     */

    /**
     * Return true if there is no filter in the autochanger. This command doesn't
     * read again the sensors.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return true if there is no filter in the autochanger. "
            + "This command doesn't read again the sensors.")
    public boolean isEmpty() {
        return latches.isEmpty();
    }

    /**
     * Moves empty to standby position and close latches on filter at standby on
     * carousel.
     *
     * @throws RejectedCommandException if latches are not OPEN.
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, alias = "grabFilter", description = "Move autochanger trucks to STANDBY position, close latches and stay at STANDBY")
    public void grabFilterAtStandby() {
        updateState();
        if (!latches.isOpened()) {
            throw new RejectedCommandException(name + " latches must be open to execute grabFilterAtStandby");
        }
        actionCounter.get(GeneralAction.GRAB_FILTER_AT_STANDBY).increment();
        final long beginTime = System.currentTimeMillis();
        try (ActionGuard g = autochangerTrucks.new ReleasedACBrakes()) {
            FCSLOG.info(name + " === About to move empty to standby position ===");
            autochangerTrucks.moveToApproachStandbyPositionWithHighVelocity();
            // Tolerance is low for the latches, needs strict alignment
            autochangerTrucks.alignFollowerStrict();
            autochangerTrucks.moveEmptyToStandbyWithLowVelocity();
        }
        FCSLOG.info(name + " ===> filter on autochanger after moveToStandbyEmptyWithLowVelocity =" + filterOnTrucksID
                + " should be != 0");
        updateFilterOnTrucksID();
        latches.close();

        subs.publishSubsystemDataOnStatusBus(
            new KeyValueData(getDurationPath(GeneralAction.GRAB_FILTER_AT_STANDBY),
                System.currentTimeMillis() - beginTime));
        FCSLOG.info(name + ": filter " + filterOnTrucksID + " is now on autochanger");
    }

    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_ROUTINE, description = "open latches then go to approachStandbyPosition with low speed "
            + "and move to HANDOFF with highSpeed.", timeout = 20000, autoAck = false)
    public void moveEmptyFromStandbyToHandoff() {
        moveEmptyFromStandbyToHandoff(false);
    }

    public void moveEmptyFromStandbyToHandoff(boolean withPrecision) {
        try (FcsUtils.AutoTimed at = new FcsUtils.AutoTimed("moveEmptyFromStandbyToHandoff")) {
            if (!this.isCarouselHoldingFilterAtStandby()) {
                throw new RejectedCommandException(name + "%s : can't open latches if carousel is not holding it.");
            }
            if (latches.isClosed()) {
                latches.open();
            }
            waitForProtectionSystemUpdate();
            try (ActionGuard g = autochangerTrucks.new ReleasedACBrakes()) {
                autochangerTrucks.moveToApproachStandbyPositionWithLowVelocity();
                autochangerTrucks.alignFollowerAndMoveEmptyFromApproachToHandoff(withPrecision);
            }
            updateFilterOnTrucksID();
        }
    }

    /**
     * set a new value for the acquisition frequency of linear rail controllers
     * current.
     *
     * @param rate
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "set a new value for the acquisition "
            + "frequency of linear rail controllers current in milliseconds.")
    public void setFastAcqRateLinearRails(int rate) {
        configurationService.change("acTruckXminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("acTruckXplus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.saveChangesForCategories("timers:fastRails");
    }

    /**
     * set a new value for the acquisition frequency of ONLINE clamps controllers
     * current.
     *
     * @param rate
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "set a new value for the acquisition "
            + "frequency of ONLINE clamps controllers current, in milliseconds.")
    public void setFastAcqRateOnlineClamps(int rate) {
        configurationService.change("onlineClampXminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("onlineClampXplus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.change("onlineClampYminus-monitorCurrent", "taskPeriodMillis", rate);
        configurationService.saveChangesForCategories("timers:fast");
    }

    /**
     * speed up online clamps current monitoring for the 3 clamps
     */
    @Deprecated
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "speed up online clamps current monitoring for the 3 clamps")
    public void increaseCurrentMonitoring() {
        configurationService.loadCategories("timers:fast");
    }

    /**
     * slow down online clamps current monitoring for the 3 clamps
     */
    @Deprecated
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "slow down online clamps current monitoring for the 3 clamps")
    public void decreaseCurrentMonitoring() {
        configurationService.loadCategories("timers:slow");
    }

    /**
     * increase current monitoring for the 2 trucks
     */
    public void increaseLinearRailsCurrentMonitoring() {
        configurationService.loadCategories("timers:fastRails");
    }

    /**
     * slow down online clamps current monitoring for the 3 clamps
     */
    public void decreaseLinearRailsCurrentMonitoring() {
        configurationService.loadCategories("timers:slowRails");
    }

    /**
     * to be used in setFilter
     */
    public void lockFilterAtOnline() {
        if (!this.isAtOnline()) {
            throw new FcsHardwareException(name + "is not at ONLINE");
        }
        if (this.isEmpty()) {
            throw new FcsHardwareException(name + " must be loaded with a filter");
        }
        onlineClamps.lockFilterAtOnline();
    }

    /**
     * Read temperatures on both seneca ( 2x 4 values)
     * @return an array of 8 doubles representing temperatures.
     *         The first 4 values are from Seneca 1, and the last 4 are from Seneca 2.
     */
    public double[] readSenecaTemperatures() {
        double[] temperatures = new double[8];
        // read seneca 1
        for (int i= 0; i < 4; i++) {
            temperatures[i] = tempSensorsDevice1.readChannel(i+1);
        }
        // read seneca 2
        for (int i= 0; i < 4; i++) {
            temperatures[i+4] = tempSensorsDevice2.readChannel(i+1);
        }
        return temperatures;
    }

    /**
     * Check autochanger temperatures and publish the values
     */
    public void checkSenecaTemperatures() {
        double[] temperatures = readSenecaTemperatures();
        subs.publishSubsystemDataOnStatusBus(new KeyValueData(name + "/temperatures", createStatusDataPublishedBySeneca(temperatures)));
    }

    /**
     * Create Autochanger temperatures data to be published at regular intervals
     */
    public StatusDataPublishedByAutochangerSeneca createStatusDataPublishedBySeneca(double[] temperatures) {
        return new StatusDataPublishedByAutochangerSeneca(temperatures);
    }

    public void registerActionDuration(GeneralAction action) {
        String path = getDurationPath(action);
        dataProviderDictionaryService.registerData(new KeyValueData(path, 0.0));
        DataProviderInfo info = dataProviderDictionaryService.getDataProviderDictionary().getDataProviderInfoForPath(path);
        info.addAttribute(DataProviderInfo.Attribute.UNITS, "millisecond");
        info.addAttribute(DataProviderInfo.Attribute.DESCRIPTION, "Duration of the action " + action.name());
    }

    public String getDurationPath(GeneralAction action) {
        return "duration/" + name + "/" + action.name();
    }
}
