
package org.lsst.ccs.subsystems.fcs.drivers;

import org.lsst.ccs.command.annotations.Command;

import org.lsst.ccs.subsystems.fcs.errors.RejectedCommandException;

/**
 * Represents a model for the CANopen device SENECA_4RTD which is used in
 * autochanger to monitor 4 temperatures.
 *
 * 2 devices SENECA_4RTD are used to monitor 8 temperatures.
 *
 * @author virieux
 */
public class CanOpenSeneca4RTD extends CanOpenDevice implements MonitorDeviceInterface {

    /* first CAN OPEN index to configure a channel */
    /* 0x2106 for channel 1, 0x2107 for channel 2, 0x2108 for channel 3, 0x2109 for channel 4 */
    final int INDEX = 0x2106;

    /* size of CAN OPEN object to configure this device */
    final int SDO_SIZE = 2;

    /* measure type is in degree °C*/
    final short MEASURE_TYPE = 0;

    /* three wires connexion */
    final short CONNEXION_TYPE = 1;

    /* frequency rejection is 60 Hz*/
    final short FREQUENCY_REJECTION = 1;

    /* filter value = average filter + High resolution */
    final short FILTER_VALUE = 2;

    /* sensor type is PT1000*/
    final int SENSOR_TYPE = 3;

    /**
     * return the canopen INDEX to be used for an channel number.
     *
     * @param channelNO
     * @return
     */
    private int getIndex(int channelNO) {
        return INDEX + channelNO - 1;
    }

    public void setSensorType(int channelNO, int sensorType) {
        this.writeSDO(getIndex(channelNO), 1, SDO_SIZE, sensorType);
    }

    public void setMeasureType(int channelNO, int measureType) {
        this.writeSDO(getIndex(channelNO), 2, SDO_SIZE, measureType);
    }

    public void setConnexionType(int channelNO, int connexionType) {
        this.writeSDO(getIndex(channelNO), 3, SDO_SIZE, connexionType);
    }

    public void setFrequencyRejection(int channelNO, int frequencyRejection) {
        this.writeSDO(getIndex(channelNO), 4, SDO_SIZE, frequencyRejection);
    }

    public void setFilterValue(int channelNO, int filterValue) {
        this.writeSDO(getIndex(channelNO), 5, SDO_SIZE, filterValue);
    }

    public void enableChannel(int channelNO) {
        this.writeSDO(getIndex(channelNO), 6, SDO_SIZE, 1);
    }

    @Override
    public void initializeAndCheckHardware() {
        if (isBooted()) {
            initialize();
        } else {
            throw new RejectedCommandException(name + " is not booted");
        }
    }

    @Override
    public void initialize() {
        for (int i = 1; i < 4; i++) {
            setSensorType(i, SENSOR_TYPE);
            setMeasureType(i, MEASURE_TYPE);
            setConnexionType(i, CONNEXION_TYPE);
            setFrequencyRejection(i, FREQUENCY_REJECTION);
            setFilterValue(i, FILTER_VALUE);
            enableChannel(i);
        }
        this.initialized = true;
        // publishData for the GUI
        this.publishData();
        FCSLOG.info(name + ": is INITIALIZED.");
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "read channel and return its value")
    @Override
    public double readChannel(int channelNO) {
        /* Object 0x6401 of device Seneca4RTD is an INTEGER16 */
        return (short) readSDO(0x6401, channelNO) / (double) 10;
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "read sensor type for a channel")
    public int readSensorType(int channelNO) {
        return (int) readSDO(getIndex(channelNO), 1);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "read measure type for a channel")
    public int readMeasureType(int channelNO) {
        return (int) readSDO(getIndex(channelNO), 2);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "read connexion type for a channel")
    public int readConnexionType(int channelNO) {
        return (int) readSDO(getIndex(channelNO), 3);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "read frequency rejection for a channel")
    public int readFrequencyRejection(int channelNO) {
        return (int) readSDO(getIndex(channelNO), 4);
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "read filter value for a channel")
    public int readFilterValue(int channelNO) {
        return (int) readSDO(getIndex(channelNO), 5);
    }

    @Override
    public void postStart() {
        if (isBooted()) {
            initialize();
        } else {
            this.raiseWarningIfMissing();
        }
    }

}
