package org.lsst.ccs.subsystems.fcs;

import java.time.Duration;
import java.util.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.PersistencyService;

import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.services.alert.AlertService;
import static org.lsst.ccs.bus.states.AlertState.ALARM;
import org.lsst.ccs.bus.states.CommandState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TOP;
import static org.lsst.ccs.commons.annotations.LookupField.Strategy.TREE;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.framework.Signal;
import org.lsst.ccs.framework.SignalHandler;
import org.lsst.ccs.framework.TreeWalkerDiag;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentStateService;
import org.lsst.ccs.services.DataProviderDictionaryService;
import org.lsst.ccs.subsystems.fcs.errors.FcsHardwareException;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.AutochangerInclination;
import static org.lsst.ccs.subsystems.fcs.FcsEnumerations.FcsAlert.UPDATE_ERROR;
import static org.lsst.ccs.subsystems.fcs.utils.FcsUtils.setDefaultTimedLoggerLevel;

import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterReadinessState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.FilterState;
import org.lsst.ccs.subsystems.fcs.FcsEnumerations.McmState;
import org.lsst.ccs.subsystems.fcs.common.AlertRaiser;
import org.lsst.ccs.subsystems.fcs.common.BridgeToHardware;
import org.lsst.ccs.subsystems.fcs.common.BridgeToLoader;
import org.lsst.ccs.subsystems.fcs.common.EPOSController;
import org.lsst.ccs.subsystems.fcs.common.MovedByEPOSController;
import org.lsst.ccs.subsystems.fcs.utils.FcsUtils;

/**
 * This is the Main Module for every software related to the Filters Exchanger :
 * - single-filter-test control-command - scale 1 prototype control-command -
 * real Filters Exchanger This class is designed to gather the commons methods
 * to all these softwares.
 *
 * @author virieux
 */
public abstract class MainModule extends Subsystem implements AlertRaiser, SignalHandler, HasLifecycle {

    // Using package level module so it is possible to setup the level of all the logger in FCS using
    // commands from the toolkit
    private static final Logger FCSLOG = Logger.getLogger(MainModule.class.getPackage().getName());

    @LookupName
    protected String name;

    @LookupField(strategy = TOP)
    protected Subsystem subs;

    @LookupField(strategy = TREE)
    private AlertService alertService;

    @LookupField(strategy = TREE)
    AgentStateService agentStateService;

    @LookupField(strategy = TREE)
    protected AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy = TREE)
    protected DataProviderDictionaryService dataProviderDictionaryService;

    @LookupField(strategy = TREE)
    private PersistencyService persistencyService;

    @LookupField(strategy = TREE)
    private final List<BridgeToHardware> bridges = new ArrayList<>();

    @LookupField(strategy = TREE)
    protected final Map<String, MovedByEPOSController> hardMovedByController = new HashMap<>();

    @LookupField(strategy = TREE)
    protected final Map<String, EPOSController> controllerList = new HashMap<>();

    /**
     * This field can be null
     */
    @LookupField(strategy = TREE)
    private FilterManager filterManager;
    protected BridgeToHardware bridge;
    protected AtomicBoolean haltRequired = new AtomicBoolean(false);
    protected AtomicBoolean stopRequired = new AtomicBoolean(false);

    public MainModule() {
        super("subsystemName-placeholder", AgentInfo.AgentType.WORKER);
    }

    @Override
    public Subsystem getSubsystem() {
        return subs;
    }

    @Override
    public AlertService getAlertService() {
        return alertService;
    }

    public boolean isHaltRequired() {
        return haltRequired.get();
    }

    public boolean isStopRequired() {
        return stopRequired.get();
    }

    /**
     * *** lifecycle methods *************************************************
     */
    @Override
    public void build() {

        agentStateService.registerState(FilterReadinessState.class, "Filter Exchange Readiness State", subs);
        agentStateService.registerState(FilterState.class, "Filter Exchange State", subs);
        agentStateService.registerState(McmState.class, "Filter Exchange State in form suitable for use by MCM", subs);
        agentStateService.registerState(AutochangerInclination.class, "Autochanger inclination state", subs);
        /**
         * What has to be done for each tick of the timer. We have to read
         * sensors, update state and publish on the * status bus to refresh the
         * GUI.
         */
        /* we can't do that is there is no clamp in carousel*/
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("main-updateGui", this::updateGui)
                .withIsFixedRate(true).withLogLevel(Level.WARNING).withPeriod(Duration.ofMinutes(1)));
        //From groovy there can be either 1 or two BridgeToHardware instances.
        //If there is only one, we use it as is. If there are two we pick the one that
        //does not implement the BridgeToLoader interface.
        //Not elegant but for now it should work.
        //If there are none, then we throw an Exception.
        switch (bridges.size()) {
            case 0:
                throw new RuntimeException("No BridgeToHardware instances were detected. Something went wrong in groovy.");
            case 1:
                bridge = bridges.get(0);
                break;
            case 2:
                BridgeToHardware b = bridges.get(0);
                if ( b instanceof BridgeToLoader ) {
                    bridge = bridges.get(1);
                } else {
                    bridge = b;
                }
                break;
            default:
                throw new RuntimeException("We expect at most 2 BridgeToHardware instances, but we got "+bridges.size()+"; something went wrong in groovy");
        }
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("main-checkControllers", this::checkControllers)
                .withIsFixedRate(true).withLogLevel(Level.WARNING).withPeriod(Duration.ofMinutes(10)));
    }

    @Override
    public void init() {
        /* The persistency service is only instantiated for subsystems that use it.
         * The loader does not use persistence so when using the loader in standalone
         * mode, we end up with a null pointer. */
        if (persistencyService != null) {
            final boolean loadAtStartup = true;
            final boolean saveAtShutdown = true;
            persistencyService.setAutomatic(loadAtStartup, saveAtShutdown);
        }

        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
            }
        };
        alertService.registerAlert(UPDATE_ERROR.getAlert(), alwaysClear);
    }

    /**
     * This methods updates the boolean hardwareReady from the hardware bridge.
     *
     * @return true if hardware is booted, identified and initialized
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Return true if all hardware is booted with the correct serial number.")
    public boolean allDevicesBooted() {
        return this.bridge.allDevicesBooted();
    }

    /**
     * Returns list of hardware names. Used by GUI.
     *
     * @return list of hardware this MainModule manages.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Returns the list of CANopen hardware managed by this subsystem.")
    public List<String> listHardwareNames() {
        return this.bridge.listHardwareNames();
    }

    /**
     * For Whole FCS GUI. Has to be overridden in FcsMainModule.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the list of LOADER CANopen hardware managed by this subsystem.")
    public List<String> listLoaderHardwareNames() {
        return Collections.emptyList();
    }

    /**
     * For whole FCS GUI or autochanger GUI.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the list of names of sensors plugged on autochanger pluto gateway.")
    public List<String> listAcSensorsNames() {
        return bridge.listAcSensorsNames();
    }

    /**
     * For loader GUI.
     * For whole FCS GUI, it is overridden in FcsMain.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Return the list of names of sensors plugged on Loader pluto gateway.")
    public List<String> listLoSensorsNames() {
        return bridge.listLoSensorsNames();
    }

    /**
     * Return a list of filter names that the subsystem manages. For the GUI or the
     * Console.
     *
     * @return
     */
    @Command(type = Command.CommandType.QUERY, description = "Return the list of names of filters  "
            + "that this subsystem manages.", level = Command.NORMAL, alias = "listFilterNames")
    public List<String> getFilterNames() {
        return (filterManager == null) ? Collections.emptyList() : filterManager.getFilterNames();
    }

    @Command(type = Command.CommandType.QUERY, description = "Return a map : couple of name of MobileItem "
            + "and name of controller.", level = Command.NORMAL)
    public Map<String, String> getMobilNameControllerNameMap() {
        Map<String, String> ctlNames = new HashMap<String, String>();
        hardMovedByController.keySet().stream().forEach((hardName) -> {
            ctlNames.put(hardName, hardMovedByController.get(hardName).getControllerName());
        });
        return ctlNames;
    }


    public Map<Integer, String> getSensorsMapPluggedOnPlutoGateway() {
        Map<Integer, String> sensorNames = new HashMap<Integer, String>();
        return sensorNames;
    }

    /**
     * Update state in reading sensors. This method has to be overridden by
     * subclasses.
     *
     * @throws FcsHardwareException
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Update state in reading sensors.")
    public abstract void updateStateWithSensors();

    /**
     * Read sensors and update state in reading sensors to update GUI.
     */
    public void updateGui() {
        if (isInState(CommandState.READY) && isInState(FilterReadinessState.READY)) {
            try {
                updateStateWithSensors();
            } catch (Exception ex) {
                this.raiseWarning(UPDATE_ERROR, "error in reading sensors can't update GUI", ex);
            }
        }
    }

    /**
     * Send command checkFault to each controllers.
     * So if there was an issue detected on a controller, it will be known immediately (every 10 minutes should be enough).
     *
     */
    public void checkControllers() {
        if (isInState(FilterReadinessState.READY)) {
            controllerList.forEach((hardName, controller) -> {
                controller.checkFault();
            });
        }
    }

    /**
     * Send command checkFault to each controller.
     * Used within Controllers viewer Panel
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL,
        description = "Check fault on each controller, after hardware initialization. Launched from Controllers overview GUI.")
    public void checkFaultForAllControllers() {
        if (isInState(FilterReadinessState.READY)) {
            controllerList.forEach((hardName, controller) -> {
                if (controller.isBooted()) { // useful if an item is not online (like the Loader)
                    controller.checkFault();
                } else {
                    controller.publishData(); // Todo necessary if a controller should be booted
                }
            });
        }
    }

    /**
     * initialize hardware after initialization. to be executed if during boot
     * process some hardware is missing. this method had to be overridden in
     * subclasses.
     */
    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE,
            description = "Check hardware after initialization. To be executed if during boot process some hardware is missing.")
    public abstract void initializeHardware();

    public void updateAgentState(FilterState filterS) {
        // agentStateService is subs.getAgentService(AgentStateService.class)
        agentStateService.updateAgentState(filterS);
    }

    public void updateAgentState(FilterReadinessState readinessS) {
        // agentStateService is subs.getAgentService(AgentStateService.class)
        agentStateService.updateAgentState(readinessS);
    }

    public void updateAgentState(AutochangerInclination inclinationS) {
        // agentStateService is subs.getAgentService(AgentStateService.class)
        agentStateService.updateAgentState(inclinationS);
    }

    public void updateAgentState(McmState mcmS) {
        // agentStateService is subs.getAgentService(AgentStateService.class)
        agentStateService.updateAgentState(mcmS);
    }

    @SuppressWarnings("rawtypes")
    public boolean isInState(Enum state) {
        return agentStateService.isInState(state);
    }

    public FilterReadinessState getFilterReadinessState() {
        return (FilterReadinessState) agentStateService.getState(FilterReadinessState.class);
    }

    /**
     * Updates the FCS state and FCS readiness state and publishes on the status bus.
     * Checks that hardware is ready to be operated and moved.
     * This means that :
     * - all CAN open devices are booted, identified and initialized,
     * - homing has been done on the controllers.
     * This updates the FCS state and FCS readiness state and publishes on the status bus.
     * This has to be overridden if there is other initializations to do on the hardware like homing.
     *
     */
    @Command(type = Command.CommandType.ACTION, level = Command.ENGINEERING_EXPERT, description = "Force FCS state and FCS readiness state to be ready and publishes on the status bus. TO BE USED ONLY IN EXTREME CASES.")
    public void updateFCSStateToReady() {
        if (this.allDevicesBooted() && !isInState(ALARM)) {
            /* The initialization has been done, so now the hardware is ready */
            updateAgentState(FilterState.READY);
            updateAgentState(FilterReadinessState.READY);
        }
    }

    @Override
    public TreeWalkerDiag signal(Signal signal) {
        switch (signal.getLevel()) {
            case HALT:
                FCSLOG.finer("HALT required");
                this.haltRequired.set(true);
                break;
            case STOP:
                FCSLOG.finer("STOP required");
                this.stopRequired.set(true);
                break;
            default:
                assert false;
        }
        // we want the signal to go to all my children.
        return TreeWalkerDiag.GO;
    }

    /**
     * for GUI
     */
    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Publish Data to populate GUI.")
    public void publishData() {
        this.bridge.publishData();
        controllerList.forEach((hardName, controller) -> {
            controller.publishData();
        });
    }

    @Command(type = Command.CommandType.QUERY, level = Command.ENGINEERING_ROUTINE, description = "Change logging level of timed actions, default is FINE (other values are SEVERE, WARNING, INFO)")
    public void setTimedActionLoggingLevel(String levelName) {
        try {
            Level.parse(levelName); // checks if levelName is a valid log level
            setDefaultTimedLoggerLevel(levelName);
        } catch (IllegalArgumentException ex) {
            throw new IllegalArgumentException("Invalid log level: " + levelName + " (valid values are: SEVERE, WARNING, INFO, FINE, FINEST)", ex);
        }
    }

    @Command(type = Command.CommandType.QUERY, level = Command.NORMAL, description = "Get logging level of timed actions")
    public String getTimedActionLoggingLevel() {
        return FcsUtils.getDefaultTimedLoggerLevel();
    }

    @Override
    public void postShutdown() {
        FCSLOG.info(()-> name + " is shutting down.");
        bridge.doShutdown();
        FCSLOG.info(()-> name + " is shutdown.");
    }

}
